#!/usr/bin/env python
from ptrace import PtraceError
from ptrace.debugger import (PtraceDebugger, Application,
    ProcessExit, ProcessSignal, NewProcessEvent)
from ptrace.syscall import SYSCALL_NAMES, SYSCALL_PROTOTYPES, FILENAME_ARGUMENTS, PtraceSyscall
from ptrace.func_call import FunctionCallOptions
from sys import stderr, exit
from optparse import OptionParser
from logging import getLogger, debug, error
from ptrace.syscall.socketcall import SOCKETCALL
from ptrace.compatibility import any
from ptrace.error import PTRACE_ERRORS, writeError
from signal import SIGTRAP

class SyscallTracer(Application):
    def __init__(self):
        Application.__init__(self)

        # Parse self.options
        self.parseOptions()

        # Setup output (log)
        self.setupLog()

    def setupLog(self):
        if self.options.output:
            fd = open(self.options.output, 'w')
        else:
            fd = stderr
        self._setupLog(fd)

    def parseOptions(self):
        parser = OptionParser(usage="%prog [options] -- program [arg1 arg2 ...]")
        self.createCommonOptions(parser)
        parser.add_option("--type", help="Display arguments type and result type (default: no)",
            action="store_true", default=False)
        parser.add_option("--name", help="Display argument name (default: no)",
            action="store_true", default=False)
        parser.add_option("--string-length", "-s", help="String max length (default: 300)",
            type="int", default=300)
        parser.add_option("--array-count", help="Maximum number of array items (default: 20)",
            type="int", default=20)
        parser.add_option("--raw-socketcall", help="Raw socketcall form",
            action="store_true", default=False)
        parser.add_option("--output", "-o", help="Write output to specified log file",
            type="str")
        parser.add_option("--address", help="Display structure addressl",
            action="store_true", default=False)
        parser.add_option("--syscalls", help="Comma separated list of shown system calls (other will be skipped)",
            type="str", default=None)
        parser.add_option("--socketcall", help="Show only socketcall (network) functions",
            action="store_true", default=False)
        parser.add_option("--filename", help="Show only syscall using filename",
            action="store_true", default=False)
        parser.add_option("--show-pid", help="Prefix line with process identifier",
            action="store_true", default=False)
        parser.add_option("--list-syscalls", help="Display system calls and exit",
            action="store_true", default=False)

        self.createLogOptions(parser)

        self.options, self.program = parser.parse_args()

        if self.options.list_syscalls:
            syscalls = SYSCALL_NAMES.items()
            syscalls.sort(key=lambda data: data[0])
            for num, name in syscalls:
                print "% 3s: %s" % (num, name)
            exit(0)

        if self.options.pid is None and not self.program:
            parser.print_help()
            exit(1)

        # Create "only" filter
        only = set()
        if self.options.syscalls:
            # split by "," and remove spaces
            for item in self.options.syscalls.split(","):
                item = item.strip()
                if not item or item in only:
                    continue
                ok = True
                valid_names = SYSCALL_NAMES.values()
                for name in only:
                    if name not in valid_names:
                        print >>stderr, "ERROR: unknow syscall %r" % name
                        ok = False
                if not ok:
                    print >>stderr
                    print >>stderr, "Use --list-syscalls options to get system calls list"
                    exit(1)
                # remove duplicates
                only.add(item)
        if self.options.filename:
            for syscall, format in SYSCALL_PROTOTYPES.iteritems():
                restype, arguments = format
                if any(argname in FILENAME_ARGUMENTS for argtype, argname in arguments):
                    only.add(syscall)
        if self.options.socketcall:
            self.options.raw_socketcall = False
            for syscall, prototype in SOCKETCALL.itervalues():
                only.add(syscall)
        self.only = only

        if self.options.fork:
            self.options.show_pid = True

        self.processOptions()

    def displaySyscall(self, process, syscall, is_exit=True):
        if (not self.only) or (syscall.name in self.only):
            pass
        else:
            return

        text = syscall.format()
        if is_exit:
            exit_text = syscall.formatExit(process)
            text = "%-40s = %s" % (text, exit_text)

        if self.options.show_pid:
            error("[%s] %s" % (process.pid, text))
        else:
            error(text)

    def syscallTrace(self, process):
        # First query to break at next syscall
        process.syscall()

        while True:
            # No more process? Exit
            if not self.debugger:
                break

            # Wait until next syscall enter
            try:
                event = self.debugger.waitSignals(SIGTRAP)
                process = event.process
            except ProcessExit, event:
                self.processExited(event)
                continue
            except ProcessSignal, event:
                event.display()
                process.syscall(event.signum)
                continue
            except NewProcessEvent, event:
                self.newProcess(event)
                continue

            # Process syscall enter or exit
            self.syscall(process)

    def syscall(self, process):
        if process in self.active_syscalls:
            # syscall exit
            syscall = self.active_syscalls[process]
            del self.active_syscalls[process]
            debug("%s exits %s" % (process, syscall))
            self.displaySyscall(process, syscall)
        else:
            # syscall enter
            syscall = PtraceSyscall(process, self.syscall_options)
            debug("%s enters %s" % (process, syscall))
            self.active_syscalls[process] = syscall

        # Break at next syscall
        process.syscall()

    def processExited(self, event):
        # Display partial syscall
        process = event.process
        if process in self.active_syscalls:
            syscall = self.active_syscalls[process]
            self.displaySyscall(process, syscall, False)

        # Display exit message
        error("*** %s ***" % event)

    def newProcess(self, event):
        process = event.process
        error("*** New process %s ***" % process.pid)
        process.syscall()
        process.parent.syscall()

    def _main(self):
        # Create debugger and traced process
        self.setupDebugger()
        process = self.createProcess()
        if not process:
            return

        self.syscall_options = FunctionCallOptions(
            write_types=self.options.type,
            write_argname=self.options.name,
            string_max_length=self.options.string_length,
            replace_socketcall=not self.options.raw_socketcall,
            write_address=self.options.address,
            max_array_count=self.options.array_count,
        )
        self.active_syscalls = {}

        self.syscallTrace(process)

    def main(self):
        self.debugger = PtraceDebugger()
        try:
            self._main()
        except ProcessExit, event:
            self.processExited(event)
        except PtraceError, err:
            error("ptrace() error: %s" % err)
        except KeyboardInterrupt:
            error("Interrupted! (CTRL+c)")
        except PTRACE_ERRORS, err:
            writeError(getLogger(), err, "Debugger error")
        self.debugger.quit(self.options.pid is None)

    def createChild(self, program):
        pid = Application.createChild(self, program)
        error("execve(%s, %s, [/* 40 vars */]) = %s" % (
            program[0], program, pid))
        return pid

if __name__ == "__main__":
    SyscallTracer().main()

