# $Id$
#
# input.py -- input editing for PLWM
#
#    Copyright (C) 2001  Mike Meyer <mwm@mired.org>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


"input - a tool for getting input from the user."

from Xlib import X, Xatom
from keys import KeyGrabKeyboard, allmap
from wmanager import Window

class InputKeyHandler(KeyGrabKeyboard):
    """Template for handling user input.

    InputKeyHandler defines the following event handler methods:

    _insert - insert the character typed. The latin1 character set is
    	bound to this by default.

    _forw, _back - move the cursor forward or backward in the input

    _delforw, _delback - delete the character forward or backward in the input

    _end, _begin - move the cursor to the end or beginning of the input

    _deltoend - remove all characters to the end of the input

    _paste - paste the current selection into the intput. The handler
             must be an instance of wmanager.Window for this to work

    _done - run action on the current string

    _abort - exit without doing anything

    _history_up - scroll to newer events

    _history_down - scroll to older events"""

    timeout = None

    def __init__(my, handler, display, history):
        """Init with a handler and display.

        handler is a handler object appropriate for KeyGrabKeyboard.
        display has show(left, right), do(string) and abort() methods.
            both abort() and do() should clean up the object.
        history is a list of strings we let the user scroll through."""

        KeyGrabKeyboard.__init__(my, handler, X.CurrentTime)
        my.display = display
        my.handler = handler
        my.history = history
        my.history_index = len(history)
        my.left = ""
        my.right = ""
        if isinstance(handler, Window):
            my.selection = my.wm.display.intern_atom("SELECTION")
            my.wm.dispatch.add_handler(X.SelectionNotify,
                                       my._paste_selection, handlerid = my)
        display.show(my.left, my.right)

    def _paste_selection(my, event):
        if event.property:
            sel = my.handler.window.get_full_property(my.selection, Xatom.STRING)
            if sel and sel.format == 8:
                my.left = my.left + sel.value
                my.display.show(my.left, my.right)

    def _paste(my, event):
        if isinstance(my.handler, Window):
            my.handler.window.convert_selection(Xatom.PRIMARY, Xatom.STRING,
                                                my.selection, X.CurrentTime)

    def _insert(my, event):
        if event.type != X.KeyPress: return
        sym = my.wm.display.keycode_to_keysym(event.detail,
                                                event.state & X.ShiftMask != 0)
        chr = my.wm.display.lookup_string(sym)
        if chr: my.left = my.left + chr
        my.display.show(my.left, my.right)

    def _forw(my, event):
        if my.right:
            my.left = my.left + my.right[0]
            my.right = my.right[1:]
        my.display.show(my.left, my.right)

    def _back(my, event):
        if my.left:
            my.right = my.left[-1] + my.right
            my.left = my.left[:-1]
        my.display.show(my.left, my.right)

    def _delforw(my, event):
        if my.right:
            my.right = my.right[1:]
        my.display.show(my.left, my.right)

    def _delback(my, event):
        if my.left:
            my.left = my.left[:-1]
        my.display.show(my.left, my.right)

    def _deltoend(my, event):
        my.right = ""
        my.display.show(my.left, my.right)

    def _end(my, event):
        my.left = my.left + my.right
        my.right = ""
        my.display.show(my.left, my.right)

    def _begin(my, event):
        my.right = my.left + my.right
        my.left = ""
        my.display.show(my.left, my.right)

    def _done(my, event):
        res = my.left + my.right
        my.history.append(res)
        my.display.do(res)
        my.wm.dispatch.remove_handler(my)
        my._cleanup()

    def _abort(my, event):
        my.display.abort()
        my.wm.dispatch.remove_handler(my)
        my._cleanup()

    def _history_up(my, event):
        if len(my.history):
            if my.history_index > 0:
                my.history_index -= 1
                my.left = my.history[my.history_index]
                my.right = ""
                my.display.show(my.left, my.right)

    def _history_down(my, event):
        if len(my.history):
            if my.history_index <(len(my.history)-1):
                my.history_index += 1
                my.left = my.history[my.history_index]
                my.right = ""
                my.display.show(my.left, my.right)

allmap(InputKeyHandler, InputKeyHandler._insert)

class inputWindow:
    "Class to get a line of user input in a window."

    fontname= "9x15"
    foreground = "black"
    background = "white"
    borderwidth = 3
    bordercolor = "black"
    history = []

    def __init__(my, prompt, screen, length=30):

        if not prompt: prompt = ' '	# We have problems if there's no prompt, so add one.
        my.string = my.prompt = prompt
        my.offset = len(my.prompt)
        my.length = length + my.offset
        my.start = 0
        fg = screen.get_color(my.foreground)
        bg = screen.get_color(my.background)
        bc = screen.get_color(my.bordercolor)
        font = screen.wm.get_font(my.fontname, 'fixed')
        size = font.query()
        my.height = size.font_ascent + size.font_descent + 1
        my.width = font.query_text_extents(prompt).overall_width + \
                   font.query_text_extents(length * 'm').overall_width
        my.baseline = size.font_ascent + 1

        window = screen.root.create_window(0, 0, my.width, my.height,
                                           my.borderwidth,
                                           X.CopyFromParent, X.InputOutput,
                                           X.CopyFromParent,
                                           background_pixel = bg,
                                           border_pixel = bc,
                                           event_mask = (X.VisibilityChangeMask |
                                                         X.ExposureMask))

        my.gc = window.create_gc(font = font, function = X.GXinvert,
                                 foreground = fg, background = bg)

        my.font = font
        my.window = screen.add_internal_window(window)
        my.window.dispatch.add_handler(X.VisibilityNotify, my.raisewindow)
        my.window.dispatch.add_handler(X.Expose, my.redraw)

    def read(my, action, handlertype, x=0, y=0):
        "Open the window at x, y, using handlertype, and doing action."

        my.action = action
        x, y, width, height = my.window.keep_on_screen(x, y, my.width, my.height)
        my.window.configure(x = x, y = y, width = width, height = height)
        my.window.map()
        my.window.get_focus(X.CurrentTime)
        handlertype(my.window, my, my.history)

    def raisewindow(my, event):
        my.window.raisewindow()

    def redraw(my, event = None):
        length = len(my.string)

        if my.offset < length:
            wide = my.font.query_text_extents(my.string[my.offset]).overall_width
        else:
            wide = my.font.query_text_extents(' ').overall_width

        if my.start >= my.offset: my.start = my.offset - 1
        left = my.font.query_text_extents(my.string[my.start:my.offset]).overall_width

        if left + wide >= my.width:
            my.start = my.offset - my.length + 1
            left = my.font.query_text_extents(my.string[my.start:my.offset]).overall_width
        
        my.window.clear_area(width = my.width, height = my.height)
        my.window.image_text(my.gc, 0, my.baseline, my.string[my.start:])
        my.window.fill_rectangle(my.gc, left, 0, wide, my.height)


    def show(my, left, right):
        if left:
            my.string = my.prompt + left
        else:	# Display the prompt in this case.
            my.string = my.prompt
            my.start = 0
        my.offset = len(my.string)
        my.string = my.string + right
        my.redraw()

    def do(my, string):
        my.action(string)
        my.window.destroy()

    def abort(my):
        my.window.destroy()


class modeInput:
    "Class to get input via the modewindow."

    history = []

    def __init__(my, prompt, screen, length = None):
	# ignore length argument
        my.prompt = prompt
        my.screen = screen

    def read(my, action, handlertype, x = 0, y = 0):
        my.action = action
        my.status_msg = my.screen.modestatus_new(my.prompt + "_")
        handlertype(my.screen.modewindow_mw.window, my, my.history)

    def show(my, left, right):
        my.status_msg.set("%s%s_%s" % (my.prompt, left, right))

    def do(my, string):
        my.action(string)
        my.status_msg.pop()

    def abort(my):
        my.status_msg.pop()

        
