# Copyright (c) 2006 Bea Lam. All rights reserved.
# 
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge,
# publish, distribute, sublicense, and/or sell copies of the Software,
# and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
# CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
# TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
# SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

# Mac OS X obex module implementation.
#
# There are several undisclosed features in this implementation, which are not
# made public because they have not been implemented on other platforms, and
# because they might change anyway. But they might prove useful.
# So, you can set USE_HIDDEN_FEATURES to True to get these features:
#   - recvfile() can take None instead of a file name/object -- the received
#     file will be written using whatever filename the client provided 
#     (existing files will be overwritten!)
#   - recvfile() can return a (filename, address) pair, where address is an 
#     (address, channel) pair showing the remote client's address details (i.e.
#     who sent you the file) and filename is the name specified by the client
#     (useful if you passed None as the file name/object and don't know what
#     filename it's saved as)


import os
import socket as _socket
import warnings
import types

import objc
import Foundation

import _BTUtil
import _macutil
import _IOBluetooth
import _lightbluecommon


# public attributes
__all__ = ("sendfile", "recvfile")


# defined in <IOBluetooth/OBEX.h>
_kOBEXSuccess = 0
_kOBEXSessionNoTransportError = -21879
_kOBEXSessionTransportDiedError = -21880
_kOBEXSessionTimeoutError = -21881
_kOBEXResponseCodeNotImplementedWithFinalBit = 0xD1
_kOBEXResponseCodeForbiddenWithFinalBit = 0xC3

# want lots of OBEX debug messages?
DEBUG = False

# turn on mac-only hidden implementation features?
USE_HIDDEN_FEATURES = False


def sendfile(address, channel, source):
    if not _lightbluecommon._isbtaddr(address):  
        raise TypeError("address '%s' is not a valid bluetooth address" \
            % address)
    if not isinstance(channel, int):
        raise TypeError("channel must be int, was %s" % type(channel))
    if not isinstance(source, (types.StringTypes, types.FileType)):
        raise TypeError("source must be string or built-in file object")
                
    if isinstance(source, types.StringTypes) and not os.path.isfile(source):
        raise IOError("File '%s' not found" % source)
        
    dev = _IOBluetooth.IOBluetoothDevice.withAddress_(
                _macutil.btaddrtochars(address))
    try:
        sender = _OBEXFileSender(address, channel, source) 
        sender.run()
    finally:    
        dev.closeConnection()
        
def recvfile(sock, dest):
    if sock is None:
        raise TypeError("Given socket is None")
    if not USE_HIDDEN_FEATURES:
        if not isinstance(dest, (types.StringTypes, types.FileType)):
            raise TypeError("dest must be string or built-in file object")    

    channelID = sock._getport()
    receiver = _OBEXFileReceiver(channelID, dest)
    recvinfo = receiver.run()
    
    if USE_HIDDEN_FEATURES:    
        return recvinfo
    else:
        return None

class _OBEXFileSender(object):
    """
    A convenience class for synchronously sending a file using OBEX. When 
    run() is called, it connects to the remote service, sends a file, then 
    disconnects.
    """
    def __init__(self, address, channel, source, timeout=10.0,
            connecttimeout=30.0):
        """
        Arguments:
            - address - address of the device to send the file to
            - channel - channel of the service to send the file to
            - source - filename of file to be sent, or a file object
            - timeout - timeout for awaiting server responses
            - connecttimeout - timeout for initial connection (might want this
              to be longer than general timeout, just for establishing conn.)
        """        
        super(_OBEXFileSender, self).__init__()
        self.address = address
        self.channel = channel
        self.source = source       
        self.timeout = timeout
        self.connecttimeout = connecttimeout

        self.client = _OBEXClient.alloc().init()
        self.client.connected = self._finishedop
        self.client.putcompleted = self._finishedop  
        self.client.disconnected = self._finishedop
        
        self._lasterrmsg = ""
        self._opresult = None
        
    def _finishedop(self, err, errmsg):
        self._opresult = err
        self._lasterrmsg = errmsg        
    
    def run(self):
        self.connect()
        try:
            self.sendfile()
            self.disconnect()
        finally:
            self.close()
    
    def connect(self):
        self._opresult = None
        self.client.connect(self.address, self.channel)
        
        # wait for connect to complete                
        ok = _macutil.waitwhile(lambda: self._opresult is None, 
                self.connecttimeout)
        if self._opresult != _kOBEXSuccess:
            raise _lightbluecommon.BluetoothError(
                self._opresult, self._lasterrmsg)
        if not ok:
            raise _lightbluecommon.BluetoothError("connect timed out")
            
    def sendfile(self):
        self._opresult = None
        try:
            if isinstance(self.source, types.StringTypes):
                self.client.putfile(self.source)
            else:
                # send file object
                self.filehandle = \
                    Foundation.NSFileHandle.alloc().initWithFileDescriptor_(
                        self.source.fileno())
                self.client.putfilehandle(self.filehandle, self.source.name, 
                                          None, -1)
        except _lightbluecommon.BluetoothError, e:
            try:
                self.disconnect()
            finally:   
                # raise the original exception, regardless of whether errors
                # occured during disconnect()
                raise e
        
        # wait for send to complete
        ok = _macutil.waitwhile(lambda: self._opresult is None, self.timeout)
        
        # check result
        if not ok or self._opresult != _kOBEXSuccess:
            if not ok:
                error = _lightbluecommon.BluetoothError("file send timed out")
            else:
                error = _lightbluecommon.BluetoothError(
                    self._opresult, self._lasterrmsg)
            try:
                self.disconnect()
            finally:
                # raise the original exception, regardless of whether errors
                # occured during disconnect()        
                raise error
    
    def disconnect(self):
        self._opresult = None
        self.client.disconnect()
        
        # wait for disconnect to complete        
        ok = _macutil.waitwhile(lambda: self._opresult is None, self.timeout)
        
        # other device may disconnect after receiving file, so ignore
        # kOBEXSessionTransportDiedError        
        if self._opresult != _kOBEXSuccess and \
                self._opresult != _kOBEXSessionTransportDiedError:
            raise _lightbluecommon.BluetoothError(
                self._opresult, self._lasterrmsg)
        if not ok:
            raise _lightbluecommon.BluetoothError("disconnect timed out")
            
    def close(self):
        self.client.close()            
        
                                      
class _OBEXClient(Foundation.NSObject):
    """
    Runs an asynchronous OBEX client. The following attributes can be 
    set for callbacks:
    
        - connected: called when a connection operation is completed
          (takes err, errmsg arguments)
        - disconnected: called when a disconnection operation is 
          completed (takes err, errmsg arguments)
        - putcompleted: called when a PUT operation is completed successfully
          (takes err, errmsg arguments)
          
    If an error occurs on the initial attempt of an operation (not on the 
    callback), _lightbluecommon.BluetoothError is raised.
    """
    
    # NSObject init, not python __init__
    def init(self):
        self = super(_OBEXClient, self).init()
        self._client = _BTUtil.BBOBEXClientSession.alloc().initWithDelegate_(self)
        
        global DEBUG
        _BTUtil.BBOBEXClientSession.setDebug_(DEBUG)
        
        self.connected = None
        self.disconnected = None
        self.putcompleted = None
        
        self.address = None
        
        return self
        
    def connect(self, addr, channel):
        """
        Connects to the OBEX service on the given address and channel.
        """
        err = self._client.connectToDeviceWithAddress_withChannelID_toTarget_(
            addr, channel, None)
        if err != _kOBEXSuccess:
            raise _lightbluecommon.BluetoothError(err, 
                    self._errdesc("Could not attempt OBEX connection"))
        self.address = addr
        
    def putfile(self, filename):
        err = self._client.putFileAtPath_(filename)
        if err != _kOBEXSuccess:
            raise _lightbluecommon.BluetoothError(err, 
                    self._errdesc("Could not attempt sending of file"))   

    def putfilehandle(self, handle, name, filetype, length):
        err = self._client.putFile_withName_type_length_(handle, name, 
                                                         filetype, length)
        if err != _kOBEXSuccess:
            raise _lightbluecommon.BluetoothError(err, 
                    self._errdesc("Could not attempt sending of file"))                      
                    
    def disconnect(self):
        err = self._client.disconnect()
        
        # other device may disconnect after receiving file, so ignore
        # kOBEXSessionTransportDiedError
        if err != _kOBEXSuccess and err != _kOBEXSessionTransportDiedError:
            raise _lightbluecommon.BluetoothError(err, 
                    self._errdesc("Could not attempt OBEX disconnection"))   
                    
    def close(self):
        if self._client.hasOpenTransportConnection():        
            # ignore returned error code
            self._client.close()
            
    # adds additional error message detail from internal OBEXClient
    def _errdesc(self, addedmsg):
        desc = self._client.lastErrorDescription()
        return "%s: %s" % (addedmsg, desc)
        
    def _docallback(self, callback, err, msg):
        if callback is not None:
            if err != _kOBEXSuccess:
                msg = self._errdesc(msg)
            callback(err, msg)
             
    #
    # Delegate methods follow. objc signatures for all methods must be set 
    # (using objc.selector) otherwise the OBEXError arg won't be received 
    # correctly (gives a segmentation fault) if it's anything other than 
    # kOBEXSuccess.
    #
    
    # - (void)obexClientSessionConnectComplete:(BBOBEXClientSession *)session
    #                                    error:(OBEXError)error
    #                                  fromWho:(NSData *)who
    #                             connectionID:(long)connectionID;
    def obexClientSessionConnectComplete_error_fromWho_connectionID_(self, 
            session, err, who, connectionID):
        self._docallback(self.connected, err, 
            "Error performing OBEX connection")
    obexClientSessionConnectComplete_error_fromWho_connectionID_ = \
        objc.selector(
            obexClientSessionConnectComplete_error_fromWho_connectionID_, 
            signature="v@:@i@l") 

    # - (void)obexClientSessionDisconnectComplete:(BBOBEXClientSession *)sess
    #                                       error:(OBEXError)error;
    def obexClientSessionDisconnectComplete_error_(self, session, err):
        self._docallback(self.disconnected, err, 
            "Error performing OBEX disconnection")    
    obexClientSessionDisconnectComplete_error_ = objc.selector(
        obexClientSessionDisconnectComplete_error_, signature="v@:@i")
                
    # - (void)obexClientSessionPutComplete:(BBOBEXClientSession *)session 
    #                                error:(OBEXError)error;
    def obexClientSessionPutComplete_error_(self, client, err):
        self._docallback(self.putcompleted, err, 
            "Error performing file PUT operation")
    obexClientSessionPutComplete_error_ = objc.selector(
        obexClientSessionPutComplete_error_, signature="v@:@i")
        
        
# -------------------------------------

class _OBEXFileReceiver(object):
    """
    A convenience class for synchronously receiving a file using OBEX. When 
    run() is called, it starts an OBEX server, and after a client has connected 
    and then sent a file and disconnected, it stops the server. 
    
    If any error occurs while running the OBEX server, the server will be
    stopped. If a file was received successfully despite the error, a warning 
    will be printed, but if a file has not been received, _lightbluecommon.BluetoothError will be
    raised.
    """
    
    def __init__(self, channelID, dest):
        # doing some testing with passing None -- if you give dest as None,
        # the file will be saved as whatever filename the client specified
        if not isinstance(dest, (types.StringTypes, types.FileType,
                types.NoneType)):
            raise TypeError("dest must be string or file object, or None")
    
        super(_OBEXFileReceiver, self).__init__()
        if isinstance(dest, types.FileType):
            self._fileobj = dest
            self._filename = ""
        else:
            # for string or None, create the file later
            self._fileobj = None
            self._filename = dest
        self._openedfile = False
        self._gotfileOK = False
        self._expectedfilesize = -1
        self._recvdfilesize = 0
        
        self._remoteaddr = ""
        self._servererr = None  # (error-code, error-desc) tuple 
        self.running = False
        
        self.server = _OBEXServer.alloc().initWithChannel_(channelID)
        self.server.disconnected = self._disconnected
        self.server.putrequested = self._putrequested
        self.server.putcompleted = self._putcompleted 
        self.server.errored = self._error
        
    def run(self):
        self.server.start()
        
        try:
            # wait until we receive a file, or an error occurs
            self.running = True
            _macutil.waitwhile(lambda: self.running and not self._gotfileOK)
            
            # if still running (i.e. no errors) wait 10 seconds for client to 
            # disconnect
            if self.running:
                ok = _macutil.waitwhile(lambda: self.running, 10)
                if not ok:
                    self._servererr = (_kOBEXSessionTimeoutError, 
                            "Timed out waiting for client disconnection")
        finally:            
            # clean up and stop the server
            self._cleanup()
            try:
                self.server.stop()  
            except _lightbluecommon.BluetoothError:
                pass    # always get error on stopping server session?
            
        # Error occured while running server. Don't complain if already got a file.
        if self._servererr:
            if self._gotfileOK or (self._recvdfilesize == self._expectedfilesize):
                # object push services often kill transport after sending file 
                # without disconnecting first, so ignore transport died error
                if self._servererr[0] != _kOBEXSessionTransportDiedError:
                    msg = "[_OBEXFileReceiver] Ignoring error " + \
                        "after got file OK: " + self._servererr
                    warnings.warn(msg)
            else:
                raise _lightbluecommon.BluetoothError(
                    self._servererr[0], self._servererr[1])

        # return (filename, address) pair
        addr = _macutil.formatdevaddr(self._remoteaddr)
        return (self._filename, (addr, self.server.channelID))
        
    def _disconnected(self, session):
        # stop the server when the client on the other end disconnects
        self.running = False
        
    def _putrequested(self, session, filename, filetype, filesize):
        # if we've already received a file, don't accept any more
        if self._gotfileOK:
            return None
            
        if self._fileobj is None:
            if self._filename is None:
                self._fileobj = file(filename, "wb")
            else:
                self._fileobj = file(self._filename, "wb")
            self._openedfile = True
        
        self._filename = filename
        self._expectedfilesize = filesize
            
        # return NSFileHandle object
        self._filehandle = \
            Foundation.NSFileHandle.alloc().initWithFileDescriptor_(
                self._fileobj.fileno())
        return self._filehandle
        
    def _putcompleted(self, session):
        self._gotfileOK = True
        
        # note the remote device that sent the file
        remotedevice = session.getRFCOMMChannel().getDevice()
        self._remoteaddr = remotedevice.getAddressString()
                
    def _cleanup(self):
        if self._fileobj is not None:
            self._recvdfilesize = self._fileobj.tell()
        if self._openedfile:
            self._fileobj.close()
                
    def _error(self, session, err, desc):
        #self._servererr = "%s (%d)" % (desc, err)
        self._servererr = (err, desc)
        self.running = False    # if waiting for a file, will stop the waiting


class _OBEXServer(Foundation.NSObject):
    """
    Runs an asynchronous OBEX server. 
          
    If an error occurs during an operation, _lightbluecommon.BluetoothError is raised.
    """

    # NSObject init, not python __init__
    def initWithChannel_(self, channelID):   
        if not isinstance(channelID, int):
            raise TypeError("channel must be int, was %s" % type(channelID)) 
            
        self = super(_OBEXServer, self).init()
        self.channelID = channelID
        self._server = _BTUtil.BBOBEXServer.alloc().initWithDelegate_(self)
        global DEBUG
        _BTUtil.BBOBEXServer.setDebug_(DEBUG)
        
        self.connected = None
        self.disconnected = None
        self.putrequested = None 
        self.putcompleted = None
        self.abortrequesed = None
        self.abortcompleted = None
        self.errored = None
        
        return self    
        
    def start(self):
        err = self._server.startOnChannelID_(self.channelID)
        if err != _kOBEXSuccess:
            raise _lightbluecommon.BluetoothError(
                err, "Could not start OBEX server")

    def stop(self):
        err = self._server.stop()
        if err != _kOBEXSuccess:
            raise _lightbluecommon.BluetoothError(
                err, "Could not stop OBEX server")
        
    #
    # Delegate methods follow.
    #
    
    # - (void)obexServerSessionConnectComplete:(BBOBEXServerSession *)session;
    def obexServerSessionConnectComplete_(self, session):
        if self.connected:
            self.connected(session)
            
    # - (void)obexServerSessionDisconnectComplete:(BBOBEXServerSession *)session;
    def obexServerSessionDisconnectComplete_(self, session):
        if self.disconnected:
            self.disconnected(session)
            
    # - (int)obexServerSessionPutRequest:(BBOBEXServerSession *)session
    #                           withName:(NSString *)name
    #                               type:(NSString *)type
    #                             length:(unsigned)totalLength
    #                        writeToFile:(NSFileHandle **)destFile;
    def obexServerSessionPutRequest_withName_type_length_writeToFile_(
            self, session, name, filetype, length, destfile):
        if self.putrequested:
            filehandle = self.putrequested(session, name, filetype, length)
            if filehandle is None:
                return (_kOBEXResponseCodeForbiddenWithFinalBit, None)
            else:
                return (_kOBEXSuccess, filehandle)
        return (_kOBEXResponseCodeNotImplementedWithFinalBit, None)
    obexServerSessionPutRequest_withName_type_length_writeToFile_ = \
        objc.selector(
            obexServerSessionPutRequest_withName_type_length_writeToFile_, 
            signature="i@:@@@IN^@")
        
    # - (void)obexServerSessionPutComplete:(BBOBEXServerSession *)session
	#  						       forFile:(NSFileHandle *)destFile;
    def obexServerSessionPutComplete_forFile_(self, session, destfile):
        if self.putcompleted:
            self.putcompleted(session)
           
    # could handle ABORT operations too

    # - (void)obexServerSessionError:(BBOBEXServerSession *)session
    #                          error:(OBEXError)error 
    #                        summary:(NSString *)summary;
    def obexServerSessionError_error_summary_(self, session, err, summary):
        if self.errored:
            self.errored(session, err, summary)
    obexServerSessionError_error_summary_ = objc.selector(
        obexServerSessionError_error_summary_, signature="v@:@i@")	   
