/*
 * Copyright (c) 2006 Bea Lam. All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

//
//  BBOBEXServer.m
//  BTUtil
//

#import "BBOBEXUtil.h"
#import "BBOBEXServer.h"
#import "BBOBEXServerSession.h"

#define DEBUG_NAME @"[BBOBEXServer] "

static BOOL debug;


@implementation BBOBEXServer

+ (void)initialize
{
	debug = NO;
}


- (id)initWithDelegate:(id)delegate
{
	self = [super init];
	mDelegate = delegate;
	
	mChannelOpenNotif = nil;
	mServerSessions = [[NSMutableArray alloc] initWithCapacity:10];
	
	return self;
}

- (OBEXError)startOnChannelID:(BluetoothRFCOMMChannelID)channelID
{
	if (debug) NSLog(DEBUG_NAME @"Starting server on channel %d", channelID);	

	mServerChannelID = channelID;
	
	// start receiving channel-opened notifications
	mChannelOpenNotif = 
		[IOBluetoothRFCOMMChannel registerForChannelOpenNotifications:self 
															 selector:@selector(newRFCOMMChannelOpened:channel:) 
														withChannelID:mServerChannelID 
															direction:kIOBluetoothUserNotificationChannelDirectionIncoming];
	[mChannelOpenNotif retain];
	
	if (mChannelOpenNotif == nil) {
		if (debug) NSLog(DEBUG_NAME @"Error registering for channel-opened notifications");
		return kOBEXGeneralError;
	}	
	
	return kOBEXSuccess;
}

- (BOOL)isRunning
{
	return (mChannelOpenNotif != nil);
}

- (OBEXError)stop
{
	// stop receiving channel-opened notifications
	[mChannelOpenNotif unregister];
	[mChannelOpenNotif release];
	mChannelOpenNotif = nil;
	
	// close any open sessions
	OBEXError status = kOBEXSuccess;
	int i; 
	for (i=0; i<[mServerSessions count]; i++) {
		BBOBEXServerSession *session = [mServerSessions objectAtIndex:i];
		if ([session isRunning]) {
			OBEXError err = [session close];
			if (status == kOBEXSuccess)
				status = err;	// don't override any existing error?
		}
	}	
	
	// I think it's safe to release the sessions now?
	[mServerSessions release];
	mServerSessions = nil;
	
	return status;
}

- (void)setDelegate:(id)delegate
{
	mDelegate = delegate;
}

- (id)delegate
{
	return mDelegate;
}

+ (void)setDebug:(BOOL)doDebug
{
	debug = doDebug;
	[BBOBEXServerSession setDebug:doDebug];
}


#pragma mark -

- (void)newRFCOMMChannelOpened:(IOBluetoothUserNotification *)notif 
					   channel:(IOBluetoothRFCOMMChannel *)newChannel
{
	if (debug) NSLog(DEBUG_NAME @"New RFCOMM channel opened");	
	
	if (newChannel && [newChannel isIncoming] && [newChannel getChannelID] == mServerChannelID) {
		
		// Create new session 
		BBOBEXServerSession *session = [[[BBOBEXServerSession alloc] initWithServer:self] autorelease];
		
		// retain new session
		if (session) {
			[mServerSessions addObject:session];

			// start an OBEX session on the new client connection
			[session runWithIncomingRFCOMMChannel:newChannel];
		}
	}
	
	// see if any old session need to be released 
	// (not sure how else to know when it's safe to release old sessions)
	int i; 
	for (i=0; i<[mServerSessions count]; i++) {
		BBOBEXServerSession *session = [mServerSessions objectAtIndex:i];
		if (![session isRunning]) {
			[mServerSessions removeObjectAtIndex:i];
			i--;
		}
	}
}


#pragma mark -

- (void)dealloc
{
	if ([self isRunning]) 
		[self stop];
	
	[mChannelOpenNotif release];
	[mServerSessions release];
	
	[super dealloc];
}


@end