# Copyright 2005 Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


"""Format amounts into human-readable units.

This module contains the class UnitFormatter, which can be used to
convert an amount such as "128974848 bytes" into "123 MB", the latter
being much easier for a human go grasp. The class is nicely generic so
that it can be used for any kinds of units. Additionally, there are 
a couple of derived classes for common situations.

Lars Wirzenius <liw@iki.fi>
"""


import math


class UnitFormatter:

    """Format amounts into human-readable units."""
    
    def __init__(self, units=None):
        if units:
            self._units = units
        
    def format(self, amount, max=None):
        result = []
        for i in range(len(self._units)):
            (factor, name) = self._units[i]
            count = long(amount / factor)
            if count > 0:
                if max is None or len(result) + 1 < max:
                    result.append("%d %s" % (count, name))
                    amount -= count * factor
                else:
                    if math.modf(float(amount) / factor)[0] >= 0.5:
                        count += 1
                    result.append("%d %s" % (count, name))
                    amount = 0
                    break

        assert amount == 0
        if result:
            result = " ".join(result)
        else:
            result = "0 %s" % self._units[-1][1]
        return result.strip()


class CountFormatter(UnitFormatter):

    _units = [
        (1, "")
    ]


class ByteFormatter(UnitFormatter):

    _units = [
        (1024L**8, "yottabyte"),
        (1024L**7, "zettabyte"),
        (1024L**6, "exabyte"),
        (1024L**5, "petabyte"),
        (1024L**4, "terabyte"),
        (1024L**3, "GB"),
        (1024L**2, "MB"),
        (1024L, "kB"),
        (1L, "B")
    ]


class TimeFormatter(UnitFormatter):

    _units = [
        (365*24*60*60, "year"),
        (30*24*60*60, "month"),
        (7*24*60*60, "week"),
        (24*60*60, "d"),
        (60*60, "h"),
        (60, "min"),
        (1, "s")
    ]


if __name__ == "__main__":
    import unittest

    class UnitFormatterTests(unittest.TestCase):

        _units = [
            (1024**3, "GB"),
            (1024**2, "MB"),
            (1024, "kB"),
            (1, "B")
        ]

        def testSimple(self):
            uf = UnitFormatter(self._units)
            self.failUnlessEqual(uf.format(0), "0 B")
            self.failUnlessEqual(uf.format(1), "1 B")
            self.failUnlessEqual(uf.format(1023), "1023 B")
            self.failUnlessEqual(uf.format(1024), "1 kB")
            self.failUnlessEqual(uf.format(1025), "1 kB 1 B")

    class ByteFormatterTests(unittest.TestCase):

        def testSimple(self):
            uf = ByteFormatter()
            self.failUnlessEqual(uf.format(0), "0 B")
            self.failUnlessEqual(uf.format(1), "1 B")
            self.failUnlessEqual(uf.format(1023), "1023 B")
            self.failUnlessEqual(uf.format(1024), "1 kB")
            self.failUnlessEqual(uf.format(1025), "1 kB 1 B")
            self.failUnlessEqual(uf.format(4096), "4 kB")

        def testRounded(self):
            uf = ByteFormatter()
            self.failUnlessEqual(uf.format(0, max=1), "0 B")
            self.failUnlessEqual(uf.format(1, max=1), "1 B")
            self.failUnlessEqual(uf.format(1023, max=1), "1023 B")
            self.failUnlessEqual(uf.format(1024, max=1), "1 kB")
            self.failUnlessEqual(uf.format(1025, max=1), "1 kB")
            self.failUnlessEqual(uf.format(2047, max=1), "2 kB")
            self.failUnlessEqual(uf.format(4096, max=1), "4 kB")
            self.failUnlessEqual(uf.format(4097, max=1), "4 kB")

    unittest.main()
