/* -*- mode: C -*-  */
/*
   IGraph library.
   Copyright (C) 2006-2012  Gabor Csardi <csardi.gabor@gmail.com>
   334 Harvard street, Cambridge, MA 02139 USA

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc.,  51 Franklin Street, Fifth Floor, Boston, MA
   02110-1301 USA

*/

#include <igraph.h>

#include <math.h>

#include "test_utilities.inc"

/* Replace modularity values which are very close to zero
 * by exact zeros, so that we can have consistent test outputs
 * across platforms. */
void fixup_modularity(igraph_vector_t *modularity) {
    igraph_integer_t i;
    igraph_integer_t len = igraph_vector_size(modularity);

    for (i=0; i < len; ++i) {
        if (fabs(VECTOR(*modularity)[i]) < 1e-15) {
            VECTOR(*modularity)[i] = 0.0;
        }
    }
}

int main() {
  igraph_t graph;

  igraph_matrix_t merges;
  igraph_vector_t modularity;
  igraph_vector_t membership;
  igraph_vector_t weights;

  /* Set default seed to get reproducible results */
  igraph_rng_seed(igraph_rng_default(), 42);

  printf("Basic test\n\n");

  /* Simple unweighted graph */
  igraph_small(&graph, 3, IGRAPH_UNDIRECTED,
      0,1, 1,2, 2,0, -1);

  igraph_matrix_init(&merges, 0, 0);
  igraph_vector_init(&modularity, 0);
  igraph_vector_init(&membership, 0);

  igraph_community_walktrap(&graph, NULL, 4, &merges, &modularity, &membership);
  printf("Merges:\n");
  print_matrix(&merges);

  printf("Modularity: ");
  fixup_modularity(&modularity);
  print_vector(&modularity);

  printf("Membership: ");
  print_vector(&membership);

  igraph_vector_destroy(&membership);
  igraph_vector_destroy(&modularity);
  igraph_matrix_destroy(&merges);

  /* Test the case when modularity=NULL and membership=NULL as this caused a crash in
   * the R interface, see https://github.com/igraph/rigraph/issues/289 */

  printf("\nWithout modularity and membership calculation\n\n");

  igraph_matrix_init(&merges, 0, 0);

  igraph_community_walktrap(&graph, NULL, 4, &merges, NULL, NULL);
  printf("Merges:\n");
  print_matrix(&merges);
  igraph_matrix_destroy(&merges);

  /* Test the case when merges=NULL but modularity is requested,
   * as the result was previously invalid due to not incrementing
   * the "merge index" during computation. */

  printf("\nWithout merges matrix calculation\n\n");

  igraph_vector_init(&modularity, 0);
  igraph_community_walktrap(&graph, NULL, 4, NULL, &modularity, NULL);
  printf("Modularity:\n");
  fixup_modularity(&modularity);
  print_vector(&modularity);
  igraph_vector_destroy(&modularity);

  igraph_destroy(&graph);

  /* Test for bug https://github.com/igraph/igraph/issues */

  printf("\nBug 2042\n\n");

  igraph_small(&graph, 3, IGRAPH_UNDIRECTED,
               0,1, -1);

  igraph_matrix_init(&merges, 0, 0);
  igraph_vector_init(&modularity, 0);
  igraph_vector_init(&membership, 0);
  igraph_vector_init_real(&weights, 1, 0.2);

  igraph_community_walktrap(&graph, &weights, 4, &merges, &modularity, &membership);
  printf("Merges:\n");
  print_matrix(&merges);

  printf("Modularity: ");
  fixup_modularity(&modularity);
  print_vector(&modularity);

  printf("Membership: ");
  print_vector(&membership);

  igraph_vector_destroy(&weights);
  igraph_vector_destroy(&membership);
  igraph_vector_destroy(&modularity);
  igraph_matrix_destroy(&merges);

  igraph_destroy(&graph);

  printf("\nSmall weighted graph\n\n");

  igraph_ring(&graph, 6, IGRAPH_UNDIRECTED, 0, 1);

  igraph_matrix_init(&merges, 0, 0);
  igraph_vector_init(&modularity, 0);
  igraph_vector_init(&membership, 0);
  igraph_vector_init_real(&weights, 6,
                          1.0, 0.5, 0.25, 0.75, 1.25, 1.5);

  igraph_community_walktrap(&graph, &weights, 4, &merges, &modularity, &membership);
  printf("Merges:\n");
  print_matrix(&merges);

  printf("Modularity: ");
  fixup_modularity(&modularity);
  print_vector(&modularity);

  printf("Membership: ");
  print_vector(&membership);

  /* Negative weights are not allowed */
  VECTOR(weights)[0] = -0.1;
  CHECK_ERROR(igraph_community_walktrap(&graph, &weights, 4, &merges, &modularity, &membership), IGRAPH_EINVAL);

  /* Invalid weight vector size */
  VECTOR(weights)[0] = 0.1;
  igraph_vector_pop_back(&weights);
  CHECK_ERROR(igraph_community_walktrap(&graph, &weights, 4, &merges, &modularity, &membership), IGRAPH_EINVAL);

  igraph_vector_destroy(&weights);
  igraph_vector_destroy(&membership);
  igraph_vector_destroy(&modularity);
  igraph_matrix_destroy(&merges);

  igraph_destroy(&graph);

  printf("\nIsolated vertices\n\n");

  igraph_empty(&graph, 5, IGRAPH_UNDIRECTED);

  igraph_matrix_init(&merges, 0, 0);
  igraph_vector_init(&modularity, 0);
  igraph_vector_init(&membership, 0);

  igraph_community_walktrap(&graph, NULL, 4, &merges, &modularity, &membership);
  printf("Merges:\n");
  print_matrix(&merges);

  printf("Modularity: ");
  fixup_modularity(&modularity);
  print_vector(&modularity);

  printf("Membership: ");
  print_vector(&membership);

  igraph_vector_destroy(&weights);
  igraph_vector_destroy(&membership);
  igraph_vector_destroy(&modularity);
  igraph_matrix_destroy(&merges);

  igraph_destroy(&graph);


  VERIFY_FINALLY_STACK();

  return 0;
}
