import os
import sys
import math
import threading

try: import pygame
except ImportError:
    raise GaspException('Pygame is not installed. Please install it.')
from pygame.locals import *
if not pygame.font: print 'Warning, fonts disabled'
if not pygame.mixer: print 'Warning, sound disabled'


# diffrent update_when values
NOTHING = "Nothing"
MOUSE_PRESSED = "Mouse_Pressed"
KEY_PRESSED = "Key_Pressed"
old_pressed = []


#============================= SCREEN FUNCTIONS ================================

def check_screen_atts(screen_obj):

    try:   # checks width attribute
        width = int(screen_obj.width)
        self.width = width
    except:
        raise GaspException("Width is not a viable type. Width = %d "  % screen_obj.width)

    try:  # checks height attribute
        height = int(height)
        self.height = height
    except:
        raise GaspException("Height is not a vaiable type. Height = %d" % screen_obj.height)


def shutdown(type, value, traceback):
    end()
    sys.__excepthook__(type, value, traceback)
    
    
def create_screen(screen_obj): # takes a Screen() objects.
    global screen, handler

    sys.excepthook = shutdown
    
    screen = screen_obj
    screen.clock = pygame.time.Clock()
    screen.frame_rate = 100
    screen.visible_sprites = pygame.sprite.RenderUpdates()

    screen.display = pygame.display.set_mode((screen.width, screen.height),
                                              pygame.DOUBLEBUF)
    pygame.display.set_caption(screen.title)

    if type(screen.background) is str:
        #screen.image = pygame.image.load(filler).convert()
        screen.color = (0, 0, 0, 255)
    elif type(screen.background) in (tuple, list):
        color = screen.background
        screen.background = pygame.Surface((screen.width, screen.height))
        screen.background.fill(color)
        screen.color = (color)
    else:
        raise GaspException('Unknown data sent to fill the screen')
   
    if screen.color != (0, 0, 0):
        screen.background.set_colorkey((0, 0, 0, 255))
    else:
        screen.background.set_colorkey((255, 255, 255, 255))
    for i in screen.visible_sprites.sprites():
        i.remove(screen.visible_sprites)
    screen.display.blit(screen.background, screen.background.get_rect())
    pygame.display.flip()

    # These are used to control the event manager thread
    screen.quitting = False  # You don't want to quit yet
    screen.update_when = NOTHING     # You don't want to wait for anything
    screen.auto_update = True  # don't use pygames clock to update
    
    screen.handler = EventHandler()
    screen.handler.start()


def require_screen(func):
    def wrapper(*args):
        if not pygame.display.get_surface():
            raise GaspException("No graphics window initialized. Please call begin_graphics().")
        return func(*args)

    return wrapper


@require_screen
def end():
    global screen
    #The event handling thread uses this as a flag
    
    #if screen:
    screen.quitting = True
    pygame.event.post(pygame.event.Event(USEREVENT))
    #screen = None


def remove(obj):
    obj.sprite.kill()
    if '' in sys.argv: update()


def clear_screen():
    global screen
    screen.visible_sprites.clear(screen.display, screen.background)
    screen.visible_sprites.empty()
    pygame.display.flip() #self.update()


@require_screen
def update():
    screen.clock.tick(screen.frame_rate)
    screen.visible_sprites.clear(screen.display, screen.background)
    rects = screen.visible_sprites.draw(screen.display)
    pygame.display.update(rects)


@require_screen
def update_now():
    screen.visible_sprites.clear(screen.display, screen.background)
    rects = screen.visible_sprites.draw(screen.display)
    pygame.display.update(rects)


#============================== SHAPE FUNCTIONS ===============================

#------Automated sprite creation and movement-----
@require_screen
def sprite_create(obj, width, height):
    width = int(math.fabs(width))
    height = int(math.fabs(height))
    obj.sprite = pygame.sprite.Sprite()
    obj.sprite.image = pygame.Surface([width, height])
    obj.sprite.rect = obj.sprite.image.get_rect()
    screen.visible_sprites.add(obj.sprite)

    obj.sprite.image.set_colorkey(screen.color)
    obj.sprite.image.fill(screen.color)
    

@require_screen
def sprite_move(obj, pos):
    try: 
        len(pos) == 2
        obj.sprite.rect.center = (pos[0], flip_coords(pos[1]))
    except:
        obj.sprite.rect.center = (pos.x, flip_coords(pos.y))


@require_screen
def rotate_sprite(obj, angle):
    center = obj.sprite.rect.center
    obj.sprite.image = pygame.transform.rotate(obj.sprite.image, angle)
    obj.sprite.rect = obj.sprite.image.get_rect(center=center)


@require_screen
def plot(obj): # must be passed a Plot()
    obj.sprite = pygame.sprite.Sprite()
    obj.sprite.image = pygame.Surface([obj.size*2, obj.size*2])
    obj.sprite.rect = obj.sprite.image.get_rect()
    screen.visible_sprites.add(obj.sprite)
    obj.sprite.rect.center = (obj.pos.x, flip_coords(obj.pos.y))
    obj.sprite.image.fill(obj.color)
    if screen.auto_update:
        update_now()


@require_screen
def create_line(obj): # Must take a line object
    obj.width = abs(int(obj.end.x - obj.start.x))
    if obj.width == 0 or obj.width == 1:
        obj.width = 2
    
    obj.height = abs(int(obj.end.y - obj.start.y))
    if obj.height == 0 or obj.height == 1:
        obj.height = 2

    dx = min(obj.start.x, obj.end.x)
    dy = min(obj.start.y, obj.end.y)
    start = (obj.start.x - dx, obj.start.y - dy)
    end = (obj.end.x - dx, obj.end.y - dy)
    
    obj.x = min(obj.start.x, obj.end.x)
    obj.y = max(obj.start.y, obj.end.y)
    #sprite_create(obj)
    #-----
    obj.sprite = pygame.sprite.Sprite()
    obj.sprite.image = pygame.Surface([obj.width, obj.height])
    obj.sprite.rect = obj.sprite.image.get_rect()
    screen.visible_sprites.add(obj.sprite)
    obj.sprite.rect.topleft = (obj.x, flip_coords(obj.y))

    obj.sprite.image.set_colorkey(screen.color)
    obj.sprite.image.fill(screen.color)
    #-----
    
    if  abs(int(obj.end.x - obj.start.x)) == 0:
        pygame.draw.rect(obj.sprite.image, obj.color, [0, 0, 2, obj.height])
    elif abs(int(obj.end.y - obj.start.y)) == 0:
        pygame.draw.rect(obj.sprite.image, obj.color, [0, 0, obj.width, 2])
    else:
        pygame.draw.aaline(obj.sprite.image,
                           obj.color,
                           (start[0], flip_coords(start[1], obj.sprite.rect)), 
                           (end[0], flip_coords(end[1], obj.sprite.rect)), False
                           )
    if screen.auto_update:
        update_now()


@require_screen
def create_box(obj): # Must take a box object
    
    sprite_create(obj, obj.width, obj.height)
    sprite_move(obj, (obj.center.x, obj.center.y))
    if obj.filled:
        obj.thickness = 0
    pygame.draw.rect(obj.sprite.image,
                     obj.color,
                     [0, 0, obj.width, obj.height],
                     obj.thickness)
    if screen.auto_update:
        update_now()


@require_screen
def create_polygon(obj): # Must take a polygon object
    
    x_values = []
    y_values = []
    for point in obj.points:
        x_values.append(point[0])
        y_values.append(point[1])
    dx = min(x_values)  
    dy = min(y_values)
    sprite_create(obj, obj.width + 1, obj.height + 1)
    relative_points = []
    for point in obj.points:
        relative_points.append((point[0] - dx, flip_coords(point[1] - dy + 1,
                                obj.sprite.rect)))
    sprite_move(obj, (obj.center.x, obj.center.y))
    if obj.filled:
        obj.thickness = 0

    sprite_move(obj, (obj.center.x, obj.center.y))
    pygame.draw.polygon(obj.sprite.image,
                        obj.color,
                        relative_points,
                        obj.thickness)

    if screen.auto_update:
        update_now()


@require_screen
def create_circle(obj): # Must take a circle objects
    sprite_create(obj, obj.radius*2, obj.radius*2)
    sprite_move(obj, (obj.center.x, obj.center.y))
    if obj.filled:
        obj.thickness = 0
    pygame.draw.circle(obj.sprite.image, obj.color, 
                       (obj.radius, obj.radius), obj.radius, obj.thickness)
    if screen.auto_update:
        update_now()


@require_screen
def create_arc(obj):
    width = obj.radius*2
    height = obj.radius*2
    sprite_create(obj, width, height)
    sprite_move(obj, (obj.center.x, obj.center.y))

    if obj.filled == True:
        obj.thickness = obj.radius

    pygame.draw.arc(obj.sprite.image,
                    obj.color, [0, 0, width, height], 
                    math.radians(obj.start_angle),
                    math.radians(obj.end_angle),
                    obj.thickness)

    if screen.auto_update:
        update_now()


@require_screen
def create_oval(obj): # Must take an oval object
    sprite_create(obj, obj.width, obj.height)
    sprite_move(obj, (obj.center.x, obj.center.y))
    if obj.filled == True:
        obj.thickness = 0
    pygame.draw.ellipse(obj.sprite.image,
                        obj.color,
                        [0, 0, obj.width, obj.height],
                        obj.thickness)
    if screen.auto_update:
        update_now()


@require_screen
def create_image(obj): # must take an image object
    obj.sprite = pygame.sprite.Sprite()
    obj.sprite.image = pygame.image.load(obj.path_name).convert()
    obj.sprite.rect = obj.sprite.image.get_rect()
    #obj.width = obj.sprite.rect.width
    #obj.height = obj.sprite.rect.height
    screen.visible_sprites.add(obj.sprite)
    obj.sprite.rect.topleft = (obj.x, flip_coords(obj.y + obj.height))
    if screen.auto_update:
        update_now()


#------object manipulaters-------

@require_screen
def move_to(obj, pos):
    sprite_move(obj, pos)
    if screen.auto_update:
        update_now()


@require_screen
def rotate_by(obj, angle):
    rotate_sprite(obj, angle)
    if screen.auto_update:
        update_now()


#================================== Text ======================================

@require_screen
def create_text(obj):
    if not pygame.font.get_init():
        pygame.font.init()
    
    font = pygame.font.Font(None, obj.size)
    obj.sprite = pygame.sprite.Sprite()
    obj.sprite.image= font.render(obj.text, 1, obj.color)
    obj.sprite.rect = obj.sprite.image.get_rect()
    screen.visible_sprites.add(obj.sprite)
    obj.sprite.rect.topleft = (obj.pos.x,
                               flip_coords(obj.pos.y + obj.sprite.rect.height))

    if screen.auto_update:
        update_now()


#================================== Sound =====================================

def create_sound(obj): # needs to be a sound object.
    try: 
        pygame.mixer.init()
    except:
        print "sound is all ready init"
        pass
    obj.sound = pygame.mixer.Sound(obj.path)


def play_sound(obj):
    obj.sound.play()


def stop_sound(obj):
    obj.sound.stop()


# ============================== Event Functions ==============================



def mouse_pos():
    a = pygame.mouse.get_pos()
    return a[0], flip_coords(a[1])


def mouse_pressed():
    pressed = pygame.mouse.get_pressed()
    return {'left': pressed[0],
            'middle': pressed[1],
            'right': pressed[2]}


@require_screen
def keys(): #returns all the keys currently being pressed
    pressed = pygame.key.get_pressed()
    name=pygame.key.name
    all=[]
    for key in [name(i) for i in xrange(len(pressed)) if pressed[i]]:
        if key[0] == '[' and key[-1] == ']': key = key[1:-1] #numpad key
        if key not in all: #is this 'if' needed?
            all.append(key)
    return all


@require_screen
def waiting_keys():
    '''
       returns all keys pressed for the first time and then waits until
       the key has been let go before it checks again.
    '''
    global old_pressed, new_pressed
    pressed = []
    new_pressed = keys()
    try:
        for key in new_pressed:
            if key not in old_pressed:
                pressed.append(key)
    except:
        pass
    old_pressed = keys()
    return pressed


#=============================== EVENT HANDLER ================================

class EventHandler(threading.Thread):

    def __init__(self):
        threading.Thread.__init__(self)
        # Limit the event handler to only looking at what we need
        pygame.event.set_blocked([
            KEYUP,
            MOUSEMOTION,
            MOUSEBUTTONUP,
            JOYAXISMOTION,
            JOYBALLMOTION,
            JOYHATMOTION,  
            JOYBUTTONUP, 
            JOYBUTTONDOWN, 
            VIDEORESIZE, 
            VIDEOEXPOSE])


    def run(self):
        global screen
        while not screen.quitting:
            e = pygame.event.wait()
            #e = pygame.event.poll()

            if e.type == ACTIVEEVENT: #The screen updates if 
                if e.state == 2 and e.gain == 0:
                    pygame.display.update()

            elif e.type == QUIT:
                end()
                pygame.quit()
                sys.exit()
                

            if screen.update_when == KEY_PRESSED:
                if e.type == KEYDOWN:
                    screen.update_when = NOTHING

            elif screen.update_when == MOUSE_PRESSED:
                if e.type == MOUSEBUTTONDOWN:
                    screen.update_when = NOTHING
            
            
            pygame.quit
            
            
            #handle network events and populate a network messege que that is handled on the front end.


# ============================ Exception Handling ============================

class GaspException(Exception):
    def __init__(self, value):
        self.value = value

    def __str__(self):
        return repr(self.value)


# =============================== BACKEND TOOLS ===============================

def screen_picture(file_name):
    pygame.image.save(screen.display, file_name)
    

def flip_coords(y, surface=None):
    """
    Changes (0, 0) from topleft bottomleft.
    """
    if not surface:
        surface = screen
   
    try:
        return surface.height - y.y
    except:
        return surface.height - y


# ============================== Time ========================================

@require_screen
def update_when(event_type):
    screen.auto_update = False
    update()
    
    if event_type == "key_pressed":
        screen.update_when = KEY_PRESSED

    elif event_type == "mouse_pressed":
        screen.update_when = MOUSE_PRESSED

    elif event_type == "next_tick":
        update() #Uses the Pygame Clock to wait till the next tick
    
    else:
        raise GaspException("An invalid event type was passed")

    while not screen.update_when == NOTHING:
        pygame.event.pump()
                    

@require_screen
def set_frame_rate(fps):
    screen.auto_update = False
    screen.frame_rate = fps


def sleep(duration):
    update_now() #updates before it starts sleeping.
    pygame.time.delay(duration * 1000)
