#include "Python.h"
#include "util.h"
#include "splitqname.h"

PyObject *DOMString_FromObject(PyObject *obj)
{
  /* sanity check */
  if (obj == NULL) {
    PyErr_BadInternalCall();
    return NULL;
  }

  if (obj == Py_None) {
    Py_INCREF(obj);
    return obj;
  }

#ifdef PyUnicode_CheckExact
  /* Python 2.2 and newer */
  if (PyUnicode_CheckExact(obj)) {
    Py_INCREF(obj);
    return obj;
  }
#endif
  if (PyUnicode_Check(obj)) {
    /* For a Unicode subtype that's not a Unicode object,
       return a true Unicode object with the same data. */
    return PyUnicode_FromUnicode(PyUnicode_AS_UNICODE(obj),
                                 PyUnicode_GET_SIZE(obj));
  }

  /* Python DOM bindings specify byte-strings (PyString) must be
     UTF-8 encoded.

     Using "utf-8" instead of "UTF-8" as it is used as the shortcut name
     for the UTF-8 codec in Python's Unicode internals.
  */
  return PyUnicode_FromEncodedObject(obj, "utf-8", "strict");
}

PyObject *DOMString_FromObjectInplace(PyObject *obj)
{
  PyObject *result;

  /* allow for inlining */
  if (obj == NULL) return obj;

  if ((result = DOMString_FromObject(obj)) != NULL) {
    Py_DECREF(obj);
  }

  return result;
}

PyObject *DOMString_ConvertArgument(PyObject *arg, char *name, int null_ok)
{
  PyObject *result;

  if (null_ok) {
    result = DOMString_FromObject(arg);
    if (result == NULL) {
      if (PyErr_ExceptionMatches(PyExc_TypeError))
        PyErr_Format(PyExc_TypeError,
                     "%s must be None, unicode or UTF-8 string, %.80s found.",
                     name, arg->ob_type->tp_name);
    }
  }
  else if (arg != Py_None) {
    result = DOMString_FromObject(arg);
    if (result == NULL) {
      if (PyErr_ExceptionMatches(PyExc_TypeError))
        PyErr_Format(PyExc_TypeError,
                     "%s must be unicode or UTF-8 string, %.80s found.",
                     name, arg->ob_type->tp_name);
    }
  }
  else {
    /* arg == Py_None and not null_ok */
    PyErr_Format(PyExc_TypeError,
                 "%s must be non-null unicode or UTF-8 string.", name);
    result = NULL;
  }

  return result;
}

void AddInternCtr(PyObject *obj, PyObject *dict)
{
  PyObject *value;

  if (obj != Py_None) {
    value = PyDict_GetItem(dict, obj);
    if (value == NULL) {
      /*Add it  also adds a ref*/
      value = PyInt_FromLong(1);
      PyDict_SetItem(dict, obj, value);
      Py_DECREF(value);
    } else {
      value = PyInt_FromLong(PyInt_AsLong(value) + 1);
      PyDict_SetItem(dict, obj, value);
      Py_DECREF(value);
    }
  }
}

int TestRefCount(PyObject *tester,PyObject *obj,int expected, const char* name)
{
  char buf[500];
  PyObject *retval = PyObject_Repr(obj);
  if (retval == NULL) return 0;
  sprintf(buf, "%.200s equal to %.200s", name, PyString_AS_STRING(retval));
  Py_DECREF(retval);

  retval = PyObject_CallMethod(tester, "startTest", "s", buf);
  if (retval == NULL) return 0;
  Py_DECREF(retval);

  retval = PyObject_CallMethod(tester, "compare", "ll", (long)expected, obj->ob_refcnt);
  if (retval == NULL) return 0;
  Py_DECREF(retval);

  retval = PyObject_CallMethod(tester, "testDone", NULL);
  if (retval == NULL) return 0;
  Py_DECREF(retval);
  return 1;
}

