/***********************************************************************
 * $Header: /var/local/cvsroot/4Suite/Ft/Xml/src/domlette/domlette.c,v 1.35 2004/08/23 18:20:29 jkloth Exp $
 ***********************************************************************/

static char module_doc[] = "\
cDomlette implementation: a very fast DOM-like library tailored for use in XPath/XSLT\n\
\n\
Copyright 2003 Fourthought, Inc. (USA).\n\
Detailed license and copyright information: http://4suite.org/COPYRIGHT\n\
Project home, documentation, distributions: http://4suite.org/\n\
";

#include "domlette.h"
#include "reader.h"
#include "nss.h"

/* needed for API verification */
#include "expat.h"

PyObject *g_xmlNamespace;
PyObject *g_xmlnsNamespace;
PyObject *g_xincludeNamespace;

/*
  These are the external interfaces
*/

#define UNICODE_FROM_CHAR(str) PyUnicode_DecodeASCII((str), strlen(str), NULL);

#define CHECK_COUNT(num,where) if (doc->ob_refcnt != (num)) { \
    sprintf(errorBuf,"%s expected %d ref count, found %d\n",(where),(num),doc->ob_refcnt); \
    PyErr_SetString(PyExc_MemoryError,errorBuf); \
  return NULL; \
}

/* The test tree XML:
test_xml = """<?xml version = "1.0"?>
<?xml-stylesheet href="addr_book1.xsl" type="text/xml"?>
<docelem xmlns:ft="http://fourthought.com">
  <child foo='bar'>Some Text</child>
  <!--A comment-->
  <ft:nschild ft:foo="nsbar">Some More Text</ft:nschild>
  <appendChild/>
</docelem>"""
*/

PyObject *PyTestTree(PyObject * self, PyObject *args)
{
  /*Build a test tree*/
  char errorBuf[256];
  PyDocumentObject *doc;
  PyProcessingInstructionObject *pi;
  PyElementObject *documentElement, *element;
  PyTextObject *text;
  PyCommentObject *comment;
  PyAttrObject *attr;
  PyObject *namespaceURI, *qualifiedName, *prefix, *localName;
  PyObject *target, *data, *value;

  unsigned long index = 0;

  if (!PyArg_ParseTuple(args, ":TestTree")) return NULL;

  doc = Document_New(&index, Py_None);
  CHECK_COUNT(1,"Doc Creation");

  /* Add processing instruction */
  target = UNICODE_FROM_CHAR("xml-stylesheet");
  data = UNICODE_FROM_CHAR("href=\"addr_book1.xsl\" type=\"text/xml\"");
  pi = Document_CreateProcessingInstruction(doc, target, data, &index);
  Py_DECREF(data);
  Py_DECREF(target);
  Node_AppendChild((PyNodeObject *)doc, (PyNodeObject *)pi);
  /*We are all done with pi, so DECREF it once*/
  Py_DECREF(pi);

  CHECK_COUNT(3, "PI Creation");

  /* Add documentElement */
  namespaceURI = Py_None;
  qualifiedName = UNICODE_FROM_CHAR("docelem");
  prefix = Py_None;
  localName = qualifiedName;
  documentElement = Document_CreateElementNS(doc, namespaceURI, qualifiedName, 
                                             prefix, localName, &index);
  Py_DECREF(qualifiedName);
  Node_AppendChild((PyNodeObject *)doc, (PyNodeObject *)documentElement);

  CHECK_COUNT(5,"Doc Elem Creation");

  /* Add documentElement namespace declartion */
  namespaceURI = g_xmlnsNamespace;
  qualifiedName = UNICODE_FROM_CHAR("xmlns:ft");
  prefix = UNICODE_FROM_CHAR("xmlns");
  localName = UNICODE_FROM_CHAR("ft");
  value = UNICODE_FROM_CHAR("http://fourthought.com");
  attr = Element_SetAttributeNS(documentElement, namespaceURI, qualifiedName,
                                prefix, localName, value);
  Py_DECREF(attr);
  Py_DECREF(value);
  Py_DECREF(localName);
  Py_DECREF(prefix);
  Py_DECREF(qualifiedName);

  CHECK_COUNT(6,"NS DECL");

  /* Add text 1 to documentElement */
  data = UNICODE_FROM_CHAR("\n  ");
  text = Document_CreateTextNode(doc, data, &index);
  Py_DECREF(data);
  Node_AppendChild((PyNodeObject *)documentElement, (PyNodeObject *)text);
  /*We are all done with text, so DECREF it once*/
  Py_DECREF(text);
  CHECK_COUNT(7,"1st Text");

  /* Add element 1 to documentElement */
  namespaceURI = Py_None;
  qualifiedName = UNICODE_FROM_CHAR("child");
  prefix = Py_None;
  localName = qualifiedName;
  element = Document_CreateElementNS(doc, namespaceURI, qualifiedName, prefix, 
                                     localName, &index);
  Py_DECREF(qualifiedName);
  Node_AppendChild((PyNodeObject *)documentElement, (PyNodeObject *)element);
  CHECK_COUNT(8,"First Child");

  /* Add element 1 attribute */
  namespaceURI = Py_None;
  qualifiedName = UNICODE_FROM_CHAR("foo");
  prefix = Py_None;
  localName = qualifiedName;
  value = UNICODE_FROM_CHAR("bar");
  attr = Element_SetAttributeNS(element, namespaceURI, qualifiedName, prefix,
                                localName, value);
  Py_DECREF(attr);
  Py_DECREF(qualifiedName);
  Py_DECREF(value);
  CHECK_COUNT(9,"First Child Attr");

  /* Add element 1 text */
  data = UNICODE_FROM_CHAR("Some Text");
  text = Document_CreateTextNode(doc, data, &index);
  Py_DECREF(data);
  Node_AppendChild((PyNodeObject *)element, (PyNodeObject *)text);
  /*We are all done with text, so DECREF it once*/
  Py_DECREF(text);
  CHECK_COUNT(10,"First Child Text");

  /*We are all done with element 1, so DECREF it once*/
  Py_DECREF(element);

  /* Add text 2 to documentElement */
  data = UNICODE_FROM_CHAR("\n  ");
  text = Document_CreateTextNode(doc, data, &index);
  Py_DECREF(data);
  Node_AppendChild((PyNodeObject *)documentElement, (PyNodeObject *)text);
  /*We are all done with text, so DECREF it once*/
  Py_DECREF(text);
  CHECK_COUNT(11,"2nd Text");

  /* Add comment to documentElement */
  data = UNICODE_FROM_CHAR("A comment");
  comment = Document_CreateComment(doc, data, &index);
  Py_DECREF(data);
  Node_AppendChild((PyNodeObject *)documentElement, (PyNodeObject *)comment);
  /*We are all done with comment so DECREF it once*/
  Py_DECREF(comment);
  CHECK_COUNT(12,"Comment");

  /* Add text 3 to documentElement */
  data = UNICODE_FROM_CHAR("\n  ");
  text = Document_CreateTextNode(doc, data, &index);
  Py_DECREF(data);
  Node_AppendChild((PyNodeObject *)documentElement, (PyNodeObject *)text);
  /*We are all done with text, so DECREF it once*/
  Py_DECREF(text);
  CHECK_COUNT(13,"3rd Text");

  /* Add element 2 to documentElement */
  namespaceURI = UNICODE_FROM_CHAR("http://fourthought.com");
  qualifiedName = UNICODE_FROM_CHAR("ft:nschild");
  prefix = UNICODE_FROM_CHAR("ft");
  localName = UNICODE_FROM_CHAR("nschild");
  element = Document_CreateElementNS(doc, namespaceURI, qualifiedName, prefix,
                                     localName, &index);
  Py_DECREF(localName);
  Py_DECREF(prefix);
  Py_DECREF(qualifiedName);
  Py_DECREF(namespaceURI);
  Node_AppendChild((PyNodeObject *)documentElement, (PyNodeObject *)element);
  CHECK_COUNT(14,"2nd Child");

  /* Add element 2 attribute */
  namespaceURI = UNICODE_FROM_CHAR("http://fourthought.com");
  qualifiedName = UNICODE_FROM_CHAR("ft:foo");
  prefix = UNICODE_FROM_CHAR("ft");
  localName = UNICODE_FROM_CHAR("foo");
  value = UNICODE_FROM_CHAR("nsbar");
  attr = Element_SetAttributeNS(element, namespaceURI, qualifiedName, prefix, 
                                localName, value);
  Py_DECREF(attr);
  Py_DECREF(value);
  Py_DECREF(localName);
  Py_DECREF(prefix);
  Py_DECREF(qualifiedName);
  Py_DECREF(namespaceURI);
  CHECK_COUNT(15,"2nd Child Attr");

  /* Add element 2 text node */
  data = UNICODE_FROM_CHAR("Some More Text");
  text = Document_CreateTextNode(doc, data, &index);
  Py_DECREF(data);
  Node_AppendChild((PyNodeObject *)element, (PyNodeObject *)text);
  /*We are all done with text, so DECREF it once*/
  Py_DECREF(text);
  CHECK_COUNT(16,"2nd Child Text");

  /*We are all done with element 2, so DECREF it once*/
  Py_DECREF(element);

  /* Add text 4 to documentElement */
  data = UNICODE_FROM_CHAR("\n  ");
  text = Document_CreateTextNode(doc, data, &index);
  Py_DECREF(data);
  Node_AppendChild((PyNodeObject *)documentElement, (PyNodeObject *)text);
  /*We are all done with text, so DECREF it once*/
  Py_DECREF(text);
  CHECK_COUNT(17,"4th Text");

  /* Add element 3 to documentElement */
  namespaceURI = Py_None;
  qualifiedName = UNICODE_FROM_CHAR("appendChild");
  prefix = Py_None;
  localName = qualifiedName;
  element = Document_CreateElementNS(doc, namespaceURI, qualifiedName, prefix, 
                                     localName, &index);
  Py_DECREF(qualifiedName);
  Node_AppendChild((PyNodeObject *)documentElement, (PyNodeObject *)element);
  CHECK_COUNT(18,"Append Child");

  /* We are all done with element 3, so DECREF it once*/
  Py_DECREF(element);

  /* Add text 5 to documentElement */
  data = UNICODE_FROM_CHAR("\n  ");
  text = Document_CreateTextNode(doc, data, &index);
  Py_DECREF(data);
  Node_AppendChild((PyNodeObject *)documentElement, (PyNodeObject *)text);
  /*We are all done with text, so DECREF it once*/
  Py_DECREF(text);
  CHECK_COUNT(19,"5th Text");

  /* We are all done with documentElement, so DECREF it once */
  Py_DECREF(documentElement);

  return (PyObject *)doc;

}


static PyObject *PyStartNodeCounting(PyObject * self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ":StartNodeCounting")) return NULL;
  g_nodeCount = 0;
  Py_INCREF(Py_None);
  return Py_None;
}


static PyObject *PyGetNodeCount(PyObject * self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ":GetNodeCount")) return NULL;
  return PyInt_FromLong(g_nodeCount);
}


static PyObject *PyTestRefCounts(PyObject * self, PyObject *args)
{
  PyObject *tester;
  PyDocumentObject *document;
  long temp = 0;

  if(!PyArg_ParseTuple(args,"OO:TestRefCounts", &tester, &document)) 
    return NULL;

  if (PyDocument_Check(document)) {
    Py_INCREF(document);
  }
  else if ((PyObject *)document == Py_None) {
    document = Document_New(&temp, Py_None);
  }
  else {
    PyErr_SetString(PyExc_TypeError, "argument 2 must be a cDocument or None");
    return NULL;
  }

  if (!document_test_ref_counts(tester, document, document->ob_refcnt)) {
    Py_DECREF(document);
    return NULL;
  }

  Py_DECREF(document);
  Py_INCREF(Py_None);
  return Py_None;
}

/** The external interface definitions ********************************/

static PyMethodDef cDomlettecMethods[] = {

  /* from reader.c */
  { "nonvalParse", (PyCFunction)Domlette_NonvalParse, 
    METH_VARARGS|METH_KEYWORDS, NonvalParse_doc },
  { "parse",        PyParse,             METH_VARARGS, PyParse_doc },

  /* from nss.c */
  { "GetAllNs",     Domlette_GetAllNs,   METH_VARARGS, GetAllNs_doc },
  { "SeekNss",      Domlette_SeekNss,    METH_VARARGS, SeekNss_doc },

  /* defined here (regression tests) */
  { "TestTree", PyTestTree, METH_VARARGS, 
    "TestTree() -> Document\n\nFor regression testing." },
  { "TestRefCounts", PyTestRefCounts, METH_VARARGS, 
    "TestRefCounts(tester, document)\n\nFor regression testing." },
  { "StartNodeCounting", PyStartNodeCounting, METH_VARARGS, 
    "StartNodeCounting()\n\nFor regression testing." },
  { "GetNodeCount", PyGetNodeCount, METH_VARARGS, 
    "GetNodeCount() -> int\n\nFor regression testing." },

  { NULL }
};

static DomletteModule_APIObject DomletteModule_API = {
  &PyDomletteDOMImplementation_Type,
  &PyDomletteDocument_Type,
  &PyDomletteElement_Type,
  &PyDomletteAttr_Type,
  &PyDomletteText_Type,
  &PyDomletteComment_Type,
  &PyDomletteProcessingInstruction_Type,
  &PyDomletteDocumentFragment_Type,
};

static void domlette_fini(void *capi)
{
  DomletteExceptions_Fini();
  DomletteReader_Fini();
  DomletteDOMImplementation_Fini();
  DomletteElement_Fini();
  DomletteAttr_Fini();
  DomletteText_Fini();
  DomletteProcessingInstruction_Fini();
  DomletteComment_Fini();
  DomletteDocument_Fini();
  DomletteDocumentFragment_Fini();

  Py_DECREF(g_xmlNamespace);
  Py_DECREF(g_xmlnsNamespace);
  Py_DECREF(g_xincludeNamespace);
}

DL_EXPORT(void) initcDomlettec(void)
{
  PyObject *module, *import;
  XML_Expat_Version version;
  PyObject *cobj;

  /* ensure that we're going to be using the proper Expat functions */
  version = XML_ExpatVersionInfo();
  if (version.major != XML_MAJOR_VERSION || 
      version.minor != XML_MINOR_VERSION ||
      version.micro != XML_MICRO_VERSION) {
    /* version mismatch */
    PyErr_Format(PyExc_ImportError,
                 "Expat version mismatch; expected %d.%d.%d, found %d.%d.%d",
                 XML_MAJOR_VERSION, XML_MINOR_VERSION, XML_MICRO_VERSION,
                 version.major, version.minor, version.micro);
    return;
  }

  module = Py_InitModule3("cDomlettec", cDomlettecMethods, module_doc);
  if (module == NULL) return;
  
  /* initialize the sub-components */
  if (DomletteExceptions_Init() == -1) return;
  if (DomletteReader_Init() == -1) return;
  if (DomletteDOMImplementation_Init() == -1) return;
  if (DomletteElement_Init() == -1) return;
  if (DomletteAttr_Init() == -1) return;
  if (DomletteText_Init() == -1) return;
  if (DomletteProcessingInstruction_Init() == -1) return;
  if (DomletteComment_Init() == -1) return;
  if (DomletteDocument_Init() == -1) return;
  if (DomletteDocumentFragment_Init() == -1) return;

  /* expose our implementation */
  if (PyModule_AddObject(module, "implementation", g_implementation) == -1)
    return;
  /* PyModule_AddObject steals a reference */
  Py_INCREF(g_implementation);

  /* get the namespace constants */
  import = PyImport_ImportModule("Ft.Xml");
  if (import == NULL) return;
  g_xmlNamespace = PyObject_GetAttrString(import, "XML_NAMESPACE");
  g_xmlNamespace = DOMString_FromObjectInplace(g_xmlNamespace);
  if (g_xmlNamespace == NULL) return;
  g_xmlnsNamespace = PyObject_GetAttrString(import, "XMLNS_NAMESPACE");
  g_xmlnsNamespace = DOMString_FromObjectInplace(g_xmlnsNamespace);
  if (g_xmlnsNamespace == NULL) return;
  Py_DECREF(import);

  import = PyImport_ImportModule("Ft.Xml.XInclude");
  if (import == NULL) return;
  g_xincludeNamespace = PyObject_GetAttrString(import, "XINCLUDE_NAMESPACE");
  g_xincludeNamespace = DOMString_FromObjectInplace(g_xincludeNamespace);
  if (g_xincludeNamespace == NULL) return;
  Py_DECREF(import);

  /* Export C API - done last to serve as a cleanup function as well */
  cobj = PyCObject_FromVoidPtr((void *)&DomletteModule_API, domlette_fini);
  if (cobj) {
    PyModule_AddObject(module, "CAPI", cobj);
  }
  return;
}
