#include "domlette.h"

#define DocumentFragment_VerifyState(ob)                      \
  if (!PyDocumentFragment_Check(ob) ||                        \
      ((PyDocumentFragmentObject *)(ob))->childNodes == NULL) \
     return DOMException_InvalidStateErr("DocumentFragment in inconsistent state");

static PyObject *my_node_type, *my_node_name;

/*Internal Methods*/

PyDocumentFragmentObject *DocumentFragment_CloneNode(
  PyObject *node, int deep, PyNodeObject *newOwnerDocument)
{
  PyDocumentFragmentObject *newNode;
  PyNodeObject *newChild;
  PyObject *childNodes;
  int count, i;

  if (!PyDocument_Check(newOwnerDocument)) {
    PyErr_SetString(PyExc_TypeError,"newOwnerDocument must be a cDocument");
    return NULL;
  }

  newNode = Document_CreateDocumentFragment((PyDocumentObject *)newOwnerDocument);
  if (!newNode) return NULL;

  if (deep) {
    childNodes = PyObject_GetAttrString(node, "childNodes");
    if (!childNodes) {
      Py_DECREF(newNode);
      return NULL;
    }

    count = PySequence_Length(childNodes);
    for (i = 0; i < count; i++) {
      PyObject *child = PySequence_GetItem(childNodes, i);
      if (!child) {
	Py_DECREF(childNodes);
	Py_DECREF(newNode);
	return NULL;
      }

      newChild = Node_CloneNode(child, deep, newOwnerDocument);
      Py_DECREF(child);
      if (!newChild) {
	Py_DECREF(childNodes);
	Py_DECREF(newNode);
	return NULL;
      }

      if (!Node_AppendChild((PyNodeObject *)newNode, newChild)) {
	Py_DECREF(childNodes);
	Py_DECREF(newNode);
	return NULL;
      }
      Py_DECREF(newChild);
    }
    Py_DECREF(childNodes);
  }
  return newNode;
}


/*External Methods*/


static struct PyMethodDef DocumentFragment_methods[] = { NODE_METHODS,
  {NULL,     NULL}      /* sentinel */
};



/*Type Methods*/

static PyObject *docfrag_getattr(PyDocumentFragmentObject *self, char *name)
{
  PyObject *rt = NULL;
  int temp;

  DocumentFragment_VerifyState(self);

  if (!strcmp(name,"nodeType")) {
    rt = my_node_type;
  }
  else if (!strcmp(name,"nodeName")) {
    rt = my_node_name;
  }
  else if (!strcmp(name,"childNodes")) {
    rt = self->childNodes;
  }
  else if (!strcmp(name,"lastChild")) {
    temp = PyList_GET_SIZE(self->childNodes);
    if (temp) {
      rt = PyList_GET_ITEM(self->childNodes,temp-1);
    } else {
      rt = Py_None;
    }
  }
  else if (!strcmp(name,"firstChild")) {
    if (PyList_GET_SIZE(self->childNodes)) {
      rt = PyList_GET_ITEM(self->childNodes,0);
    } else {
      rt = Py_None;
    }
  }
  if (rt) {
    Py_INCREF(rt);
    return rt;
  }
  return node_getattr((PyNodeObject*)self,name,DocumentFragment_methods);
}

static int docfrag_setattr(PyDocumentFragmentObject *self, char *name, PyObject *v)
{
  /* Set attribute 'name' to value 'v'. v==NULL means delete */
  if (v == NULL) {
    PyErr_Format(PyExc_AttributeError,
                 "Cannot delete attribute '%.400s' on '%.50s' object",
                 name, self->ob_type->tp_name);
  } else {
    PyErr_Format(PyExc_AttributeError,
                 "Cannot set attribute '%.400s' on '%.50s' object",
                 name, self->ob_type->tp_name);
  }
  return -1;
}

static void docfrag_dealloc(PyDocumentFragmentObject *node)
{
  PyObject_GC_UnTrack((PyObject *) node);

  Py_XDECREF(node->childNodes);
  node->childNodes = NULL;

  Node_Del(node);
}

static int docfrag_clear(PyDocumentFragmentObject *self)
{
  /*Called when we need to break cycles*/
  (void) PyList_SetSlice(self->childNodes, 0, -1, 0);

  return node_clear((PyNodeObject *)self);
}

static int docfrag_traverse(PyDocumentFragmentObject *self, visitproc visit, void *arg)
{
  int rt;
  if (self->childNodes != NULL) {
    rt = visit(self->childNodes, arg);
    if (rt) return rt;
  }
  return node_traverse((PyNodeObject *)self,visit,arg);
}



static PyObject *docfrag_repr(PyDocumentFragmentObject *docfrag)
{
  char buf[256];
  sprintf(buf, "<cDocumentFragment at %p>", docfrag);
  return PyString_FromString(buf);
}

PyTypeObject PyDomletteDocumentFragment_Type = {
    PyObject_HEAD_INIT(0)
    0,
    "cDomlette.DocumentFragment",
    GC_TP_BASICSIZE(PyDocumentFragmentObject),
    0,
    (destructor)docfrag_dealloc,    /*tp_dealloc*/
    (printfunc)0,   /*tp_print*/
    (getattrfunc)docfrag_getattr,   /*tp_getattr*/
    (setattrfunc)docfrag_setattr,   /*tp_setattr*/
    0,                          /*tp_compare*/
    (reprfunc)docfrag_repr,     /*tp_repr*/
    0,                          /*tp_as_number*/
    0,              /*tp_as_sequence*/
    0,              /*tp_as_mapping*/
    0,                             /*tp_hash*/
    0,          /*tp_call*/
    0,          /*tp_str*/
    0,                      /*tp_getattro*/
    0,          /*tp_setattro*/
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_HAVE_GC,
    0,                          /* tp_doc */
    (traverseproc)docfrag_traverse,  /* tp_traverse */
    (inquiry)docfrag_clear,          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
};

int DomletteDocumentFragment_Init(void)
{
  PyDomletteDocumentFragment_Type.ob_type = &PyType_Type;

  my_node_type = PyInt_FromLong(DOCUMENT_FRAGMENT_NODE);
  if (my_node_type == NULL) return 0;

  my_node_name = PyUnicode_DecodeASCII("#document-fragment", 18, NULL);
  if (my_node_name == NULL) return 0;

  return 1;
}

void DomletteDocumentFragment_Fini(void)
{
  Py_DECREF(my_node_type);
  Py_DECREF(my_node_name);
}
