#include "domlette.h"

static PyObject *my_node_type, *my_node_name;

/* Internal Methods */

/* No need to intern the empty string, Python already does this */
#define UNICODE_EMPTY_STRING PyUnicode_FromUnicode(NULL, 0)

static PyObject *InternUnicode(PyDocumentObject *doc, PyObject *unistr)
{
#ifdef DISABLE_INTERN
  Py_INCREF(unistr);
  return unistr;
#else
  PyObject *v = unistr;

  if (unistr != Py_None) {
    PyObject *interned = doc->internedUnicode;
    if ((v = PyDict_GetItem(interned, unistr)) == NULL) {
      PyErr_Clear();
      if (PyDict_SetItem(interned, unistr, unistr)) {
	PyErr_SetString(PyExc_SystemError, "InternUnicode: intern failed");
	return NULL;
      }
      v = unistr;
    }
  }


  Py_INCREF(v);
  return v;
#endif
}

/** C API **************************************************************/

PyDocumentObject *Document_New(unsigned long *docIndex, PyObject *documentUri)
{
  PyDocumentObject *self;

  self = Node_New(PyDocumentObject, &PyDomletteDocument_Type, Py_None);
  if (self == NULL)
    return NULL;

  /* assign members */
  self->childNodes = PyList_New(0);
  if (self->childNodes == NULL) {
    Node_Del(self);
    return NULL;
  }

  self->internedUnicode = PyDict_New();
  if (self->internedUnicode == NULL) {
    Py_DECREF(self->childNodes);
    Node_Del(self);
    return NULL;
  }

  self->unparsedEntities = PyDict_New();
  if (self->unparsedEntities == NULL) {
    Py_DECREF(self->internedUnicode);
    Py_DECREF(self->childNodes);
    Node_Del(self);
    return NULL;
  }

  if (documentUri == Py_None) {
    documentUri = UNICODE_EMPTY_STRING;
    /* just in case */
    if (documentUri == NULL) {
      Py_DECREF(self->unparsedEntities);
      Py_DECREF(self->internedUnicode);
      Py_DECREF(self->childNodes);
      Node_Del(self);
      return NULL;
    }
  }
  else
    Py_INCREF(documentUri);
  self->documentUri = documentUri;

  self->docIndex = (*docIndex)++;
  self->nextIndex = *docIndex;

  self->publicId = Py_None;
  Py_INCREF(Py_None);

  self->systemId = Py_None;
  Py_INCREF(Py_None);

  PyObject_GC_Track(self);

  return self;
}


PyElementObject *Document_CreateElementNS(PyDocumentObject *doc,
					  PyObject *namespaceURI,
                                          PyObject *qualifiedName,
					  PyObject *prefix,
					  PyObject *localName,
					  unsigned long *docIx)
{
  PyElementObject *element;

  if (!PyDocument_Check(doc)) {
    PyErr_BadInternalCall();
    return NULL;
  }

  DOMString_NullCheck(prefix);
  DOMString_NullCheck(namespaceURI);

  element = Node_New(PyElementObject, &PyDomletteElement_Type, doc);
  if (!element)
    return NULL;

  element->namespaceURI = InternUnicode(doc, namespaceURI);
  if (!element->namespaceURI) {
    Node_Del(element);
    return NULL;
  }

  element->prefix = InternUnicode(doc, prefix);
  if (!element->prefix) {
    Py_DECREF(element->namespaceURI);
    Node_Del(element);
    return NULL;
  }

  element->localName = InternUnicode(doc, localName);
  if (!element->localName) {
    Py_DECREF(element->namespaceURI);
    Py_DECREF(element->prefix);
    Node_Del(element);
    return NULL;
  }

  element->nodeName = InternUnicode(doc, qualifiedName);
  if (!element->nodeName) {
    Py_DECREF(element->namespaceURI);
    Py_DECREF(element->prefix);
    Py_DECREF(element->localName);
    Node_Del(element);
    return NULL;
  }

  element->attributes = PyDict_New();
  if (!element->attributes) {
    Py_DECREF(element->namespaceURI);
    Py_DECREF(element->prefix);
    Py_DECREF(element->localName);
    Py_DECREF(element->nodeName);
    Node_Del(element);
    return NULL;
  }

  element->childNodes = PyList_New(0);
  if (!element->childNodes) {
    Py_DECREF(element->namespaceURI);
    Py_DECREF(element->prefix);
    Py_DECREF(element->localName);
    Py_DECREF(element->nodeName);
    Py_DECREF(element->attributes);
    Node_Del(element);
    return NULL;
  }

  element->docIndex = *docIx;
  /* Make space for namespace and attribute nodes index */
  (*docIx) += 3;

  PyElement_BASE_URI(element) = PyDocument_BASE_URI(doc);
  Py_INCREF(PyElement_BASE_URI(element));

  PyObject_GC_Track(element);

  return element;
}


PyAttrObject *Document_CreateAttributeNS(PyDocumentObject *doc,
					 PyObject *namespaceURI,
                                         PyObject *qualifiedName,
					 PyObject *prefix,
					 PyObject *localName,
                                         PyObject *value)
{
  PyAttrObject *attr;

  if (!PyDocument_Check(doc)) {
    PyErr_BadInternalCall();
    return NULL;
  }

  DOMString_NullCheck(prefix);
  DOMString_NullCheck(namespaceURI);

  attr = Node_New(PyAttrObject, &PyDomletteAttr_Type, doc);
  if (attr == NULL)
    return NULL;

  attr->namespaceURI = InternUnicode(doc, namespaceURI);
  if (!attr->namespaceURI) {
    Node_Del(attr);
    return NULL;
  }

  attr->prefix = InternUnicode(doc, prefix);
  if (!attr->prefix) {
    Py_DECREF(attr->namespaceURI);
    Node_Del(attr);
    return NULL;
  }

  attr->localName = InternUnicode(doc, localName);
  if (!attr->localName) {
    Py_DECREF(attr->namespaceURI);
    Py_DECREF(attr->prefix);
    Node_Del(attr);
    return NULL;
  }

  attr->nodeName = InternUnicode(doc, qualifiedName);
  if (!attr->nodeName) {
    Py_DECREF(attr->namespaceURI);
    Py_DECREF(attr->prefix);
    Py_DECREF(attr->localName);
    Node_Del(attr);
    return NULL;
  }

  if (value == NULL)
    attr->nodeValue = UNICODE_EMPTY_STRING;
  else
    attr->nodeValue = InternUnicode(doc, value);
  if (!attr->nodeValue) {
    Py_DECREF(attr->namespaceURI);
    Py_DECREF(attr->prefix);
    Py_DECREF(attr->localName);
    Py_DECREF(attr->nodeName);
    Node_Del(attr);
    return NULL;
  }

  /* Attribute indices are handled by their ownerElement */
  attr->docIndex = 0;

  PyObject_GC_Track(attr);

  return attr;
}


PyTextObject *Document_CreateTextNode(PyDocumentObject *doc,
                                      PyObject *data,
                                      unsigned long *docIx)
{
  PyTextObject *text;

  if (!PyDocument_Check(doc)) {
    PyErr_BadInternalCall();
    return NULL;
  }

  text = Node_New(PyTextObject, &PyDomletteText_Type, doc);
  if (text == NULL)
    return NULL;

  text->nodeValue = InternUnicode(doc, data);
  if (text->nodeValue == NULL) {
    Node_Del(text);
    return NULL;
  }

  text->docIndex = (*docIx)++;

  PyObject_GC_Track(text);

  return text;
}


PyCommentObject *Document_CreateComment(PyDocumentObject *doc,
                                        PyObject *data,
                                        unsigned long *docIx)
{
  PyCommentObject *comment;

  if (!PyDocument_Check(doc)) {
    PyErr_BadInternalCall();
    return NULL;
  }

  comment = Node_New(PyCommentObject, &PyDomletteComment_Type, doc);
  if (comment == NULL)
    return NULL;

  comment->nodeValue = InternUnicode(doc, data);
  if (comment->nodeValue == NULL) {
    Node_Del(comment);
    return NULL;
  }

  comment->docIndex = (*docIx)++;

  PyObject_GC_Track(comment);

  return comment;
}


PyProcessingInstructionObject *Document_CreateProcessingInstruction(
                                    PyDocumentObject *doc,
                                    PyObject *target,
                                    PyObject *data,
				    unsigned long *docIx)
{
  PyProcessingInstructionObject *pi;

  if (!PyDocument_Check(doc)) {
    PyErr_BadInternalCall();
    return NULL;
  }

  pi = Node_New(PyProcessingInstructionObject,
		&PyDomletteProcessingInstruction_Type, doc);
  if (pi == NULL)
    return NULL;

  pi->nodeName = InternUnicode(doc, target);
  if (pi->nodeName == NULL) {
    Node_Del(pi);
    return NULL;
  }

  pi->nodeValue = InternUnicode(doc, data);
  if (pi->nodeValue == NULL) {
    Py_DECREF(pi->nodeName);
    Node_Del(pi);
    return NULL;
  }

  pi->docIndex = (*docIx)++;

  PyObject_GC_Track(pi);

  return pi;
}


PyDocumentFragmentObject *
Document_CreateDocumentFragment(PyDocumentObject *doc)
{
  /*Returns new reference*/
  PyDocumentFragmentObject *df;

  if (!PyDocument_Check(doc)) {
    PyErr_BadInternalCall();
    return NULL;
  }

  df = Node_New(PyDocumentFragmentObject, &PyDomletteDocumentFragment_Type,
		doc);
  if (df == NULL)
    return NULL;

  df->childNodes = PyList_New(0);
  if (df->childNodes == NULL) {
    Node_Del(df);
    return NULL;
  }

  df->docIndex = 0;

  PyObject_GC_Track(df);

  return df;
}

/** Python API *********************************************************/

static char PyDocument_CreateElementNS__doc__[] =
"createElementNS(namespaceURI, qualifiedName) -> new Element\n\
\n\
Creates an element of the given qualified name and namespace URI.";

static PyObject *PyDocument_CreateElementNS(PyObject *self, PyObject *args)
{
  PyObject *res;
  PyObject *namespaceURI, *qualifiedName, *localName, *prefix;

  if(!PyArg_ParseTuple(args, "OO:createElementNS",
		       &namespaceURI, &qualifiedName))
    return NULL;

  namespaceURI = DOMString_ConvertArgument(namespaceURI, "namespaceURI", 1);
  if (namespaceURI == NULL) return NULL;

  qualifiedName = DOMString_ConvertArgument(qualifiedName, "qualifiedName", 0);
  if (qualifiedName == NULL) {
    Py_DECREF(namespaceURI);
    return NULL;
  }

  if (!SplitQName(qualifiedName, &prefix, &localName)) {
    Py_DECREF(namespaceURI);
    Py_DECREF(qualifiedName);
    return NULL;
  }

  res = (PyObject *)Document_CreateElementNS((PyDocumentObject *)self,
                                             namespaceURI, qualifiedName, 
                                             prefix, localName,
                                             &PyDocument_INDEX(self));
  Py_DECREF(namespaceURI);
  Py_DECREF(qualifiedName);
  Py_DECREF(prefix);
  Py_DECREF(localName);
  return res;
}


static char PyDocument_CreateAttributeNS__doc__[] =
"createAttributeNS(namespaceURI, qualifiedName) -> new Attribute\n\
\n\
Creates an attribute of the given qualified name and namespace URI.";

static PyObject *PyDocument_CreateAttributeNS(PyObject *self, PyObject *args)
{
  PyObject *res;
  PyObject *namespaceURI, *qualifiedName, *localName, *prefix;

  if (!PyArg_ParseTuple(args, "OO:createAttributeNS",
			&namespaceURI, &qualifiedName))
    return NULL;

  namespaceURI = DOMString_ConvertArgument(namespaceURI, "namespaceURI", 1);
  if (namespaceURI == NULL) return NULL;

  qualifiedName = DOMString_ConvertArgument(qualifiedName, "qualifiedName", 0);
  if (qualifiedName == NULL) {
    Py_DECREF(namespaceURI);
    return NULL;
  }

  if (!SplitQName(qualifiedName, &prefix, &localName)) {
    Py_DECREF(namespaceURI);
    Py_DECREF(qualifiedName);
    return NULL;
  }

  res = (PyObject *)Document_CreateAttributeNS((PyDocumentObject *)self,
                                               namespaceURI, qualifiedName, 
                                               prefix, localName, NULL);

  Py_DECREF(namespaceURI);
  Py_DECREF(qualifiedName);
  Py_DECREF(prefix);
  Py_DECREF(localName);
  return res;
}


static char PyDocument_CreateTextNode__doc__[] =
"createTextNode(data) -> new Text\n\
\n\
Creates a Text node given the specified string.";

static PyObject *PyDocument_CreateTextNode(PyObject *self, PyObject *args)
{
  PyObject *res;
  PyObject *data;

  if(!PyArg_ParseTuple(args, "O:createTextNode", &data))
    return NULL;

  data = DOMString_ConvertArgument(data, "data", 0);
  if (data == NULL) return NULL;

  res = (PyObject *)Document_CreateTextNode((PyDocumentObject *)self, data,
                                            &PyDocument_INDEX(self));
  Py_DECREF(data);
  return res;
}


static char PyDocument_CreateProcessingInstruction__doc__[] =
"createProcessingInstruction(target, data) -> new ProcessingInstruction\n\
\n\
Creates a ProcessingInstruction node given the specified name and data\n\
strings.";

static PyObject *PyDocument_CreateProcessingInstruction(PyObject *self,
							PyObject *args)
{
  PyProcessingInstructionObject *pi;
  PyObject *target, *data;

  if (!PyArg_ParseTuple(args,"OO:createProcessingInstruction", &target, &data))
    return NULL;

  target = DOMString_ConvertArgument(target, "target", 0);
  if (target == NULL) return NULL;

  data = DOMString_ConvertArgument(data, "data", 0);
  if (data == NULL) {
    Py_DECREF(target);
    return NULL;
  }

  pi = Document_CreateProcessingInstruction((PyDocumentObject *)self, target,
					    data, &PyDocument_INDEX(self));
  Py_DECREF(data);
  Py_DECREF(target);
  return (PyObject *)pi;
}


static char PyDocument_CreateComment__doc__[] =
"createComment(data) -> new Comment\n\
\n\
Creates a Comment node given the specified string.";

static PyObject *PyDocument_CreateComment(PyObject *self, PyObject *args)
{
  PyCommentObject *comment;
  PyObject *data;

  if (!PyArg_ParseTuple(args, "O:createComment", &data))
    return NULL;

  data = DOMString_ConvertArgument(data, "data", 0);
  if (data == NULL) return NULL;

  comment = Document_CreateComment((PyDocumentObject *)self, data,
				   &PyDocument_INDEX(self));
  Py_DECREF(data);
  return (PyObject *)comment;
}

static char PyDocument_CreateDocumentFragment__doc__[] =
"createDocumentFragment() -> new DocumentFragment\n\
\n\
Creates an empty DocumentFragment object.";

static PyObject *PyDocument_CreateDocumentFragment(PyObject *self,
						   PyObject *args)
{
  if (!PyArg_ParseTuple(args, ":createDocumentFragment"))
    return NULL;

  return (PyObject *)Document_CreateDocumentFragment((PyDocumentObject *)self);
}

static char PyDocument_ImportNode__doc__[] =
"importNode(importedNode, deep) -> Node\n\
\n\
Imports a node from another document to this document. The returned node\n\
has no parent; (parentNode is None). The source node is not altered or\n\
removed from the original document; this method creates a new copy of the\n\
source node.";

static PyObject *PyDocument_ImportNode(PyObject *self, PyObject *args)
{
  PyObject *node, *boolean_deep = Py_False;
  int deep;
    
  if (!PyArg_ParseTuple(args,"O|O:importNode", &node, &boolean_deep))
    return NULL;

  deep = PyObject_IsTrue(boolean_deep);
  if (deep == -1)
    return NULL;

  return (PyObject *)Node_CloneNode(node, deep, (PyNodeObject *)self);
}


static struct PyMethodDef Document_methods[] = {
  NODE_METHODS,

  { "createElementNS", PyDocument_CreateElementNS, METH_VARARGS,
    PyDocument_CreateElementNS__doc__ },
  { "createAttributeNS", PyDocument_CreateAttributeNS, METH_VARARGS,
    PyDocument_CreateAttributeNS__doc__ },
  { "createTextNode", PyDocument_CreateTextNode, METH_VARARGS,
    PyDocument_CreateTextNode__doc__ },
  { "createProcessingInstruction", PyDocument_CreateProcessingInstruction,
    METH_VARARGS, PyDocument_CreateProcessingInstruction__doc__ },
  { "createComment", PyDocument_CreateComment, METH_VARARGS,
    PyDocument_CreateComment__doc__ },
  { "createDocumentFragment", PyDocument_CreateDocumentFragment, METH_VARARGS,
    PyDocument_CreateDocumentFragment__doc__ },
  { "importNode", PyDocument_ImportNode, METH_VARARGS,
    PyDocument_ImportNode__doc__ },

  { NULL, NULL}      /* sentinel */
};


/** Python Slot Methods ************************************************/

static PyObject *document_getattr(PyDocumentObject *self, char *name)
{
  PyObject *rt = NULL;
  int temp;

  if (!strcmp(name, "ownerDocument")) {
    rt = Py_None;
  }
  else if (!strcmp(name, "rootNode")) {
    rt = (PyObject *)self;
  }
  else if (!strcmp(name, "documentElement")) {
    int i;
    rt = Py_None;
    for (i = 0; i < PyList_GET_SIZE(self->childNodes); i++) {
      PyObject *child = PyList_GET_ITEM(self->childNodes, i);
      if (PyElement_Check(child)) {
	rt = child;
	break;
      }
    }
  }
  else if (!strcmp(name, "implementation")) {
    rt = g_implementation;
  }
  else if (!strcmp(name, "doctype")) {
    rt = Py_None;
  }
  else if (!strcmp(name, "publicId")) {
    rt = self->publicId;
  }
  else if (!strcmp(name, "systemId")) {
    rt = self->systemId;
  }
  /* DOM3 documentURI - The location of the document or null if undefined. */
  else if (!strcmp(name, "documentURI") ||
	   !strcmp(name, "baseURI") ||
	   !strcmp(name, "refUri")) {
    rt = self->documentUri;
  }
  else if (!strcmp(name,"nodeType")) {
    rt = my_node_type;
  }
  else if (!strcmp(name,"nodeName")) {
    rt = my_node_name;
  }
  else if (!strcmp(name,"childNodes")) {
    rt = self->childNodes;
  }
  else if (!strcmp(name,"lastChild")) {
    temp = PyList_GET_SIZE(self->childNodes);
    if (temp) {
      rt = PyList_GET_ITEM(self->childNodes,temp-1);
    } else {
      rt = Py_None;
    }
  }
  else if (!strcmp(name,"firstChild")) {
    if (PyList_GET_SIZE(self->childNodes)) {
      rt = PyList_GET_ITEM(self->childNodes,0);
    } else {
      rt = Py_None;
    }
  }
  else if (!strcmp(name, "unparsedEntities")) {
    rt = self->unparsedEntities;
  }
  if (rt) {
    Py_INCREF(rt);
    return rt;
  }
  return node_getattr((PyNodeObject*)self, name, Document_methods);
}


static int document_setattr(PyDocumentObject *self, char *name, PyObject *v)
{
  PyObject *value;

  /* Set attribute 'name' to value 'v'. v==NULL means delete */
  if (v == NULL) {
    PyErr_Format(PyExc_AttributeError,
                 "Cannot delete attribute '%.400s' on '%.50s' object",
                 name, self->ob_type->tp_name);
  }
  else if (PyDocument_BASE_URI(self) == Py_None &&
	   (!strcmp(name, "baseURI") || !strcmp(name, "documentURI"))) {
    if ((value = DOMString_ConvertArgument(v, name, 0)) == NULL) return -1;

    Py_DECREF(PyDocument_BASE_URI(self));
    PyDocument_BASE_URI(self) = value;
    return 0;
  }
  else if (strcmp(name, "publicId") == 0) {
    if ((value = DOMString_ConvertArgument(v, name, 0)) == NULL) return -1;

    Py_DECREF(self->publicId);
    self->publicId = value;
    return 0;
  }
  else if (strcmp(name, "systemId") == 0) {
    if ((value = DOMString_ConvertArgument(v, name, 0)) == NULL) return -1;

    Py_DECREF(self->systemId);
    self->systemId = value;
    return 0;
  }
  else {
    PyErr_Format(PyExc_AttributeError,
                 "Cannot set attribute '%.400s' on '%.50s' object",
                 name, self->ob_type->tp_name);
  }
  return -1;
}


static void document_dealloc(PyDocumentObject *self)
{
  PyObject_GC_UnTrack((PyObject *) self);

  Py_XDECREF(self->documentUri);
  self->documentUri = NULL;

  Py_XDECREF(self->unparsedEntities);
  self->unparsedEntities = NULL;

  if (self->internedUnicode) {
    PyDict_Clear(self->internedUnicode);
    Py_DECREF(self->internedUnicode);
    self->internedUnicode = NULL;
  }

  Py_XDECREF(self->childNodes);
  self->childNodes = NULL;

  Node_Del(self);
}


static int document_clear(PyDocumentObject *self)
{
  (void) PyList_SetSlice(self->childNodes, 0, -1, 0);

  return node_clear((PyNodeObject *)self);
}


static int document_traverse(PyDocumentObject *self, visitproc visit,
			     void *arg)
{
  int rt;

  if (self->childNodes != NULL) {
    rt = visit(self->childNodes, arg);
    if (rt) return rt;
  }

  /* Node */
  return node_traverse((PyNodeObject *)self,visit,arg);
}


static char *document_format_str = "<cDocument at 0x%p>";

static PyObject *document_repr(PyDocumentObject *document)
{
  char buf[256];
  sprintf(buf, document_format_str, document);
  return PyString_FromString(buf);
}


/** Testing ************************************************************/

int document_test_ref_counts(PyObject *tester, PyDocumentObject *doc, int base)
{
  /*To test a document, we must make sure that the ref counts to our self are correct
    To do this test (and count as we go) the number of children and attrs in the tree.  Also
    It is our job to test that the interned strings have the correct ref count */
  long childCount = 0;
  PyObject *internCtr = PyDict_New();
  int ctr;
  PyObject *curChild;

#if 0 && !defined(DISABLE_INTERN)
  char buff[512];
  PyObject *v1, *v2, *temp;
  int i;
#endif

  for (ctr=0;ctr<PyList_GET_SIZE(doc->childNodes);ctr++) {
    curChild = PyList_GET_ITEM(doc->childNodes,ctr);
    if (!node_test_ref_counts(tester, (PyNodeObject *)curChild, &childCount,
			      internCtr, 0)) {
      Py_DECREF(internCtr);
      return 0;
    };
  }

  PyObject_CallMethod(tester,"startTest","s","Node");
  /*Base additional*/
  if (!PyObject_CallMethod(tester,"compare","ll",childCount + base + PyList_GET_SIZE(doc->childNodes),doc->ob_refcnt)){
    Py_DECREF(internCtr);
    return 0;
  }
  PyObject_CallMethod(tester,"testDone","");

#if 0 && !defined(DISABLE_INTERN)
  PyObject_CallMethod(tester,"startTest","s","Intern Ref Count");
  i = 0;
  while (PyDict_Next(doc->internedUnicode, &i, &v1, &v2)) {
    /*Each entry in here will be 3 more then in the count.  2 for key,value in dict and one for key in the count dict*/
    temp = PyDict_GetItem(internCtr,v1);
    if (temp == NULL) {
      sprintf(buff,"String %s was in interned dict but not in the couting dict",PyString_AS_STRING(PyObject_Repr((PyObject *)v1)));
      PyObject_CallMethod(tester,"error","s",buff);
      return 0;
    }
    /*Make sure the internCtr + 3 = refCount of v1*/
    if (!PyObject_CallMethod(tester,"compare","lO",PyInt_AsLong(temp)+3,PyInt_FromLong((long)v1->ob_refcnt))) {
      return 0;
    }
  }

  printf("Check Backwards\n");

  PyObject_CallMethod(tester,"testDone","");
#endif

  Py_DECREF(internCtr);
  return 1;
}


/** Type Object ********************************************************/

static char document_doc[] =
"The Document interface represents the entire XML document. Conceptually,\n\
it is the root of the document tree, and provides the primary access to the\n\
document's data.";

PyTypeObject PyDomletteDocument_Type = {
  /* PyObject_HEAD  */ PyObject_HEAD_INIT(NULL)
  /* ob_size        */ 0,
  /* tp_name        */ "cDomlette.Document",
  /* tp_basicsize   */ sizeof(PyDocumentObject),
  /* tp_itemsize    */ 0,
  /* tp_dealloc     */ (destructor) document_dealloc,
  /* tp_print       */ (printfunc) 0,
  /* tp_getattr     */ (getattrfunc) document_getattr,
  /* tp_setattr     */ (setattrfunc) document_setattr,
  /* tp_compare     */ (cmpfunc) 0,
  /* tp_repr        */ (reprfunc) document_repr,
  /* tp_as_number   */ (PyNumberMethods *) 0,
  /* tp_as_sequence */ (PySequenceMethods *) 0,
  /* tp_as_mapping  */ (PyMappingMethods *) 0,
  /* tp_hash        */ (hashfunc) 0,
  /* tp_call        */ (ternaryfunc) 0,
  /* tp_str         */ (reprfunc) 0,
  /* tp_getattro    */ (getattrofunc) 0,
  /* tp_setattro    */ (setattrofunc) 0,
  /* tp_as_buffer   */ (PyBufferProcs *) 0,
  /* tp_flags       */ Py_TPFLAGS_DEFAULT | Py_TPFLAGS_HAVE_GC,
  /* tp_doc         */ (char *) document_doc,
  /* tp_traverse    */ (traverseproc) document_traverse,
  /* tp_clear       */ (inquiry) document_clear,
};

int DomletteDocument_Init(void)
{
  PyDomletteDocument_Type.ob_type = &PyType_Type;

  my_node_type = PyInt_FromLong(DOCUMENT_NODE);
  if (my_node_type == NULL) return 0;

  my_node_name = PyUnicode_DecodeASCII("#document", 9, NULL);
  if (my_node_name == NULL) return 0;

  return 1;
}

void DomletteDocument_Fini(void)
{
  Py_DECREF(my_node_type);
  Py_DECREF(my_node_name);
}
