#include "domlette.h"
/*
   External Methods
*/

#define Attr_VerifyState(ob)                          \
  if (!PyAttr_Check(ob) ||                            \
      ((PyAttrObject *)(ob))->nodeValue == NULL ||    \
      ((PyAttrObject *)(ob))->namespaceURI == NULL || \
      ((PyAttrObject *)(ob))->prefix == NULL ||       \
      ((PyAttrObject *)(ob))->localName == NULL ||    \
      ((PyAttrObject *)(ob))->nodeName == NULL)       \
     return DOMException_InvalidStateErr("Attr in inconsistent state");

static PyObject *my_node_type;

PyAttrObject *Attr_CloneNode(PyObject *node, int deep,
                             PyNodeObject *newOwnerDocument)
{
  PyObject *namespaceURI, *qualifiedName, *prefix, *localName, *value;
  PyAttrObject *attr;
  
  namespaceURI = PyObject_GetAttrString(node, "namespaceURI");
  namespaceURI = DOMString_FromObjectInplace(namespaceURI);
  qualifiedName = PyObject_GetAttrString(node, "nodeName");
  qualifiedName = DOMString_FromObjectInplace(qualifiedName);
  prefix = PyObject_GetAttrString(node, "prefix");
  prefix = DOMString_FromObjectInplace(prefix);
  localName = PyObject_GetAttrString(node, "localName");
  localName = DOMString_FromObjectInplace(localName);
  value = PyObject_GetAttrString(node, "value");
  value = DOMString_FromObjectInplace(value);
  if (namespaceURI == NULL || qualifiedName == NULL || prefix == NULL ||
      localName == NULL || value == NULL) {
    Py_XDECREF(value);
    Py_XDECREF(localName);
    Py_XDECREF(prefix);
    Py_XDECREF(qualifiedName);
    Py_XDECREF(namespaceURI);
    return NULL;
  }

  attr = Document_CreateAttributeNS((PyDocumentObject*) newOwnerDocument,
                                    namespaceURI, qualifiedName, prefix, 
                                    localName, value);
  Py_DECREF(value);
  Py_DECREF(localName);
  Py_DECREF(prefix);
  Py_DECREF(qualifiedName);
  Py_DECREF(namespaceURI);
  return attr;
}


static struct PyMethodDef Attr_methods[] = { NODE_METHODS,
  {NULL,     NULL}      /* sentinel */
};

/*
  Methods for our type
*/

static PyObject *attr_getattr(PyAttrObject *self, char *name)
     /* Returns a new ref */
{
  PyObject *rt = NULL;

  Attr_VerifyState(self);

  if (!strcmp(name, "name")) {
    rt = (PyObject *)self->nodeName;
  }
  else if (!strcmp(name,"childNodes")) {
    /* XPath Data Model - attributes do not have children */
    return PyList_New(0);
  }
  else if (!strcmp(name,"lastChild") || !strcmp(name,"firstChild") ) {
    rt = Py_None;
  }
  else if (!strcmp(name, "nodeName")) {
    rt = (PyObject *)self->nodeName;
  }
  else if (!strcmp(name, "namespaceURI")) {
    rt = self->namespaceURI;
  }
  else if (!strcmp(name, "prefix")) {
    rt = (PyObject *)self->prefix;
  }
  else if (!strcmp(name, "localName")) {
    rt = (PyObject *)self->localName;
  }
  else if (!strcmp(name, "value")) {
    rt = (PyObject *)self->nodeValue;
  }
  else if (!strcmp(name, "nodeValue")) {
    rt = (PyObject *)self->nodeValue;
  }
  else if (!strcmp(name, "ownerElement") || !strcmp(name, "parentNode")) {
    /* XPath Data Model - attributes parent is the element containing it */
    rt = self->parentNode;
  }
  else if (!strcmp(name, "nodeType")) {
    rt = my_node_type;
  }
  else if (!strcmp(name, "specified")) {
    /* Until the DTD information is used, assume it was from the document */
    return PyInt_FromLong(1);
  }

  if (rt) {
    Py_INCREF(rt);
    return rt;
  }
  return node_getattr((PyNodeObject*)self, name, Attr_methods);
}

static void attr_dealloc(PyAttrObject *node)
{

  PyObject_GC_UnTrack((PyObject *) node);

  Py_XDECREF(node->namespaceURI);
  node->namespaceURI = NULL;

  Py_XDECREF(node->prefix);
  node->prefix = NULL;

  Py_XDECREF(node->localName);
  node->localName = NULL;

  Py_XDECREF(node->nodeName);
  node->nodeName = NULL;

  Py_XDECREF(node->nodeValue);
  node->nodeValue = NULL;

  Node_Del(node);
}

static int attr_clear(PyAttrObject *self)
{
  return node_clear((PyNodeObject *)self);
}

static int attr_traverse(PyAttrObject *self, visitproc visit, void *arg)
{
  return node_traverse((PyNodeObject *)self,visit,arg);
}


static PyObject *attr_repr(PyAttrObject *attr)
{
  char buf[256];
  PyObject *name = PyObject_Repr((PyObject *)(attr->nodeName));
  PyObject *value = PyObject_Repr((PyObject *)(attr->nodeValue));
  sprintf(buf, "<cAttr at %p: name %.50s, value %.100s>",
          attr, PyString_AS_STRING(name), PyString_AS_STRING(value));
  Py_DECREF(name);
  Py_DECREF(value);
  return PyString_FromString(buf);
}

int attr_test_ref_counts(PyObject *tester,PyAttrObject *node,long *childCtr,PyObject *internDict,int base) {
  /*Ref count is base + 1.  Out nodeValue is interned*/

  PyObject_CallMethod(tester,"startTest","s","Attr Node Ref Count");
  if (!PyObject_CallMethod(tester,"compare","ll",base+1,node->ob_refcnt)) return 0;
  PyObject_CallMethod(tester,"testDone","");

#ifdef DISABLE_INTERN
  /*Namespace and local name are two because they are used in our parents key*/
  /*nodeName and localName could be the same*/

  /*if (!TestRefCount(tester,(PyObject *)node->nodeValue,1, "nodeValue")) return 0;*/

  if (node->namespaceURI != Py_None)
    if (!TestRefCount(tester,(PyObject *)node->namespaceURI,2, "namespaceURI")) return 0;

  /*if (PyUnicode_GET_SIZE(node->prefix) != 0) {*/
    /*Prefixes get shared sooo much they can only really be tested in Intern
    if (node->prefix != g_xmlnsUnicode)
    if (!TestRefCount(tester,(PyObject *)node->prefix,1, "prefix")) return 0;*/
    /*if (!TestRefCount(tester,(PyObject *)node->nodeName,1, "nodeName")) return 0;*/
    /*if (!TestRefCount(tester,(PyObject *)node->localName,2, "localName")) return 0;*/
    /*} else {
    if (!TestRefCount(tester,(PyObject *)node->nodeName,3, "nodeName")) return 0;
    if (!TestRefCount(tester,(PyObject *)node->localName,3, "localName")) return 0;
    }*/
#else
  PyObject_CallMethod(tester,"startTest","s","Add Intern Counters");
  AddInternCtr((PyObject *)node->nodeValue,internDict);
  AddInternCtr(node->namespaceURI,internDict);
  AddInternCtr((PyObject *)node->prefix,internDict);
  AddInternCtr((PyObject *)node->localName,internDict);
  AddInternCtr((PyObject *)node->nodeName,internDict);
  PyObject_CallMethod(tester,"testDone","");
#endif
  return 1;
}

static int
attr_setattr(PyAttrObject *self, char *name, PyObject *v)
{
  /* Set attribute 'name' to value 'v'. v==NULL means delete */
  if (v == NULL) {
    PyErr_Format(PyExc_AttributeError,
                 "Cannot delete attribute '%.400s' on '%.50s' object",
                 name, self->ob_type->tp_name);
    return -1;
  }

  if (strcmp(name, "value") == 0 || strcmp(name, "nodeValue") == 0) {
    PyObject *nodeValue = DOMString_ConvertArgument(v, name, 0);
    if (nodeValue == NULL) return -1;

    Py_DECREF(self->nodeValue);
    self->nodeValue = nodeValue;
  } else {
    PyErr_Format(PyExc_AttributeError,
                 "Cannot set attribute '%.400s' on '%.50s' object",
                 name, self->ob_type->tp_name);
    return -1;
  }
  return 0;
}

/*
  Our type
 */

PyTypeObject PyDomletteAttr_Type = {
    PyObject_HEAD_INIT(0)
    0,
    "cDomlette.Attr",
    GC_TP_BASICSIZE(PyAttrObject),
    0,
    (destructor)attr_dealloc,   /*tp_dealloc*/
    (printfunc)0,          /*tp_print*/
    (getattrfunc)attr_getattr,  /*tp_getattr*/
    (setattrfunc)attr_setattr,  /*tp_setattr*/
    0,                          /*tp_compare*/
    (reprfunc)attr_repr,        /*tp_repr*/
    0,                          /*tp_as_number*/
    0,              /*tp_as_sequence*/
    0,              /*tp_as_mapping*/
    0,                             /*tp_hash*/
    0,          /*tp_call*/
    0,          /*tp_str*/
    0,                      /*tp_getattro*/
    0,          /*tp_setattro*/
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_HAVE_GC,
    0,                          /* tp_doc */
    (traverseproc)attr_traverse,  /* tp_traverse */
    (inquiry)attr_clear,          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
};

int DomletteAttr_Init(void)
{
  PyDomletteAttr_Type.ob_type = &PyType_Type;

  my_node_type = PyInt_FromLong(ATTRIBUTE_NODE);
  if (my_node_type == NULL) return 0;

  return 1;
}

void DomletteAttr_Fini(void)
{
  Py_DECREF(my_node_type);
}
