########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Xml/Xslt/StylesheetHandler.py,v 1.43 2004/09/06 23:37:08 mbrown Exp $
"""
Stylesheet tree generator

Copyright 2004 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

import operator, urllib, time, warnings
from Ft.Lib import Truncate, UriException
from Ft.Xml import XML_NAMESPACE, EMPTY_NAMESPACE
from Ft.Xml import Xslt
from Ft.Xml.Domlette import XmlStrStrip
from Ft.Xml.XInclude import XINCLUDE_NAMESPACE, XIncludeException
from Ft.Xml.Xslt import XSL_NAMESPACE, MessageSource
from Ft.Xml.Xslt import XsltException, XsltParserException, Error
from Ft.Xml.Xslt import CategoryTypes, BuiltInExtElements
from Ft.Xml.Xslt import Exslt
from Ft.Xml.InputSource import DefaultFactory
from Ft.Xml import ReaderException

from LiteralElement import LiteralElement

import StylesheetTree, ContentInfo, AttributeInfo


# Space is definitely illegal in a element name
NS_SPLIT_CHAR = ' '

XML_SPACE = XML_NAMESPACE + NS_SPLIT_CHAR + 'space'
XINCLUDE_NAME = XINCLUDE_NAMESPACE + NS_SPLIT_CHAR + 'include'

# Table for load-on-demand of the XSLT elements
_ELEMENT_MAPPING = {
    'apply-templates' : 'ApplyTemplatesElement.ApplyTemplatesElement',
    'apply-imports' : 'ApplyImportsElement.ApplyImportsElement',
    'attribute' : 'AttributeElement.AttributeElement',
    'attribute-set' : 'AttributeSetElement.AttributeSetElement',
    'call-template' : 'CallTemplateElement.CallTemplateElement',
    'choose' : 'ChooseElement.ChooseElement',
    'when' : 'ChooseElement.WhenElement',
    'otherwise' : 'ChooseElement.OtherwiseElement',
    'copy' : 'CopyElement.CopyElement',
    'copy-of' : 'CopyOfElement.CopyOfElement',
    'comment' : 'CommentElement.CommentElement',
    'element' : 'ElementElement.ElementElement',
    'for-each' : 'ForEachElement.ForEachElement',
    'if' : 'IfElement.IfElement',
    'message' : 'MessageElement.MessageElement',
    'number' : 'NumberElement.NumberElement',
    'param' : 'ParamElement.ParamElement',
    'processing-instruction' : 'ProcessingInstructionElement.ProcessingInstructionElement',
    'sort' : 'SortElement.SortElement',
    'stylesheet' : 'Stylesheet.StylesheetElement',
    'transform' : 'Stylesheet.StylesheetElement',
    'template' : 'TemplateElement.TemplateElement',
    'text' : 'TextElement.TextElement',
    'variable' : 'VariableElement.VariableElement',
    'value-of' : 'ValueOfElement.ValueOfElement',
    'with-param' : 'WithParamElement.WithParamElement',
    'import' : 'OtherXslElement.ImportElement',
    'include' : 'OtherXslElement.IncludeElement',
    'decimal-format' : 'OtherXslElement.DecimalFormatElement',
    'key' : 'OtherXslElement.KeyElement',
    'namespace-alias' : 'OtherXslElement.NamespaceAliasElement',
    'output' : 'OtherXslElement.OutputElement',
    'fallback' : 'OtherXslElement.FallbackElement',
    'preserve-space' : 'WhitespaceElements.PreserveSpaceElement',
    'strip-space' : 'WhitespaceElements.StripSpaceElement',
    }

# The XSL attributes allowed on literal elements
_RESULT_ELEMENT_XSL_ATTRS = {
    'exclude-result-prefixes' : AttributeInfo.Prefixes(),
    'extension-element-prefixes' : AttributeInfo.Prefixes(),
    'use-attribute-sets' : AttributeInfo.QNames(),
    'version' : AttributeInfo.Number(),
    }

# The XSL attributes allowed on extension elements
_EXT_ELEMENT_XSL_ATTRS = {
    'extension-element-prefixes' : AttributeInfo.Prefixes(),
    }

# helper function
def _SplitName(name):
    index = name.rfind(NS_SPLIT_CHAR)
    if index == -1:
        # no namespace, name is the localName
        namespace = None
        local = name
    else:
        namespace = name[:index]
        local = name[index+1:]
    return (namespace, local)


class StylesheetHandler:
    """
    Handles SAX-like events coming from the stylesheet parser,
    in order to build the stylesheet tree.
    """

    def __init__(self, importIndex=0, globalVars=None, extElements=None,
                 visitedStyUris=None):
        self._import_index = importIndex
        self._extElements = BuiltInExtElements.ExtElements.copy()
        self._extElements.update(Exslt.ExtElements)
        self._processXIncludes = True
        #_ignoreMarkup is designed as a flag for ignoring markup
        #inside xsl:import, xsl:include and xi:include elements
        #The code that sets it is currently disabled because of problems
        #reusing the handler/reader instance when set
        self._ignoreMarkup = 0
        if extElements:
            self._extElements.update(extElements)

        if globalVars is None:
            # We need to make sure that the same dictionary is used
            # through the entire processing (even if empty)
            self._global_vars = {}
        else:
            self._global_vars = globalVars

        self._visited_stylesheet_uris = visitedStyUris or {}
        return

    def reset(self):
        self._global_vars = {}
        self._import_index = 0
        self._ignoreMarkup = 0
        self._visited_stylesheet_uris = {}
        return

    def clone(self):
        return self.__class__(self._import_index, self._global_vars,
                              self._extElements, self._visited_stylesheet_uris)

    def startDocument(self):
        """
        ownerDoc is supplied when processing an XSLT import or include.
        """
        # Our root is always a document
        # We use a document for this because of error checking and
        # because we explicitly pass ownerDocument to the nodes as
        # they are created
        root = StylesheetTree.XsltRoot(self._input_source.uri)

        if not self._ownerDoc:
            self._ownerDoc = root

        self._node_stack = [root]
        self._validation = [root.validator.getValidation()]
        self._local_vars = [{}]

        # Stylesheets ignore any whitespace expect in xsl:text or elements
        # with an xml:space='preserve'
        self._preserveSpaceStack = [0]

        # for external entities and XIncludes
        self._entity_stack = []

        # for recursive include checks for xsl:include/xsl:import
        self._visited_stylesheet_uris[self._input_source.uri] = True

        # for recursive include checks for XIncludes
        self._visited_xinclude_hrefs = []

        # Initial set of namespaces is the xml and null namespaces
        self._namespaces = [{'xml' : XML_NAMESPACE,
                             None : None,
                             }]
        self._reverseNss = [{XML_NAMESPACE : 'xml',
                             None : None,
                             }]

        # the literal text
        self._whole_text = []

        # namespace uris for extension elements and excluded result prefixes
        # format: [(extension_nss, exclude_nss)...]
        self._processing_nss = [([], [XML_NAMESPACE, XSL_NAMESPACE])]
        return

    def endDocument(self):
        return

    def externalEntityRef(self, context, base, sysid, pubid):
        self._entity_stack.append(self.parser)
        self.parser = self.parser.ExternalEntityParserCreate(context)

        # default behaviour is to use the SystemId as the source
        # source = urllib.basejoin(base, sysid)
        # stream = Uri.UrlOpen(source)
        stream = self.resolve(sysid, base)
        self.parser.SetBase(self.normalize(sysid, base))
        self.parser.ParseFile(stream)
        stream.close()

        self.parser = self._entity_stack.pop()
        return 1

    def startNamespaceDecl(self, prefix, uri):
        try:
            # setup prefix to uri lookup
            nss = self._namespaces[-1].copy()
            self._namespaces.append(nss)
            nss[prefix] = uri
            # Do the same for the uri to prefix lookup
            reverse = self._reverseNss[-1].copy()
            self._reverseNss.append(reverse)
            reverse[uri] = prefix
        except:
            import traceback
            traceback.print_exc()
        return

    def endNamespaceDecl(self, prefix):
        del self._namespaces[-1]
        del self._reverseNss[-1]
        return

    def _debug_validation(self, token=None, next=None):
        from pprint import pprint
        parent = self._node_stack[-1]
        print '='*60
        print 'content expression =', parent.validator
        print 'initial validation'
        pprint (parent.validator.getValidation())
        print 'current validation'
        pprint (self._validation[-1])
        if token:
            print 'token', token
        if next:
            print 'next validation'
            pprint (next)
        print '='*60
        return

    def _make_display_name(self, namespace, local):
        prefix = self._reverseNss[-1].get(namespace)
        if prefix:
            qname = u':'.join([prefix, local])
        else:
            qname = local
        return qname

    def startElement(self, name, attribs, *args):

        if self._ignoreMarkup: return
        self._whole_text and self._completeTextNode(self._node_stack[-1])

        # check for XInclude
        if self._processXIncludes and name == XINCLUDE_NAME:
            self._handle_xinclude(attribs)
            # throw away this element
            self._node_stack.append(None)
            #self._ignoreMarkup = 1
            return

        #[start inline] (namespace, local) = _SplitName(name)
        index = name.rfind(NS_SPLIT_CHAR)
        if index == -1:
            # no namespace, name is the localName
            namespace = None
            local = name
            if local.rfind(":") != -1:
                index = local.rfind(":")
                raise ReaderException(ReaderException.XMLNS_UNKNOWN_PREFIX,
                                      local[:index])
        else:
            namespace = name[:index]
            local = name[index+1:]
        #[end inline]

        xsl_class, ext_class, category = self._get_element_class(namespace,
                                                                 local)
        # verify that this element can be declared here
        validation = self._validation[-1]

        if category is not None:
            tokens = [category, (namespace, local)]
        else:
            tokens = [(namespace, local)]

        for token in tokens:
            next = validation.get(token)
            if next is None and validation.has_key(ContentInfo.ELSE):
                next = validation[ContentInfo.ELSE].get(token)
            if next is not None:
                break

        if next is None:
            child_name = self._make_display_name(namespace, local)

            parent = self._node_stack[-1]
            parent_name = apply(self._make_display_name, parent.expandedName)
            from Ft.Xml.Xslt import Stylesheet
            if isinstance(parent,Stylesheet.StylesheetElement):
                if (XSL_NAMESPACE,'import') == (namespace,local):

                    raise XsltParserException(Error.ILLEGAL_IMPORT,
                                              self.parser)
            elif (XSL_NAMESPACE,'choose') == parent.expandedName:
                if (XSL_NAMESPACE,'otherwise') == (namespace,local):
                    raise XsltParserException(Error.ILLEGAL_CHOOSE_CHILD,
                                              self.parser)



            raise XsltParserException(Error.ILLEGAL_ELEMENT_CHILD,
                                      self.parser, child_name, parent_name)



        else:
            # save this state for next go round
            self._validation[-1] = next

        extension_nss, exclude_nss = self._processing_nss[-1]

        # setup space rules
        space = attribs.get(XML_SPACE)
        if space == 'preserve':
            preserve_space = 1
        elif space == 'default':
            preserve_space = 0
        else:
            preserve_space = self._preserveSpaceStack[-1]

        if xsl_class:
            # all text within xsl:text elements is preserved
            if local == 'text':
                preserve_space = 1

            instance = xsl_class(self._ownerDoc, namespace, local, self._input_source.uri)
            inst_dict = instance.__dict__
            instance.namespaces = self._namespaces[-1]
            instance.lineNumber = self.parser.ErrorLineNumber
            instance.columnNumber = self.parser.ErrorColumnNumber

            # Handle attributes in the null-namespace
            for attr_name, attr_info in xsl_class.legalAttrs.items():
                value = attribs.get(attr_name)
                if value is not None:
                    del attribs[attr_name]
                elif attr_info.required:  # value is None
                    element_name = self._make_display_name(namespace, local)
                    raise XsltParserException(Error.MISSING_REQUIRED_ATTRIBUTE,
                                              self.parser, element_name, attr_name)
                try:
                    value = attr_info.prepare(instance, value)
                except XsltException, e:
                    e.message = MessageSource.POSITION_INFO % (
                                    self.parser.GetBase(),
                                    self.parser.ErrorLineNumber,
                                    self.parser.ErrorColumnNumber,
                                    e.message)
                    raise e
                if attr_name == 'extension-element-prefixes':
                    extension_nss = self._handle_prefix_list(extension_nss,
                                                             value)
                    # Extension namespaces are automatically excluded
                    exclude_nss = self._handle_prefix_list(exclude_nss,
                                                           value)
                elif attr_name == 'exclude-result-prefixes':
                    exclude_nss = self._handle_prefix_list(exclude_nss,
                                                           value)
                else:
                    instance_var_name = '_' + attr_name.replace('-', '_')
                    inst_dict[instance_var_name] = value

            # Process attributes with a namespace-uri and check for
            # any illegal attributes in the null-namespace
            for attr_name, value in attribs.items():
                expanded =  _SplitName(attr_name)
                if expanded[0] is EMPTY_NAMESPACE:
                    # expanded is a tuple of (namespace-uri, local-name)
                    element_name = self._make_display_name(namespace, local)
                    raise XsltParserException(Error.ILLEGAL_NULL_NAMESPACE_ATTR,
                                              self.parser, attr_name, element_name)
                else:
                    instance.attributes[expanded] = value

            # XSLT Spec 2.6 - Combining Stylesheets
            if local in ['import', 'include']:
                self._combine_stylesheet(instance._href, (local == 'import'))
                # throw away this element
                self._node_stack.append(None)
                #self._ignoreMarkup = 1
                return

        elif ext_class:
            # an extension element
            instance = ext_class(self._ownerDoc, namespace, local, self._input_source.uri)
            instance.lineNumber = self.parser.ErrorLineNumber
            instance.columnNumber = self.parser.ErrorColumnNumber
            inst_dict = instance.__dict__
            instance.namespaces = self._namespaces[-1]

            if ext_class.legalAttrs is not None:
                # Handle attributes in the null-namespace
                for attr_name, attr_info in ext_class.legalAttrs.items():
                    value = attribs.get(attr_name)
                    if value is not None:
                        del attribs[attr_name]
                    elif attr_info.required:  # value is None
                        element_name = self._make_display_name(namespace, local)
                        raise XsltParserException(Error.MISSING_REQUIRED_ATTRIBUTE,
                                                  self.parser, element_name, attr_name)
                    value = attr_info.prepare(instance, value)
                    inst_dict['_' + attr_name.replace('-', '_')] = value

            # Process attributes with a namespace-uri and check for
            # any illegal attributes in the null-namespace
            for attr_name, value in attribs.items():
                expanded = attr_ns, attr_local = _SplitName(attr_name)
                if attr_ns is EMPTY_NAMESPACE and ext_class.legalAttrs is not None:
                    element_name = self._make_display_name(namespace, local)
                    raise XsltParserException(Error.ILLEGAL_NULL_NAMESPACE_ATTR,
                                              self.parser, attr_name, element_name)
                elif attr_ns == XSL_NAMESPACE:
                    attr_info = _EXT_ELEMENT_XSL_ATTRS.get(attr_local, -1)
                    if attr_info == -1:
                        element_name = self._make_display_name(namespace, local)
                        raise XsltParserException(Error.ILLEGAL_XSL_NAMESPACE_ATTR,
                                                  self.parser, attr_local, element_name)

                    # this can only be 'extension-element-prefixes'
                    prefixes = attr_info.prepare(instance, value)
                    extension_nss = self._handle_prefix_list(extension_nss,
                                                             prefixes)
                    # Extension namespaces are automatically excluded
                    exclude_nss = self._handle_prefix_list(exclude_nss,
                                                           prefixes)
                else:
                    instance.attributes[expanded] = value
        else:
            # a literal result element
            instance = LiteralElement(self._ownerDoc, namespace, local,
                                      self._input_source.uri)

            instance.lineNumber = self.parser.ErrorLineNumber
            instance.columnNumber = self.parser.ErrorColumnNumber
            instance.namespaces = self._namespaces[-1]
            for attr_name, value in attribs.items():
                expanded = attr_ns, attr_local = _SplitName(attr_name)
                if attr_ns == XSL_NAMESPACE:
                    attr_info = _RESULT_ELEMENT_XSL_ATTRS.get(attr_local, -1)
                    if attr_info == -1:
                        element_name = self._make_display_name(namespace, local)
                        raise XsltParserException(Error.ILLEGAL_XSL_NAMESPACE_ATTR,
                                                  self.parser, attr_local, element_name)

                    value = attr_info.prepare(instance, value)

                    if attr_local == 'extension-element-prefixes':
                        extension_nss = self._handle_prefix_list(extension_nss,
                                                                 value)
                        # Extension namespaces are automatically excluded
                        exclude_nss = self._handle_prefix_list(exclude_nss,
                                                               value)
                    elif attr_local == 'exclude-result-prefixes':
                        exclude_nss = self._handle_prefix_list(exclude_nss,
                                                               value)
                    else:
                        inst_attr = '_' + attr_local.replace('-', '_')
                        instance.__dict__[inst_attr] = value
                else:
                    instance.attributes[expanded] = value

            instance.excludedNss = exclude_nss


        # Add additional information
        instance.importIndex = self._import_index

        # update our stacks
        self._node_stack.append(instance)
        self._validation.append(instance.validator.getValidation())
        self._processing_nss.append((extension_nss, exclude_nss))
        self._preserveSpaceStack.append(preserve_space)
        self._local_vars.append(self._local_vars[-1].copy())

        # update root stacks
        if instance.doesPrime:
            self._ownerDoc.primeInstructions.append(instance)
        if instance.doesIdle:
            self._ownerDoc.idleInstructions.append(instance)
        return

    def endElement(self, name):
        if self._ignoreMarkup: return
        element = self._node_stack.pop()
        if element is None:
            self._ignoreMarkup = 0
            # handled an xsl:import/xsl:include or xi:include
            return

        self._whole_text and self._completeTextNode(element)

        del self._preserveSpaceStack[-1]
        del self._processing_nss[-1]
        del self._validation[-1]
        del self._local_vars[-1]

        if len(self._node_stack) == 1 and isinstance(element, LiteralElement):
            # a literal result element as stylesheet
            try:
                version = element._version
            except:
                raise XsltParserException(Error.LITERAL_RESULT_MISSING_VERSION,
                                          self.parser)

            stylesheet = XSL_NAMESPACE + NS_SPLIT_CHAR + u'stylesheet'
            self.startElement(stylesheet, {u'version': version})

            template = XSL_NAMESPACE + NS_SPLIT_CHAR + u'template'
            self.startElement(template, {u'match': u'/'})

            # make this element the template's content
            # If self._node_stack[-1] is None, then an xsl:import/include
            # was not closed
            try:
                self._node_stack[-1].appendChild(element)
            except AttributeError:
                self._node_stack[-2].appendChild(element)
            self.endElement(template)
            self.endElement(stylesheet)
        else:
            try:
                self._node_stack[-1].appendChild(element)
            except AttributeError:
                self._node_stack[-2].appendChild(element)
            if name in [XSL_NAMESPACE + NS_SPLIT_CHAR + u'variable',
                        XSL_NAMESPACE + NS_SPLIT_CHAR + u'param']:
                self._handle_variables(element._name)
        return

    def characterData(self, data):
        if self._ignoreMarkup: return
        self._whole_text.append(data)
        return

    def _completeTextNode(self, parent):
        text = u''.join(self._whole_text)
        # Stylesheets ignore any all-whitespace text except in xsl:text
        # or in elements with an xml:space='preserve'
        # S ::= \x20 | \x09 | \x0A | \x0D
        if self._preserveSpaceStack[-1] or XmlStrStrip(text):
            # verify that the current element can have text children
            validation = self._validation[-1]
            next = validation.get(ContentInfo.TEXT_NODE)
            if next is None and validation.has_key(ContentInfo.ELSE):
                next = validation[ContentInfo.ELSE].get(ContentInfo.TEXT_NODE)
            if next is None:
                # If the parent can have element children, but not text nodes,
                # ignore pure whitespace nodes. This clarification is from
                # XSLT 2.0 [3.4] Whitespace Stripping.
                # e.g. xsl:stylesheet, xsl:apply-templates, xsl:choose
                if validation.has_key(ContentInfo.EMPTY) or XmlStrStrip(text):
                    namespace, local = parent.expandedName
                    parent_name = self._make_display_name(namespace, local)
                    raise XsltParserException(Error.ILLEGAL_TEXT_CHILD_PARSE,
                                              self.parser,
                                              repr(Truncate(text, 10)),
                                              parent_name)
            else:
                # save this state for next go round
                self._validation[-1] = next

                node = StylesheetTree.XsltText(self._ownerDoc,
                                               self._input_source.uri, text)
                parent.appendChild(node)

        self._whole_text = []
        return

    def _get_element_class(self, namespaceUri, localName):
        xsl_class = ext_class = None
        category = CategoryTypes.RESULT_ELEMENT
        if namespaceUri == XSL_NAMESPACE:
            xsl_class = _ELEMENT_MAPPING.get(localName)
            if xsl_class is None:
                raise XsltParserException(Error.XSLT_ILLEGAL_ELEMENT,
                                          self.parser, localName)
            elif type(xsl_class) == type(''):
                # We need to import it
                parts = xsl_class.split('.')
                path = '.'.join(['Ft.Xml.Xslt'] + parts[:-1])
                module = __import__(path, {}, {}, '*')
                try:
                    xsl_class = module.__dict__[parts[-1]]
                except KeyError:
                    raise ImportError('.'.join(parts))
                _ELEMENT_MAPPING[localName] = xsl_class
                xsl_class.validator = ContentInfo.Validator(xsl_class.content)
            category = xsl_class.category
        elif namespaceUri in self._processing_nss[-1][0]:
            # Default XsltElement behavior effects fallback
            ext_class = self._extElements.get((namespaceUri, localName),
                                              StylesheetTree.XsltElement)
            if ext_class == StylesheetTree.XsltElement and \
                   not self._extElements.has_key((namespaceUri, localName)):
                warnings.warn("No implementation available for extension"
                              " element %s.  Fallback will be effected."
                              % repr((namespaceUri, localName)))
        return (xsl_class, ext_class, category)

    def _combine_stylesheet(self, href, is_import):
        hint = is_import and 'STYLESHEET IMPORT' or 'STYLESHEET INCLUDE'
        try:
            new_source = self._input_source.resolve(href, pubid=None, hint=hint)
        except (OSError, UriException):
            new_source = None
            for uri in self._alt_base_uris:
                try:
                    new_href = self._input_source.getUriResolver().normalize(href, uri)
                    #Do we need to figure out a way to pass the hint here?
                    new_source = self._input_source.factory.fromUri(new_href)
                    break
                except (OSError, UriException):
                    pass
            if not new_source:
                raise XsltParserException(Error.INCLUDE_NOT_FOUND,
                                          self.parser, href,
                                          self._input_source.uri)

        if self._visited_stylesheet_uris.has_key(new_source.uri):
            raise XsltParserException(Error.CIRCULAR_INCLUDE,
                                      self.parser, new_source.uri)
        else:
            self._visited_stylesheet_uris[new_source.uri] = True

        # the last node on the stack should always be the stylesheet because
        # xsl:import and xsl:include are top-level-elements only
        stylesheet = self._node_stack[-1]

        # Create a new reader to handle the inclusion
        include = self.clone().fromSrc(new_source)

        # The stylesheet containing the import will always be one higher
        # than the imported stylesheet.
        #
        # For example:
        #  stylesheet A imports stylesheets B and C in that order,
        #  stylesheet B imports stylesheet D,
        #  stylesheet C imports stylesheet E.
        # The resulting import precedences are:
        #  A=4, C=3, E=2, B=1, D=0

        # Always update the precedence from the included stylesheet
        # because it may have contained imports thus increasing its
        # import precedence.
        import_index = include.importIndex + is_import
        stylesheet.importIndex = self._import_index = import_index


        # merge the top-level elements
        stylesheet.children.extend(include.children)
        for child in include.children:
            child.parent = stylesheet
        return

    def _handle_variables(self, name):
        # one for the root and one for the stylesheet or
        # a literal result element as stylesheet
        if len(self._node_stack) > 2 or \
           isinstance(self._node_stack[-1], LiteralElement):
            # local variables
            # it is safe to ignore import precedence here
            if self._local_vars[-1].has_key(name):
                raise XsltParserException(Error.ILLEGAL_SHADOWING,
                                          self.parser, name)
            self._local_vars[-1][name] = 1
        else:
            # global variables
            existing = self._global_vars.get(name, -1)
            if self._import_index > existing:
                self._global_vars[name] = self._import_index
            elif self._import_index == existing:
                raise XsltParserException(Error.DUPLICATE_TOP_LEVEL_VAR,
                                          self.parser, name)
        return

    def _handle_prefix_list(self, namespace_list, prefixes):
        # A whitespace separated list of prefixes
        result = namespace_list[:]
        for prefix in prefixes:
            try:
                uri = self._namespaces[-1][prefix]
            except KeyError:
                raise XsltParserException(Error.UNDEFINED_PREFIX,
                                          self.parser, prefix or '#default')
            if uri not in result:
                result.append(uri)
        return result

    def _handle_xinclude(self, attribs):
        # At this point attributes are not split apart
        href = attribs.get('href')
        if not href:
            raise XIncludeException(XIncludeException.MISSING_HREF)

        # resolve given uri to absolute uri
        new_source = self._input_source.resolve(href, pubid=None, hint='Stylesheet XINCLUDE')

        if new_source.uri in self._visited_xinclude_hrefs:
            raise XIncludeException(XIncludeException.CIRCULAR_INCLUDE_ERROR,
                                    attribs['href'])


        # At this point attributes are not split apart
        parse = attribs.get('parse', 'xml')
        if parse == 'xml':
            self._entity_stack.append(self.parser)
            self._visited_xinclude_hrefs.append(new_source.uri)

            self.parser = self.parser.ExternalEntityParserCreate('include')
            self.parser.SetBase(new_source.uri)
            self.parser.ParseFile(new_source.stream)
            new_source.stream.close()

            self._visited_xinclude_hrefs.pop()
            self.parser = self._entity_stack.pop()
        elif parse == 'text':
            text = new_source.stream.read()
            new_source.stream.close()
            # Now try and determine the encoding (XInclude sect 4.3):
            # - external encoding information, if available, otherwise
            # - if the media type matches the conventions text/*+xml or
            #   application/*+xml as described in XML Media Types
            #   [IETF RFC 3023], the encoding is recognized as specified
            #   in XML 1.0, otherwise
            # - the value of the encoding attribute if one exists, otherwise
            # - UTF-8
            # FIXME - support cases 1 and 2
            encoding = attribs.get('encoding', 'utf-8')
            self._whole_text.append(unicode(text, encoding))
        else:
            raise XIncludeException(XIncludeException.INVALID_PARSE_ATTR, parse)

        return
