########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Xml/FtMiniDom/DomTree.py,v 1.6 2004/10/04 03:49:50 mbrown Exp $
"""
A Python implementation for the Domlette interface.  It is provided solely
as a fallback when cDomlette doesn't work for a particular installation.

Copyright 2004 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

from types import UnicodeType
from xml.dom import Node as _Node
from xml.dom import NotSupportedErr, HierarchyRequestErr, NotFoundErr
from xml.dom import IndexSizeErr
from Ft.Xml import SplitQName, XMLNS_NAMESPACE

# Number of characters to truncate to for Text and Comment repr's
CDATA_REPR_LIMIT = 20

try:
    property
except NameError:
    def _defproperty(klass, name):
        pass

    class _ComputedAttributes:
        def __getattr__(self, name):
            # Prevent infinite recursion
            if name.startswith("_get_"):
                raise AttributeError(name)
            try:
                func = getattr(self, "_get_" + name)
            except AttributeError:
                raise AttributeError(name)
            else:
                return func()

else:
    # class properties supported
    def _defproperty(klass, name):
        fget = getattr(klass, ("_get_" + name)).im_func
        setattr(klass, name, property(fget))

    class _ComputedAttributes:
        pass


class DOMImplementation:

    def createDocument(self, namespaceURI, qualifiedName, doctype):
        if doctype is not None:
            raise NotSupportedErr("doctype must be None for Domlettes")
        doc = Document(u'')
        if qualifiedName:
            elem = doc.createElementNS(namespaceURI, qualifiedName)
            doc.appendChild(elem)
        return doc

    def createRootNode(self, documentURI=u''):
        return Document(documentURI)

    def hasFeature(self, feature, version):
        if feature.lower() != 'core':
            return 0

        if not version or version == '2.0':
            # From DOM Level 2 Core 1.0: Section 1.2: DOMImplementation:
            # If the version is not specified, supporting any version of
            # the feature causes the method to return true.
            return 1
        return 0

    def __repr__(self):
        return "<pDOMImplementation at %X>" % id(self)

class Node(_Node, _ComputedAttributes):
    nodeName = None
    nodeValue = None
    nodeType = None
    parentNode = None
    childNodes = None
    firstChild = None
    lastChild = None
    previousSibling = None
    nextSibling = None
    attributes = None
    ownerDocument = None
    namespaceURI = None
    prefix = None
    localName = None

    # DOM Level 3
    baseURI = None

    # XPath Data Model
    rootNode = None

    def insertBefore(self, newChild, refChild):
        if isinstance(newChild, DocumentFragment):
            for c in tuple(newChild.childNodes):
                self.insertBefore(c, refChild)
            return newChild

        # Already owned, remove from its current parent
        if newChild.parentNode is not None:
            newChild.parentNode.removeChild(newChild)
        if refChild is None:
            self.appendChild(newChild)
        else:
            try:
                index = self.childNodes.index(refChild)
            except ValueError:
                raise NotFoundErr()
            self.childNodes.insert(index, newChild)
            newChild.nextSibling = refChild
            refChild.previousSibling = newChild
            if index:
                node = self.childNodes[index-1]
                node.nextSibling = newChild
                newChild.previousSibling = newChild
            else:
                newChild.previousSibling = None
                self.firstChild = newChild
            newChild.parentNode = self
        return newChild

    def replaceChild(self, newChild, oldChild):
        if isinstance(newChild, DocumentFragment):
            refChild = oldChild.nextSibling
            self.removeChild(oldChild)
            return self.insertBefore(newChild, refChild)

        try:
            index = self.childNodes.index(oldChild)
        except ValueError:
            raise NotFoundErr()

        if newChild is oldChild:
            # Nothing to do
            return

        if newChild.parentNode is not None:
            newChild.parentNode.removeChild(newChild)

        self.childNodes[index] = newChild
        newChild.parentNode = self
        newChild.previousSibling = oldChild.previousSibling
        newChild.nextSibling = oldChild.nextSibling
        if newChild.previousSibling is not None:
            newChild.previousSibling.nextSibling = newChild
        else:
            self.firstChild = newChild

        if newChild.nextSibling is not None:
            newChild.nextSibling.previousSibling = newChild
        else:
            self.lastChild = newChild

        oldChild.nextSibling = oldChild.previousSibling = None
        oldChild.parentNode = None
        return oldChild

    def removeChild(self, oldChild):
        try:
            self.childNodes.remove(oldChild)
        except ValueError:
            raise NotFoundErr()

        if oldChild.nextSibling is not None:
            oldChild.nextSibling.previousSibling = oldChild.previousSibling
        else:
            self.lastChild = oldChild.previousSibling

        if oldChild.previousSibling is not None:
            oldChild.previousSibling.nextSibling = oldChild.nextSibling
        else:
            self.firstChild = oldChild.nextSibling

        oldChild.nextSibling = oldChild.previousSibling = None
        oldChild.parentNode = None
        return oldChild

    def appendChild(self, newChild):
        if isinstance(newChild, DocumentFragment):
            for c in tuple(newChild.childNodes):
                self.appendChild(c)
            return newChild

        if newChild.parentNode is not None:
            newChild.parentNode.removeChild(newChild)

        if not self.firstChild:
            self.firstChild = newChild
        else:
            self.lastChild.nextSibling = newChild
            newChild.previousSibling = self.lastChild
        self.lastChild = newChild
        self.childNodes.append(newChild)
        newChild.parentNode = self
        return newChild

    def hasChildNodes(self):
        # Force boolean result
        return not not self.childNodes

    def normalize(self):
        node = self.firstChild
        while node:
            if isinstance(node, Text):
                next = node.nextSibling
                while isinstance(next, Text):
                    node.nodeValue = node.data = (node.data + next.data)
                    node.parentNode.removeChild(next)
                    next = node.nextSibling
                if not node.data:
                    # Remove any empty text nodes
                    next = node.nextSibling
                    node.parentNode.removeChild(node)
                    node = next
                    # Just in case this was the last child
                    continue
            elif isinstance(node, Element):
                node.normalize()
            node = node.nextSibling
        return

    # DOM Level 3
    def isSameNode(self, other):
        return self is other

    def xpath(self, expr, **kw_args):
        from Ft.Xml.XPath import SimpleEvaluate
        return SimpleEvaluate(expr, self, **kw_args)


class Document(Node):

    nodeType = Node.DOCUMENT_NODE
    nodeName = u'#document'
    implementation = DOMImplementation()
    doctype = None

    # Moved from DocumentType interface (since it is not supported)
    publicId = None
    systemId = None

    def __init__(self, documentURI):
        self.unparsedEntities = {}
        self.childNodes = []
        self.documentURI = self.baseURI = documentURI and \
            unicode(documentURI) or u''
        self.docIndex = 0
        self.nextIndex = 1
        self.rootNode = self
        return

    def _get_documentElement(self):
        for node in self.childNodes:
            if isinstance(node, Element):
                return node
        # No element children
        return None

    def createDocumentFragment(self):
        df = DocumentFragment()
        df.ownerDocument = df.rootNode = self
        return df

    def createTextNode(self, data):
        text = Text(data)
        text.ownerDocument = text.rootNode = self
        text.baseURI = self.baseURI
        text.docIndex = self.nextIndex
        self.nextIndex += 1
        return text

    def createComment(self, data):
        comment = Comment(data)
        comment.ownerDocument = comment.rootNode = self
        comment.baseURI = self.baseURI
        comment.docIndex = self.nextIndex
        self.nextIndex += 1
        return comment

    def createProcessingInstruction(self, target, data):
        pi = ProcessingInstruction(target, data)
        pi.ownerDocument = pi.rootNode = self
        pi.baseURI = self.baseURI
        pi.docIndex = self.nextIndex
        self.nextIndex += 1
        return pi

    def createElementNS(self, namespaceURI, qualifiedName):
        prefix, localName = SplitQName(qualifiedName)
        element = Element(qualifiedName, namespaceURI, prefix, localName)
        element.ownerDocument = element.rootNode = self
        element.baseURI = self.baseURI
        element.docIndex = self.nextIndex
        self.nextIndex += 3  # room for namespace and attribute nodes
        return element

    def createAttributeNS(self, namespaceURI, qualifiedName):
        prefix, localName = SplitQName(qualifiedName)
        attr = Attr(qualifiedName, namespaceURI, prefix, localName, u'')
        attr.ownerDocument = attr.rootNode = self.ownerDocument
        return attr

    def cloneNode(self, deep):
        raise NotSupportedErr("cloning document nodes")

    def importNode(self, node, deep):
        # Alien node, use nodeType checks only
        if node.nodeType == Node.ELEMENT_NODE:
            element = self.createElementNS(node.namespaceURI, node.nodeName)
            for attr in node.attributes.values():
                if attr.specified:
                    element.setAttributeNS(attr.namespaceURI, attr.name,
                                           attr.value)
            if deep:
                for child in node.childNodes:
                    element.appendChild(self.importNode(child, deep))
            return element

        if node.nodeType == Node.TEXT_NODE:
            return self.createTextNode(node.data)

        if node.nodeType == Node.COMMENT_NODE:
            return self.createComment(node.data)

        if node.nodeType == Node.PROCESSING_INSTRUCTION_NODE:
            return self.createProcessingInstruction(node.target, node.data)

        raise NotSupportedErr("importing nodeType %d" % node.nodeType)


    def __repr__(self):
        return '<pDocument at %X: %d children>' % (id(self),
                                                   len(self.childNodes))

_defproperty(Document, "documentElement")


class DocumentFragment(Node):

    nodeType = Node.DOCUMENT_FRAGMENT_NODE
    nodeName = "#document-fragment"

    def __init__(self):
        self.childNodes = []

    def __repr__(self):
        return '<pDocumentFragment at %X: %d children>' % (
            id(self), len(self.childNodes))

class Element(Node):

    nodeType = Node.ELEMENT_NODE

    def __init__(self, nodeName, namespaceURI, prefix, localName):
        if type(nodeName) is not UnicodeType:
            nodeName = unicode(nodeName, errors='strict')
        if namespaceURI is not None and type(namespaceURI) is not UnicodeType:
            namespaceURI = unicode(namespaceURI, errors='strict')
        if prefix is not None and type(prefix) is not UnicodeType:
            prefix = unicode(prefix, errors='strict')
        if type(localName) is not UnicodeType:
            localName = unicode(localName, errors='strict')
        self.nodeName = self.tagName = nodeName
        self.namespaceURI = namespaceURI
        self.prefix = prefix
        self.localName = localName
        self.attributes = {}
        self.childNodes = []
        return

    def _get_attrkey(self, namespaceURI, localName):
        """Helper function to create the key into the attributes dictionary"""
        key = (namespaceURI, localName)
        if key == (XMLNS_NAMESPACE, 'xmlns'):
            # Default namespace declaration
            key = (namespaceURI, None)
        return key

    def _set_attribute(self, key, attr):
        """Helper function for setAttributeNS/setAttributeNodeNS"""
        self.attributes[key] = attr

        attr.ownerElement = attr.parentNode = self
        attr.ownerDocument = attr.rootNode = self.ownerDocument
        attr.baseURI = self.baseURI

        # Namespace nodes take self.docIndex + 1
        # Attributes are unordered so they all share the same docIndex
        attr.docIndex = self.docIndex + 2
        return

    def getAttributeNS(self, namespaceURI, localName):
        if namespaceURI is not None and type(namespaceURI) is not UnicodeType:
            namespaceURI = unicode(namespaceURI, errors='strict')
        if type(localName) is not UnicodeType:
            localName = unicode(localName, errors='strict')
        key = self._get_attrkey(namespaceURI, localName)
        try:
            attr = self.attributes[key]
        except KeyError:
            return u''
        else:
            return attr.nodeValue

    def setAttributeNS(self, namespaceURI, qualifiedName, value):
        if namespaceURI is not None and type(namespaceURI) is not UnicodeType:
            namespaceURI = unicode(namespaceURI, errors='strict')
        if type(qualifiedName) is not UnicodeType:
            qualifiedName = unicode(qualifiedName, errors='strict')
        if type(value) is not UnicodeType:
            value = unicode(value, errors='strict')
        prefix, localName = SplitQName(qualifiedName)
        key = self._get_attrkey(namespaceURI, localName)
        attr = self.attributes.get(key)
        if attr:
            # Reuse existing attribute node
            attr.prefix = prefix
            attr.nodeValue = attr.value = value
            return

        attr = Attr(qualifiedName, namespaceURI, prefix, localName, value)
        self._set_attribute(key, attr)
        return

    def removeAttributeNS(self, namespaceURI, localName):
        if namespaceURI is not None and type(namespaceURI) is not UnicodeType:
            namespaceURI = unicode(namespaceURI, errors='strict')
        if type(localName) is not UnicodeType:
            localName = unicode(localName, errors='strict')
        key = self._get_attrkey(namespaceURI, localName)
        try:
            del self.attributes[key]
        except KeyError:
            pass
        return

    def hasAttributeNS(self, namespaceURI, localName):
        if namespaceURI is not None and type(namespaceURI) is not UnicodeType:
            namespaceURI = unicode(namespaceURI, errors='strict')
        if type(localName) is not UnicodeType:
            localName = unicode(localName, errors='strict')
        key = self._get_attrkey(namespaceURI, localName)
        return self.attributes.has_key(key)

    def getAttributeNodeNS(self, namespaceURI, localName):
        if namespaceURI is not None and type(namespaceURI) is not UnicodeType:
            namespaceURI = unicode(namespaceURI, errors='strict')
        if type(localName) is not UnicodeType:
            localName = unicode(localName, errors='strict')
        key = self._get_attrkey(namespaceURI, localName)
        return self.attributes.get(key)

    def setAttributeNodeNS(self, newAttr):
        key = self._get_attrkey(newAttr.namespaceURI, newAttr.localName)
        oldAttr = self.attributes.get(key)
        self._set_attribute(key, newAttr)
        return oldAttr

    def removeAttributeNode(self, oldAttr):
        for key, attr in self.attributes.items():
            if attr is oldAttr:
                del self.attributes[key]
                attr.ownerElement = attr.parentNode = None
                return attr

        raise NotFoundErr()

    def cloneNode(self, deep):
        doc = self.ownerDocument
        element = doc.createElementNS(self.namespaceURI, self.nodeName)
        for attr in self.attributes.values():
            new_attr = doc.createAttributeNS(attr.namespaceURI, attr.nodeName)
            new_attr.value = attr.value
            element.setAttributeNodeNS(new_attr)

        if deep:
            for child in self.childNodes:
                element.appendChild(child.cloneNode(deep))
        return element

    def __repr__(self):
        return "<pElement at %X: %s, %d attributes, %d children>" % (
            id(self),
            repr(self.nodeName),
            len(self.attributes),
            len(self.childNodes)
            )

class _Childless:
    """
    Mixin that makes childless-ness easy to implement and avoids
    the complexity of the Node methods that deal with children.
    """

    childNodes = []

    def insertBefore(self, newChild, refChild):
        raise HierarchyRequestErr(self.nodeName + " nodes cannot have children")

    def replaceChild(self, newChild, oldChild):
        raise HierarchyRequestErr(self.nodeName + " nodes cannot have children")

    def removeChild(self, oldChild):
        raise HierarchyRequestErr(self.nodeName + " nodes cannot have children")

    def appendChild(self, newChild):
        raise HierarchyRequestErr(self.nodeName + " nodes cannot have children")


class Attr(_Childless, Node):

    nodeType = Node.ATTRIBUTE_NODE
    specified = 1

    # Default when created via Document.createAttributeNS
    docIndex = -1

    def __init__(self, nodeName, namespaceURI, prefix, localName, value):
        if type(nodeName) is not UnicodeType:
            nodeName = unicode(nodeName, errors='strict')
        if namespaceURI is not None and type(namespaceURI) is not UnicodeType:
            namespaceURI = unicode(namespaceURI, errors='strict')
        if prefix is not None and type(prefix) is not UnicodeType:
            prefix = unicode(prefix, errors='strict')
        if type(localName) is not UnicodeType:
            localName = unicode(localName, errors='strict')
        self.nodeName = self.name = nodeName
        self.namespaceURI = namespaceURI
        self.prefix = prefix
        self.localName = localName
        self.nodeValue = self.value = value
        self.ownerElement = None
        # XPath Data Model
        self.parentNode = None
        return

    def cloneNode(self, deep):
        raise NotSupportedErr("cloning of attribute nodes")

    def __repr__(self):
        return "<pAttr at %X: name %s, value %s>" % (id(self),
                                                     repr(self.nodeName),
                                                     repr(self.nodeValue))

class _CharacterData(_Childless, Node):

    def __init__(self, data):
        if type(data) is not UnicodeType:
            data = unicode(data, errors='strict')
        self.__dict__['data'] = data
        self.__dict__['nodeValue'] = data
        return

    def __setattr__(self, name, value):
        if name == 'data':
            if type(value) is not UnicodeType:
                value = unicode(value, errors='strict')
            self.__dict__['nodeValue'] = value
        elif name == 'nodeValue':
            if type(value) is not UnicodeType:
                value = unicode(value, errors='strict')
            self.__dict__['data'] = value
        self.__dict__[name] = value
        return

    def substringData(self, offset, count):
        if offset < 0:
            raise IndexSizeErr("offset cannot be negative")
        if offset >= len(self.data):
            raise IndexSizeErr("offset cannot be beyond end of data")
        if count < 0:
            raise IndexSizeErr("count cannot be negative")
        return self.data[offset:offset+count]

    def appendData(self, arg):
        if type(arg) is not UnicodeType:
            arg = unicode(arg, errors='strict')
        self.data = self.data + arg
        return

    def insertData(self, offset, arg):
        if type(arg) is not UnicodeType:
            arg = unicode(arg, errors='strict')
        if offset < 0:
            raise IndexSizeErr("offset cannot be negative")
        if offset >= len(self.data):
            raise IndexSizeErr("offset cannot be beyond end of data")
        if arg:
            self.data = "%s%s%s" % (
                self.data[:offset], arg, self.data[offset:])
        return

    def deleteData(self, offset, count):
        if offset < 0:
            raise IndexSizeErr("offset cannot be negative")
        if offset >= len(self.data):
            raise IndexSizeErr("offset cannot be beyond end of data")
        if count < 0:
            raise IndexSizeErr("count cannot be negative")
        if count:
            self.data = self.data[:offset] + self.data[offset+count:]
        return

    def replaceData(self, offset, count, arg):
        if type(arg) is not UnicodeType:
            arg = unicode(arg, errors='strict')
        if offset < 0:
            raise IndexSizeErr("offset cannot be negative")
        if offset >= len(self.data):
            raise IndexSizeErr("offset cannot be beyond end of data")
        if count < 0:
            raise IndexSizeErr("count cannot be negative")
        if count:
            self.data = "%s%s%s" % (
                self.data[:offset], arg, self.data[offset+count:])
        return

    def __repr__(self):
        if len(self.data) > CDATA_REPR_LIMIT:
            data = self.data[:CDATA_REPR_LIMIT] + '...'
        else:
            data = self.data
        return "<p%s at %X: %s>" % (self.__class__.__name__, id(self), `data`)


class Text(_CharacterData):

    nodeType = Node.TEXT_NODE
    nodeName = u'#text'

    def cloneNode(self, deep):
        return self.ownerDocument.createTextNode(self.data)


class Comment(_CharacterData):

    nodeType = Node.COMMENT_NODE
    nodeName = u'#comment'

    def cloneNode(self, deep):
        return self.ownerDocument.createComment(self.data)


class ProcessingInstruction(_Childless, Node):

    nodeType = Node.PROCESSING_INSTRUCTION_NODE

    def __init__(self, target, data):
        if type(target) is not UnicodeType:
            target = unicode(target, errors='strict')
        if type(data) is not UnicodeType:
            targdataet = unicode(data, errors='strict')
        self.target = self.nodeName = target
        self.nodeValue = self.data = data
        return

    def __setattr__(self, name, value):
        if name == 'data':
            if type(value) is not UnicodeType:
                value = unicode(value, errors='strict')
            self.__dict__['nodeValue'] = value
        elif name == 'nodeValue':
            if type(value) is not UnicodeType:
                value = unicode(value, errors='strict')
            self.__dict__['data'] = value
        elif name == 'target':
            if type(value) is not UnicodeType:
                value = unicode(value, errors='strict')
            self.__dict__['nodeName'] = value
        elif name == 'nodeName':
            if type(value) is not UnicodeType:
                value = unicode(value, errors='strict')
            self.__dict__['target'] = value
        self.__dict__[name] = value
        return

    def cloneNode(self, deep):
        return self.ownerDocument.createProcessingInstruction(self.nodeName,
                                                              self.nodeValue)

    def __repr__(self):
        return "<pProcessingInstruction at %X: %s %s>" % (id(self),
                                                          repr(self.nodeName),
                                                          repr(self.nodeValue))

implementation = Document.implementation
