########################################################################
#
# File Name:            ResourceMetaData.py
#
"""
XSLT and XPath extensions supporting the 4SS ResourceMetaData API
WWW: http://4suite.org        e-mail: support@4suite.org

Copyright 2000-2001 Fourthought Inc, USA.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""

import cStringIO
from Ft.Server.Common import ResourceTypes, Schema, AclConstants
from Ft.Xml.XPath import Conversions
from Ft.Xml.Xslt import XsltElement, ContentInfo, AttributeInfo

from Ft.Server.Server import FtServerServerException

from Ns import SCORE_NS
import FtssXsltBase



def ResourceType(context, path=None):
    """
    Get the resource type of the resource
    """
    path = Conversions.StringValue(path)
    base = FtssXsltBase.FetchBaseObject(context.processor, path)
    return Schema.g_rdfResourceTypes[base.resourceType]


###########################################
##def GetPath(context):
##    """
##    Get the path of this resource
##    """
##    raise NotImplementedError("Get Path")

##def GetBasePath(context):
##    """
##    Get the path of this object as a base path
##    """
##    raise NotImplementedError("Get Base Path")


##def GetDisplayPath(context):
##    """
##    Get the path relative to the document root
##    """
##    raise NotImplementedError("Get Display Path")

def GetAbsolutePath(context,path=None):
    """
    Get the full path into the system, resolving aliases if needed
    """
    path = Conversions.StringValue(path)
    #print path
    return FtssXsltBase.FetchBaseObject(context.processor,path).getAbsolutePath()


def GetName(context,path=None):
    """
    Get the name of the resource.
    """
    path = Conversions.StringValue(path)
    #print path
    return FtssXsltBase.FetchBaseObject(context.processor,path).getName()


def GetCreationDate(context,path=None):
    """
    Get the date the resource was created
    """
    path = Conversions.StringValue(path)
    #print path
    return FtssXsltBase.FetchBaseObject(context.processor,path).getCreationDate()

def GetLastModifiedDate(context,path=None):
    """
    Get the date the resource was last modified
    """
    path = Conversions.StringValue(path)
    #print path
    return FtssXsltBase.FetchBaseObject(context.processor,path).getLastModifiedDate()

def GetSize(context,path=None):
    """
    Get the size of the resource
    """
    path = Conversions.StringValue(path)
    #print path
    return FtssXsltBase.FetchBaseObject(context.processor,path).getSize()


##    def getResourceType(self):
##        """
##        Get the type of this resource.  Types are defined in Core.Constants.ResourceType
##        """
##        self._verifyTx()
##        return self.resourceType

##    def isResourceType(self,testType):
##        """
##        Determine if this resource is of a certain type
##        """
##        self._verifyTx()
##        return ResourceTypes._IsResourceType(self.getResourceType(),testType)


def GetContent(context, path=None, encoding="LATIN-1"):
    """
    Get the string content of this resource as an XPath string object

    path - the repository path to the object to retrieve
    encoding - the encoding to use in converting the content to Unicode
    """
    #In future, this function may/should validate that the result is
    #A good XPath string
    path = Conversions.StringValue(path)
    base = FtssXsltBase.FetchBaseObject(context.processor, path)
    return unicode(base.getContent(), encoding)

def GetRawContent(context, path=None, encoding="LATIN-1"):
    """
    Get the string content of this resource as a raw Unicode object

    path - the repository path to the object to retrieve
    encoding - the encoding to use in converting the content to Unicode
    """
    #In current implementation, this is identical to GetContent
    #But this function does signal a different intent: to return an
    #Arbitrary Unicode object rather than a conforming XPath string
    path = Conversions.StringValue(path)
    base = FtssXsltBase.FetchBaseObject(context.processor, path)
    return unicode(base.getContent(), encoding)



##    def getContentSlice(self,start=None,end=None):
##        """
##        Get a piece of our content.  Really only makes sense for large raw files
##        """
##        if start:
##            if end:
##                return self.getContent()[start:end]
##            return self.getContent()[start:]
##        elif end:
##            return self.getContent()[:end]
##        return self.getContent()

def GetMetaDataResource(context,path=None):
    """
    Get the string meta data of this resource
    """
    path = Conversions.StringValue(path)
    base = FtssXsltBase.FetchBaseObject(context.processor, path)
    return base.getMetaDataResource().getAbsolutePath()

##    def getContentResource(self):
##        """
##        Get the resource the represents the content of this object
##        """
##        self._verifyTx()
##        p = self._path.normalize(".;content",0)
##        return self._fetchResource(p)


##    def getValidationInfo(self):
##        """
##        Get the validation information associated with this resource.
##        """
##        self._verifyTx()
##        return None


def GetParent(context, path=None):
    """
    Get the parent container of this resource
    """
    path = Conversions.StringValue(path)
    base = FtssXsltBase.FetchBaseObject(context.processor, path)
    parent = base.getParent()
    if parent:
        return parent.getAbsolutePath()
    return ""

##    def getRoot(self):
##        """
##        Get a reference to the system repository
##        """
##        self._verifyTx()
##        return self._cache.getResource(self._path._documentRoot)

def GetAliases(context,path=None):
    """
    Get a list of alias objects that reference this object
    """
    path = Conversions.StringValue(path)
    base = FtssXsltBase.FetchBaseObject(context.processor, path)

    doc = None
    if hasattr(context.node,'ownerDocument'):
        doc = context.node.ownerDocument or context.node
    else:
        doc = context.node
        while doc.parentNode:
            doc = doc.parentNode

    df = doc.createDocumentFragment()
    for token in base.getAliases():
        df.appendChild(doc.createTextNode(token.getAbsolutePath()))

    return df.childNodes


##    #####################################
##    #ACL Interfaces
##    #####################################
class AddAclElement(XsltElement):
    """
    Add access for an given identifer and level to the ACL of a resource
    """

    content = ContentInfo.Empty

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(
            default=None,
            description='The path of the resource '
            'where to add the ACL. If not given, '
            'the source document is used.'),
        'ident' : AttributeInfo.StringAvt(
            required=1,
            description='The ACL identifier (a user or group identifier)'),
        'access' : AttributeInfo.StringAvt(
            required=1,
            description='The access level or type.  The built-in ACL levels are "read" | "write" | "write user model" | "execute" | "delete" | "change owner" | "change permissions"'),
        'allowed' : AttributeInfo.YesNoAvt(
            default='yes',
            description='Allow or deny this type of access (yes/no)'),
        }

    childArgument = None

    def instantiate(self, context, processor):
        context.setProcessState(self)

        path = self._path.evaluate(context)
        ident = self._ident.evaluate(context)
        access = self._access.evaluate(context)
        allowed = self._allowed.evaluate(context)

        obj = FtssXsltBase.FetchBaseObject(processor, path)
        obj.addAcl(access, ident, allowed)

        return (context,)


def GetAclByKey(context,aclKey,path=None):
    aclKey = Conversions.StringValue(aclKey)
    return _GetAcl(context,aclKey,path)

def GetAllAcl(context,path=None):
    return _GetAcl(context,None,path)

def _GetAcl(context,aclKey=None,path=None):
    """
    Get a list of users and groups specified by the aclKey
    """
    path = Conversions.StringValue(path)
    base = FtssXsltBase.FetchBaseObject(context.processor, path)

    proc = context.processor
    proc.pushResultTree(context.currentInstruction.baseUri)
    try:
	def _serializeAclKey(writer,key,data):
	    writer.startElement("AclKey")
	    writer.attribute("key",key)
	    for ident,allowed in data.items():
		
		writer.startElement("Ident")
		writer.attribute("name",ident)
		writer.attribute("allowed",allowed == AclConstants.ALLOWED and "1" or "0")
		writer.endElement("Ident")
	    writer.endElement("AclKey")

	if aclKey:
	    _serializeAclKey(proc.writers[-1],aclKey,base.getAcl(aclKey))
	else:
	    proc.writers[-1].startElement('Acl')
	    acl = base.getAcl()
	    for key,data in acl.items():
		_serializeAclKey(proc.writers[-1],key,data)
	    proc.writers[-1].endElement('Acl')
    finally:
        rtf = proc.popResult()
    return rtf.childNodes
	

class SetAclElement(XsltElement):
    """
    Reset the ACL for a resource, and then optionally add access for an
    given identifer and level to the ACL of a resource
    """

    content = ContentInfo.Empty

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(
            default=None,
            description='The path of the resource '
            'where to set the ACL. If not given, '
            'the source document is used.'),
        'ident' : AttributeInfo.StringAvt(
            required=1,
            description='The ACL identifier (user or group)'),
        'access' : AttributeInfo.StringAvt(
            required=1,
            description='The access level or type.  The built-in ACL levels are "read" | "write" | "write user model" | "execute" | "delete" | "change owner" | "change permissions"'),
        'allowed' : AttributeInfo.YesNoAvt(
            default='yes',
            description='Allow or deny this type of access'),
        }

    childArgument = None

    def instantiate(self, context, processor):
        context.setProcessState(self)

        path = self._path.evaluate(context)
        ident = self._ident.evaluate(context)
        access = self._access.evaluate(context)
        allowed = self._allowed.evaluate(context)

        obj = FtssXsltBase.FetchBaseObject(processor, path)
        obj.setAcl(access, ident, allowed)

        return (context,)



def VerifyAcl(context,aclKey,path=None,verifyTraverse=1):
    """
    Verify that the testAclIdent is in the acl list specified by the acl key for this object.
    """
    path = Conversions.StringValue(path)
    aclKey = Conversions.StringValue(aclKey)
    verifyTraverse = Conversions.BooleanValue(verifyTraverse)
    try:
        FtssXsltBase.FetchBaseObject(context.processor,path).verifyAcl(aclKey,verifyTraverse)
    except FtServerServerException, e:
        return 0
    return 1



##    def getAclIdentifiers(self):
##        """
##        Get a list of all acl identifiers for the current users
##        """
##        self._verifyTx()
##        return self._driver.getAclIdentifiers()

##    def getAclIdent(self):
##        """
##        Get the ACL identifier for the current user
##        """
##        self._verifyTx()
##        return self._driver.getAclIdent()

class InheritAclElement(XsltElement):
    """
    Set the inheritance ACL of a resource
    """

    content = ContentInfo.Empty

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(default=None,
					       description='The path of the resource '
					       'where to set the ACL. If not given, '
					       'the source document is used.'),
        'access' : AttributeInfo.StringAvt(required=1,
					   description='The access type'),
        }

    childArgument = None

    def instantiate(self, context, processor):
        context.setProcessState(self)

        path = self._path.evaluate(context)
        access = self._access.evaluate(context)

        obj = FtssXsltBase.FetchBaseObject(processor, path)
        obj.inheritAcl(access)

        return (context,)



class RemoveAclElement(XsltElement):
    """
    Remove the ACL from a resource
    """

    content = ContentInfo.Empty

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(
            default=None,
            description='The path of the resource '
            'where to add the ACL. If not given, '
            'the source document is used.'),
        'ident' : AttributeInfo.StringAvt(
            required=1,
            description='The ACL identifier (user or group)'),
        'access' : AttributeInfo.StringAvt(
            required=1,
            description='The access type'),
        }

    childArgument = None

    def instantiate(self, context, processor):
        context.setProcessState(self)

        path = self._path.evaluate(context)
        ident = self._ident.evaluate(context)
        access = self._access.evaluate(context)

        obj = FtssXsltBase.FetchBaseObject(processor, path)
        obj.removeAcl(access, ident)

        return (context,)


def GetOwner(context,path=None):
    """
    Get the owner of the resource
    """
    path = Conversions.StringValue(path)
    #print path
    owner = FtssXsltBase.FetchBaseObject(context.processor,path).getOwner()
    if owner == None:
        return ""
    return owner.getAbsolutePath()



class SetOwnerElement(XsltElement):
    """
    Change the owner of a resource
    """
    content = None

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(
        default=None,
        description='The path of the resource to change the IMT of.  Defaults to current.'),
        'owner' : AttributeInfo.UriReferenceAvt(required=1,
                                                description='The path to the new owner'),
        }

    def instantiate(self, context, processor):
        context.setProcessState(self)

        # This may return None (the 'default' from AttributeInfo)
        path          = self._path.evaluate(context)
        owner          = FtssXsltBase.FetchBaseObject(processor, self._owner.evaluate(context))
        
        obj = FtssXsltBase.FetchBaseObject(processor, path)
        obj.setOwner(owner)

        return (context,)


##    ##########################################
##    #XML Interfaces
##    ##########################################

##    def applyXslt(self, stylesheets, params=None, ignorePis=1,
##                  extParams=None, extModules=None):
##        """
##        applies the specified stylesheets (with the specified transformation parameters)
##        on the DOM representation of the raw file resource
##        """
##        extModules = extModules or []

##        params = params or {}
##        params[(RESERVED_NAMESPACE, 'display-path')] = self._path.displayPath
##        params[(RESERVED_NAMESPACE, 'absolute-path')] = self._path.absolutePath
##        extParams = extParams or {}
##        extParams[(RESERVED_NAMESPACE, 'path')] = self._path.absolutePath

##        #First map the stylesheets
##        if type(stylesheets) not in [type([]),type(())]:
##            stylesheets = (stylesheets,)

##        p = Ft.Xml.Xslt.Processor.Processor()
##        p.extensionParams = extParams

##        p._repository = self.getRoot()
##        mods = p.registerExtensionModules(
##            ['Ft.Server.Server.Xslt']+extModules
##            )

##        p.setDocumentReader(
##            FtssInputSource.NonvalidatingReader
##            )

##        for s in stylesheets:
##            if type(s) == type(''):
##                #It is a URI
##                isrc = FtssInputSource.FtssInputSourceFactory.fromUri(self.getAbsolutePath(),self._driver)
##                isrc = isrc.resolve(s,'','Repo ApplyXlst')
##                p.appendStylesheet(isrc)
##            elif hasattr(s,'toStylesheet'):
##                #A document reference
##                p.appendStylesheetInstance(s.toStylesheet(self))
##            elif hasattr(s,'asStylesheet'):
##                #A reference to a document in the system
##                p.appendStylesheetInstance(s.asStylesheet())

##        isrc = FtssInputSource.FtssInputSourceFactory.fromUri(self.getAbsolutePath(),self._driver)
##        rt = p.run(isrc, ignorePis=ignorePis, topLevelParams=params)
##        if extParams is not None and hasattr(p, 'extensionParams'):
##            extParams.update(p.extensionParams)

##        imt = p._lastOutputParams.mediaType
##        return rt, imt

def AsDom(context,path=None):
    """Get an RTF that is as a live version of the document function.  NOTE: This does not work like document(), where document() caches RTFs and always returns the same one per the spec, as-dom will return the latest in the repo."""
    
    path = Conversions.StringValue(path)
    base = FtssXsltBase.FetchBaseObject(context.processor, path)
    return [base.asDom()]
   

##    def toXPathContext(self,nss=None):
##        """
##        Create an XPath Context with the src of this RawFile
##        """
##        self._verifyTx()
##        return self._driver.getContext(self._path,nss)

class XUpdateElement(XsltElement):
    """
    Allows XML content to be updated with the XUpdate protocol
    """

    content = ContentInfo.Template

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(
            default=None,
            description='The path of the resource to apply XUpdate to'),
        }

    childArgument = ('src', 1, None, 'The XUpdate document')

    def instantiate(self, context, processor):
        context.setProcessState(self)

        # This may return None (the 'default' from AttributeInfo)
        path = self._path.evaluate(context)

        stream = cStringIO.StringIO()
        params = processor.writers[-1]._outputParams.clone()
        params.method = (None, u'xml')
        processor.addHandler(params, stream)
        for child in self.children:
            context = child.instantiate(context, processor)[0]
        processor.removeHandler()
        content = stream.getvalue()

        #print content
        
        obj = FtssXsltBase.FetchBaseObject(processor, path)
        obj.xUpdate(content)

        
        return (context,)


##    #####################################
##    #Query Interfaces
##    #####################################
##    def hasResource(self,path):
##        """
##        Query if the system has a resource specified by the path
##        """
##        self._verifyTx()
##        path = self._basePath.normalize(path,0)
##        if self._driver.hasResource(path):
##            return self._driver.getType(path)
##        return 0


##    #####################################
##    #Retrieval Interfaces
##    #####################################
##    def fetchResource(self,path):
##        """
##        Fetch an resource from the system.
##        """
##        self._verifyTx()
##        path = self._basePath.normalize(path)
##        return self._fetchResource(path)


##    ##########################################
##    #Mutation Interfaces
##    ##########################################
class SetContentElement(XsltElement):
    """
    Set the string content of this resource
    """

    content = ContentInfo.Template

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(
            default=None,
            description='The path of the resource to set content on'),
            
        'content-as-string' : AttributeInfo.YesNo(
            default=None,
            description='Specifies whether or not to assume the child argument is a string')
        }

    childArgument = ('src', 1, None, 'The string to set the content too')

    def instantiate(self, context, processor):
        context.setProcessState(self)

        # This may return None (the 'default' from AttributeInfo)
        path          = self._path.evaluate(context)        
        inputString = self._content_as_string
        
        stream = cStringIO.StringIO()
        params = processor.writers[-1]._outputParams.clone()
        
        if not inputString:
            params.method = (None, u'xml')
            
        processor.addHandler(params, stream)

        for child in self.children:           
            context = child.instantiate(context, processor)[0]
        processor.removeHandler()
        
        
        text = stream.getvalue()

        obj = FtssXsltBase.FetchBaseObject(processor, path)
        obj.setContent(text)

        return (context,)




class DeleteElement(XsltElement):
    """
    Delete this resource
    """

    content = ContentInfo.Template

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(
        default=None,
        description='The path of the resource to delete'),
            
        }

    def instantiate(self, context, processor):
        context.setProcessState(self)

        # This may return None (the 'default' from AttributeInfo)
        path          = self._path.evaluate(context)        
        print "Deleting resource", path

        obj = FtssXsltBase.FetchBaseObject(processor, path)
        obj.delete()

        return (context,)


##    ###################################
##    #Creation Interfaces
##    ###################################

class AddAliasElement(XsltElement):
    """
    Adds an alias for this resource to the repository.
    """
    content = None

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(
            default=None,
            description='The path of the resource to add an alias too.  Defaults to current.'),
        'alias' : AttributeInfo.UriReferenceAvt(required=1,
                                               description='The path of the new alias'),
        'doc-def' : AttributeInfo.UriReferenceAvt(default=None,
                                                  description='The document definition to use for the new alias'),
        }

    childArgument = None
    def instantiate(self, context, processor):
        context.setProcessState(self)

        # This may return None (the 'default' from AttributeInfo)
        path          = self._path.evaluate(context)        
        alias = self._alias.evaluate(context)
        dd = self._doc_def.evaluate(context)
        
        obj = FtssXsltBase.FetchBaseObject(processor, path)

        obj.addAlias(alias,dd)

        return (context,)

##    ######################################
##    #Delete Interface
##    ######################################
class RemoveAliasElement(XsltElement):
    """
    Remove an alias for this resource from the repository.
    """
    content = None

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(
            default=None,
            description='The path of the resource to add an alias too.  Defaults to current.'),
        'alias' : AttributeInfo.UriReferenceAvt(required=1,
                                               description='The path of the new alias'),
        }

    childArgument = None
    def instantiate(self, context, processor):
        context.setProcessState(self)

        # This may return None (the 'default' from AttributeInfo)
        path          = self._path.evaluate(context)        
        alias = self._alias.evaluate(context)
        
        obj = FtssXsltBase.FetchBaseObject(processor, path)

        obj.removeAlias(alias)
        
        return (context,)


##    ########################################
##    #RDF Interfaces
##    ########################################

##    def getModel(self):
##        """
##        Retrieves the common model associated w/ the repository
##        """
##        self._verifyTx()
##        return self._driver.getModel(self._path)

##    def addLiteralProperty(self, predicate, object, domain='', statementUri=''):
##        """
##        Adds a statement about this repository object, with a literal property, to
##        the model
##        """
##        stmt = Statement.Statement(self.getAbsolutePath(),
##                                   predicate,
##                                   object,
##                                   domain = domain,
##                                   statementUri=statementUri,
##                                   objectType=OBJECT_TYPE_LITERAL)
##        self._driver.getModel(self._path).add([stmt])
##        return stmt

##    def addResourceProperty(self, predicate, object, domain='', statementUri=''):
##        """
##        Adds a statement about this repository object, with a resource
##        property, to the model
##        """
##        stmt = Statement.Statement(self.getAbsolutePath(),
##                                   predicate,
##                                   object.getUri(),
##                                   domain=domain,
##                                   statementUri=statementUri,
##                                   objectType=OBJECT_TYPE_RESOURCE)
##        self._driver.getModel(self._path).add([stmt])
##        return stmt

##    def removeResourceProperty(self, predicate, object):
##        """
##        Remove a statement about this repository object, with a resource preoperty, from
##        the model
##        """
##        self._driver.getModel(self._path).removePattern(self.getAbsolutePath(),predicate,object.getUri())

##    def removeLiteralProperty(self, predicate, object):
##        """
##        Removes a statement about this repository object, with a literal property, from
##        the model
##        """
##        self._driver.getModel(self._path).removePattern(self.getAbsolutePath(),predicate,object)

##    def getStatementsAbout(self):
##        """
##        Returns all statements about this object in the model
##        """
##        return self._driver.getModel(self._path).complete(self.getPath(), None, None)

##    def getStatementsGeneratedBy(self):
##        """
##        Returns all statements in the model generated by this object
##        """
##        return self._driver.getModel(self._path).complete(None, None, None, domain=self.getPath())


##    ########################################
##    #Temporary file Interfaces
##    ########################################
class MarkTemporaryElement(XsltElement):
    """
    Mark a resource as temporary
    """

    content = None

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(
            default=None,
            description='The path of the resource to mark as temporary.  Defaults to current.'),
        'time-to-live' : AttributeInfo.NumberAvt(
            default=0,
            required= 1,
            description='The time in seconds from now that the resource is to live.'),
        }

    childArgument = None

    def instantiate(self, context, processor):
        context.setProcessState(self)

        # This may return None (the 'default' from AttributeInfo)
        path = self._path.evaluate(context)
        ttl = int(self._time_to_live.evaluate(context))

        obj = FtssXsltBase.FetchBaseObject(processor, path)
        #print obj
        #print ttl
        obj.markTemporary(ttl)
        
        return (context,)


ExtFunctions = {
    (SCORE_NS, 'get-content'): GetContent,    
    (SCORE_NS, 'get-raw-content'): GetRawContent,
    (SCORE_NS, 'resource-type'): ResourceType,    
    (SCORE_NS, 'get-metadata-resource'): GetMetaDataResource,    
    (SCORE_NS, 'get-creation-date'): GetCreationDate,    
    (SCORE_NS, 'get-last-modified-date'): GetLastModifiedDate,    
    (SCORE_NS, 'get-size'): GetSize,    
    (SCORE_NS, 'get-absolute-path'): GetAbsolutePath,    
    (SCORE_NS, 'get-name'): GetName,    
    (SCORE_NS, 'get-owner'): GetOwner,    
    (SCORE_NS, 'get-acl-by-key'): GetAclByKey,    
    (SCORE_NS, 'get-all-acl'): GetAllAcl,    
    (SCORE_NS, 'get-parent'): GetParent,    
    (SCORE_NS, 'get-aliases'): GetAliases,    
    (SCORE_NS, 'verify-acl'): VerifyAcl,    
    (SCORE_NS, 'as-dom'): AsDom,    
}

ExtElements = {
    (SCORE_NS, 'set-content'): SetContentElement,
    (SCORE_NS, 'x-update'): XUpdateElement,
    (SCORE_NS, 'set-acl'): SetAclElement,
    (SCORE_NS, 'inherit-acl'): InheritAclElement,
    (SCORE_NS, 'add-acl'): AddAclElement,
    (SCORE_NS, 'remove-acl'): RemoveAclElement,
    (SCORE_NS, 'mark-temporary'): MarkTemporaryElement,
    (SCORE_NS, 'add-alias'): AddAliasElement,
    (SCORE_NS, 'remove-alias'): RemoveAliasElement,
    (SCORE_NS, 'set-owner'): SetOwnerElement,    
    (SCORE_NS, 'delete-resource'): DeleteElement,    
}
