########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Server/Server/Drivers/FtssDriver.py,v 1.61 2004/08/29 14:01:58 cogbuji Exp $
"""
4Suite repository driver

Copyright 2004 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

import time, cStringIO, cPickle

import pAclObject as AclObject
import FtssModel, FtssInputSource
import PathImp

from Ft.Lib import Time, Uuid
from Ft.Rdf import Statement, Model
from Ft.Rdf import OBJECT_TYPE_UNKNOWN, OBJECT_TYPE_LITERAL, OBJECT_TYPE_RESOURCE
from Ft.Rdf.Drivers import Memory as RdfMemory
from Ft.Rdf.Serializers import Dom as RdfDom
import Ft.Rdf.Util
from Ft.Server import FTSERVER_NAMESPACE, RESERVED_NAMESPACE
from Ft.Server.Common import AclConstants, ResourceTypes
from Ft.Server.Common import Schema, XmlLib
from Ft.Server.Server import FtServerServerException, Error
from Ft.Xml import Domlette, InputSource, XPath, XLink, XUpdate
from Ft.Xml.Domlette import Print
from Ft.Xml.XLink import XLINK_NAMESPACE
from Ft.Xml.XPath import Conversions
from Ft.Xml.Xslt import StylesheetReader, XSL_NAMESPACE
import Ft.Xml.Xslt.Processor


DEFAULT_SESSION_TIME_TO_LIVE = 60*15 #15 mins

def CurrentTime():
    return str(Time.FromPythonTime())

class DataStoreContainer:
    """
    Repository XML Container object.  createContainer/getContent/setContent/_getChildren are overidden
    to use low level api's to implement containment.  Should catch Exceptions more ferverently.
    
    NOTE
    self._driver is the FtssDriver instance
    self._driver._driver is the Data store driver instance
    
    """
    def _createContainer(self,path,content):
        """
        Overidden by underlying Data Store to do what it needs / wants to in order to represent a container.
        Those that don't will create a 'Resource' with the provided content (regardless of whether or not it
        is automatically generated later
        """
        self.createFile(path.absolutePath,
                                 ResourceTypes.RESOURCE_CONTENT,
                                 content)        
    
    def createContainer(self, path, createParents=0, docDef=None,
                        actualContent=None):
        """
        Creates the specified container, if createParents is true
        then all its parents along the path are created (if they
        dont exist). If actualContent is given NOT_SUPPORTED Exception is raised
        Needs to mimick (mostly) and override FtssDriver.createResource
        in order to call _createContainer
        """
        self._verifyTx()

        if not hasattr(path, 'displayPath'):
            path = self._basePath.normalize(path)

        parentPath = path.getParentPath()
        
        if not self._driver.hasResource(parentPath):
            if createParents:
                self.createContainer(parentPath.displayPath, createParents=1)
            else:
                raise FtServerServerException(
                    Error.INVALID_PATH, path=parentPath.absolutePath,
                    type='Container'
                    )
        if self._driver.getType(parentPath) not in ResourceTypes.CONTAINERS:
            raise FtServerServerException(Error.INVALID_PATH,
                                          path=parentPath,
                                          type='Container')
        
        if actualContent:
            raise FtServerServerException(Error.NOT_SUPPORTED,
                                          reason='Cant provide XML content for datastore containers')

        md, content = self._driver.newContainerXml(
            path.name, {}, self._driver.getAclIdent(),
            docDef
            )
        
        if self._driver._driver.hasFile(path.absolutePath, ResourceTypes.RESOURCE_METADATA):
            raise FtServerServerException(Error.PATH_EXISTS,
                                          path=path.displayPath)

        #self.__acl.verifyCreate(path)

        # Add this child to the parent container
        self._driver._modifyParentContainer(path.getParentPath(), addPath=path.name)

        # Create the resource metadata first to give underlying datastore the oppurtunity to
        #determine the nature of the content (from resourceType) before it is stored
        metadata_path = path.normalize('.;metadata;no-traverse')
        self._driver._driver.createFile(metadata_path.absolutePath,
                                ResourceTypes.RESOURCE_METADATA,
                                md)
        
        # Create the resource content
        self._driver._driver._createContainer(path,content)
        
        self._driver._setMetaData(path, md, content)

        return self._fetchResource(path)

    
    
    
    def _getChildren(self):
        """
        Uses underlying datastore implementation
        """
        self._verifyTx()
        return self._driver._driver.fetchChildren(self.getPath())

    def childReferenceXML(self,path):
        """
        Returns an XML representation of the specified container (path is a PathImp)
        Uses fetchChildren (implemented by the data store) to retrieve the names of the children and generates
        a repository container XML document.
        """
        containerSkeleton = """<?xml version="1.0" encoding="ISO-8859-1"?>
        <ftss:Container
             xmlns:ftss="%s"
             xmlns:xlink="%s">
             <ftss:Children>
                %s
             </ftss:Children>
        </ftss:Container>"""
        childEntry = """<ftss:ChildReference xlink:type='simple' xlink:actuate='onLoad' xlink:show='embed' xlink:href='%s'/>\n"""
        childrenXML = ""
        for child in self.fetchChildren(path):
            childrenXML += childEntry%(child+';metadata')
        rt=containerSkeleton%(str(FTSERVER_NAMESPACE), str(XLINK_NAMESPACE), childrenXML)        
        return rt
    
    
class FtssDriver:
    """
    The FtssDriver wraps the lowest-level repository database driver,
    providing functions commonly used by higher-level APIs.
    """

    def __init__(self, logger, driverMod, driverProperties):

        self.__acl = None
        self.__aclIdent = None

        self._driver = None
        self._driverMod = driverMod
        self._driverProperties = driverProperties

        self._cache = {}
        self._domCache = {}
        self._logger = logger

        self.__xupdateReader = XUpdate.Reader()
        self.__xuProcessor = XUpdate.Processor()

        self.__tempFileDelete = 0  #Set when temp files are being removed
        return

    def exists(self):
        """A very special-case function.  This can be called without being logged in"""
        rt = self._driverMod.Exists(self._driverProperties)
        if rt == 1:
            #Make sure there is a user
            driver = self._driverMod.Begin(**self._driverProperties)
            return len(driver.getSystemModel().complete(None,
                                                        Schema.TYPE,
                                                        Schema.g_rdfResourceTypes[ResourceTypes.ResourceType.USER])) > 0
        return rt

    def initialize(self):
        """Another very special case that will create a new '/'.  It cannot be called if they are logged in"""
        if self.__acl:
            raise FtServerServerException(Error.PERMISSION_DENIED)
        self._driverMod.Initialize(self._driverProperties)

        self._driver = apply(self._driverMod.Begin,(),self._driverProperties)

        acl = self.defaultAcl(AclConstants.SUPER_USER_GROUP_NAME,
                              AclConstants.SUPER_USER_GROUP_NAME,
                              )

        acl[AclConstants.READ_ACCESS][AclConstants.WORLD_GROUP_NAME] = AclConstants.ALLOWED
        acl[AclConstants.EXECUTE_ACCESS][AclConstants.WORLD_GROUP_NAME] = AclConstants.ALLOWED

        md, content = self.newContainerXml('/', acl,
                                           AclConstants.SUPER_USER_GROUP_NAME,
                                           None)

        self._driver.createFile('/', ResourceTypes.RESOURCE_METADATA,md)
        self._driver.createFile('/', ResourceTypes.RESOURCE_CONTENT,content)

        #Do a pseudo login
        self.__acl = AclObject.AclObject(self, 0, [AclConstants.SUPER_USER_GROUP_NAME])
        self.__aclIdent = AclConstants.SUPER_USER_GROUP_NAME

        p = PathImp.QuickCreate('/')
        self._setMetaData(p, md, content)
        return

    def destroy(self):
        """Another very special case that will destroy a '/'.
        It cannot be called if they are logged in"""
        if self.__acl:
            raise FtServerServerException(Error.PERMISSION_DENIED)
        return self._driverMod.Destroy(self._driverProperties)

    _passwordExpression = XPath.Compile('/ftss:User/ftss:PasswdHash')

    def login(self, userName, password, verify=0):
        if verify and self.exists() != 1:
            raise FtServerServerException(Error.CONFIG_INVALID_REPOSITORY)

        self._driver = apply(self._driverMod.Begin, (), self._driverProperties)
        keep = 0
        try:
            (aclIdent, aclIdents, anon) = self.checkLogin(userName, password)
            keep = 1
        finally:
            if not keep:
                self._driver.rollback()
                self._driver = None

        self.__aclIdent = aclIdent
        self.__acl = AclObject.AclObject(self, anon, aclIdents)
        return

    def checkLogin(self, userName, password):
        if userName is None or userName == AclConstants.ANONYMOUS_USER_NAME:
            #Anon
            aclIdents = [AclConstants.WORLD_GROUP_NAME]
            aclIdent = AclConstants.WORLD_GROUP_NAME
            anon = 1
        else:
            model = self._driver.getSystemModel()
            stmts = model.complete(None, Schema.USER_NAME, userName)
            if len(stmts) != 1:
                raise FtServerServerException(Error.INVALID_LOGIN)

            userUri = stmts[0].subject

            #Make sure it is a user
            userMetaData = self._driver.fetchFile(userUri, ResourceTypes.RESOURCE_METADATA)
            if self._getType(userUri) != ResourceTypes.ResourceType.USER:
                raise FtServerServerException(Error.INVALID_LOGIN)

            userPath = PathImp.QuickCreate(userUri)
            con = self.getContext(userPath)

            res = self._passwordExpression.evaluate(con)

            if not res:
                raise FtServerServerException(Error.INVALID_LOGIN)

            pw = Conversions.StringValue(res[0])

            aclIdent = userName
            if pw != password:
                raise FtServerServerException(Error.INVALID_LOGIN)

            aclIdents = [aclIdent,
                         AclConstants.WORLD_GROUP_NAME,
                         AclConstants.USERS_GROUP_NAME]
            aclIdents.extend(self.__expandGroups(userUri))
            anon = 0
        #If we get this far, login is valid
        return (aclIdent, aclIdents, anon)

    def getContext(self, path, nss=None):
        defaultNss = {'ftss':FTSERVER_NAMESPACE,
                      'xlink':XLINK_NAMESPACE,
                      'xsl':XSL_NAMESPACE,
                      }
        if nss:
            defaultNss.update(nss)
        if self._hasDomCache(path):
            dom = self._getDomCache(path)
        else:
            #Return the generated content if a container.  Avoid metadata paths
            if not path.resourceType == ResourceTypes.RESOURCE_METADATA and self.getType(path) in ResourceTypes.CONTAINERS:
                content = str(self._driver.childReferenceXML(path))
            else:
                content = self._driver.fetchFile(path.absolutePath,
                                             path.resourceType)
            if content is None:
                raise FtServerServerException(Error.UNKNOWN_PATH,
                                              path=path.absolutePath)
            isrc = FtssInputSource.FtssInputSourceFactory.fromString(content,path.absolutePathAsUri, driver=self)
            dom = FtssInputSource.NonvalidatingReader.parse(isrc)
            self._setDomCache(path, dom)
        return XPath.Context.Context(dom, processorNss=defaultNss)

    def getModel(self,path):
        self._verifyTx()
        s =  self._driver.getSystemModel()
        u = self._driver.getUserModel()
        return FtssModel.FtssModel(s, u, self.__acl, path)

    def getSystemModel(self):
        self._verifyTx()
        return self._driver.getSystemModel()

    def getAclIdentifiers(self):
        self._verifyTx()
        return self.__acl.getAclIdentifiers()

    def getAclIdent(self):
        self._verifyTx()
        return self.__aclIdent

    def getSystemContainerPath(self):
        self._verifyTx()
        return self.__systemContainer

    def getAcl(self,path,access=None):
        self._verifyTx()
        self.__acl.verifyFetch(path)
        return self.__acl.getAcl(path,access)

    def setAcl(self,path,acl):
        self._verifyTx()
        self.__acl.verifyChangePermissions(path)
        self.__acl.clearCache()
        aclXml = self.aclToXml(acl)

        xu = UPDATE_ACL % (str(FTSERVER_NAMESPACE), aclXml)
        xu = XmlLib.MakeString(xu)
        self.xupdateMetaData(path,xu)

    def verifyAcl(self,path,access, verifyTraverse):
        self._verifyTx()
        self.__acl.verifyAcl(path, access, verifyTraverse)

    def commit(self):
        self._verifyTx()
        self._driver.commit()

        self.__acl = self._driver = None
    def rollback(self):
        self._verifyTx()
        self._driver.rollback()
        self.__acl = self._driver = None
        
    ##############################################
    #    Resource/Container management interfaces
    ##############################################

    def _modifyParentContainer(self, path, addPath='', removePath=''):
        """
        Update the parent container content and metadata.  Only one of
        addPath or removePath may be specified.

        path is the PathImp for the parent container
        addPath and removePath specify the child path to add or remove
        """
        if self.getType(path)  not in ResourceTypes.CONTAINERS:
            raise FtServerServerException(Error.INVALID_PATH,
                                          path=path.displayPath,
                                          type='Container')

        if addPath and removePath:
            raise TypeError("Only one of addPath or removePath allowed")
        elif addPath:
            self._driver.manageChildren(path,addPath)
        elif removePath:
            self._driver.manageChildren(path,removePath,False)
        else:
            raise TypeError("One of addPath or removePath are required")

        content = self._driver.childReferenceXML(path)
        
        metadata_path = path.normalize('.;metadata;no-traverse')
        modified_date = CurrentTime()
        content_size = str(len(content))
        metadata = self.__applyCompiledXUpdate(metadata_path,
                                                UPDATE_LAST_MODIFIED_AND_SIZE,
                                                {(None, 'lmd') : modified_date,
                                                 (None, 'size') : content_size,
                                                 })
        self._driver.updateFile(metadata_path.absolutePath,
                                metadata_path.resourceType,
                                metadata)

        # Uncache the parent
        self._clearCache(path)
        self._clearCache(metadata_path)
        self._clearDomCache(path)
        self._clearDomCache(metadata_path)

        # From the content we add a statement for the new child
        # From the metadata we change ftss:LastModified and ftss:Size

        model = self._driver.getSystemModel()

        # Delete the existing ftss:LastModified
        model.removePattern(path.absolutePath, Schema.MODIFIED_DATE, None,
                            scope=Schema.SYSTEM_SOURCE_URI)

        # Delete the existing ftss:Size
        model.removePattern(path.absolutePath, Schema.CONTENT_SIZE, None,
                            scope=Schema.SYSTEM_SOURCE_URI)

        # Add statements for ftss:LastModified, ftss:Size.
        statements = [Statement.Statement(path.absolutePath,
                                          Schema.MODIFIED_DATE,
                                          modified_date,
                                          scope=Schema.SYSTEM_SOURCE_URI,
                                          objectType=OBJECT_TYPE_LITERAL),
                      Statement.Statement(path.absolutePath,
                                          Schema.CONTENT_SIZE,
                                          content_size,
                                          scope=Schema.SYSTEM_SOURCE_URI,
                                          objectType=OBJECT_TYPE_LITERAL),
                      ]

        model.add(statements)                
        
        docdefs = model.complete(path.absolutePath, Schema.DOCDEF, None)
        if docdefs and docdefs[0].object != Schema.NULL_DOCDEF:
            model.removePattern(None, None, None,
                                scope=path.absolutePath)
            self.__generateUserDocDefStmts(docdefs[0].object, path)

        return

    def createResource(self, path, metadata, content):
        """
        Create a new resource.  This function:
            - modifies the parent container
            - creates the metadata / content for the resource            
            - set's up the system metadata (via associated system doc defs)
        """
        self._verifyTx()

        if self._driver.hasFile(path.absolutePath, ResourceTypes.RESOURCE_METADATA):
            raise FtServerServerException(Error.PATH_EXISTS,
                                          path=path.displayPath)

        self.__acl.verifyCreate(path)

        # Add this child to the parent container
        self._modifyParentContainer(path.getParentPath(), addPath=path.name)

        # Create the resource metadata first to give underlying datastore the oppurtunity to
        #determine the nature of the content (from resourceType) before it is stored
        metadata_path = path.normalize('.;metadata;no-traverse')
        self._driver.createFile(metadata_path.absolutePath,
                                ResourceTypes.RESOURCE_METADATA,
                                metadata)
        
        # Create the resource content
        self._driver.createFile(path.absolutePath,
                                 ResourceTypes.RESOURCE_CONTENT,
                                 content)        
        
        self._setMetaData(path, metadata, content)
        return

    def setResourceCache(self, path, cache):
        """Pickle and store the cache"""
        if self._driver.hasFile(path.absolutePath,
                                ResourceTypes.RESOURCE_CACHE):
            self._driver.updateFile(path.absolutePath,
                                    ResourceTypes.RESOURCE_CACHE,
                                    cPickle.dumps(cache))
        else:
            #self.__find_cdomlette(cache)
            self._driver.createFile(path.absolutePath,
                                    ResourceTypes.RESOURCE_CACHE,
                                    cPickle.dumps(cache))
        return

    def fetchResourceCache(self, path):
        """Un Pickle and return the cache"""
        if hasattr(path, 'displayPath'):
            path = path.absolutePath
        if self._driver.hasFile(path,
                                ResourceTypes.RESOURCE_CACHE):
            content = self._driver.fetchFile(path, ResourceTypes.RESOURCE_CACHE)
            return cPickle.loads(content)
        raise FtServerServerException(Error.UNKNOWN_PATH,
                                      path=path)

    def hasResource(self, path):
        self._verifyTx()
##        try:
##            self.__acl.verifyFetch(path)
##        except FtServerServerException:
##            raise
##            return 0
        if self._hasCache(path): return 1
        return self._driver.hasFile(path.absolutePath,path.resourceType)

    def fetchResource(self, path):
        """Fetch the content (caching the results)"""
        self._verifyTx()

        if not self.__tempFileDelete:
            self.__acl.verifyFetch(path)

        #See if it is cached
        if self._hasCache(path):
            return self._getCache(path)
        if not path.resourceType == ResourceTypes.RESOURCE_METADATA and self.getType(path) in ResourceTypes.CONTAINERS:        
            content = self._driver.childReferenceXML(path)
        else:
            content = self._driver.fetchFile(path.absolutePath, path.resourceType)

        if content is None:
            raise FtServerServerException(Error.UNKNOWN_PATH,
                                          path=path.displayPath)
        self._setCache(path,content)
        return content

    def xupdateContent(self, path, xu):
        """
        Apply XUpdate to the content and store it (updatng the size and last modified system metadata)
        """
        self._verifyTx()
        self.__acl.verifyWrite(path)

        rt = self.getType(path)  #Checks to see if it is a resource

        newContent = self.__applyXUpdate(path,xu)

        #update Last modified time stamp and size
        mdPath = path.normalize('.;metadata;no-traverse')

        newMd = self.__applyCompiledXUpdate(mdPath,
                                             UPDATE_LAST_MODIFIED_AND_SIZE,
                                             {(None,'lmd') : CurrentTime(),
                                              (None,'size') : str(len(newContent))
                                              }
                                             )


        self._driver.updateFile(path.absolutePath,
                                 ResourceTypes.RESOURCE_CONTENT,
                                 newContent)
        self._driver.updateFile(path.absolutePath,
                                 ResourceTypes.RESOURCE_METADATA,
                                 newMd)

        self._clearCache(path)
        self._clearCache(mdPath)
        self._clearDomCache(path)
        self._clearDomCache(mdPath)

        #Update the system metadata
        self._updateMetaData(path,newMd,newContent)

    def xupdateMetaData(self, path, xu):
        """
        Apply XUpdate to the metadata associated with the specified resource and store it (updating the last modified
        time stamp and size)
        """

        self._verifyTx()
        self.__acl.verifyWrite(path)

        mdPath = path.normalize('.;metadata;no-traverse')

        content = self.fetchResource(path)

        newMd = self.__applyXUpdate(mdPath, xu)

        self._driver.updateFile(mdPath.absolutePath,
                                ResourceTypes.RESOURCE_METADATA,
                                newMd)

        #Update the last modified system metadata with the current time
        xu = XmlLib.MakeString(UPDATE_LAST_MODIFIED%(str(FTSERVER_NAMESPACE),
                                                       CurrentTime(),
                                                       ))

        newMd = self.__applyXUpdate(mdPath,xu)

        self._driver.updateFile(mdPath.absolutePath,
                                ResourceTypes.RESOURCE_METADATA,
                                newMd)

        self._clearCache(mdPath)
        self._clearDomCache(mdPath)

        #Update the system metadata
        self._updateMetaData(path,newMd,content)
        return

    def updateResourceContent(self, path, content):
        """
        Update the specified resource with the given content (updating the
        last modified time and size)
        """
        self._verifyTx()
        self.__acl.verifyWrite(path)

        mdPath = path.normalize('.;metadata;no-traverse')

        newMd = self.__applyCompiledXUpdate(mdPath,
                                             UPDATE_LAST_MODIFIED_AND_SIZE,
                                             {(None,'lmd') : CurrentTime(),
                                              (None,'size') : str(len(content))
                                              }
                                             )

        self._driver.updateFile(path.absolutePath,
                                ResourceTypes.RESOURCE_CONTENT, content)
        self._driver.updateFile(path.absolutePath,
                                ResourceTypes.RESOURCE_METADATA, newMd)

        self._clearCache(mdPath)
        self._clearCache(path)
        self._clearDomCache(mdPath)
        self._clearDomCache(path)
        
        #Update the system metadata
        self._updateMetaData(path, newMd, content)
        return

    def updateResourceMetadata(self, path, newMd, changeDocDef=0):
        """Apply Update the md and store it (also updating the last modified and size system metadata)"""
        self._verifyTx()
        self.__acl.verifyWrite(path)

        mdPath = path.normalize('.;metadata;no-traverse')

        contentPath = path.normalize('.;content;no-traverse')
        content = self.fetchResource(contentPath)

        #FIXME: Redundant?
        self._driver.updateFile(path.absolutePath,
                                ResourceTypes.RESOURCE_CONTENT, content)
        
        #Update metadata
        self._driver.updateFile(path.absolutePath,
                                ResourceTypes.RESOURCE_METADATA, newMd)

        self._clearCache(mdPath)
        self._clearDomCache(mdPath)

        #Might be redundant if the newMD already reflects the new time/size
        newMd = self.__applyCompiledXUpdate(mdPath,
                                             UPDATE_LAST_MODIFIED_AND_SIZE,
                                             {(None,'lmd') : CurrentTime(),
                                              (None,'size') : str(len(content))
                                              }
                                             )

        self._driver.updateFile(path.absolutePath,
                                ResourceTypes.RESOURCE_METADATA, newMd)

        #Regenerate the system metadata (deleting it first if the user doc def changes)
        if changeDocDef:
            #If there is a change to the doc def, we need to do this the slow way
            #so that the second stage of generation uses the correct doc def
            self._deleteMetaData(path.absolutePath)
            self._setMetaData(path, newMd, content)
        else:
            self._updateMetaData(path, newMd, content)

        return

    def deleteResource(self, path):
        """
        Delete the resource (clear it's cache, delete all of it's data-store components, remove from it's
        parent, and delete it's system metadata)
        """
        self._verifyTx()

        if path == '/':
            raise FtServerServerException(Error.PERMISSION_DENIED,
                                          level = 'delete',
                                          path = '/')

        if not self.__tempFileDelete:
            self.__acl.verifyDelete(path)

        # Remove the content and metadata from the cache
        mdPath = path.normalize('.;metadata;no-traverse')
        self._clearCache(path)
        self._clearCache(mdPath)
        self._clearDomCache(path)
        self._clearDomCache(mdPath)

        # Remove it from the repository
        self._driver.deleteFile(path.absolutePath,ResourceTypes.RESOURCE_METADATA)
        self._driver.deleteFile(path.absolutePath,ResourceTypes.RESOURCE_CONTENT)
        if self._driver.hasFile(path.absolutePath,ResourceTypes.RESOURCE_CACHE):
            self._driver.deleteFile(path.absolutePath,ResourceTypes.RESOURCE_CACHE)

        # Remove from the parent container
        self._modifyParentContainer(path.getParentPath(), removePath=path.name)

        # Remove the statements for this from the model
        self._deleteMetaData(path.absolutePath)
        return

    def verifyDeleteResource(self,path):
        """Delete the resource"""
        self._verifyTx()
        if not self.__tempFileDelete:
            self.__acl.verifyDelete(path)


    def open(self,href):
        ###Bits of a stream interface
        #Turn it into a path
        p = PathImp.CreateInitialPath('/',
                                      self)
        p = p.normalize(href+';no-traverse') #Should we not traverse here?
        src = self.fetchResource(p)
        return cStringIO.StringIO(str(src))



    #########################################################
    #
    #  Session Interfaces
    #
    #########################################################
    def createSession(self, key, userName, password,
                      ttl=DEFAULT_SESSION_TIME_TO_LIVE):
        """
        Create a new session object

        key - session key to use
        userName - user name for log in
        password - password for log in
        ttl - session time to live, in seconds, default 15 minutes
        """
        ttl = ttl or DEFAULT_SESSION_TIME_TO_LIVE
        sid = Uuid.UuidAsString(Uuid.GenerateUuid())

        session = {'USERNAME':userName,
                   'PASSWORD':password,
                   'EXPIRE':int(time.time() + ttl + 0.5),
                   'TTL': ttl + 0.5,
                   'KEY':key,
                   'DATA':{}}

        if self._driver.hasFile(sid, ResourceTypes.SESSION_DATA):
            #Its time for it to go
            self._driver.deleteFile(sid, ResourceTypes.SESSION_DATA)
        self._driver.createFile(sid, ResourceTypes.SESSION_DATA,
                                cPickle.dumps(session))
        return sid

    def retrieveSession(self, sid, key):
        """Get and validate a session"""
        driver = apply(self._driverMod.Begin, (), self._driverProperties)

        try:
            if not driver.hasFile(sid, ResourceTypes.SESSION_DATA):
                raise FtServerServerException(Error.INVALID_SESSION)

            session = cPickle.loads(driver.fetchFile(sid, ResourceTypes.SESSION_DATA).encode('latin1'))

            if session['EXPIRE'] < time.time():
                driver.deleteFile(sid,ResourceTypes.SESSION_DATA)
                raise FtServerServerException(Error.INVALID_SESSION)

            if session['KEY'] != key:
                driver.deleteFile(sid,ResourceTypes.SESSION_DATA)
                raise FtServerServerException(Error.INVALID_SESSION)

            session['EXPIRE'] = time.time() + session.get('TTL', 0)

            driver.updateFile(sid, ResourceTypes.SESSION_DATA,
                              cPickle.dumps(session))
        finally:
            driver.commit()

        #Now, do a login
        self.login(session['USERNAME'], session['PASSWORD'])
        return

    def setSessionData(self, sid, key, value):
        self._verifyTx()
        if not self._driver.hasFile(sid, ResourceTypes.SESSION_DATA):
            raise FtServerServerException(Error.INVALID_SESSION)

        session = cPickle.loads(self._driver.fetchFile(
            sid, ResourceTypes.SESSION_DATA))
        session['DATA'][key] = value
        self._driver.updateFile(sid, ResourceTypes.SESSION_DATA,
                                cPickle.dumps(session))

    def deleteSessionData(self, sid, key, value):
        self._verifyTx()
        if not self._driver.hasFile(sid, ResourceTypes.SESSION_DATA):
            raise FtServerServerException(Error.INVALID_SESSION)

        session = cPickle.loads(self._driver.fetchFile(
            sid, ResourceTypes.SESSION_DATA
            ))
        if session['DATA'].has_key(key):
            del session['DATA'][key]

        self._driver.updateFile(
            sid, ResourceTypes.SESSION_DATA, cPickle.dumps(session)
            )

    def getSessionData(self,sid,key):
        self._verifyTx()
        if not self._driver.hasFile(sid,ResourceTypes.SESSION_DATA):
            raise FtServerServerException(Error.INVALID_SESSION)

        session = cPickle.loads(self._driver.fetchFile(
            sid, ResourceTypes.SESSION_DATA))

        return session['DATA'].get(key)

    def getSessionExpiration(self,sid):
        self._verifyTx()
        if not self._driver.hasFile(sid, ResourceTypes.SESSION_DATA):
            raise FtServerServerException(Error.INVALID_SESSION)

        session = cPickle.loads(self._driver.fetchFile(
            sid, ResourceTypes.SESSION_DATA
            ))

        return session['EXPIRE']

    def invalidateSession(self,sid):

        self._verifyTx()
        if not self._driver.hasFile(sid, ResourceTypes.SESSION_DATA):
            raise FtServerServerException(Error.INVALID_SESSION)
        self._driver.deleteFile(sid, ResourceTypes.SESSION_DATA)



    ###############################################
    #
    #  Interface to help with Temp files
    #
    ###############################################
    def setTempFileDelete(self,flag):
        self.__tempFileDelete = flag

    _typeExpression = XPath.Compile('//ftss:MetaData/@type')

    def getType(self,path):
        """Get the resource type from the metadata. path is a Path object"""

        res = self._driver.getSystemModel().complete(path.absolutePath, Schema.TYPE, None)
        if not res:
            raise FtServerServerException(Error.UNKNOWN_PATH,
                                          path=path.displayPath)
        return Schema.g_resourceTypeFromRdf[res[0].object]

    def _getType(self,path):
        """Get the resource type from the metadata. path is an absolute path string"""

        res = self._driver.getSystemModel().complete(path,Schema.TYPE,None)
        if not res:
            raise FtServerServerException(Error.UNKNOWN_PATH,
                                          path=path)
        return Schema.g_resourceTypeFromRdf[res[0].object]

    ############################################################################################
    #
    #  Interfaces for managing system RDF metadata via associated system xpath doc defs expressions
    #
    ############################################################################################

    def _updateMetaData(self,path,metaData,content):
        newStmts = self._setMetaData(path,metaData,content,1)
        oldStmts = []
        model = self._driver.getSystemModel()
        oldStmts.extend(model._driver.complete(path.absolutePath, None, None, None,Schema.SYSTEM_SOURCE_URI,{}))
        oldStmts.extend(model._driver.complete(None, None, None, None, path.absolutePath,{}))

        toAdd = []
        toDel = []
        for s in newStmts:
            if s not in oldStmts:
                toAdd.append(s)
        for s in oldStmts:
            if s not in newStmts:
                toDel.append(s)
        model._driver.add(toAdd)
        model._driver.remove(toDel)

    def _setMetaData(self, path, metaData, content, justBuild=0):
        mPath = path.normalize('.;metadata;no-traverse')
        con = self.getContext(mPath)

        res = self._typeExpression.evaluate(con)
        if not res:
            raise FtServerServerException(Error.INVALID_XML,
                                          message='No Type Attribute')
        rt = Schema.g_resourceTypeFromRdf[res[0].value]

        #First Set System MetaData
        exprs = self._systemDocDefs[rt][ResourceTypes.RESOURCE_METADATA]
        results = []
        if justBuild:
            results.extend(self.__buildExpressions(path,con,exprs,scope=Schema.SYSTEM_SOURCE_URI))
        else:
            self.__applyExpressions(path, con, exprs,
                                    self._driver.getSystemModel(),
                                    scope=Schema.SYSTEM_SOURCE_URI)

        if rt not in (ResourceTypes.ResourceType.RAW_FILE,
                      ResourceTypes.ResourceType.URI_REFERENCE_FILE):
            con = self.getContext(path)

            exprs = self._systemDocDefs[rt][ResourceTypes.RESOURCE_CONTENT]
            if justBuild:
                results.extend(self.__buildExpressions(path, con, exprs,
                                                       scope = Schema.SYSTEM_SOURCE_URI))
            else:
                self.__applyExpressions(path, con, exprs,
                                        self._driver.getSystemModel(),
                                        scope = Schema.SYSTEM_SOURCE_URI)

            dd = self._driver.getSystemModel()._driver.complete(path.absolutePath,
                                                                Schema.DOCDEF, None,None,None,{})
            if dd and dd[0][2] != Schema.NULL_DOCDEF:
                #Update the user docdefs
                results.extend(self.__generateUserDocDefStmts(dd[0][2], path,justBuild))
        return results

    def _deleteMetaData(self, path):
        model = self._driver.getSystemModel()
        model.removePattern(path, None, None, scope=Schema.SYSTEM_SOURCE_URI)
        model.removePattern(None, None, None, scope=path)
        
    #######################################################
    #
    #  Interfaces for managing user defined doc definitions
    #
    #######################################################

    def __generateUserDocDefStmts(self, ddUri, path,justbuild):
        if self._getType(ddUri) == ResourceTypes.ResourceType.XPATH_DOCUMENT_DEFINITION:
            return self.__generateUserXPathDocDefStmts(ddUri, path,justbuild)
        else:
            return self.__generateUserXsltDocDefStmts(ddUri, path,justbuild)

    def __generateUserXPathDocDefStmts(self, ddUri, path,justBuild):
        ddPath = PathImp.QuickCreate(ddUri)
        con = self.getContext(ddPath)
        from Ft.Server.Common import DocumentDefinitionSerialization
        nsMap = DocumentDefinitionSerialization._DeserializeNsMaps(con)
        rdfMaps = DocumentDefinitionSerialization._DeserializeRdfMaps(con)
        baseNames = map(lambda x:x.object,self._driver.getSystemModel().complete(ddUri, Schema.BASE_DOCDEF, None))

        exprs = []
        for s, p, o, t in rdfMaps:
            exprs.append((XPath.Compile(s),
                          XPath.Compile(p),
                          XPath.Compile(o), t))

        con = self.getContext(path, nsMap)

        results = []
        if justBuild:
            results.extend(self.__buildExpressions(path, con, exprs,
                                                   scope = path.absolutePath))
        else:
            self.__applyExpressions(path, con, exprs,
                                    self._driver.getSystemModel(),
                                    scope = path.absolutePath)

        for baseUri in baseNames:
            results.extend(self.__generateUserDocDefStmts(baseUri,
                                                          path,
                                                          justBuild))
        return results

    def __generateUserXsltDocDefStmts(self, ddUri, path, justBuild):
        baseNames = map(lambda x: x.object,
                        self._driver.getSystemModel().complete(
            ddUri, Schema.BASE_DOCDEF, None))

        r = StylesheetReader.StylesheetReader()
        st = r.fromInstant(self.fetchResourceCache(ddUri), path.absolutePath)

        extModules = []

        params = {}
        params[(RESERVED_NAMESPACE, 'display-path')] = path.displayPath
        params[(RESERVED_NAMESPACE, 'absolute-path')] = path.absolutePath

        p = None
        try:
            p = Ft.Xml.Xslt.Processor.Processor()

            mods = p.registerExtensionModules(
                ['Ft.Server.Server.Xslt']+extModules
                )

            p.appendStylesheetInstance(st)

            content = self._driver.fetchFile(path.absolutePath,
                                             path.resourceType)

            isrc = FtssInputSource.FtssInputSourceFactory.fromString(
                content, path.absolutePathAsUri, self)

            #p.inputSourceFactory = FtssInputSource._FtssInputSourceFactory(
            #    self._driver)
            p.inputSourceFactory = FtssInputSource.FtssInputSourceFactory
            rt = p.run(isrc, ignorePis=1, topLevelParams=params)
        except:
            import traceback;traceback.print_exc()
            raise FtServerServerException(Error.USER_XSLT_DOCDEF_EXCEPTION)

        #So we got a result.  Cool, deserialize it and add it.
        model,db = Ft.Rdf.Util.DeserializeFromString(rt, create=1, scope=path.absolutePath)

        results = []
        stmts = model.statements()
        res = []
        for stmt in stmts:
            stmt.scope = path.absolutePath
            res.append((unicode(stmt.subject),
                        unicode(stmt.predicate),
                        unicode(stmt.object),
                        unicode(stmt.uri),
                        unicode(path.absolutePath),
                        unicode(stmt.objectType)))
        stmts = res

        if justBuild:
            results.extend(stmts)
        else:
            self._driver.getSystemModel()._driver.add(stmts)

        baseNames = map(lambda x:x.object,self._driver.getSystemModel().complete(ddUri,Schema.BASE_DOCDEF,None))

        for baseUri in baseNames:
            results.extend(self.__generateUserDocDefStmts(baseUri,
                                                          path,justBuild))
        return results

    #############################
    #
    #  Xupdate helper interfaces
    #
    #############################


    def __applyXUpdate(self, path, xu):
        con = self.getContext(path)
        isrc = FtssInputSource.FtssInputSourceFactory.fromString(
            xu, path.absolutePathAsUri, self
        )
        xu = self.__xupdateReader.fromSrc(isrc)
        self.__xuProcessor.execute(con.node, xu)

        #This node has been changed its DOM cache gets Magically updated
        st = cStringIO.StringIO()
        Print(con.node, stream=st)
        res = st.getvalue()

        self._clearDomCache(path)
        return res


    def __applyCompiledXUpdate(self, path, xu, vars=None):
        con = self.getContext(path)
        self.__xuProcessor.execute(con.node, xu,variables=vars)

        #This node has been changed its DOM cache gets Magically updated

        st = cStringIO.StringIO()
        Print(con.node, stream=st)
        res = st.getvalue()

        #self._clearDomCache(path)
        return res

    ###########################################################################
    #
    #  Interfaces for managing expressions generated from system xpath doc defs
    #
    ###########################################################################

    def __buildExpressions(self, path, context, exprs, scope):
        stmts = []
        context.varBindings[(None,'uri')] = path.absolutePath
        for s, p, o, t in exprs:
            subjects = s.evaluate(context)
            if type(subjects) not in [type(()),type([])]:
                subjects = [subjects]
            predicates = p.evaluate(context)
            if type(predicates) not in [type(()),type([])]:
                predicates = [predicates]
            objects = o.evaluate(context)
            if type(objects) not in [type(()),type([])]:
                objects = [objects]
            for ns in subjects:
                for np in predicates:
                    for no in objects:
                        stmts.append((unicode(Conversions.StringValue(ns)),
                                      unicode(Conversions.StringValue(np)),
                                      unicode(Conversions.StringValue(no)),
                                      unicode(''),
                                      unicode(scope),
                                      unicode(t)))
        return stmts
    def __applyExpressions(self, path, context, exprs, model, scope):
        model._driver.add(self.__buildExpressions(path,context,exprs,scope))
        
    
    ###################
    #
    #  Misc. interfaces
    #
    ###################


    def __expandGroups(self,userUri):
        """Expand a user's groups"""
        model = self._driver.getSystemModel()
        res = model.complete(None,Schema.GROUP_MEMBER,userUri+';metadata')
        groupUris = map(lambda x:x.subject,res)
        groups = map(lambda x:x.split('/')[-1],groupUris)
        for g in groupUris:
            groups.extend(self.__expandGroups(g))
        return groups




    def _verifyTx(self):
        if self._driver == None or self.__acl == None:
            raise FtServerServerException(Error.TRANSACTION_NOT_IN_PROGRESS)



    def defaultAcl(self,user,owner):
        if user == owner:
            v = {owner:AclConstants.ALLOWED,
                 AclConstants.OWNER:AclConstants.ALLOWED}
        else:
            v = {user:AclConstants.ALLOWED,
                 owner:AclConstants.ALLOWED,
                 AclConstants.OWNER:AclConstants.ALLOWED,
                 }
        return {AclConstants.READ_ACCESS:v.copy(),
                AclConstants.WRITE_ACCESS:v.copy(),
                AclConstants.EXECUTE_ACCESS:v.copy(),
                AclConstants.DELETE_ACCESS:v.copy(),
                AclConstants.CHANGE_OWNER_ACCESS:v.copy(),
                AclConstants.CHANGE_PERMISSIONS_ACCESS:v.copy(),
                AclConstants.WRITE_USER_MODEL_ACCESS:v.copy(),
                #AclConstants.READ_METADATA_ACCESS:v.copy(),
                }


    def newContainerXml(self,path,acl,owner,docDef,content=None):
        rt = ResourceTypes.ResourceType.CONTAINER
        tagName = 'Container'
        if content is None:
            content = """<ftss:%s xmlns:ftss="%s">
  <ftss:Children/>
</ftss:%s>""" % (tagName, str(FTSERVER_NAMESPACE), tagName)


        a = self.aclToXml(acl)
        t = CurrentTime()
        docDef = docDef or Schema.NULL_DOCDEF
        md = """<ftss:MetaData xmlns:ftss="%s" path='%s' type='%s' creation-date='%s' document-definition='%s'>
  %s
<ftss:LastModifiedDate>%s</ftss:LastModifiedDate>
<ftss:Owner>%s</ftss:Owner>
<ftss:Imt>text/xml</ftss:Imt>
<ftss:Size>%d</ftss:Size>
</ftss:MetaData>
        """ % (str(FTSERVER_NAMESPACE), path, Schema.g_rdfResourceTypes[rt],
        t, docDef, a, t, owner, len(content))

        return XmlLib.MakeString(md),XmlLib.MakeString(content)

    def aclToXml(self,acl):
        st = "<ftss:Acl>"
        for key in acl.keys():
            for name,allowed in acl[key].items():
                st = st + '  <ftss:Access type="%s" ident="%s" allowed="%d"/>' % (key,
                                                                                name,
                                                                                allowed == AclConstants.ALLOWED)
        return st + '</ftss:Acl>'



    #Cache interfaces
    def _hasCache(self,path):
        return self._cache.has_key((path.absolutePath,path.resourceType))
    def _setCache(self,path,content):
        self._cache[(path.absolutePath,path.resourceType)] = content
    def _getCache(self,path):
        return self._cache[(path.absolutePath,path.resourceType)]
    def _clearCache(self,path):
        k = (path.absolutePath,path.resourceType)
        if self._cache.has_key(k):
            del self._cache[k]


    def _hasDomCache(self,path):
        return self._domCache.has_key((path.absolutePath,path.resourceType))
    def _setDomCache(self,path,content):
        self._domCache[(path.absolutePath,path.resourceType)] = content
    def _getDomCache(self,path):
        return self._domCache[(path.absolutePath,path.resourceType)]
    def _clearDomCache(self,path):
        k = (path.absolutePath,path.resourceType)
        if self._domCache.has_key(k):
            del self._domCache[k]

    def __find_cdomlette(self,object,parent="",visited = None):
        from Ft.Xml import cDomlettec
        import types
        visited = visited or []
        if object in visited:return
        visited.append(object)
        if type(object) == types.InstanceType:
            for name in dir(object):
                value = getattr(object,name)
                if type(value) == cDomlettec.DocumentType:
                    print "FOUND:" + parent + "." + name
                else:
                    self.__find_cdomlette(value,parent  + "." + name,visited)
        elif type(object) == types.DictionaryType:
            for name in object.keys():
                value = object[name]
                if type(value) == cDomlettec.DocumentType:
                    print "FOUND:" + parent + "." + str(name)
                else:
                    self.__find_cdomlette(value,parent  + "." + name,visited)
        elif type(object) in [types.ListType,types.TupleType]:
            for value in object:
                name = "LIST:%d" * object.index(value)
                if type(value) == cDomlettec.DocumentType:
                    print "FOUND:" + parent + "." + name
                else:
                    self.__find_cdomlette(value,parent  + "." + name,visited)

        elif type(object) in [types.MethodType,types.StringType,types.UnicodeType]:
            pass
        else:
            print type(object)



    #Set up our system docdefs
    ###########################################################################
    #
    #  System Xpath Document Definitions for managing system metadata (RDF)
    #
    ###########################################################################
    
    systemDocDefs = {}

    for rt in ResourceTypes.RAW_FILES:
        systemDocDefs[rt] = {}
        systemDocDefs[rt][ResourceTypes.RESOURCE_METADATA] = []
        systemDocDefs[rt][ResourceTypes.RESOURCE_CONTENT] = []
    del rt

    cur = systemDocDefs[ResourceTypes.ResourceType.RAW_FILE][ResourceTypes.RESOURCE_METADATA]
    cur.extend([('$uri', '"%s"'%Schema.TYPE, '/ftss:MetaData/@type', OBJECT_TYPE_RESOURCE),
                ('$uri', '"%s"'%Schema.CREATION_DATE, '/ftss:MetaData/@creation-date', OBJECT_TYPE_LITERAL),
                ('$uri', '"%s"'%Schema.MODIFIED_DATE, '/ftss:MetaData/ftss:LastModifiedDate', OBJECT_TYPE_LITERAL),
                ('$uri', '"%s"'%Schema.IMT, '/ftss:MetaData/ftss:Imt', OBJECT_TYPE_LITERAL),
                ('$uri', '"%s"'%Schema.CONTENT_SIZE, '/ftss:MetaData/ftss:Size', OBJECT_TYPE_LITERAL),
                ('$uri', '"%s"'%Schema.OWNER, '/ftss:MetaData/ftss:Owner', OBJECT_TYPE_RESOURCE),
                ('$uri', '"%s"'%Schema.TIME_TO_LIVE, '/ftss:MetaData/ftss:TimeToLive', OBJECT_TYPE_LITERAL),
                ])
    del cur

    #URI Ref Documents
    cur = systemDocDefs[ResourceTypes.ResourceType.URI_REFERENCE_FILE][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.RAW_FILE][ResourceTypes.RESOURCE_METADATA][:]
    cur.append(('$uri', '"%s"'%Schema.URI_REFERENCE_LOCATION, "/ftss:MetaData/ftss:Reference", OBJECT_TYPE_RESOURCE))
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.URI_REFERENCE_FILE][ResourceTypes.RESOURCE_CONTENT] = systemDocDefs[ResourceTypes.ResourceType.RAW_FILE][ResourceTypes.RESOURCE_CONTENT][:]
    del cur


    #XML Documents
    cur = systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.RAW_FILE][ResourceTypes.RESOURCE_METADATA][:]
    cur.append(('$uri', '"%s"'%Schema.DOCDEF, "/ftss:MetaData/@document-definition", OBJECT_TYPE_RESOURCE))
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_CONTENT]= systemDocDefs[ResourceTypes.ResourceType.RAW_FILE][ResourceTypes.RESOURCE_CONTENT][:]
    del cur

    #CONTAINERS
    cur = systemDocDefs[ResourceTypes.ResourceType.CONTAINER][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    #Group
    cur = systemDocDefs[ResourceTypes.ResourceType.GROUP][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.GROUP][ResourceTypes.RESOURCE_CONTENT] = systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_CONTENT][:]
    cur.append(('$uri','"%s"'%Schema.GROUP_MEMBER,"/ftss:*/ftss:Members/ftss:MemberReference/@xlink:href", OBJECT_TYPE_RESOURCE))
    cur.append(('$uri','"%s"'%Schema.GROUP_NAME,"/ftss:Group/@name", OBJECT_TYPE_LITERAL))
    del cur

    #User
    cur = systemDocDefs[ResourceTypes.ResourceType.USER][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.USER][ResourceTypes.RESOURCE_CONTENT] = systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_CONTENT][:]
    cur.append(('$uri','"%s"'%Schema.USER_NAME,"/ftss:User/@userName", OBJECT_TYPE_LITERAL))
    del cur

    #Document Definition
    systemDocDefs[ResourceTypes.ResourceType.DOCUMENT_DEFINITION] = {}
    systemDocDefs[ResourceTypes.ResourceType.DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_METADATA] = []
    systemDocDefs[ResourceTypes.ResourceType.DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_CONTENT] = []

    cur = systemDocDefs[ResourceTypes.ResourceType.DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_CONTENT] = systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_CONTENT][:]
    cur.append(('$uri','"%s"'%Schema.FULL_TEXT_INDEX,"/*/ftss:CreationParams/@FullTextIndex", OBJECT_TYPE_LITERAL))
    cur.append(('$uri','"%s"'%Schema.XSLT_EXT_MODULE,"/*/ftss:CreationParams/ftss:ExtensionModule/@PyImport", OBJECT_TYPE_LITERAL))
    cur.append(('$uri','"%s"'%Schema.ENFORCE_RDF_SCHEMA,"/*/ftss:CreationParams/@EnforceSchema", OBJECT_TYPE_LITERAL))
    cur.append(('$uri','"%s"'%Schema.BASE_DOCDEF,"/*/ftss:BaseNames/ftss:Base/@xlink:href", OBJECT_TYPE_RESOURCE))
    cur.append(('$uri','"%s"'%Schema.XML_VALIDATION_TYPE,"/*/ftss:CreationParams/ftss:Validator/@type", OBJECT_TYPE_RESOURCE))

    del cur


    #XPath Doc Def
    cur = systemDocDefs[ResourceTypes.ResourceType.XPATH_DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.XPATH_DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_CONTENT] = systemDocDefs[ResourceTypes.ResourceType.DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_CONTENT][:]
    del cur

    #Xslt Doc Def
    cur = systemDocDefs[ResourceTypes.ResourceType.XSLT_DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.XSLT_DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_CONTENT] = systemDocDefs[ResourceTypes.ResourceType.DOCUMENT_DEFINITION][ResourceTypes.RESOURCE_CONTENT][:]
    cur.append(('$uri','"%s"'%Schema.STYLESHEET_DEPENDENCY,"//xsl:include/@href", OBJECT_TYPE_RESOURCE))
    cur.append(('$uri','"%s"'%Schema.STYLESHEET_DEPENDENCY,"//xsl:import/@href", OBJECT_TYPE_RESOURCE))
    del cur


    #Xslt Document
    cur = systemDocDefs[ResourceTypes.ResourceType.XSLT_DOCUMENT][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.XSLT_DOCUMENT][ResourceTypes.RESOURCE_CONTENT] = systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_CONTENT][:]
    cur.append(('$uri','"%s"'%Schema.STYLESHEET_DEPENDENCY,"//xsl:include/@href", OBJECT_TYPE_RESOURCE))
    cur.append(('$uri','"%s"'%Schema.STYLESHEET_DEPENDENCY,"//xsl:import/@href", OBJECT_TYPE_RESOURCE))
    del cur

    #Commands
    cur = systemDocDefs[ResourceTypes.ResourceType.COMMAND][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.COMMAND][ResourceTypes.RESOURCE_CONTENT] = systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_CONTENT][:]
    cur.append(('$uri','"%s"'%Schema.COMMAND_NAME,"/ftss:Command/@name", OBJECT_TYPE_LITERAL))
    cur.append(('$uri','"%s"'%Schema.COMMAND_FULL_NAME,"/ftss:Command/@full-name", OBJECT_TYPE_LITERAL))
    cur.append(('$uri','"%s"'%Schema.COMMAND_SUBCOMMAND,"/ftss:Command/ftss:SubCommands/ftss:CommandReference/@xlink:href", OBJECT_TYPE_RESOURCE))
    del cur

    #Server
    cur = systemDocDefs[ResourceTypes.ResourceType.SERVER][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.SERVER][ResourceTypes.RESOURCE_CONTENT] = systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_CONTENT][:]
    cur.append(('$uri','"%s"'%Schema.SERVER_NAME,"/ftss:Server/@name", OBJECT_TYPE_LITERAL))
    cur.append(('$uri','"%s"'%Schema.SERVER_HANDLER,"/ftss:Server/ftss:Handler", OBJECT_TYPE_LITERAL))
    cur.append(('$uri','"%s"'%Schema.SERVER_RUNNING,"/ftss:Server/ftss:Status/@running", OBJECT_TYPE_LITERAL))
    del cur


    #RDF Documents
    cur = systemDocDefs[ResourceTypes.ResourceType.RDF_DOCUMENT][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.RDF_DOCUMENT][ResourceTypes.RESOURCE_CONTENT]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_CONTENT][:]
    del cur

    #Schematron Documents
    cur = systemDocDefs[ResourceTypes.ResourceType.SCHEMATRON_DOCUMENT][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.SCHEMATRON_DOCUMENT][ResourceTypes.RESOURCE_CONTENT]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_CONTENT][:]
    del cur

    #Alias
    cur = systemDocDefs[ResourceTypes.ResourceType.ALIAS][ResourceTypes.RESOURCE_METADATA]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_METADATA][:]
    del cur

    cur = systemDocDefs[ResourceTypes.ResourceType.ALIAS][ResourceTypes.RESOURCE_CONTENT]= systemDocDefs[ResourceTypes.ResourceType.XML_DOCUMENT][ResourceTypes.RESOURCE_CONTENT][:]
    cur.append(('$uri', '"%s"'%Schema.ALIAS_REFERENCE, "/ftss:Alias/@reference", OBJECT_TYPE_RESOURCE))
    del cur


    #Compile the system docdefs
    _systemDocDefs = {}
    for typ, modes in systemDocDefs.items():
        _systemDocDefs[typ] = {}

        for mode, exprs in modes.items():
            _systemDocDefs[typ][mode] = []
            for s, p, o, t in exprs:
                _systemDocDefs[typ][mode].append(
                    (XPath.Compile(s), XPath.Compile(p), XPath.Compile(o), t)
                    )
    del typ, modes
    del systemDocDefs


xu_reader = XUpdate.Reader()
NEW_CONTAINER_CHILD_XUPDATE_STRING = """<?xml version="1.0"?>
<xupdate:modifications
  version="1.0"
  xmlns:xupdate="http://www.xmldb.org/xupdate"
  xmlns:ftss="%s"
  xmlns:xlink="%s"
>
  <xupdate:append select="(ftss:Repository | ftss:Container)/ftss:Children" child="last()">
    <ftss:ChildReference xlink:type="simple" xlink:actuate="onLoad" xlink:show="embed">
      <xupdate:attribute name='xlink:href'><xupdate:value-of select='concat($child-name,";metadata")'/></xupdate:attribute>
    </ftss:ChildReference>
  </xupdate:append>
</xupdate:modifications>"""%(str(FTSERVER_NAMESPACE), str(XLINK_NAMESPACE))

DUMMY_URI = "urn:bogus:4suite-%s"
isrc = InputSource.DefaultFactory.fromString(NEW_CONTAINER_CHILD_XUPDATE_STRING, DUMMY_URI % 'xupdate-new-container-child')
NEW_CONTAINER_CHILD_XUPDATE = xu_reader.fromSrc(isrc)


DELETE_CONTAINER_CHILD_XUPDATE = """<xupdate:modifications
  version="1.0"
  xmlns:xupdate="http://www.xmldb.org/xupdate"
  xmlns:ftss="%s"
  xmlns:xlink="%s"
>
  <xupdate:remove select="/ftss:*/ftss:Children/ftss:ChildReference[@xlink:href=concat($child-name, ';metadata')]"/>
</xupdate:modifications>
""" % (str(FTSERVER_NAMESPACE), str(XLINK_NAMESPACE))
isrc = InputSource.DefaultFactory.fromString(DELETE_CONTAINER_CHILD_XUPDATE, DUMMY_URI % 'xupdate-del-container-child')
DELETE_CONTAINER_CHILD_XUPDATE = xu_reader.fromSrc(isrc)


UPDATE_LAST_MODIFIED_AND_SIZE_STRING = """<xupdate:modifications
  version="1.0"
  xmlns:xupdate="http://www.xmldb.org/xupdate"
  xmlns:ftss="%s"
>
  <xupdate:update select="/ftss:MetaData/ftss:LastModifiedDate"><xupdate:value-of select='$lmd'/></xupdate:update>
  <xupdate:update select="/ftss:MetaData/ftss:Size"><xupdate:value-of select='$size'/></xupdate:update>
</xupdate:modifications>
""" % str(FTSERVER_NAMESPACE)

isrc = InputSource.DefaultFactory.fromString(UPDATE_LAST_MODIFIED_AND_SIZE_STRING, DUMMY_URI % 'xupdate-lastmod-and-size')
UPDATE_LAST_MODIFIED_AND_SIZE = xu_reader.fromSrc(isrc)


UPDATE_LAST_MODIFIED = """<xupdate:modifications
  version="1.0"
  xmlns:xupdate="http://www.xmldb.org/xupdate"
  xmlns:ftss="%s"
>
  <xupdate:update select="/ftss:MetaData/ftss:LastModifiedDate">%s</xupdate:update>
</xupdate:modifications>
"""

UPDATE_SIZE = """<xupdate:modifications
  version="1.0"
  xmlns:xupdate="http://www.xmldb.org/xupdate"
  xmlns:ftss="%s"
>
  <xupdate:update select="/ftss:MetaData/ftss:Size">%s</xupdate:update>
</xupdate:modifications>
"""

UPDATE_ACL = """<xupdate:modifications
  version="1.0"
  xmlns:xupdate="http://www.xmldb.org/xupdate"
  xmlns:ftss="%s"
>
  <xupdate:remove select="/ftss:MetaData/ftss:Acl"/>
  <xupdate:append select="/ftss:MetaData">%s</xupdate:append>
</xupdate:modifications>
"""


def RecurPrint(obj, done=[]):
    import types
    import FtssDriver
    if type(obj) is types.InstanceType:
        done.append(obj)
        print "Instance: %s" % repr(obj)
        for name,value in obj.__dict__.items():
            print "Class Member %s:" % name
            if value in done:
                repr(obj)
            else:
                RecurPrint(value)
    elif type(obj) == type(FtssDriver):
        print repr(obj)
        print obj
        raise obj
    else:
        print repr(obj)



