########################################################################
#
#       License: BSD
#       Created: June 15, 2005
#       Author:  Antonio Valentino
#       Modified by:  Francesc Altet
#
#       $Source: /cvsroot/pytables/pytables/tables/Table.py,v $
#       $Id: Table.py 1009 2005-06-15 13:39:15Z faltet $
#
########################################################################

"""Here is defined the CArray class.

See CArray class docstring for more info.

Classes:

    CArray

Functions:


Misc variables:

    __version__


"""

import sys, warnings

import numarray
import numarray.records as records

try:
    import Numeric
    Numeric_imported = True
except ImportError:
    Numeric_imported = False

from tables.Atom import Atom
from tables.Array import Array
from tables.utils import processRangeRead


__version__ = "$Revision: 1.0 $"


# default version for CARRAY objects
obversion = "1.0"    # Support for time & enumerated datatypes.



class CArray(Array):
    """Represent an homogeneous dataset in HDF5 file.

    It enables to create new datasets on-disk from Numeric, numarray,
    lists, tuples, strings or scalars, or open existing ones.

    All Numeric and numarray typecodes are supported except for complex
    datatypes.

    Methods:

      Common to all Array's:
        read(start, stop, step)
        iterrows(start, stop, step)

    Instance variables:

      Common to all Array's:

        type -- The type class for the array.
        itemsize -- The size of the atomic items. Specially useful for
            CharArrays.
        flavor -- The flavor of this object.
        nrow -- On iterators, this is the index of the row currently
            dealed with.


    """

    # Class identifier.
    _c_classId = 'CARRAY'


    # <undo-redo support>
    _c_canUndoCreate = True  # Can creation/copying be undone and redone?
    _c_canUndoRemove = True  # Can removal be undone and redone?
    _c_canUndoMove   = True  # Can movement/renaming be undone and redone?
    # </undo-redo support>


    def __init__(self, shape = None, atom = None,
                 title = "", filters = None):
        """Create CArray instance.

        Keyword arguments:

        shape -- The shape of the chunked array to be saved.

        atom -- An Atom object representing the shape, type and flavor
            of the atomic objects to be saved. One of the shape
            dimensions must be 0. The dimension being 0 means that the
            resulting EArray object can be extended along it.

        title -- Sets a TITLE attribute on the array entity.

        filters -- An instance of the Filters class that provides
            information about the desired I/O filters to be applied
            during the life of this object.

        """
        self._v_new_title = title
        self._v_new_filters = filters
        self.extdim = -1   # An CArray object is not enlargeable
        # Check if we have to create a new object or read their contents
        # from disk
        if shape is not None and atom is not None:
            self._v_new = 1
            self.shape = tuple(shape)
            self.atom = atom
        else:
            self._v_new = 0
            if shape is not None:
                warnings.warn("""atom is None: shape ignored.""")
            if atom is not None:
                warnings.warn("""shape is None: atom ignored.""")

    def _calcBufferSize(self, atom, nrows, compress=None):
        """Calculate the buffer size and the maximum number of tuples.

        """

        # The buffer size
        expectedfsizeinKb = numarray.product(self.shape) * atom.itemsize / 1024
        buffersize = self._g_calcBufferSize(expectedfsizeinKb)

        # Max Tuples to fill the buffer
        maxTuples = buffersize // numarray.product(self.shape[1:])

        # Check if at least 1 tuple fits in buffer
        if maxTuples == 0:
            maxTuples = 1

        return (buffersize, maxTuples)

    def _create(self):
        """Create a fresh array (i.e., not present on HDF5 file)."""

        global obversion

        assert isinstance(self.shape, tuple), \
            "The shape passed to the CArray constructor has to be a tuple for CArrays, and you passed a '%s' object." % (self.shape)
        assert isinstance(self.atom, Atom), \
            "The object passed to the CArray constructor must be a descendent of the Atom class."
        assert isinstance(self.atom.shape, tuple), \
            "The Atom shape has to be a tuple for CArrays, and you passed a '%s' object." % (self.atom.shape)

        # Version, type, shape, flavor, byteorder
        self._v_version = obversion
        self.type = self.atom.type
        self.stype = self.atom.stype
        #self.shape = self.atom.shape
        self.flavor = self.atom.flavor
        if self.type == "CharType" or isinstance(self.type, records.Char):
            self.byteorder = "non-relevant"
        else:
            # Only support for creating objects in system byteorder
            self.byteorder  = sys.byteorder

        # Compute some values for buffering and I/O parameters
        # Compute the rowsize for each element
        self.rowsize = self.atom.itemsize
        for i in self.shape:
            if i>0:
                self.rowsize *= i
            else:
                raise ValueError, \
                      "A CArray object cannot have zero-dimensions."
        if self.shape:
            self._v_expectedrows = self.shape[0]
            self.nrows = self.shape[0]
        else:
            self._v_expectedrows = 1  # Scalar case
            self.nrows = 1    # Scalar case

        self.itemsize = self.atom.itemsize

        if min(self.atom.shape) < 1:
            raise ValueError, \
                  "Atom in CArray object cannot have zero-dimensions."

        self._v_chunksize = tuple(self.atom.shape)
        if len(self.shape) != len(self._v_chunksize):
            raise ValueError, "The Carray rank and atom rank must be equal:" \
                              "shape = %s, atom.shape = %s." % \
                                    (self.shape, self.atom.shape)

        # Compute the buffer chunksize
        (self._v_buffersize, self._v_maxTuples) = \
           self._calcBufferSize(self.atom, self.nrows, self.filters.complevel)

        try:
            self._createEArray(self._v_new_title)
        except:  #XXX
            # Problems creating the Array on disk. Close node and re-raise.
            self.close(flush=0)
            raise

    def _open(self):
        """Get the metadata info for an array in file."""
        (self.type, self.stype, self.shape, self.itemsize, self.byteorder,
         self._v_chunksize) = self._openArray()

        # Post-condition
        assert self.extdim == -1, "extdim != -1: this should never happen!"
        assert numarray.product(self._v_chunksize) > 0, \
                "product(self._v_chunksize) > 0: this should never happen!"

        # Compute the rowsize for each element
        self.rowsize = self.itemsize
        if self._v_chunksize:
            for i in xrange(len(self._v_chunksize)):
                self.rowsize *= self._v_chunksize[i]
        else:
            for i in xrange(len(self.shape)):
                self.rowsize *= self.shape[i]

        # Assign a value to nrows in case we are a non-enlargeable object
        if self.shape:
            self.nrows = self.shape[0]
        else:
            self.nrows = 1   # Scalar case

        # Compute the real shape for atom:
        shape = list(self._v_chunksize)
        if self.type == "CharType" or isinstance(self.type, records.Char):
            # Add the length of the array at the end of the shape for atom
            shape.append(self.itemsize)
        shape = tuple(shape)

        # Create the atom instance
        self.atom = Atom(dtype=self.stype, shape=shape, flavor=self.flavor)

        # Compute the buffer chunksize
        (self._v_buffersize, self._v_maxTuples) = \
           self._calcBufferSize(self.atom, self.nrows)

    def _g_copyWithStats(self, group, name, start, stop, step, title, filters):
        "Private part of Leaf.copy() for each kind of leaf"
        # Now, fill the new carray with values from source
        nrowsinbuf = self._v_maxTuples
        # The slices parameter for self.__getitem__
        slices = [slice(0, dim, 1) for dim in self.shape]
        # This is a hack to prevent doing innecessary conversions
        # when copying buffers
        (start, stop, step) = processRangeRead(self.nrows, start, stop, step)
        self._v_convert = 0
        shape = list(self.shape)
        shape[0] = ((stop - start - 1) / step) + 1
        # Build the new CArray object
        object = self._v_file.createCArray(group, name, shape, atom=self.atom,
                                title=title, filters=filters, _log = False)
        # Start the copy itself
        for start2 in range(start, stop, step*nrowsinbuf):
            # Save the records on disk
            stop2 = start2+step*nrowsinbuf
            if stop2 > stop:
                stop2 = stop
            # Set the proper slice in the first dimension
            slices[0] = slice(start2, stop2, step)
            start3 = (start2-start)/step
            stop3 = start3 + nrowsinbuf
            if stop3 > shape[0]:
                stop3 = shape[0]
            object[start3:stop3] = self.__getitem__(tuple(slices))
        # Active the conversion again (default)
        self._v_convert = 1
        nbytes = numarray.product(self.shape)*self.itemsize

        return (object, nbytes)

    def __repr__(self):
        """This provides more metainfo in addition to standard __str__"""

        return """%s
  atom = %r
  nrows = %s
  flavor = %r
  byteorder = %r""" % (self, self.atom, self.nrows, self.flavor, self.byteorder)
