#include "H5TB-opt.h"

#include <stdlib.h>
#include <string.h>

/*-------------------------------------------------------------------------
 * Function: H5TBOopen_read
 *
 * Purpose: Open a table to be read
 *
 * Return: Success: 0, Failure: -1
 *
 * Programmer: Francesc Altet, faltet@carabos.com
 *
 * Date: April 19, 2003
 *
 * Comments:
 *
 * Modifications:
 *
 *
 *-------------------------------------------------------------------------
 */

herr_t H5TBOopen_read( hid_t *dataset_id,
		       hid_t *space_id,
		       hid_t loc_id,
		       const char *dset_name)
{

 /* Open the dataset. */
 if ( (*dataset_id = H5Dopen( loc_id, dset_name )) < 0 )
  return -1;

 /* Get the dataspace handle */
 if ( (*space_id = H5Dget_space( *dataset_id )) < 0 )
  goto out;

return 0;

out:
 H5Dclose( *dataset_id );
 return -1;

}

/*-------------------------------------------------------------------------
 * Function: H5TBOread_records
 *
 * Purpose: Read records from an opened table
 *
 * Return: Success: 0, Failure: -1
 *
 * Programmer: Francesc Altet, faltet@carabos.com
 *
 * Date: April 19, 2003
 *
 * Comments:
 *
 * Modifications:
 *
 *
 *-------------------------------------------------------------------------
 */

herr_t H5TBOread_records( hid_t *dataset_id,
			  hid_t *space_id,
			  hid_t mem_type_id,
			  hsize_t start,
			  hsize_t nrecords,
			  void *data )
{

 hsize_t  count[1];
 hssize_t offset[1];
 hid_t    mem_space_id;

 /* Define a hyperslab in the dataset of the size of the records */
 offset[0] = start;
 count[0]  = nrecords;
 if ( H5Sselect_hyperslab( *space_id, H5S_SELECT_SET, offset, NULL, count, NULL) < 0 )
  goto out;

 /* Create a memory dataspace handle */
 if ( (mem_space_id = H5Screate_simple( 1, count, NULL )) < 0 )
  goto out;

 if ( H5Dread( *dataset_id, mem_type_id, mem_space_id, *space_id, H5P_DEFAULT, data ) < 0 )
  goto out;

 /* Terminate access to the memory dataspace */
 if ( H5Sclose( mem_space_id ) < 0 )
  goto out;

return 0;

out:
 H5Dclose( *dataset_id );
 return -1;

}

/*-------------------------------------------------------------------------
 * Function: H5TBOread_records
 *
 * Purpose: Read selected records from an opened table
 *
 * Return: Success: 0, Failure: -1
 *
 * Programmer: Francesc Altet, faltet@carabos.com
 *
 * Date: April 19, 2003
 *
 * Comments:
 *
 * Modifications:
 *
 *
 *-------------------------------------------------------------------------
 */

herr_t H5TBOread_elements( hid_t *dataset_id,
			   hid_t *space_id,
			   hid_t mem_type_id,
			   hsize_t nrecords,
			   void *coords,
			   void *data )
{

 hid_t    mem_space_id;
 hsize_t  count[1];

 /* Define a selection of points in the dataset */
 H5Sselect_none(*space_id); 	/* Delete the previous selection */

 if ( H5Sselect_elements(*space_id, H5S_SELECT_SET, (size_t)nrecords, (const hsize_t **)coords) < 0 )
  goto out;

 /* Create a memory dataspace handle */
 count[0] = nrecords;
 if ( (mem_space_id = H5Screate_simple( 1, count, NULL )) < 0 )
  goto out;

 if ( H5Dread( *dataset_id, mem_type_id, mem_space_id, *space_id, H5P_DEFAULT, data ) < 0 )
  goto out;

 /* Terminate access to the memory dataspace */
 if ( H5Sclose( mem_space_id ) < 0 )
  goto out;

return 0;

out:
 H5Dclose( *dataset_id );
 return -1;

}

/*-------------------------------------------------------------------------
 * Function: H5TBOclose_read
 *
 * Purpose: Close a table that has been opened for reading
 *
 * Return: Success: 0, Failure: -1
 *
 * Programmer: Francesc Altet, faltet@carabos.com
 *
 * Date: April 19, 2003
 *
 * Comments:
 *
 * Modifications:
 *
 *
 *-------------------------------------------------------------------------
 */

herr_t H5TBOclose_read(hid_t *dataset_id,
		       hid_t *space_id)
{

 /* Terminate access to the dataspace */
 if ( H5Sclose( *space_id ) < 0 )
  goto out;

 /* End access to the dataset */
 if ( H5Dclose( *dataset_id ) < 0 )
  return -1;

return 0;

out:
 H5Dclose( *dataset_id );
 return -1;

}

/* From here on, similar funtions are provided for appending.
 */

/*-------------------------------------------------------------------------
 * Function: H5TBOopen_append
 *
 * Purpose: Prepare a table to append records
 *
 * Return: Success: 0, Failure: -1
 *
 * Programmer: Francesc Altet, faltet@carabos.com
 *
 * Date: April 20, 2003
 *
 * Comments:
 *
 * Modifications:
 *
 *
 *-------------------------------------------------------------------------
 */

herr_t H5TBOopen_append( hid_t *dataset_id,
			 hid_t loc_id,
			 const char *dset_name)
{

 /* Open the dataset. */
 if ( (*dataset_id = H5Dopen( loc_id, dset_name )) < 0 )
  goto out;

 return 0;

out:
 H5Dclose( *dataset_id );
 return -1;

}


/*-------------------------------------------------------------------------
 * Function: H5TBOappend_records
 *
 * Purpose: Appends records to a table
 *
 * Return: Success: 0, Failure: -1
 *
 * Programmers:
 *  Francesc Altet, faltet@carabos.com
 *
 * Date: April 20, 2003
 *
 * Comments: Uses memory offsets
 *
 * Modifications:
 *
 *
 *-------------------------------------------------------------------------
 */


herr_t H5TBOappend_records( hid_t *dataset_id,
			    hid_t mem_type_id,
			    hsize_t nrecords,
			    hsize_t nrecords_orig,
			    const void *data )
{
 hid_t    space_id = -1; 	/* Shut up the compiler */
 hsize_t  count[1];
 hssize_t offset[1];
 hid_t    mem_space_id = -1;    /* Shut up the compiler */
 hsize_t  dims[1];


 /* Extend the dataset */
 dims[0] = nrecords_orig;
 dims[0] += nrecords;
 if ( H5Dextend ( *dataset_id, dims ) < 0 )
  goto out;

 /* Create a simple memory data space */
 count[0]=nrecords;
 if ( (mem_space_id = H5Screate_simple( 1, count, NULL )) < 0 )
  return -1;

 /* Get the file data space */
 if ( (space_id = H5Dget_space( *dataset_id )) < 0 )
  return -1;

 /* Define a hyperslab in the dataset */
 offset[0] = nrecords_orig;
 if ( H5Sselect_hyperslab( space_id, H5S_SELECT_SET, offset, NULL, count, NULL) < 0 )
  goto out;

 if ( H5Dwrite( *dataset_id, mem_type_id, mem_space_id, space_id, H5P_DEFAULT, data ) < 0 )
  goto out;

 /* Terminate access to the dataspace */
 if ( H5Sclose( mem_space_id ) < 0 )
  goto out;

 if ( H5Sclose( space_id ) < 0 )
  goto out;

return 0;

out:
 H5E_BEGIN_TRY {
  H5Dclose(*dataset_id);
  H5Sclose(mem_space_id);
  H5Sclose(space_id);
 } H5E_END_TRY;
 return -1;

}

/*-------------------------------------------------------------------------
 * Function: H5TBOclose_append
 *
 * Purpose: Close a table that has been opened for append
 *
 * Return: Success: 0, Failure: -1
 *
 * Programmer: Francesc Altet, faltet@carabos.com
 *
 * Date: April 20, 2003
 *
 * Comments:
 *
 * Modifications:
 *
 *
 *-------------------------------------------------------------------------
 */

herr_t H5TBOclose_append(hid_t *dataset_id,
			 hsize_t ntotal_records,
			 const char *dset_name,
			 hid_t parent_id)
{

 /* End access to the dataset */
 if ( H5Dclose( *dataset_id ) < 0 )
  return -1;

/*-------------------------------------------------------------------------
 * Store the new dimension as an attribute
 *-------------------------------------------------------------------------
 */

 /* Set the attribute */
 if (H5LT_set_attribute_numerical(parent_id, dset_name, "NROWS",
				  H5T_NATIVE_LLONG, &ntotal_records)<0)
   return -1;

return 0;

out:
 H5Dclose( *dataset_id );
 return -1;

}

/*-------------------------------------------------------------------------
 * Function: H5TBOwrite_records
 *
 * Purpose: Writes records
 *
 * Return: Success: 0, Failure: -1
 *
 * Programmer: Pedro Vicente, pvn@ncsa.uiuc.edu
 *
 * Date: November 19, 2001
 *
 * Comments: Uses memory offsets
 *
 * Modifications:
 * -  Added a step parameter in order to support strided writing.
 *    Francesc Altet, faltet@carabos.com. 2004-08-12
 *
 *
 *-------------------------------------------------------------------------
 */


herr_t H5TBOwrite_records( hid_t loc_id,
			   const char *dset_name,
			   hsize_t start,
			   hsize_t nrecords,
			   hsize_t step,
			   size_t type_size,
			   hid_t mem_type_id,
			   const void *data )
{

 hid_t    dataset_id;
 hsize_t  count[1];
 hsize_t  stride[1];
 hssize_t offset[1];
 hid_t    space_id;
 hid_t    mem_space_id;
 hsize_t  dims[1];

 /* Open the dataset. */
 if ( (dataset_id = H5Dopen( loc_id, dset_name )) < 0 )
  return -1;

 /* Get the dataspace handle */
 if ( (space_id = H5Dget_space( dataset_id )) < 0 )
  goto out;

  /* Get records */
 if ( H5Sget_simple_extent_dims( space_id, dims, NULL) < 0 )
  goto out;

/*  if ( start + nrecords > dims[0] ) */
 if ( start + (nrecords-1) * step + 1 > dims[0] )
  goto out;

 /* Define a hyperslab in the dataset of the size of the records */
 offset[0] = start;
 stride[0] = step;
 count[0] = nrecords;
 if ( H5Sselect_hyperslab( space_id, H5S_SELECT_SET, offset, stride, count, NULL) < 0 )
  goto out;

 /* Create a memory dataspace handle */
 if ( (mem_space_id = H5Screate_simple( 1, count, NULL )) < 0 )
  goto out;

 if ( H5Dwrite( dataset_id, mem_type_id, mem_space_id, space_id, H5P_DEFAULT, data ) < 0 )
  goto out;

 /* Terminate access to the memory dataspace */
 if ( H5Sclose( mem_space_id ) < 0 )
  goto out;

 /* Terminate access to the dataspace */
 if ( H5Sclose( space_id ) < 0 )
  goto out;

 /* End access to the dataset */
 if ( H5Dclose( dataset_id ) < 0 )
  return -1;

return 0;

out:
 H5Dclose( dataset_id );
 return -1;

}


