#emacs: -*- mode: python-mode; py-indent-offset: 4; indent-tabs-mode: nil -*-
#ex: set sts=4 ts=4 sw=4 et:
### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ##
#
#   See COPYING file distributed along with the PyMVPA package for the
#   copyright and license terms.
#
### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ### ##
"""Base classes for all classifiers.

Base Classifiers can be grouped according to their function as

:group Basic Classifiers: Classifier BoostedClassifier ProxyClassifier
:group BoostedClassifiers: CombinedClassifier MulticlassClassifier
  SplitClassifier
:group ProxyClassifiers: BinaryClassifier MappedClassifier
  FeatureSelectionClassifier
:group PredictionsCombiners for CombinedClassifier: PredictionsCombiner
  MaximalVote MeanPrediction

"""

__docformat__ = 'restructuredtext'

import operator, sys
import numpy as N

# We have to use deepcopy from python 2.5, since otherwise it fails to
# copy sensitivity analyzers with assigned combiners which are just
# functions not functors
if sys.version_info[0] > 2 or sys.version_info[1] > 4:
    from mvpa.misc.copy import deepcopy
else:
    from mvpa.misc.copy import deepcopy

import time
from sets import Set

from mvpa.misc.args import group_kwargs
from mvpa.misc.support import idhash
from mvpa.mappers.mask import MaskMapper
from mvpa.datasets.splitter import NFoldSplitter
from mvpa.misc.state import StateVariable, Stateful, Harvestable, Parametrized
from mvpa.misc.param import Parameter

from mvpa.clfs.transerror import ConfusionMatrix, RegressionStatistics
from mvpa.misc.transformers import FirstAxisMean, SecondAxisSumOfAbs

from mvpa.measures.base import \
    BoostedClassifierSensitivityAnalyzer, ProxyClassifierSensitivityAnalyzer, \
    MappedClassifierSensitivityAnalyzer
from mvpa.base import warning

if __debug__:
    import traceback
    from mvpa.base import debug


def _deepcopyclf(clf):
    """Deepcopying of a classifier.

    If deepcopy fails -- tries to untrain it first so that there is no
    swig bindings attached
    """
    try:
        return deepcopy(clf)
    except:
        clf.untrain()
        return deepcopy(clf)


class Classifier(Parametrized):
    """Abstract classifier class to be inherited by all classifiers
    """

    # Kept separate from doc to don't pollute help(clf), especially if
    # we including help for the parent class
    _DEV__doc__ = """
    Required behavior:

    For every classifier is has to be possible to be instanciated without
    having to specify the training pattern.

    Repeated calls to the train() method with different training data have to
    result in a valid classifier, trained for the particular dataset.

    It must be possible to specify all classifier parameters as keyword
    arguments to the constructor.

    Recommended behavior:

    Derived classifiers should provide access to *values* -- i.e. that
    information that is finally used to determine the predicted class label.

    Michael: Maybe it works well if each classifier provides a 'values'
             state member. This variable is a list as long as and in same order
             as Dataset.uniquelabels (training data). Each item in the list
             corresponds to the likelyhood of a sample to belong to the
             respective class. However the sematics might differ between
             classifiers, e.g. kNN would probably store distances to class-
             neighbours, where PLR would store the raw function value of the
             logistic function. So in the case of kNN low is predictive and for
             PLR high is predictive. Don't know if there is the need to unify
             that.

             As the storage and/or computation of this information might be
             demanding its collection should be switchable and off be default.

    Nomenclature
     * predictions  : corresponds to the quantized labels if classifier spits
                      out labels by .predict()
     * values : might be different from predictions if a classifier's predict()
                   makes a decision based on some internal value such as
                   probability or a distance.
    """
    # Dict that contains the parameters of a classifier.
    # This shall provide an interface to plug generic parameter optimizer
    # on all classifiers (e.g. grid- or line-search optimizer)
    # A dictionary is used because Michael thinks that access by name is nicer.
    # Additonally Michael thinks ATM that additonal information might be
    # necessary in some situations (e.g. reasonably predefined parameter range,
    # minimal iteration stepsize, ...), therefore the value to each key should
    # also be a dict or we should use mvpa.misc.param.Parameter'...

    trained_labels = StateVariable(enabled=True,
        doc="Set of unique labels it has been trained on")

    trained_dataset = StateVariable(enabled=False,
        doc="The dataset it has been trained on")

    training_confusion = StateVariable(enabled=False,
        doc="Confusion matrix of learning performance")

    predictions = StateVariable(enabled=True,
        doc="Most recent set of predictions")

    values = StateVariable(enabled=True,
        doc="Internal classifier values the most recent " +
            "predictions are based on")

    training_time = StateVariable(enabled=True,
        doc="Time (in seconds) which took classifier to train")

    predicting_time = StateVariable(enabled=True,
        doc="Time (in seconds) which took classifier to predict")

    feature_ids = StateVariable(enabled=False,
        doc="Feature IDS which were used for the actual training.")

    _clf_internals = []
    """Describes some specifics about the classifier -- is that it is
    doing regression for instance...."""

    regression = Parameter(False, allowedtype='bool',
        doc="""Either to use 'regression' as regression. By default any
        Classifier-derived class serves as a classifier, so regression
        does binary classification. TODO:""")

    retrainable = Parameter(False, allowedtype='bool',
        doc="""Either to enable retraining for 'retrainable' classifier.
        TODO: make it available only for actually retrainable classifiers""")


    def __init__(self, **kwargs):
        """Cheap initialization.
        """
        Parametrized.__init__(self, **kwargs)


        self.__trainednfeatures = None
        """Stores number of features for which classifier was trained.
        If None -- it wasn't trained at all"""

        self._setRetrainable(self.params.retrainable, force=True)

        if self.params.regression:
            for statevar in [ "trained_labels"]: #, "training_confusion" ]:
                if self.states.isEnabled(statevar):
                    if __debug__:
                        debug("CLF",
                              "Disabling state %s since doing regression, " %
                              statevar + "not classification")
                    self.states.disable(statevar)
            self._summaryClass = RegressionStatistics
        else:
            self._summaryClass = ConfusionMatrix
            if 'regression' in self._clf_internals:
                # regressions are used as binary classifiers if not asked to perform
                # regression explicitely
                self._clf_internals.append('binary')

        # deprecate
        #self.__trainedidhash = None
        #"""Stores id of the dataset on which it was trained to signal
        #in trained() if it was trained already on the same dataset"""


    def __str__(self):
        if __debug__ and 'CLF_' in debug.active:
            return "%s / %s" % (repr(self), super(Classifier, self).__str__())
        else:
            return repr(self)

    def __repr__(self, prefixes=[]):
        return super(Classifier, self).__repr__(prefixes=prefixes)


    def _pretrain(self, dataset):
        """Functionality prior to training
        """
        # So we reset all state variables and may be free up some memory
        # explicitely
        params = self.params
        if not params.retrainable:
            self.untrain()
        else:
            # just reset the states, do not untrain
            self.states.reset()
            if not self.__changedData_isset:
                self.__resetChangedData()
                _changedData = self._changedData
                __idhashes = self.__idhashes
                __invalidatedChangedData = self.__invalidatedChangedData

                # if we don't know what was changed we need to figure
                # them out
                if __debug__:
                    debug('CLF_', "IDHashes are %s" % (__idhashes))

                # Look at the data if any was changed
                for key, data_ in (('traindata', dataset.samples),
                                   ('labels', dataset.labels)):
                    _changedData[key] = self.__wasDataChanged(key, data_)
                    # if those idhashes were invalidated by retraining
                    # we need to adjust _changedData accordingly
                    if __invalidatedChangedData.get(key, False):
                        if __debug__ and not _changedData[key]:
                            debug('CLF_', 'Found that idhash for %s was '
                                  'invalidated by retraining' % key)
                        _changedData[key] = True

                # Look at the parameters
                for col in self._paramscols:
                    changedParams = self._collections[col].whichSet()
                    if len(changedParams):
                        _changedData[col] = changedParams

                self.__invalidatedChangedData = {} # reset it on training

                if __debug__:
                    debug('CLF_', "Obtained _changedData is %s" % (self._changedData))

        if not params.regression and 'regression' in self._clf_internals \
           and not self.states.isEnabled('trained_labels'):
            # if classifier internally does regression we need to have
            # labels it was trained on
            if __debug__:
                debug("CLF", "Enabling trained_labels state since it is needed")
            self.states.enable('trained_labels')


    def _posttrain(self, dataset):
        """Functionality post training

        For instance -- computing confusion matrix
        :Parameters:
          dataset : Dataset
            Data which was used for training
        """
        if self.states.isEnabled('trained_labels'):
            self.trained_labels = dataset.uniquelabels

        self.trained_dataset = dataset

        # needs to be assigned first since below we use predict
        self.__trainednfeatures = dataset.nfeatures

        # XXX seems to be not even needed
        # self.__trained_labels_map = dataset.labels_map

        if __debug__ and 'CHECK_TRAINED' in debug.active:
            self.__trainedidhash = dataset.idhash

        if self.states.isEnabled('training_confusion') and \
               not self.states.isSet('training_confusion'):
            # we should not store predictions for training data,
            # it is confusing imho (yoh)
            self.states._changeTemporarily(
                disable_states=["predictions"])
            if self.params.retrainable:
                # we would need to recheck if data is the same,
                # XXX think if there is a way to make this all
                # efficient. For now, probably, retrainable
                # classifiers have no chance but not to use
                # training_confusion... sad
                self.__changedData_isset = False
            predictions = self.predict(dataset.samples)
            self.states._resetEnabledTemporarily()
            self.training_confusion = self._summaryClass(
                targets=dataset.labels,
                predictions=predictions)

            try:
                self.training_confusion.labels_map = dataset.labels_map
            except:
                pass

        if self.states.isEnabled('feature_ids'):
            self.feature_ids = self._getFeatureIds()


    def _getFeatureIds(self):
        """Virtual method to return feature_ids used while training

        Is not intended to be called anywhere but from _posttrain,
        thus classifier is assumed to be trained at this point
        """
        # By default all features are used
        return range(self.__trainednfeatures)


    def summary(self):
        """Providing summary over the classifier"""

        s = "Classifier %s" % self
        states = self.states
        states_enabled = states.enabled

        if self.trained:
            s += "\n trained"
            if states.isSet('training_time'):
                s += ' in %.3g sec' % states.training_time
            s += ' on data with'
            if states.isSet('trained_labels'):
                s += ' labels:%s' % list(states.trained_labels)
            if states.isSet('trained_dataset'):
                td = states.trained_dataset
                s += ' #samples:%d #chunks:%d' % (td.nsamples, len(td.uniquechunks))
            s += " #features:%d" % self.__trainednfeatures
            if states.isSet('feature_ids'):
                s += ", used #features:%d" % len(states.feature_ids)
            if states.isSet('training_confusion'):
                s += ", training error:%.3g" % states.training_confusion.error
        else:
            s += "\n not yet trained"

        if len(states_enabled):
            s += "\n enabled states:%s" % ', '.join([str(states[x]) for x in states_enabled])
        return s


    def _train(self, dataset):
        """Function to be actually overriden in derived classes
        """
        raise NotImplementedError


    def train(self, dataset):
        """Train classifier on a dataset

        Shouldn't be overriden in subclasses unless explicitely needed
        to do so
        """
        if __debug__:
            debug("CLF", "Training classifier %(clf)s on dataset %(dataset)s",
                  msgargs={'clf':self, 'dataset':dataset})
            #if 'CLF_TB' in debug.active:
            #    tb = traceback.extract_stack(limit=5)
            #    debug("CLF_TB", "Traceback: %s" % tb)

        self._pretrain(dataset)

        # remember the time when started training
        t0 = time.time()

        if dataset.nfeatures > 0:
            result = self._train(dataset)
        else:
            warning("Trying to train on dataset with no features present")
            if __debug__:
                debug("CLF",
                      "No features present for training, no actual training " \
                      "is called")
            result = None

        self.training_time = time.time() - t0
        self._posttrain(dataset)
        return result


    def _prepredict(self, data):
        """Functionality prior prediction
        """
        if not ('notrain2predict' in self._clf_internals):
            # check if classifier was trained if that is needed
            if not self.trained:
                raise ValueError, \
                      "Classifier %s wasn't yet trained, therefore can't " \
                      "predict" % self
            nfeatures = data.shape[1]
            # check if number of features is the same as in the data
            # it was trained on
            if nfeatures != self.__trainednfeatures:
                raise ValueError, \
                      "Classifier %s was trained on data with %d features, " % \
                      (self, self.__trainednfeatures) + \
                      "thus can't predict for %d features" % nfeatures


        if self.params.retrainable:
            if not self.__changedData_isset:
                self.__resetChangedData()
                _changedData = self._changedData
                _changedData['testdata'] = \
                                        self.__wasDataChanged('testdata', data)
                if __debug__:
                    debug('CLF_', "prepredict: Obtained _changedData is %s" % (_changedData))


    def _postpredict(self, data, result):
        """Functionality after prediction is computed
        """
        self.predictions = result
        if self.params.retrainable:
            self.__changedData_isset = False

    def _predict(self, data):
        """Actual prediction
        """
        raise NotImplementedError


    def predict(self, data):
        """Predict classifier on data

        Shouldn't be overriden in subclasses unless explicitely needed
        to do so. Also subclasses trying to call super class's predict
        should call _predict if within _predict instead of predict()
        since otherwise it would loop
        """
        data = N.asarray(data)
        if __debug__:
            debug("CLF", "Predicting classifier %(clf)s on data %(data)s",
                msgargs={'clf':self, 'data':data.shape})
            #if 'CLF_TB' in debug.active:
            #    tb = traceback.extract_stack(limit=5)
            #    debug("CLF_TB", "Traceback: %s" % tb)

        # remember the time when started computing predictions
        t0 = time.time()

        states = self.states
        # to assure that those are reset (could be set due to testing
        # post-training)
        states.reset(['values', 'predictions'])

        self._prepredict(data)

        if self.__trainednfeatures > 0 \
               or 'notrain2predict' in self._clf_internals:
            result = self._predict(data)
        else:
            warning("Trying to predict using classifier trained on no features")
            if __debug__:
                debug("CLF",
                      "No features were present for training, prediction is " \
                      "bogus")
            result = [None]*data.shape[0]

        states.predicting_time = time.time() - t0

        if 'regression' in self._clf_internals and not self.params.regression:
            # We need to convert regression values into labels
            # XXX unify may be labels -> internal_labels conversion.
            #if len(self.trained_labels) != 2:
            #    raise RuntimeError, "XXX Ask developer to implement for " \
            #        "multiclass mapping from regression into classification"

            # must be N.array so we copy it to assign labels directly
            # into labels.
            # XXX or should we just recreate "result"
            result_ = N.array(result)
            if states.isEnabled('values'):
                # values could be set by now so assigning 'result' would
                # be misleading
                if not states.isSet('values'):
                    states.values = result_.copy()
                else:
                    # it might be the values are pointing to result at
                    # the moment, so lets assure this silly way that
                    # they do not overlap
                    states.values = states.values.copy()

            trained_labels = self.trained_labels
            for i, value in enumerate(result):
                dists = N.abs(value - trained_labels)
                result[i] = trained_labels[N.argmin(dists)]

            if __debug__:
                debug("CLF_", "Converted regression result %(result_)s "
                      "into labels %(result)s for %(self_)s",
                      msgargs={'result_':result_, 'result':result,
                               'self_': self})

        self._postpredict(data, result)
        return result

    # XXX deprecate ?
    def isTrained(self, dataset=None):
        """Either classifier was already trained.

        MUST BE USED WITH CARE IF EVER"""
        if dataset is None:
            # simply return if it was trained on anything
            return not self.__trainednfeatures is None
        else:
            res = (self.__trainednfeatures == dataset.nfeatures)
            if __debug__ and 'CHECK_TRAINED' in debug.active:
                res2 = (self.__trainedidhash == dataset.idhash)
                if res2 != res:
                    raise RuntimeError, \
                          "isTrained is weak and shouldn't be relied upon. " \
                          "Got result %b although comparing of idhash says %b" \
                          % (res, res2)
            return res


    def _regressionIsBogus(self):
        """Some classifiers like BinaryClassifier can't be used for
        regression"""

        if self.params.regression:
            raise ValueError, "Regression mode is meaningless for %s" % \
                  self.__class__.__name__ + " thus don't enable it"


    @property
    def trained(self):
        """Either classifier was already trained"""
        return self.isTrained()

    def untrain(self):
        """Reset trained state"""
        self.__trainednfeatures = None
        # probably not needed... retrainable shouldn't be fully untrained
        # XXX or should be??
        #if self.params.retrainable:
        #    # XXX don't duplicate the code ;-)
        #    self.__idhashes = {'traindata': None, 'labels': None,
        #                       'testdata': None, 'testtraindata': None}
        super(Classifier, self).reset()


    def getSensitivityAnalyzer(self, **kwargs):
        """Factory method to return an appropriate sensitivity analyzer for
        the respective classifier."""
        raise NotImplementedError


    #
    # Methods which are needed for retrainable classifiers
    #
    def _setRetrainable(self, value, force=False):
        """Assign value of retrainable parameter

        If retrainable flag is to be changed, classifier has to be
        untrained.  Also internal attributes such as _changedData,
        __changedData_isset, and __idhashes should be initialized if
        it becomes retrainable
        """
        pretrainable = self.params['retrainable']
        if (force or value != pretrainable.value) and 'retrainable' in self._clf_internals:
            if __debug__:
                debug("CLF_", "Setting retrainable to %s" % value)
            if 'meta' in self._clf_internals:
                warning("Retrainability is not yet crafted/tested for "
                        "meta classifiers. Unpredictable behavior might occur")
            # assure that we don't drag anything behind
            if self.trained:
                self.untrain()
            states = self.states
            if not value and states.isKnown('retrained'):
                states.remove('retrained')
                states.remove('repredicted')
            if value:
                if not 'retrainable' in self._clf_internals:
                    warning("Setting of flag retrainable for %s has no effect"
                            " since classifier has no such capability. It would"
                            " just lead to resources consumption and slowdown"
                            % self)
                states.add(StateVariable(enabled=True,
                        name='retrained',
                        doc="Either retrainable classifier was retrained"))
                states.add(StateVariable(enabled=True,
                        name='repredicted',
                        doc="Either retrainable classifier was repredicted"))

            pretrainable.value = value

            # if retrainable we need to keep track of things
            if value:
                self.__idhashes = {'traindata': None, 'labels': None,
                                   'testdata': None} #, 'testtraindata': None}
                if __debug__ and 'CHECK_RETRAIN' in debug.active:
                    # XXX it is not clear though if idhash is faster than
                    # simple comparison of (dataset != __traineddataset).any(),
                    # but if we like to get rid of __traineddataset then we should
                    # use idhash anyways
                    self.__trained = self.__idhashes.copy() # just the same Nones ;-)
                self.__resetChangedData()
                self.__invalidatedChangedData = {}
            elif 'retrainable' in self._clf_internals:
                #self.__resetChangedData()
                self.__changedData_isset = False
                self._changedData = None
                self.__idhashes = None
                if __debug__ and 'CHECK_RETRAIN' in debug.active:
                    self.__trained = None

    def __resetChangedData(self):
        """For retrainable classifier we keep track of what was changed
        This function resets that dictionary
        """
        if __debug__:
            debug('CLF_', 'Resetting flags on either data was changed (for retrainable)')
        keys = self.__idhashes.keys() + self._paramscols
        # XXX we might like just to reinit values to False
        #_changedData = self._changedData
        #if isinstance(_changedData, dict):
        #    for key in _changedData.keys():
        #        _changedData[key] = False
        self._changedData = dict(zip(keys, [False]*len(keys)))
        self.__changedData_isset = False


    def __wasDataChanged(self, key, entry, update=True):
        """Check if given entry was changed from what known prior. If so -- store

        needed only for retrainable beastie
        """
        idhash_ = idhash(entry)
        __idhashes = self.__idhashes

        changed = __idhashes[key] != idhash_
        if __debug__ and 'CHECK_RETRAIN' in debug.active:
            __trained = self.__trained
            changed2 = entry != __trained[key]
            if isinstance(changed2, N.ndarray):
                changed2 = changed2.any()
            if changed != changed2 and not changed:
                raise RuntimeError, \
                  'idhash found to be weak for %s. Though hashid %s!=%s %s, '\
                  'values %s!=%s %s' % \
                  (key, idhash_, __idhashes[key], changed,
                   entry, __trained[key], changed2)
            if update:
                __trained[key] = entry

        if __debug__ and changed:
            debug('CLF_', "Changed %s from %s to %s.%s"
                      % (key, __idhashes[key], idhash_,
                         ('','updated')[int(update)]))
        if update:
            __idhashes[key] = idhash_

        return changed


    # def __updateHashIds(self, key, data):
    #     """Is twofold operation: updates hashid if was said that it changed.
    #
    #     or if it wasn't said that data changed, but CHECK_RETRAIN and it found
    #     to be changed -- raise Exception
    #     """
    #
    #     check_retrain = __debug__ and 'CHECK_RETRAIN' in debug.active
    #     chd = self._changedData
    #
    #     # we need to updated idhashes
    #     if chd[key] or check_retrain:
    #         keychanged = self.__wasDataChanged(key, data)
    #     if check_retrain and keychanged and not chd[key]:
    #         raise RuntimeError, \
    #               "Data %s found changed although wasn't " \
    #               "labeled as such" % key


    #
    # Additional API which is specific only for retrainable classifiers.
    # For now it would just puke if asked from not retrainable one XXX
    #
    # Might come useful and efficient for statistics testing, so if just
    # labels of dataset changed, then
    #  self.retrain(dataset, labels=True)
    # would cause efficient retraining (no kernels recomputed etc)
    # and subsequent self.repredict(data) should be also quite fase ;-)

    def retrain(self, dataset, **kwargs):
        """Helper to avoid check if data was changed actually changed

        Useful if just some aspects of classifier were changed since
        its previous training. For instance if dataset wasn't changed
        but only classifier parameters, then kernel matrix does not
        have to be computed.

        Words of caution: classifier must be previousely trained,
        results always should first be compared to the results on not
        'retrainable' classifier (without calling retrain). Some
        additional checks are enabled if debug id 'CHECK_RETRAIN' is
        enabled, to guard against obvious mistakes.

        :Parameters:
          kwargs
            that is what _changedData gets updated with. So, smth like
            ``(params=['C'], labels=True)`` if parameter C and labels
            got changed
        """
        # Note that it also demolishes anything for repredicting,
        # which should be ok in most of the cases
        if __debug__:
            if not self.params.retrainable:
                raise RuntimeError, \
                      "Do not use retrain/repredict on non-retrainable classifiers"

            if kwargs.has_key('params') or kwargs.has_key('kernel_params'):
                raise ValueError, "Retraining for changed params yet not working"

        self.__resetChangedData()

        # local bindings
        chd = self._changedData
        ichd = self.__invalidatedChangedData

        chd.update(kwargs)
        # mark for future 'train()' items which are explicitely
        # mentioned as changed
        for key, value in kwargs.iteritems():
            if value: ichd[key] = True
        self.__changedData_isset = True

        # To check if we are not fooled
        if __debug__ and 'CHECK_RETRAIN' in debug.active:
            for key, data_ in (('traindata', dataset.samples),
                               ('labels', dataset.labels)):
                # so it wasn't told to be invalid
                if not chd[key] and not ichd.get(key, False):
                    if self.__wasDataChanged(key, data_, update=False):
                        raise RuntimeError, \
                              "Data %s found changed although wasn't " \
                              "labeled as such" % key

        # TODO: parameters of classifiers... for now there is explicit
        # 'forbidance' above

        # Below check should be superseeded by check above, thus never occur.
        # XXX remove later on
        if __debug__ and 'CHECK_RETRAIN' in debug.active and self.trained \
               and not self._changedData['traindata'] \
               and self.__trained['traindata'].shape != dataset.samples.shape:
            raise ValueError, "In retrain got dataset with %s size, " \
                  "whenever previousely was trained on %s size" \
                  % (dataset.samples.shape, self.__trained['traindata'].shape)
        self.train(dataset)


    def repredict(self, data, **kwargs):
        """Helper to avoid check if data was changed actually changed

        Useful if classifier was (re)trained but with the same data
        (so just parameters were changed), so that it could be
        repredicted easily (on the same data as before) without
        recomputing for instance train/test kernel matrix. Should be
        used with caution and always compared to the results on not
        'retrainable' classifier. Some additional checks are enabled
        if debug id 'CHECK_RETRAIN' is enabled, to guard against
        obvious mistakes.

        :Parameters:
          data
            data which is conventionally given to predict
          kwargs
            that is what _changedData gets updated with. So, smth like
            ``(params=['C'], labels=True)`` if parameter C and labels
            got changed
        """
        if len(kwargs)>0:
            raise RuntimeError, \
                  "repredict for now should be used without params since " \
                  "it makes little sense to repredict if anything got changed"
        if __debug__ and not self.params.retrainable:
            raise RuntimeError, \
                  "Do not use retrain/repredict on non-retrainable classifiers"

        self.__resetChangedData()
        chd = self._changedData
        chd.update(**kwargs)
        self.__changedData_isset = True


        # check if we are attempted to perform on the same data
        if __debug__ and 'CHECK_RETRAIN' in debug.active:
            for key, data_ in (('testdata', data),):
                # so it wasn't told to be invalid
                #if not chd[key]:# and not ichd.get(key, False):
                    if self.__wasDataChanged(key, data_, update=False):
                        raise RuntimeError, \
                              "Data %s found changed although wasn't " \
                              "labeled as such" % key

        # Should be superseeded by above
        # XXX remove in future
        if __debug__ and 'CHECK_RETRAIN' in debug.active \
               and not self._changedData['testdata'] \
               and self.__trained['testdata'].shape != data.shape:
            raise ValueError, "In repredict got dataset with %s size, " \
                  "whenever previousely was trained on %s size" \
                  % (data.shape, self.__trained['testdata'].shape)

        return self.predict(data)


    # TODO: callback into retrainable parameter
    #retrainable = property(fget=_getRetrainable, fset=_setRetrainable,
    #                  doc="Specifies either classifier should be retrainable")


#
# Base classifiers of various kinds
#

class BoostedClassifier(Classifier, Harvestable):
    """Classifier containing the farm of other classifiers.

    Should rarely be used directly. Use one of its childs instead
    """

    # should not be needed if we have prediction_values upstairs
    # TODO : should be handled as Harvestable or smth like that
    raw_predictions = StateVariable(enabled=False,
        doc="Predictions obtained from each classifier")

    raw_values = StateVariable(enabled=False,
        doc="Values obtained from each classifier")


    def __init__(self, clfs=None, propagate_states=True,
                 harvest_attribs=None, copy_attribs='copy',
                 **kwargs):
        """Initialize the instance.

        :Parameters:
          clfs : list
            list of classifier instances to use (slave classifiers)
          propagate_states : bool
            either to propagate enabled states into slave classifiers.
            It is in effect only when slaves get assigned - so if state
            is enabled not during construction, it would not necessarily
            propagate into slaves
          harvest_attribs : list of basestr
            What attributes of call to store and return within
            harvested state variable
          copy_attribs : None or basestr
            Force copying values of attributes on harvesting
          kwargs : dict
            dict of keyworded arguments which might get used
            by State or Classifier
        """
        if clfs == None:
            clfs = []

        Classifier.__init__(self, **kwargs)
        Harvestable.__init__(self, harvest_attribs, copy_attribs)

        self.__clfs = None
        """Pylint friendly definition of __clfs"""

        self.__propagate_states = propagate_states
        """Enable current enabled states in slave classifiers"""

        self._setClassifiers(clfs)
        """Store the list of classifiers"""


    def __repr__(self, prefixes=[]):
        if self.__clfs is None or len(self.__clfs)==0:
            #prefix_ = "clfs=%s" % repr(self.__clfs)
            prefix_ = []
        else:
            prefix_ = ["clfs=[%s,...]" % repr(self.__clfs[0])]
        return super(BoostedClassifier, self).__repr__(prefix_ + prefixes)


    def _train(self, dataset):
        """Train `BoostedClassifier`
        """
        for clf in self.__clfs:
            clf.train(dataset)


    def _posttrain(self, dataset):
        """Custom posttrain of `BoostedClassifier`

        Harvest over the trained classifiers if it was asked to so
        """
        Classifier._posttrain(self, dataset)
        if self.states.isEnabled('harvested'):
            for clf in self.__clfs:
                self._harvest(locals())
        if self.params.retrainable:
            self.__changedData_isset = False


    def _getFeatureIds(self):
        """Custom _getFeatureIds for `BoostedClassifier`
        """
        # return union of all used features by slave classifiers
        feature_ids = Set([])
        for clf in self.__clfs:
            feature_ids = feature_ids.union(Set(clf.feature_ids))
        return list(feature_ids)


    def _predict(self, data):
        """Predict using `BoostedClassifier`
        """
        raw_predictions = [ clf.predict(data) for clf in self.__clfs ]
        self.raw_predictions = raw_predictions
        assert(len(self.__clfs)>0)
        if self.states.isEnabled("values"):
            # XXX pylint complains that numpy has no array member... weird
            if N.array([x.states.isEnabled("values")
                        for x in self.__clfs]).all():
                values = [ clf.values for clf in self.__clfs ]
                self.raw_values = values
            else:
                warning("One or more classifiers in %s has no 'values' state" %
                        self + "enabled, thus BoostedClassifier can't have" +
                        " 'raw_values' state variable defined")

        return raw_predictions


    def _setClassifiers(self, clfs):
        """Set the classifiers used by the boosted classifier

        We have to allow to set list of classifiers after the object
        was actually created. It will be used by
        MulticlassClassifier
        """
        self.__clfs = clfs
        """Classifiers to use"""

        for flag in ['regression']:
            values = N.array([clf.params[flag].value for clf in self.__clfs])
            value = values.any()
            if __debug__:
                debug("CLFBST", "Setting %(flag)s=%(value)s for classifiers "
                      "%(clfs)s with %(values)s",
                      msgargs={'flag' : flag, 'value' : value,
                               'clfs' : self.__clfs,
                               'values' : values})
            # set flag if it needs to be trained before predicting
            self.params[flag].value = value

        # enable corresponding states in the slave-classifiers
        if self.__propagate_states:
            for clf in self.__clfs:
                clf.states.enable(self.states.enabled, missingok=True)

        # adhere to their capabilities + 'multiclass'
        # XXX do intersection across all classifiers!
        self._clf_internals = [ 'binary', 'multiclass', 'meta' ]
        if len(clfs)>0:
            self._clf_internals += self.__clfs[0]._clf_internals

    def untrain(self):
        """Untrain `BoostedClassifier`

        Has to untrain any known classifier
        """
        if not self.trained:
            return
        for clf in self.clfs:
            clf.untrain()
        super(BoostedClassifier, self).untrain()

    def getSensitivityAnalyzer(self, **kwargs):
        """Return an appropriate SensitivityAnalyzer"""
        return BoostedClassifierSensitivityAnalyzer(
                self,
                **kwargs)


    clfs = property(fget=lambda x:x.__clfs,
                    fset=_setClassifiers,
                    doc="Used classifiers")



class ProxyClassifier(Classifier):
    """Classifier which decorates another classifier

    Possible uses:

     - modify data somehow prior training/testing:
       * normalization
       * feature selection
       * modification

     - optimized classifier?

    """

    def __init__(self, clf, **kwargs):
        """Initialize the instance

        :Parameters:
          clf : Classifier
            classifier based on which mask classifiers is created
          """

        Classifier.__init__(self, regression=clf.regression, **kwargs)

        self.__clf = clf
        """Store the classifier to use."""

        # adhere to slave classifier capabilities
        # XXX test test test
        self._clf_internals = self._clf_internals[:] + ['meta']
        if clf is not None:
            self._clf_internals += clf._clf_internals


    def __repr__(self, prefixes=[]):
        return super(ProxyClassifier, self).__repr__(
            ["clf=%s" % repr(self.__clf)] + prefixes)

    def summary(self):
        s = super(ProxyClassifier, self).summary()
        if self.trained:
            s += "\n Slave classifier summary:" + \
                 '\n + %s' % \
                 (self.__clf.summary().replace('\n', '\n |'))
        return s



    def _train(self, dataset):
        """Train `ProxyClassifier`
        """
        # base class does nothing much -- just proxies requests to underlying
        # classifier
        self.__clf.train(dataset)

        # for the ease of access
        # TODO: if to copy we should exclude some states which are defined in
        #       base Classifier (such as training_time, predicting_time)
        # YOH: for now _copy_states_ would copy only set states variables. If
        #      anything needs to be overriden in the parent's class, it is welcome
        #      to do so
        #self.states._copy_states_(self.__clf, deep=False)


    def _predict(self, data):
        """Predict using `ProxyClassifier`
        """
        clf = self.__clf
        if self.states.isEnabled('values'):
            clf.states.enable(['values'])

        result = clf.predict(data)
        # for the ease of access
        self.states._copy_states_(self.__clf, ['values'], deep=False)
        return result


    def untrain(self):
        """Untrain ProxyClassifier
        """
        if not self.__clf is None:
            self.__clf.untrain()
        super(ProxyClassifier, self).untrain()


    @group_kwargs(prefixes=['slave_'], passthrough=True)
    def getSensitivityAnalyzer(self, slave_kwargs, **kwargs):
        """Return an appropriate SensitivityAnalyzer"""
        return ProxyClassifierSensitivityAnalyzer(
                self,
                analyzer=self.__clf.getSensitivityAnalyzer(**slave_kwargs),
                **kwargs)


    clf = property(lambda x:x.__clf, doc="Used `Classifier`")



#
# Various combiners for CombinedClassifier
#

class PredictionsCombiner(Stateful):
    """Base class for combining decisions of multiple classifiers"""

    def train(self, clfs, dataset):
        """PredictionsCombiner might need to be trained

        :Parameters:
          clfs : list of Classifier
            List of classifiers to combine. Has to be classifiers (not
            pure predictions), since combiner might use some other
            state variables (value's) instead of pure prediction's
          dataset : Dataset
            training data in this case
        """
        pass


    def __call__(self, clfs, dataset):
        """Call function

        :Parameters:
          clfs : list of Classifier
            List of classifiers to combine. Has to be classifiers (not
            pure predictions), since combiner might use some other
            state variables (value's) instead of pure prediction's
        """
        raise NotImplementedError



class MaximalVote(PredictionsCombiner):
    """Provides a decision using maximal vote rule"""

    predictions = StateVariable(enabled=True,
        doc="Voted predictions")
    all_label_counts = StateVariable(enabled=False,
        doc="Counts across classifiers for each label/sample")

    def __init__(self):
        """XXX Might get a parameter to use raw decision values if
        voting is not unambigous (ie two classes have equal number of
        votes
        """
        PredictionsCombiner.__init__(self)


    def __call__(self, clfs, dataset):
        """Actuall callable - perform voting

        Extended functionality which might not be needed actually:
        Since `BinaryClassifier` might return a list of possible
        predictions (not just a single one), we should consider all of those

        MaximalVote doesn't care about dataset itself
        """
        if len(clfs)==0:
            return []                   # to don't even bother

        all_label_counts = None
        for clf in clfs:
            # Lets check first if necessary state variable is enabled
            if not clf.states.isEnabled("predictions"):
                raise ValueError, "MaximalVote needs classifiers (such as " + \
                      "%s) with state 'predictions' enabled" % clf
            predictions = clf.predictions
            if all_label_counts is None:
                all_label_counts = [ {} for i in xrange(len(predictions)) ]

            # for every sample
            for i in xrange(len(predictions)):
                prediction = predictions[i]
                if not operator.isSequenceType(prediction):
                    prediction = (prediction,)
                for label in prediction: # for every label
                    # XXX we might have multiple labels assigned
                    # but might not -- don't remember now
                    if not all_label_counts[i].has_key(label):
                        all_label_counts[i][label] = 0
                    all_label_counts[i][label] += 1

        predictions = []
        # select maximal vote now for each sample
        for i in xrange(len(all_label_counts)):
            label_counts = all_label_counts[i]
            # lets do explicit search for max so we know
            # if it is unique
            maxk = []                   # labels of elements with max vote
            maxv = -1
            for k, v in label_counts.iteritems():
                if v > maxv:
                    maxk = [k]
                    maxv = v
                elif v == maxv:
                    maxk.append(k)

            assert len(maxk) >= 1, \
                   "We should have obtained at least a single key of max label"

            if len(maxk) > 1:
                warning("We got multiple labels %s which have the " % maxk +
                        "same maximal vote %d. XXX disambiguate" % maxv)
            predictions.append(maxk[0])

        self.all_label_counts = all_label_counts
        self.predictions = predictions
        return predictions



class MeanPrediction(PredictionsCombiner):
    """Provides a decision by taking mean of the results
    """

    predictions = StateVariable(enabled=True,
        doc="Mean predictions")

    def __call__(self, clfs, dataset):
        """Actuall callable - perform meaning

        """
        if len(clfs)==0:
            return []                   # to don't even bother

        all_predictions = []
        for clf in clfs:
            # Lets check first if necessary state variable is enabled
            if not clf.states.isEnabled("predictions"):
                raise ValueError, "MeanPrediction needs classifiers (such as " + \
                      "%s) with state 'predictions' enabled" % clf
            all_predictions.append(clf.predictions)

        # compute mean
        predictions = N.mean(N.asarray(all_predictions), axis=0)
        self.predictions = predictions
        return predictions


class ClassifierCombiner(PredictionsCombiner):
    """Provides a decision using training a classifier on predictions/values

    TODO
    """

    predictions = StateVariable(enabled=True,
        doc="Trained predictions")


    def __init__(self, clf, variables=None):
        """Initialize `ClassifierCombiner`

        :Parameters:
          clf : Classifier
            Classifier to train on the predictions
          variables : list of basestring
            List of state variables stored in 'combined' classifiers, which
            to use as features for training this classifier
        """
        PredictionsCombiner.__init__(self)

        self.__clf = clf
        """Classifier to train on `variables` states of provided classifiers"""

        if variables == None:
            variables = ['predictions']
        self.__variables = variables
        """What state variables of the classifiers to use"""


    def untrain(self):
        """It might be needed to untrain used classifier"""
        if self.__clf:
            self.__clf.untrain()

    def __call__(self, clfs, dataset):
        """
        """
        if len(clfs)==0:
            return []                   # to don't even bother

        # XXX TODO
        raise NotImplementedError



class CombinedClassifier(BoostedClassifier):
    """`BoostedClassifier` which combines predictions using some
    `PredictionsCombiner` functor.
    """

    def __init__(self, clfs=None, combiner=None, **kwargs):
        """Initialize the instance.

        :Parameters:
          clfs : list of Classifier
            list of classifier instances to use
          combiner : PredictionsCombiner
            callable which takes care about combining multiple
            results into a single one (e.g. maximal vote for
            classification, MeanPrediction for regression))
          kwargs : dict
            dict of keyworded arguments which might get used
            by State or Classifier

        NB: `combiner` might need to operate not on 'predictions' descrete
            labels but rather on raw 'class' values classifiers
            estimate (which is pretty much what is stored under
            `values`
        """
        if clfs == None:
            clfs = []

        BoostedClassifier.__init__(self, clfs, **kwargs)

        # assign default combiner
        if combiner is None:
            combiner = (MaximalVote, MeanPrediction)[int(self.regression)]()
        self.__combiner = combiner
        """Functor destined to combine results of multiple classifiers"""


    def __repr__(self, prefixes=[]):
        return super(CombinedClassifier, self).__repr__(
            ["combiner=%s" % repr(self.__combiner)] + prefixes)


    def summary(self):
        s = super(CombinedClassifier, self).summary()
        if self.trained:
            s += "\n Slave classifiers summaries:"
            for i, clf in enumerate(self.clfs):
                s += '\n + %d clf: %s' % \
                     (i, clf.summary().replace('\n', '\n |'))
        return s


    def untrain(self):
        """Untrain `CombinedClassifier`
        """
        try:
            self.__combiner.untrain()
        except:
            pass
        super(CombinedClassifier, self).untrain()

    def _train(self, dataset):
        """Train `CombinedClassifier`
        """
        BoostedClassifier._train(self, dataset)
        # combiner might need to train as well
        self.__combiner.train(self.clfs, dataset)


    def _predict(self, data):
        """Predict using `CombinedClassifier`
        """
        BoostedClassifier._predict(self, data)
        # combiner will make use of state variables instead of only predictions
        # returned from _predict
        predictions = self.__combiner(self.clfs, data)
        self.predictions = predictions

        if self.states.isEnabled("values"):
            if self.__combiner.states.isActive("values"):
                # XXX or may be we could leave simply up to accessing .combiner?
                self.values = self.__combiner.values
            else:
                if __debug__:
                    warning("Boosted classifier %s has 'values' state" % self +
                            " enabled, but combiner has it active, thus no" +
                            " values could be provided directly, access .clfs")
        return predictions


    combiner = property(fget=lambda x:x.__combiner,
                        doc="Used combiner to derive a single result")



class BinaryClassifier(ProxyClassifier):
    """`ProxyClassifier` which maps set of two labels into +1 and -1
    """

    def __init__(self, clf, poslabels, neglabels, **kwargs):
        """
        :Parameters:
          clf : Classifier
            classifier to use
          poslabels : list
            list of labels which are treated as +1 category
          neglabels : list
            list of labels which are treated as -1 category
        """

        ProxyClassifier.__init__(self, clf, **kwargs)

        self._regressionIsBogus()

        # Handle labels
        sposlabels = Set(poslabels) # so to remove duplicates
        sneglabels = Set(neglabels) # so to remove duplicates

        # check if there is no overlap
        overlap = sposlabels.intersection(sneglabels)
        if len(overlap)>0:
            raise ValueError("Sets of positive and negative labels for " +
                "BinaryClassifier must not overlap. Got overlap " %
                overlap)

        self.__poslabels = list(sposlabels)
        self.__neglabels = list(sneglabels)

        # define what values will be returned by predict: if there is
        # a single label - return just it alone, otherwise - whole
        # list
        # Such approach might come useful if we use some classifiers
        # over different subsets of data with some voting later on
        # (1-vs-therest?)

        if len(self.__poslabels) > 1:
            self.__predictpos = self.__poslabels
        else:
            self.__predictpos = self.__poslabels[0]

        if len(self.__neglabels) > 1:
            self.__predictneg = self.__neglabels
        else:
            self.__predictneg = self.__neglabels[0]


    def __repr__(self, prefixes=[]):
        prefix = "poslabels=%s, neglabels=%s" % (
            repr(self.__poslabels), repr(self.__neglabels))
        return super(BinaryClassifier, self).__repr__([prefix] + prefixes)


    def _train(self, dataset):
        """Train `BinaryClassifier`
        """
        idlabels = [(x, +1) for x in dataset.idsbylabels(self.__poslabels)] + \
                    [(x, -1) for x in dataset.idsbylabels(self.__neglabels)]
        # XXX we have to sort ids since at the moment Dataset.selectSamples
        #     doesn't take care about order
        idlabels.sort()
        # select the samples
        orig_labels = None

        # If we need all samples, why simply not perform on original
        # data, an just store/restore labels. But it really should be done
        # within Dataset.selectSamples
        if len(idlabels) == dataset.nsamples \
            and [x[0] for x in idlabels] == range(dataset.nsamples):
            # the last condition is not even necessary... just overly
            # cautious
            datasetselected = dataset   # no selection is needed
            orig_labels = dataset.labels # but we would need to restore labels
            if __debug__:
                debug('CLFBIN',
                      "Assigned all %d samples for binary " %
                      (dataset.nsamples) +
                      " classification among labels %s/+1 and %s/-1" %
                      (self.__poslabels, self.__neglabels))
        else:
            datasetselected = dataset.selectSamples([ x[0] for x in idlabels ])
            if __debug__:
                debug('CLFBIN',
                      "Selected %d samples out of %d samples for binary " %
                      (len(idlabels), dataset.nsamples) +
                      " classification among labels %s/+1 and %s/-1" %
                      (self.__poslabels, self.__neglabels) +
                      ". Selected %s" % datasetselected)

        # adjust the labels
        datasetselected.labels = [ x[1] for x in idlabels ]

        # now we got a dataset with only 2 labels
        if __debug__:
            assert((datasetselected.uniquelabels == [-1, 1]).all())

        self.clf.train(datasetselected)

        if not orig_labels is None:
            dataset.labels = orig_labels

    def _predict(self, data):
        """Predict the labels for a given `data`

        Predicts using binary classifier and spits out list (for each sample)
        where with either poslabels or neglabels as the "label" for the sample.
        If there was just a single label within pos or neg labels then it would
        return not a list but just that single label.
        """
        binary_predictions = ProxyClassifier._predict(self, data)
        self.values = binary_predictions
        predictions = [ {-1: self.__predictneg,
                         +1: self.__predictpos}[x] for x in binary_predictions]
        self.predictions = predictions
        return predictions



class MulticlassClassifier(CombinedClassifier):
    """`CombinedClassifier` to perform multiclass using a list of
    `BinaryClassifier`.

    such as 1-vs-1 (ie in pairs like libsvm doesn) or 1-vs-all (which
    is yet to think about)
    """

    def __init__(self, clf, bclf_type="1-vs-1", **kwargs):
        """Initialize the instance

        :Parameters:
          clf : Classifier
            classifier based on which multiple classifiers are created
            for multiclass
          bclf_type
            "1-vs-1" or "1-vs-all", determines the way to generate binary
            classifiers
          """
        CombinedClassifier.__init__(self, **kwargs)
        self._regressionIsBogus()
        if not clf is None:
            clf._regressionIsBogus()

        self.__clf = clf
        """Store sample instance of basic classifier"""

        # Some checks on known ways to do multiclass
        if bclf_type == "1-vs-1":
            pass
        elif bclf_type == "1-vs-all": # TODO
            raise NotImplementedError
        else:
            raise ValueError, \
                  "Unknown type of classifier %s for " % bclf_type + \
                  "BoostedMulticlassClassifier"
        self.__bclf_type = bclf_type

    # XXX fix it up a bit... it seems that MulticlassClassifier should
    # be actually ProxyClassifier and use BoostedClassifier internally
    def __repr__(self, prefixes=[]):
        prefix = "bclf_type=%s, clf=%s" % (repr(self.__bclf_type),
                                            repr(self.__clf))
        return super(MulticlassClassifier, self).__repr__([prefix] + prefixes)


    def _train(self, dataset):
        """Train classifier
        """
        # construct binary classifiers
        ulabels = dataset.uniquelabels
        if self.__bclf_type == "1-vs-1":
            # generate pairs and corresponding classifiers
            biclfs = []
            for i in xrange(len(ulabels)):
                for j in xrange(i+1, len(ulabels)):
                    clf = _deepcopyclf(self.__clf)
                    biclfs.append(
                        BinaryClassifier(
                            clf,
                            poslabels=[ulabels[i]], neglabels=[ulabels[j]]))
            if __debug__:
                debug("CLFMC", "Created %d binary classifiers for %d labels" %
                      (len(biclfs), len(ulabels)))

            self.clfs = biclfs

        elif self.__bclf_type == "1-vs-all":
            raise NotImplementedError

        # perform actual training
        CombinedClassifier._train(self, dataset)



class SplitClassifier(CombinedClassifier):
    """`BoostedClassifier` to work on splits of the data

    """

    """
    TODO: SplitClassifier and MulticlassClassifier have too much in
          common -- need to refactor: just need a splitter which would
          split dataset in pairs of class labels. MulticlassClassifier
          does just a tiny bit more which might be not necessary at
          all: map sets of labels into 2 categories...
    """

    # TODO: unify with CrossValidatedTransferError which now uses
    # harvest_attribs to expose gathered attributes
    confusion = StateVariable(enabled=False,
        doc="Resultant confusion whenever classifier trained " +
            "on 1 part and tested on 2nd part of each split")

    splits = StateVariable(enabled=False, doc=
       """Store the actual splits of the data. Can be memory expensive""")

    # XXX couldn't be training_confusion since it has other meaning
    #     here, BUT it is named so within CrossValidatedTransferError
    #     -- unify
    # YYY decided to go with overriding semantics tiny bit. For split
    #     classifier training_confusion would correspond to summary
    #     over training errors across all splits. Later on if need comes
    #     we might want to implement global_training_confusion which would
    #     correspond to overall confusion on full training dataset as it is
    #     done in base Classifier
    #global_training_confusion = StateVariable(enabled=False,
    #    doc="Summary over training confusions acquired at each split")

    def __init__(self, clf, splitter=NFoldSplitter(cvtype=1), **kwargs):
        """Initialize the instance

        :Parameters:
          clf : Classifier
            classifier based on which multiple classifiers are created
            for multiclass
          splitter : Splitter
            `Splitter` to use to split the dataset prior training
          """

        CombinedClassifier.__init__(self, regression=clf.regression, **kwargs)
        self.__clf = clf
        """Store sample instance of basic classifier"""

        if isinstance(splitter, type):
            raise ValueError, \
                  "Please provide an instance of a splitter, not a type." \
                  " Got %s" % splitter

        self.__splitter = splitter


    def _train(self, dataset):
        """Train `SplitClassifier`
        """
        # generate pairs and corresponding classifiers
        bclfs = []

        # local binding
        states = self.states

        clf_template = self.__clf
        if states.isEnabled('confusion'):
            states.confusion = clf_template._summaryClass()
        if states.isEnabled('training_confusion'):
            clf_template.states.enable(['training_confusion'])
            states.training_confusion = clf_template._summaryClass()

        clf_hastestdataset = hasattr(clf_template, 'testdataset')

        # for proper and easier debugging - first define classifiers and then
        # train them
        for split in self.__splitter.splitcfg(dataset):
            if __debug__:
                debug("CLFSPL",
                      "Deepcopying %(clf)s for %(sclf)s",
                      msgargs={'clf':clf_template,
                               'sclf':self})
            clf = _deepcopyclf(clf_template)
            bclfs.append(clf)
        self.clfs = bclfs

        self.splits = []

        for i,split in enumerate(self.__splitter(dataset)):
            if __debug__:
                debug("CLFSPL", "Training classifier for split %d" % (i))

            if states.isEnabled("splits"):
                self.splits.append(split)

            clf = self.clfs[i]

            # assign testing dataset if given classifier can digest it
            if clf_hastestdataset:
                clf.testdataset = split[1]

            clf.train(split[0])

            # unbind the testdataset from the classifier
            if clf_hastestdataset:
                clf.testdataset = None

            if states.isEnabled("confusion"):
                predictions = clf.predict(split[1].samples)
                self.confusion.add(split[1].labels, predictions,
                                   clf.states.get('values', None))
            if states.isEnabled("training_confusion"):
                states.training_confusion += \
                                               clf.states.training_confusion
        # XXX hackish way -- so it should work only for ConfusionMatrix
        try:
            if states.isEnabled("confusion"):
                states.confusion.labels_map = dataset.labels_map
            if states.isEnabled("training_confusion"):
                states.training_confusion.labels_map = dataset.labels_map
        except:
            pass


    @group_kwargs(prefixes=['slave_'], passthrough=True)
    def getSensitivityAnalyzer(self, slave_kwargs, **kwargs):
        """Return an appropriate SensitivityAnalyzer for `SplitClassifier`

        :Parameters:
          combiner
            If not provided, FirstAxisMean is assumed
        """
        kwargs.setdefault('combiner', FirstAxisMean)
        return BoostedClassifierSensitivityAnalyzer(
                self,
                analyzer=self.__clf.getSensitivityAnalyzer(**slave_kwargs),
                **kwargs)

    splitter = property(fget=lambda x:x.__splitter,
                        doc="Splitter user by SplitClassifier")


class MappedClassifier(ProxyClassifier):
    """`ProxyClassifier` which uses some mapper prior training/testing.

    `MaskMapper` can be used just a subset of features to
    train/classify.
    Having such classifier we can easily create a set of classifiers
    for BoostedClassifier, where each classifier operates on some set
    of features, e.g. set of best spheres from SearchLight, set of
    ROIs selected elsewhere. It would be different from simply
    applying whole mask over the dataset, since here initial decision
    is made by each classifier and then later on they vote for the
    final decision across the set of classifiers.
    """

    def __init__(self, clf, mapper, **kwargs):
        """Initialize the instance

        :Parameters:
          clf : Classifier
            classifier based on which mask classifiers is created
          mapper
            whatever `Mapper` comes handy
          """
        ProxyClassifier.__init__(self, clf, **kwargs)

        self.__mapper = mapper
        """mapper to help us our with prepping data to
        training/classification"""


    def _train(self, dataset):
        """Train `MappedClassifier`
        """
        # first train the mapper
        # XXX: should training be done using whole dataset or just samples
        # YYY: in some cases labels might be needed, thus better full dataset
        self.__mapper.train(dataset)

        # for train() we have to provide dataset -- not just samples to train!
        wdataset = dataset.applyMapper(featuresmapper = self.__mapper)
        ProxyClassifier._train(self, wdataset)


    def _predict(self, data):
        """Predict using `MappedClassifier`
        """
        return ProxyClassifier._predict(self, self.__mapper.forward(data))


    @group_kwargs(prefixes=['slave_'], passthrough=True)
    def getSensitivityAnalyzer(self, slave_kwargs, **kwargs):
        """Return an appropriate SensitivityAnalyzer"""
        return MappedClassifierSensitivityAnalyzer(
                self,
                analyzer=self.__clf.getSensitivityAnalyzer(**slave_kwargs),
                **kwargs)


    mapper = property(lambda x:x.__mapper, doc="Used mapper")



class FeatureSelectionClassifier(ProxyClassifier):
    """`ProxyClassifier` which uses some `FeatureSelection` prior training.

    `FeatureSelection` is used first to select features for the classifier to
    use for prediction. Internally it would rely on MappedClassifier which
    would use created MaskMapper.

    TODO: think about removing overhead of retraining the same classifier if
    feature selection was carried out with the same classifier already. It
    has been addressed by adding .trained property to classifier, but now
    we should expclitely use isTrained here if we want... need to think more
    """

    _clf_internals = [ 'does_feature_selection', 'meta' ]

    def __init__(self, clf, feature_selection, testdataset=None, **kwargs):
        """Initialize the instance

        :Parameters:
          clf : Classifier
            classifier based on which mask classifiers is created
          feature_selection : FeatureSelection
            whatever `FeatureSelection` comes handy
          testdataset : Dataset
            optional dataset which would be given on call to feature_selection
          """
        ProxyClassifier.__init__(self, clf, **kwargs)

        self.__maskclf = None
        """Should become `MappedClassifier`(mapper=`MaskMapper`) later on."""

        self.__feature_selection = feature_selection
        """`FeatureSelection` to select the features prior training"""

        self.__testdataset = testdataset
        """`FeatureSelection` might like to use testdataset"""


    def untrain(self):
        """Untrain `FeatureSelectionClassifier`

        Has to untrain any known classifier
        """
        if not self.trained:
            return
        if not self.__maskclf is None:
            self.__maskclf.untrain()
        super(FeatureSelectionClassifier, self).untrain()


    def _train(self, dataset):
        """Train `FeatureSelectionClassifier`
        """
        # temporarily enable selected_ids
        self.__feature_selection.states._changeTemporarily(
            enable_states=["selected_ids"])

        if __debug__:
            debug("CLFFS", "Performing feature selection using %s" %
                  self.__feature_selection + " on %s" % dataset)

        (wdataset, tdataset) = self.__feature_selection(dataset,
                                                        self.__testdataset)
        if __debug__:
            add_ = ""
            if "CLFFS_" in debug.active:
                add_ = " Selected features: %s" % \
                       self.__feature_selection.selected_ids
            debug("CLFFS", "%(fs)s selected %(nfeat)d out of " +
                  "%(dsnfeat)d features.%(app)s",
                  msgargs={'fs':self.__feature_selection,
                           'nfeat':wdataset.nfeatures,
                           'dsnfeat':dataset.nfeatures,
                           'app':add_})

        # create a mask to devise a mapper
        # TODO -- think about making selected_ids a MaskMapper
        mappermask = N.zeros(dataset.nfeatures)
        mappermask[self.__feature_selection.selected_ids] = 1
        mapper = MaskMapper(mappermask)

        self.__feature_selection.states._resetEnabledTemporarily()

        # create and assign `MappedClassifier`
        self.__maskclf = MappedClassifier(self.clf, mapper)
        # we could have called self.__clf.train(dataset), but it would
        # cause unnecessary masking
        self.__maskclf.clf.train(wdataset)

        # for the ease of access
        # TODO see for ProxyClassifier
        #self.states._copy_states_(self.__maskclf, deep=False)

    def _getFeatureIds(self):
        """Return used feature ids for `FeatureSelectionClassifier`

        """
        return self.__feature_selection.selected_ids

    def _predict(self, data):
        """Predict using `FeatureSelectionClassifier`
        """
        clf = self.__maskclf
        if self.states.isEnabled('values'):
            clf.states.enable(['values'])

        result = clf._predict(data)
        # for the ease of access
        self.states._copy_states_(clf, ['values'], deep=False)
        return result

    def setTestDataset(self, testdataset):
        """Set testing dataset to be used for feature selection
        """
        self.__testdataset = testdataset

    maskclf = property(lambda x:x.__maskclf, doc="Used `MappedClassifier`")
    feature_selection = property(lambda x:x.__feature_selection,
                                 doc="Used `FeatureSelection`")

    @group_kwargs(prefixes=['slave_'], passthrough=True)
    def getSensitivityAnalyzer(self, slave_kwargs, **kwargs):
        """Return an appropriate SensitivityAnalyzer

        TODO: had to clone from mapped classifier... XXX
        """
        return MappedClassifierSensitivityAnalyzer(
                self,
                analyzer=self.clf.getSensitivityAnalyzer(**slave_kwargs),
                **kwargs)



    testdataset = property(fget=lambda x:x.__testdataset,
                           fset=setTestDataset)
