"""
module to access local filesystem pathes
(mostly filename manipulations but also file operations)
"""
import os, sys, stat

import py
#__________________________________________________________
#
# Local Path Posix Mixin
#__________________________________________________________

class PosixMixin:
    # an instance needs to be a local path instance
    def owner(self):
        """ return owner name of file. """
        uid = self.stat().st_uid
        entry = self._callex(py.std.pwd.getpwuid, uid)
        return entry[0]

    def group(self):
        """ return group name of file. """
        gid = self.stat().st_gid
        entry = self._callex(py.std.grp.getgrgid, gid)
        return entry[0]

    def mode(self):
        """ return permission mode of the path object """
        return self.stat().st_mode

    def chmod(self, mode, rec=0):
        """ change permissions to the given mode. If mode is an
            integer it directly encodes the os-specific modes.
            (xxx if mode is a string then it specifies access rights
            in '/bin/chmod' style, e.g. a+r).
            if rec is True perform recursively.
        """
        if not isinstance(mode, int):
            raise TypeError("mode %r must be an integer" % (mode,))
        if rec:
            for x in self.visit(rec=rec):
                self._callex(os.chmod, str(x), mode)
        self._callex(os.chmod, str(self), mode)

    def chown(self, user, group, rec=0):
        """ change ownership to the given user and group.
            user and group may be specified by a number or
            by a name.  if rec is True change ownership
            recursively.
        """
        uid = getuserid(user)
        gid = getgroupid(group)
        if rec:
            for x in self.visit(rec=lambda x: x.check(link=0)): 
                if x.check(link=0):
                    self._callex(os.chown, str(x), uid, gid)
        self._callex(os.chown, str(self), uid, gid)

    def readlink(self):
        """ return value of a symbolic link. """
        return self._callex(os.readlink, self.strpath)

    def mklinkto(self, oldname):
        """ posix style hard link to another name. """
        self._callex(os.link, str(oldname), str(self))

    def mksymlinkto(self, value, absolute=1):
        """ create a symbolic link with the given value (pointing to another name). """
        if absolute:
            self._callex(os.symlink, str(value), self.strpath)
        else:
            base = self.common(value)
            # with posix local paths '/' is always a common base
            relsource = self.__class__(value).relto(base)
            reldest = self.relto(base)
            n = reldest.count(self.sep)
            target = self.sep.join(('..', )*n + (relsource, ))
            self._callex(os.symlink, target, self.strpath)

def getuserid(user):
    import pwd
    if isinstance(user, int):
        return user
    entry = pwd.getpwnam(user)
    return entry[2]

def getgroupid(group):
    import grp
    if isinstance(group, int):
        return group
    entry = grp.getgrnam(group)
    return entry[2]
