#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009, 2011-2012  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

# Ported from GNUbik
# Original filename: main.c
# Original copyright and license: 1998, 2003--2004  John Darrington, GPL3+

from __future__ import print_function, division, unicode_literals

import sys, os
from collections import namedtuple

from . import debug, config

N_ = lambda s: s


def format_opts(opts):
    opts = ((o + o.strip('--')[0].upper() if o.endswith('=') else o) for o in opts)
    return '  ' + ', '.join(opts)
    
def format_help(text, indent=0):
    try:
        width = int(os.environ['COLUMNS']) - 2
    except (KeyError, ValueError):
        width = 78
    width -= indent
    def split(text):
        lines = text.split('\n')
        for line in lines:
            res = None
            words = line.split(' ')
            for word in words:
                if res is None:
                    res = word
                elif len(res + word) + 1 <= width:
                    res = ' '.join((res, word))
                else:
                    yield res
                    res = word
            yield res
    return '\n'.ljust(indent+1).join(split(text))
    
def print_usage(unused_value, unused_opts):
    print('Usage:', sys.argv[0], '[options]')
    for args, unused_func, text in arg_info:
        if args is None:
            print()
            print(format_help(text))
        else:
            opts = format_opts(args)
            if len(opts) > 18:
                text = '\n' + text
            else:
                opts = opts.ljust(20)
            text = format_help(text, 20)
            print(opts + text)
    sys.exit(0)
    
def print_version(*unused_args):
    print(config.APPNAME, config.VERSION)
    print()
    print(config.COPYRIGHT.replace('©', '(C)')) #XXX: help2man cannot handle unicode
    print()
    print(config.wrap(config.LICENSE_INFO))
    print()
    print(config.wrap(config.LICENSE_NOT_FOUND))
    sys.exit(0)
    
debug_level_names = [__a[6:].lower() for __a in debug.__all__ if __a.startswith('DEBUG_')]

arg_info = [
        (None, None, config.LONG_DESCRIPTION.replace('ő', 'o')), #XXX: help2man cannot handle unicode
        (None, None, 'Pybik can be fully controlled and configured via the graphical user interface.'
                     ' The options listed here are intended primarily for testing and debugging.'),
        (None, None, 'Options:'),
        (['-h', '--help'], print_usage,
                            'Show help message and exit'),
        (['--version'], print_version,
                            'Show version number and exit'),
        (['--config-file='], lambda value, opts: opts.setdefault('config-file', value),
                            'Specify the configuration file to use instead of the default location'),
        (['--defaultconfig'], lambda value, opts: opts.setdefault('defaultconfig', True),
                            'Print default settings to stdout and exit'),
        (['--pyside'], lambda value, opts: opts.setdefault('pyside', True),
                            'Use PySide, cannot combined with --pyqt4'),
        (['--pyqt4'], lambda value, opts: opts.setdefault('pyqt4', True),
                            'Use PyQt4, cannot be combined with --pyside,'
                            ' is neither --pyside nor --pyqt4 is given, the application'
                            ' first imports PySide and if that fails it imports PyQt4'),
        (['--pure-python'], lambda value, opts: opts.setdefault('pure-python', True),
                            'Use python module for rendering (very slow)'),
        (['--debug='], lambda value, opts: opts.setdefault('debug', value),
                            'Enable debug output, D is a comma-separated list of debug flags:\n{0}'
                            .format(' '.join(debug_level_names))),
        (None, None,  'Qt-Options, for a full list refer to the Qt-Reference,'
                ' but not all options make sense for this application:'),
        (['-style S'], None,
                            'sets the application GUI style. Possible values are'
                            ' motif, windows, and platinum.'),
        (['-widgetcount'], None,
                            'prints debug message at the end about number of widgets left'
                            ' undestroyed and maximum number of widgets existed at the same time'),
        (['-reverse'], None,
                            "sets the application's layout direction to left-to-right"),
        (['-graphicssystem G'], None,
                            'sets the backend to be used for on-screen widgets and pixmaps.'
                            ' Available options are raster and opengl.'),
    ]
    
def parse_args(args):
    arg_functs = {o: f for opts, f, h in arg_info for o in opts or [] if f is not None}
    opts = {}
    ui_args = []
    for arg in args:
        try:
            index = arg.index('=')
        except ValueError:
            value = None
        else:
            arg, value = arg[:index+1], arg[index+1:]
        try:
            func = arg_functs[arg]
        except KeyError:
            ui_args.append(arg)
        else:
            func(value, opts)
            
    if 'pyside' in opts:
        if 'pyqt4' in opts:
            print('You cannot use both --pyside and --pyqt4')
            sys.exit(1)
        else:
            pythonqt = 'pyside'
    else:
        if 'pyqt4' in opts:
            pythonqt = 'pyqt4'
        else:
            pythonqt = None
            
    debug_flags = opts.get('debug', None)
    if debug_flags is not None:
        debug_flags = [f for f in debug_flags.split(',') if f]
        for d in debug_flags:
            if d not in debug_level_names:
                print('unknown debug option:', d)
                sys.exit(1)
            
    Opts = namedtuple('Opts', 'debug_flags pythonqt pure_python ui_args config_file defaultconfig')
    return Opts(debug_flags, pythonqt, opts.get('pure-python', False), ui_args,
                opts.get('config-file', config.USER_SETTINGS_FILE), opts.get('defaultconfig', False))
    
def import_ext(pythonqt, pure_python):
    if pythonqt != 'pyqt4':
        debug.debug('Importing PySide')
        try:
            import PySide, PySide.QtCore, PySide.QtGui, PySide.QtOpenGL
        except ImportError as e:
            debug.debug('  Failed')
            if pythonqt == 'pyside':
                print('PySide not found.', e)
                sys.exit(1)
            pythonqt = 'pyqt4'
    if pythonqt == 'pyqt4':
        debug.debug('Importing PyQt4')
        try:
            import sip
            sip.setapi('QString', 2)
            sip.setapi('QVariant', 2)
            import PyQt4
            import PyQt4.QtCore
            import PyQt4.QtGui
            import PyQt4.QtOpenGL
        except ImportError as e:
            debug.debug('  Failed')
            print('Python bindings for Qt4 not found.', e)
            sys.exit(1)
        sys.modules[b'PySide'] = PyQt4
        sys.modules[b'PySide.QtCore'] = PyQt4.QtCore
        sys.modules[b'PySide.QtGui'] = PyQt4.QtGui
        sys.modules[b'PySide.QtOpenGL'] = PyQt4.QtOpenGL
        # Monkey patch QtCore
        # pylint: disable=E1101
        PyQt4.QtCore.Signal = PyQt4.QtCore.pyqtSignal
        PyQt4.QtCore.Slot = PyQt4.QtCore.pyqtSlot
        PyQt4.QtCore.Qt.CursorShape.CrossCursor = PyQt4.QtCore.Qt.CrossCursor
        PyQt4.QtCore.Qt.TransformationMode.SmoothTransformation = \
                        PyQt4.QtCore.Qt.SmoothTransformation
        # Monkey patch QtOpenGL
        PyQt4.QtOpenGL.QGLContext.BindOption.LinearFilteringBindOption = \
                            PyQt4.QtOpenGL.QGLContext.LinearFilteringBindOption
        PyQt4.QtOpenGL.QGLContext.BindOption.MipmapBindOption = PyQt4.QtOpenGL.QGLContext.MipmapBindOption
        # pylint: enable=E1101
        
    if debug.DEBUG:
        import PySide, PySide.QtCore, PySide.QtGui, PySide.QtOpenGL # pylint: disable=W0404
        debug.debug('Qt modules:', PySide.__name__, PySide.QtCore.__name__,
                                   PySide.QtGui.__name__, PySide.QtOpenGL.__name__)
        
    if not pure_python:
        try:
            import _gldraw
            sys.modules[b'pybiklib.gldraw'] = _gldraw
            sys.modules[b'_gldraw'] = _gldraw
        except ImportError as e:
            print(e)
            print('Unable to import module _gldraw, using slow fallback.')
            pure_python = True
    if not pure_python:
        try:
            import _glarea
            sys.modules[b'pybiklib.glarea'] = _glarea
        except ImportError as e:
            del sys.modules[b'pybiklib.gldraw']
            del sys.modules[b'_gldraw']
            print(e)
            print('Unable to import module _glarea, using slow fallback.')
    if pure_python:
        try:
            from OpenGL import GL as unused_GL
        except ImportError as e:
            print('Python bindings for OpenGL not found.', e)
            sys.exit(1)


def run_app(root_dir, args, config_file):
    import gettext
    from PySide.QtCore import QLocale, QTranslator, QTimer
    from PySide.QtGui import QApplication, QPalette, QColor
    
    # initialize QApplication
    app = QApplication(args)
    args = app.arguments()[1:]
    if args:
        print('Unknown arguments:', ' '.join(args))
        sys.exit(1)
    app.setOrganizationName(config.PACKAGE)
    app.setApplicationName(config.APPNAME)
    app.setApplicationVersion(config.VERSION)
    
    # Workaround for whatsThis-Text (white text on light background)
    palette = app.palette()
    colorfg = palette.color(QPalette.Active, QPalette.ToolTipText)
    colorbg = palette.color(QPalette.Active, QPalette.ToolTipBase)
    valuefg = colorfg.value()
    valuebg = (valuefg + 255) // 2 if valuefg < 128 else valuefg // 2
    colorbg = QColor.fromHsv(colorbg.hue(), colorbg.saturation(), valuebg)
    palette.setColor(QPalette.Active, QPalette.ToolTipBase, colorbg)
    app.setPalette(palette)
    
    # initialize translation
    language = QLocale.system().name()
    # standard Qt translation, used for e.g. standard buttons and shortcut names
    translator = QTranslator()
    translator.load('qt_' + language, config.QT_LOCALE_DIR)
    app.installTranslator(translator)
    # the rest of the app use gettext for translation
    if root_dir == sys.prefix:
        # normal installation
        LOCALEDIR = None
    else:
        # different root, e.g. /usr/local, source directory
        LOCALEDIR = config.LOCALE_DIR
    t = gettext.translation(config.PACKAGE, LOCALEDIR, languages=[language], fallback=True)
    t.install(unicode=True, names=['ngettext'])
    
    # initialize settings
    from .settings import settings
    try:
        settings.load(config_file)
    except EnvironmentError as e:
        error_message = N_('An error occurred while reading the settings:\n'
                        '{error_message}').format(error_message=e)
        debug.debug(error_message)
        settings.load('')
        QTimer.singleShot(0, lambda: window.error_dialog(_(error_message)))
        
    # create main window
    # The application module can be imported only if a QApplication object is created
    from .application import MainWindow
    window = MainWindow()
    
    # run the application
    app.exec_()
    
    
def run(root_dir=None):
    opts = parse_args(sys.argv)
    
    if opts.debug_flags is not None:
        print(config.PACKAGE, config.VERSION)
        debug.set_flags(opts.debug_flags)
        print('debug flags:', *opts.debug_flags)
        debug.debug('Qt args:', opts.ui_args)
    
    if opts.defaultconfig:
        from .settings import settings
        settings.load('')
        settings.keystore.dump(sys.stdout, all=True)
        sys.exit(0)
        
    import_ext(opts.pythonqt, opts.pure_python)
    
    run_app(root_dir, opts.ui_args, opts.config_file)
    

