#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2012  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

# Ported from GNUbik
# Original filename: glarea.c
# Original copyright and license: 2003, 2004  John Darrington,  GPL3+

from __future__ import print_function, division, unicode_literals

import os
from collections import namedtuple
import math

# pylint: disable=W0614,W0401
from .debug import *

from PySide.QtOpenGL import *
from PySide.QtCore import *
from PySide.QtGui import *
# pylint: enable=W0614,W0401

# no need for OpenGL module for just two constants
GL_RGBA, GL_TEXTURE_2D = 6408, 3553
if DEBUG:
    from OpenGL import GL
    assert GL.GL_RGBA == GL_RGBA and GL.GL_TEXTURE_2D == GL_TEXTURE_2D

from . import config
import gldraw
import glarea
from .textures import textures
from .settings import settings
from . import model


class CubeArea (QGLWidget):
    end_animation = Signal(bool)
    request_rotation = Signal((int, int, bool))
    request_swap_blocks = Signal((int, int, int, bool))
    request_rotate_block = Signal((int, bool))
    drop_color = Signal((int, str, str))
    drop_file = Signal((int, str, str))
    
    def __init__(self):
        glformat = QGLFormat()
        if settings.draw.samples > 0:
            glformat.setSampleBuffers(True)
            glformat.setSamples(2**settings.draw.samples)
        if settings.draw.accum > 0:
            glformat.setAccum(True)
        else:
            accum_buffers = 0
        QGLWidget.__init__(self, glformat)
        
        self.model = model.empty_model
        
        self.selection_debug_info = None
        
        self.last_mouse_x = -1
        self.last_mouse_y = -1
        self.button_down_background = False
        self.timer_animate = QTimer() # the animate timer
        self.timer_animate.timeout.connect(self._on_animate)
        self.mouse_xy = -1, -1
        # Structure to hold copy of the last selection taken or None
        self.pickdata = None
        self.selection_mode = None
        self.editing_model = False
        self.stop_requested = False
        settings.keystore.changed.connect(self.on_settings_changed, Qt.QueuedConnection)
        
        if DEBUG_FPS:
            self.monotonic_time = QElapsedTimer()
            self.monotonic_time.start()
            self.render_count = 0
            self.fps = 0.
            
        self.speed = settings.draw.speed
        glarea.init_module()
        gldraw.init_module()
        self.rotation_x, self.rotation_y = glarea.set_rotation_xy(*settings.draw.default_rotation)
        accum_buffers = settings.draw.accum if self.format().accum() else 0
        sample_buffers = self.format().sampleBuffers()
        glarea.set_antialiasing(accum_buffers, sample_buffers)
        
        self.setAcceptDrops(True)
        self.setFocusPolicy(Qt.StrongFocus)
        self.setMinimumSize(200, 200)
        
        self.cursors = None
        self.load_cursors()
        self.update_selection_pending = False
        self.set_cursor()
        self.setMouseTracking(True)
        
    def load_cursors(self):
        cursors = []
        # Load 3 cursors from file (n - ne)
        for i, (x, y) in enumerate([(8, 0), (15, 0), (15, 0)]):
            filename = os.path.join(config.UI_DIR, 'mouse_{}.png'.format(i))
            image = QImage(filename)
            cursors.append((image, x, y))
        # 1 cursor (nnw)
        image, x, y = cursors[1]
        cursors.insert(0, (image.mirrored(True, False), 15-x, y))
        # 12 cursors (ene - nw)
        transform = QTransform()
        transform.rotate(90)
        for i in range(4, 16):
            image, x, y = cursors[-4]
            cursors.append((image.transformed(transform), 15-y, x))
        cursors.append(cursors[0])
        self.cursors = [QCursor(QPixmap.fromImage(image), x, y) for image, x, y in cursors[1:]]
        # cursor for center faces
        filename = os.path.join(config.UI_DIR, 'mouse_ccw.png')
        cursor = QCursor(QPixmap(filename), 7, 7)
        self.cursors.append(cursor)
        # background cursor
        cursor = QCursor()
        cursor.setShape(Qt.CursorShape.CrossCursor)
        self.cursors.append(cursor)
        
    def apply_design(self):
        for i in xrange(6):
            color = settings.theme.face[i].color
            imagefile = settings.theme.face[i].image
            imagemode = settings.theme.face[i].mode
            self.set_face_design(i, color, imagefile, imagemode)
        self.set_lighting(settings.draw.lighting)
        self.set_background_color(settings.theme.bgcolor)
        
    stock_texnames = [-1, 0]
    def set_face_texture(self, faceidx, imagefile):
        if imagefile.startswith('/'):
            pixbuf = textures.create_pixbuf_from_file(imagefile)
        else:
            pixbuf = textures.get_stock_pixbuf(imagefile)
        if pixbuf is None:
            texname = -1
        else:
            texname = self.bindTexture(pixbuf, GL_TEXTURE_2D, GL_RGBA,
                                   QGLContext.BindOption.LinearFilteringBindOption|
                                   QGLContext.BindOption.MipmapBindOption)
        if not imagefile.startswith('/') and texname not in self.stock_texnames:
            self.stock_texnames.append(texname)
        texname = gldraw.set_face_texture(faceidx, texname)
        if texname not in self.stock_texnames:
            self.deleteTexture(texname)
        
    def set_face_design(self, i, color, imagefile, imagemode):
        rgba = QColor()
        rgba.setNamedColor(color)
        gldraw.set_face_rendering(i, red=rgba.redF(), green=rgba.greenF(), blue=rgba.blueF(),
                                     imagemode=imagemode)
        self.set_face_texture(i, imagefile)
        
    def set_model(self, model_):
        self.model = model_
        glarea.set_frustrum(self.model.bounding_sphere_radius, settings.draw.zoom)
        gldraw.set_model(self.model)
        self.set_selection_mode(settings.draw.selection)
        
    @staticmethod
    def apply_to_glmodel(cubestate):
        gldraw.set_transformations(cubestate.blocks)
        
    def initializeGL(self):
        if DEBUG_MSG:
            glcontext = self.context()
            glformat = glcontext.format()
            glrformat = glcontext.requestedFormat()
            def printglattr(name, *attrs):
                print('  {}: '.format(name), end='')
                def get_value(glformat, attr):
                    if isinstance(attr, basestring):
                        return getattr(glformat, attr)()
                    else:
                        return attr(glformat)
                values = [get_value(glformat, a) for a in attrs]
                rvalues = [get_value(glrformat, a) for a in attrs]
                if values == rvalues:
                    print(*values)
                else:
                    print(*values, end=' (')
                    print(*rvalues, end=')\n')
            print('OpenGL format:')
            printglattr('accum', 'accum', 'accumBufferSize')
            printglattr('alpha', 'alpha', 'alphaBufferSize')
            printglattr('rgb', 'redBufferSize', 'greenBufferSize', 'blueBufferSize')
            printglattr('depth', 'depth', 'depthBufferSize')
            printglattr('directRendering', 'directRendering')
            printglattr('doubleBuffer', 'doubleBuffer')
            printglattr('hasOverlay', 'hasOverlay')
            printglattr('version', lambda glformat: '{}.{} 0x{:x}'.format(
                                                            glformat.majorVersion(),
                                                            glformat.minorVersion(),
                                                            int(glformat.openGLVersionFlags())))
            printglattr('plane', 'plane')
            printglattr('profile', 'profile')
            printglattr('rgba', 'rgba')
            printglattr('samples', 'sampleBuffers', 'samples')
            printglattr('stencil', 'stencil', 'stencilBufferSize')
            printglattr('stereo', 'stereo')
            printglattr('swapInterval', 'swapInterval')
        glarea.init_glarea()
        self.apply_design()
        
    def draw_face_debug(self):
        maxis, mslice, mdir, block, symbol, face, center, angle = self.pickdata
        self.qglColor(QColor(255, 255, 255))
        self.renderText(2, 18, "block %s, face %s (%s), center %s" % (block.index, symbol, face, center))
        self.renderText(2, 34, "axis %s, slice %s, dir %s" % (maxis, mslice, mdir))
        self.renderText(2, 50, "angle %s" % (angle))
        
    def paintGL(self):
        glarea.display()
        if DEBUG:
            if DEBUG_DRAW:
                if self.pickdata is not None:
                    self.draw_face_debug()
                if self.selection_debug_info is not None:
                    gldraw.draw_select_debug(*self.selection_debug_info)
            if DEBUG_FPS:
                self.render_count += 1
                if self.monotonic_time.elapsed() > 1000:
                    elapsed = self.monotonic_time.restart()
                    self.fps = 1000. / elapsed * self.render_count
                    self.render_count = 0
                self.qglColor(QColor(255, 255, 255))
                self.renderText(2, 18, "FPS %.1f" % self.fps)
        
    def resizeGL(self, width, height):
        glarea.resize(width, height)
        
    MODIFIER_MASK = int(Qt.ShiftModifier | Qt.ControlModifier)
    def keyPressEvent(self, event):
        modifiers = int(event.modifiers()) & self.MODIFIER_MASK
        if modifiers:
            return QGLWidget.keyPressEvent(self, event)
        elif event.key() == Qt.Key_Right:
            self.rotation_x += 2
        elif event.key() == Qt.Key_Left:
            self.rotation_x -= 2
        elif event.key() == Qt.Key_Up:
            self.rotation_y -= 2
        elif event.key() == Qt.Key_Down:
            self.rotation_y += 2
        else:
            return QGLWidget.keyPressEvent(self, event)
            
        self.rotation_x, self.rotation_y = glarea.set_rotation_xy(self.rotation_x, self.rotation_y)
        self.update()
        self.update_selection()
        
    PickData = namedtuple('PickData', 'maxis mslice mdir block symbol face center angle')
    def pick_polygons(self, x, y):
        '''Identify the block at screen co-ordinates x,y.'''
        
        height = self.height()
        index = glarea.pick_polygons(x, height-y, 1)
        if not index:
            self.selection_debug_info = None
            self.pickdata = None
            return
        maxis, mslice, mdir, face, center, block, symbol, face_center, edge_center = self.model.pick_data[index]
        
        if center:
            angle, self.selection_debug_info = None, None
        else:
            angle, self.selection_debug_info = glarea.get_cursor_angle(face_center, edge_center)
        self.pickdata = self.PickData(maxis, mslice, mdir, block, symbol, face, center, angle)
        
    def update_selection(self):
        '''This func determines which block the mouse is pointing at'''
        if self.timer_animate.isActive():
            if self.pickdata is not None:
                self.pickdata = None
                self.set_cursor()
            return
        if self.update_selection_pending:
            return
        QTimer.singleShot(0, self.on_idle_update_selection)
        self.update_selection_pending = True
        
    def on_idle_update_selection(self):
        if self.timer_animate.isActive():
            if self.pickdata is not None:
                self.pickdata = None
                self.set_cursor()
        else:
            self.pick_polygons(*self.mouse_xy)
            self.set_cursor()
            if DEBUG_DRAW:
                if not self.timer_animate.isActive():
                    self.updateGL()
        self.update_selection_pending = False
        
    def mouseMoveEvent(self, event):
        self.mouse_xy = event.x(), event.y()
        
        if not self.button_down_background:
            self.update_selection()
            return
            
        # perform rotation
        offset_x = event.x() - self.last_mouse_x
        offset_y = event.y() - self.last_mouse_y
        self.rotation_x, self.rotation_y = glarea.set_rotation_xy(
                                                        self.rotation_x + offset_x,
                                                        self.rotation_y + offset_y)
        self.rotation_x -= offset_x
        self.rotation_y -= offset_y
        self.update()
        
    def mousePressEvent(self, event):
        if self.pickdata is not None:
            if self.timer_animate.isActive():
                return
            # make a move
            if self.editing_model:
                if event.modifiers() & Qt.ControlModifier:
                    if event.button() == Qt.LeftButton:
                        self.request_rotate_block.emit(self.pickdata.block.index, False)
                    elif event.button() == Qt.RightButton:
                        self.request_rotate_block.emit(self.pickdata.block.index, True)
                else:
                    if event.button() == Qt.LeftButton:
                        self.request_swap_blocks.emit(self.pickdata.block.index,
                                    self.pickdata.maxis, self.pickdata.mslice, self.pickdata.mdir)
            else:
                mslice = -1 if event.modifiers() & Qt.ControlModifier else self.pickdata.mslice
                if event.button() == Qt.LeftButton:
                    self.request_rotation.emit(self.pickdata.maxis, mslice, self.pickdata.mdir)
                elif event.button() == Qt.RightButton and settings.draw.selection_nick == 'simple':
                    self.request_rotation.emit(self.pickdata.maxis, mslice, not self.pickdata.mdir)
        elif event.button() == Qt.LeftButton:
            # start rotation
            self.button_down_background = True
            self.last_mouse_x = event.x()
            self.last_mouse_y = event.y()
        self.update()
        
    def mouseReleaseEvent(self, event):
        if event.button() != Qt.LeftButton:
            return
            
        if self.button_down_background:
            # end rotation
            self.rotation_x += event.x() - self.last_mouse_x
            self.rotation_y += event.y() - self.last_mouse_y
            self.button_down_background = False
            self.update_selection()
            
    def wheelEvent(self, event):
        if event.orientation() == Qt.Vertical:
            zoom = settings.draw.zoom * math.pow(1.1, -event.delta() / 120)
            zoom_min, zoom_max = settings.draw.zoom_range
            if zoom < zoom_min:
                zoom = zoom_min
            if zoom > zoom_max:
                zoom = zoom_max
            settings.draw.zoom = zoom
            
    def dragEnterEvent(self, event):
        debug('drag enter:', event.mimeData().formats())
        if (event.mimeData().hasFormat("text/uri-list") or
                event.mimeData().hasFormat("application/x-color")):
            event.acceptProposedAction()
            
    def dropEvent(self, event):
        # when a drag is in progress the pickdata is not updated, so do it now
        self.pick_polygons(event.pos().x(), event.pos().y())
        
        mime_data = event.mimeData()
        if mime_data.hasFormat("application/x-color"):
            color = mime_data.colorData()
            if color is None:
                debug("Received invalid color data")
                return
                
            if self.pickdata is not None:
                self.drop_color.emit(self.pickdata.block.index, self.pickdata.symbol, color.name())
            else:
                self.drop_color.emit(-1, '', color.name())
        elif mime_data.hasFormat("text/uri-list"):
            if self.pickdata is None:
                debug('Background image is not supported.')
                return
            uris = mime_data.urls()
            for uri in uris:
                if not uri.isLocalFile():
                    debug('filename "%s" not found or not a local file.' % uri.toString())
                    continue
                filename = uri.toLocalFile()
                if not filename or not os.path.exists(filename):
                    debug('filename "%s" not found.' % filename)
                    continue
                self.drop_file.emit(self.pickdata.block.index, self.pickdata.symbol, filename)
                break # For now,  just use the first one.
        # Ignore all others
        
    def set_cursor(self):
        if self.pickdata is None or self.button_down_background:
            index = -1
        elif self.pickdata.angle is None:
            index = -2
        else:
            index = int((self.pickdata.angle+180) / 22.5 + 0.5) % 16
        self.setCursor(self.cursors[index])
        
    def set_std_cursor(self):
        QTimer.singleShot(0, lambda: self.unsetCursor())
        
    @staticmethod
    def set_lighting(enable):
        glarea.set_lighting(enable)
        
    def set_selection_mode(self, mode):
        self.selection_mode = mode
        gldraw.set_pick_model(*self.model.gl_pick_data(mode))
        self.update_selection()
        
    def set_editing_mode(self, enable):
        self.editing_model = enable
        if enable:
            gldraw.set_pick_model(*self.model.gl_pick_data(0))
        else:
            gldraw.set_pick_model(*self.model.gl_pick_data(self.selection_mode))
        self.update_selection()
        
    @staticmethod
    def set_background_color(color):
        rgba = QColor()
        rgba.setNamedColor(color)
        glarea.set_background_color(rgba.redF(), rgba.greenF(), rgba.blueF())
        
    def reset_rotation(self):
        '''Reset cube rotation'''
        self.rotation_x, self.rotation_y = glarea.set_rotation_xy(*settings.draw.default_rotation)
        self.update()
        
    ### Animation
    
    def animate_rotation(self, move_data, blocks, stop_after):
        self.stop_requested = stop_after
        axis = (self.model.axesI if move_data.dir else self.model.axes)[move_data.axis]
        angle = self.model.symmetry[move_data.axis]
        gldraw.start_animate(angle, *axis)
        for block_id, selected in blocks:
            gldraw.set_animation_block(block_id, selected)
        self.timer_animate.start(0 if DEBUG_VFPS else 20)
        self.update_selection()
        
    def _on_animate(self):
        increment = self.speed * 1e-02 * 20
        increment = min(increment, 45)
        unfinished = gldraw.step_animate(increment)
        if unfinished:
            self.updateGL()
            return
            
        # we have finished the animation sequence now
        gldraw.end_animate()
        self.timer_animate.stop()
        self.end_animation.emit(self.stop_requested)
        self.updateGL()
        self.update_selection()
        self.stop_requested = False
        
    def animate_abort(self, update=True):
        self.timer_animate.stop()
        gldraw.end_animate()
        if update:
            self.end_animation.emit(self.stop_requested)
            self.update()
            self.update_selection()
        self.stop_requested = False
        
    def on_settings_changed(self, key):
        if key == 'draw.speed':
            self.speed = settings.draw.speed
        elif key == 'draw.lighting':
            self.set_lighting(settings.draw.lighting)
        elif key == 'draw.accum':
            if self.format().accum():
                glarea.set_antialiasing(settings.draw.accum)
        elif key == 'draw.samples':
            if self.format().samples():
                samples = 2**settings.draw.samples
                if samples == 1:
                    glarea.set_antialiasing(None, False)
                elif samples == self.format().samples():
                    glarea.set_antialiasing(None, True)
        elif key == 'draw.selection':
            self.set_selection_mode(settings.draw.selection)
        elif key.startswith('theme.face.'):
            i = int(key.split('.')[2])
            if key == 'theme.face.{}.color'.format(i):
                color = QColor()
                color.setNamedColor(settings.theme.face[i].color)
                gldraw.set_face_rendering(i, red=color.redF(), green=color.greenF(), blue=color.blueF())
            elif key == 'theme.face.{}.image'.format(i):
                self.set_face_texture(i, settings.theme.face[i].image)
            elif key == 'theme.face.{}.mode'.format(i):
                imagemode = settings.theme.face[i].mode
                gldraw.set_face_rendering(i, imagemode=imagemode)
        elif key == 'theme.bgcolor':
            self.set_background_color(settings.theme.bgcolor)
        elif key == 'draw.zoom':
            glarea.set_frustrum(self.model.bounding_sphere_radius, settings.draw.zoom)
        else:
            debug('Unknown settings key changed:', key)
        self.update()
        
        
