#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2011  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


import random
from copy import deepcopy

from .debug import *

class ParseBlockException (Exception): pass


# This class is currently used to store the initial state of the cube
# and to provide the cube to plugin scripts.
class CubeState (object):
    flubrd = 'FLUBRD'
    rand = random.Random()
    if DEBUG_LEVEL > 0:
        print 'rand.seed(123)'
        rand.seed(123)
    
    def __init__(self):
        self.type = 'Cube'
        # Information is stored in module cube 
        self._dimension = None
        self._faces = None
        self._blocks = None
        
    def copy(self):
        other = CubeState()
        other._dimension = self._dimension
        other._faces = deepcopy(self._faces)
        other._blocks = deepcopy(self._blocks)
        return other
        
    @property
    def dimension(self):
        return self._dimension
    @property
    def faces(self):
        if self._faces is None:
            self._faces = self._make_faces_cube()
        return self._faces
    @property
    def blocks(self):
        assert self._blocks
        return self._blocks
    
    def set_solved(self, dimension=None):
        assert self.type == 'Cube'
        if dimension:
            assert 0 < dimension <= 10
            self._dimension = dimension
        self._set_cube_dimension()
        self._faces = None
        
    def _set_cube_dimension(self):
        if not self._dimension:
            return
        
        def block_index_to_coord(i, d):
            return float(2 * i - d + 1)
        def block_index_to_coords(i, d):
            return [ block_index_to_coord(i % d, d),
                     block_index_to_coord((i / d) % d, d),
                     block_index_to_coord((i / (d * d)) % d, d),
                     1.0
                   ]
        
        number_blocks = self._dimension ** 3
        # Initialize all transformations to the identity matrix,  then set the
        # translation part to correspond to the initial position of the block.
        self._blocks = []
        for i in range(number_blocks):
            self._blocks.append([[1.0, 0.0, 0.0, 0.0],
                                 [0.0, 1.0, 0.0, 0.0],
                                 [0.0, 0.0, 1.0, 0.0],
                                 block_index_to_coords(i, self._dimension),
                                ])
    def _make_faces_cube(self):
        if self.dimension is None:
            return None
        
        MATRIX_DIM = 4
        number_blocks = self.dimension ** 3
        def transform(M, x):
            q = [0] * MATRIX_DIM
            
            for i in range(MATRIX_DIM):
                q[i] = 0
                for j in range(MATRIX_DIM):
                    q[i] += M[j][i] * x[j]
            return q
        def block_coords_to_index(i):
            return int((i + self.dimension - 1) / 2.0)
        
        #char colours [6] [self.dimension] [self.dimension]
        colours = []
        for j in range(6):
            colour = []
            for i in range(self.dimension):
                colour.append(list(' '*self.dimension))
            colours.append(colour)

        # Loop over all blocks and faces,  but only process if the face is on the
        # outside of the cube.

        _visible_faces = [ (lambda i, d: 0 == i / (d * d)),
                           (lambda i, d: d - 1 == i / (d * d)),
                           (lambda i, d: 0 == i / d % d),
                           (lambda i, d: d - 1 == i / d % d),
                           (lambda i, d: 0 == i % d),
                           (lambda i, d: d - 1 == i % d),
                         ]
            
        _cube_blocks_face = [[ 0.0, 0.0,-1.0, 0.0],
                             [ 0.0, 0.0, 1.0, 0.0],
                             [ 0.0,-1.0, 0.0, 0.0],
                             [ 0.0, 1.0, 0.0, 0.0],
                             [-1.0, 0.0, 0.0, 0.0],
                             [ 1.0, 0.0, 0.0, 0.0]]
        for i in range(number_blocks-1, -1, -1):
            for face in range(6):
                if _visible_faces[face](i, self.dimension):
                    # Apply the rotation part of the block transformation to the
                    # offset of the face centre from the centre of the block. This
                    # will tell us the direction the face is now facing.
                    
                    face_direction_ = _cube_blocks_face[face]
                    face_direction = transform(self._blocks[i], face_direction_)

                    # The face direction will have exactly one non-zero component
                    # this is in the direction of the normal,  so we can infer which
                    # side of the cube we are on. Further,  if we look at the other
                    # two components of the location part of the block
                    # transformation,  we can infer the position of the block in the
                    # face. We set the appropriate element in the colours array to
                    # the colour of this face,  which corresponds exactly with the
                    # original face the block was at.

                    if abs(face_direction[0]) > 0.1:
                        if face_direction[0] < 0.0: f = 4
                        else:                       f = 5
                        colours[f] \
                               [block_coords_to_index(self._blocks[i][3][1])] \
                               [block_coords_to_index(self._blocks[i][3][2])] = face
                    elif abs(face_direction[1]) > 0.1:
                        if face_direction[1] < 0.0: f = 2
                        else:                       f = 3
                        colours[f] \
                               [block_coords_to_index(self._blocks[i][3][0])] \
                               [block_coords_to_index(self._blocks[i][3][2])] = face
                    else:
                        if face_direction[2] < 0.0: f = 0
                        else:                       f = 1
                        colours[f] \
                               [block_coords_to_index(self._blocks[i][3][0])] \
                               [block_coords_to_index(self._blocks[i][3][1])] = face

        # Now the colours array should be completely populated
        
        face_vector = []
        for face in range(6):
            block_vector = []
            for j in range(self.dimension):
                for i in range(self.dimension):
                    block_vector.append(colours[face][i][j])
            face_vector.append(block_vector)

        return face_vector
        
    def identify_rotation_blocks(self, axis, slice_):
        assert self._blocks
        
        slice_pos = slice_ * 2 - self.dimension + 1
        for i, block in enumerate(self._blocks):
            if slice_ == -1 or abs(block[3][axis] - slice_pos) < 0.1:
                yield i
                
    def _rotate_slice(self, axis, slice_, dir_):
        assert self._blocks
        
        MATRIX_DIM = 4
        def _cos_quadrant(quarters):
            quarters = abs(quarters) % 4
            if quarters == 0:
                return 1
            elif quarters == 2:
                return -1
            else:
                return 0
        def _sin_quadrant(quarters):
            return _cos_quadrant(quarters - 1)
        def _pre_mult(M, block):
            N = self._blocks[block]
            self._blocks[block] = T = []
            for k in range(MATRIX_DIM):
                T.append([])
                for i in range(MATRIX_DIM):
                    T[k].append(0)
                    T[k][i] = sum(M[j][i] * N[k][j] for j in range(MATRIX_DIM))
        
        # Create a matrix describing the rotation of we are about to perform.
        # -90 degrees equals +270 degrees
        rotation = [[1,0,0,0], [0,1,0,0], [0,0,1,0], [0,0,0,1]]
        turns = 1 if dir_ else 3
        rotation[(axis + 1) % 3][(axis + 1) % 3] = _cos_quadrant(turns)
        rotation[(axis + 2) % 3][(axis + 2) % 3] = _cos_quadrant(turns)
        rotation[(axis + 2) % 3][(axis + 1) % 3] = _sin_quadrant(turns)
        rotation[(axis + 1) % 3][(axis + 2) % 3] = -_sin_quadrant(turns)

        # Apply the rotation matrix to all the blocks in this slice.
        if DEBUG_ROTATE:
            debug('rotate axis={} slice={} dir={!s:5} blocks:'.format(axis, slice_, dir_), end=' ')
        for i in self.identify_rotation_blocks(axis, slice_):
            if DEBUG_ROTATE: debug('{!s:2}'.format(i), end=' ')
            _pre_mult(rotation, i)
        if DEBUG_ROTATE: debug()
        self._faces = None
        
    def get_color(self, face, x, y):
        return self.faces[face][x + y*self._dimension]
        
    def dump(self):
        for i, f in enumerate(self.faces):
            print 'Face %d:'%i, str(f)
        print
        #for i, b in enumerate(self._blocks):
        #    debug('Block %d:'%i, str(b))
        
    rotation_matrices = [
        [[ 1,0,0,0],  [0, 1,0,0],  [0,0, 1,0]], # 0 empty
        [[ 1,0,0,0],  [0,-1,0,0],  [0,0,-1,0]], # 1 ff lluu uull
        [[-1,0,0,0],  [0, 1,0,0],  [0,0,-1,0]], # 2 ll ffuu uuff
        [[-1,0,0,0],  [0,-1,0,0],  [0,0, 1,0]], # 3 uu ffll llff
        
        [[ 1,0,0,0],  [0,0, 1,0],  [0,-1,0,0]], # 4 f
        [[ 1,0,0,0],  [0,0,-1,0],  [0, 1,0,0]], # 5 f'
        [[-1,0,0,0],  [0,0, 1,0],  [0, 1,0,0]], # 6 fuu uuf' f'll llf
        [[-1,0,0,0],  [0,0,-1,0],  [0,-1,0,0]], # 7 fll llf' f'uu uuf
        
        [[0, 1,0,0],  [ 1,0,0,0],  [0,0,-1,0]], # 8 ffu ull llu' u'ff
        [[0, 1,0,0],  [-1,0,0,0],  [0,0, 1,0]], # 9 u
        [[0,-1,0,0],  [ 1,0,0,0],  [0,0, 1,0]], #10 u'
        [[0,-1,0,0],  [-1,0,0,0],  [0,0,-1,0]], #11 ffu' u'll llu uff
        
        [[0, 1,0,0],  [0,0, 1,0],  [ 1,0,0,0]], #12 fu ul lf
        [[0, 1,0,0],  [0,0,-1,0],  [-1,0,0,0]], #13 f'u ul' l'f'
        [[0,-1,0,0],  [0,0, 1,0],  [-1,0,0,0]], #14 fu' u'l' l'f
        [[0,-1,0,0],  [0,0,-1,0],  [ 1,0,0,0]], #15 f'u' u'l lf'
        
        [[0,0, 1,0],  [ 1,0,0,0],  [0, 1,0,0]], #16 f'l' l'u' u'f'
        [[0,0, 1,0],  [-1,0,0,0],  [0,-1,0,0]], #17 uf fl' l'u
        [[0,0,-1,0],  [ 1,0,0,0],  [0,-1,0,0]], #18 fl lu' u'f
        [[0,0,-1,0],  [-1,0,0,0],  [0, 1,0,0]], #19 lu uf' f'l
        
        [[0,0, 1,0],  [0, 1,0,0],  [-1,0,0,0]], #20 l'
        [[0,0, 1,0],  [0,-1,0,0],  [ 1,0,0,0]], #21 ffl' l'uu uul lff
        [[0,0,-1,0],  [0, 1,0,0],  [ 1,0,0,0]], #22 l
        [[0,0,-1,0],  [0,-1,0,0],  [-1,0,0,0]], #23 ffl luu uul' l'ff
    ]
    
    rotation_symbolic = [
        "",     # 0 empty
        "ff",   # 1 lluu uull
        "ll",   # 2 ffuu uuff
        "uu",   # 3 ffll llff
        
        "f",    # 4
        "f'",   # 5
        "fuu",  # 6 uuf' f'll llf
        "fll",  # 7 llf' f'uu uuf
        
        "ffu",  # 8 ull llu' u'ff
        "u",    # 9 
        "u'",   #10 
        "ffu'", #11 u'll llu uff
        
        "fu",   #12 ul lf
        "f'u",  #13 ul' l'f'
        "fu'",  #14 u'l' l'f
        "f'u'", #15 u'l lf'
        
        "f'l'", #16 l'u' u'f'
        "uf",   #17 fl' l'u
        "fl",   #18 lu' u'f
        "lu",   #19 uf' f'l
        
        "l'",   #20 
        "ffl'", #21 l'uu uul lff
        "l",    #22 
        "ffl",  #23 luu uul' l'ff
    ]
    
    @staticmethod
    def coord_to_slice(coord, size):
        assert -size < coord < size
        return int(coord+size-1)//2
        
    def format_block_compact(self):
        assert self._blocks
        res = '%s %d blocks_compact:' % (self.type, self.dimension)
        # every block is stored as 4-tuple (x,y,z,rot),
        # where rot is an index to a list of all (24) rotation matrices.
        for block in self._blocks:
            rot = self.rotation_matrices.index(block[:-1])
            x, y, z = [self.coord_to_slice(i, self.dimension) for i in block[3][:3]]
            res += ' %d,%d,%d,%s' % (x, y, z, self.rotation_symbolic[rot])
        return res
        
    @staticmethod
    def slice_to_coord(slice_, size):
        assert 0 <= slice_ < size, (slice_, size)
        return float(slice_ * 2 - size + 1)
        
    def parse_block(self, code):
        tdf, blocks = code.split(':', 1)
        t, d, f = tdf.split(' ', 2)
        if t != 'Cube':
            raise ParseBlockException('Unknown model type: {}'.format(t))
        self._dimension = int(d)
        if f == 'identity':
            #self._blocks = 1
            self.set_solved()
            assert self._blocks
            return
        if f != 'blocks_compact':
            raise ParseBlockException('Unknown data format: {}'.format(f))
        blocks = blocks.strip().split(' ')
        if len(blocks) != self._dimension**3:
            raise ParseBlockException('Wrong block count: {}, expected: {}'.format(
                                                    len(blocks), self._dimension**3))
        for i, block in enumerate(blocks):
            block = block.strip().split(',')
            if len(block) != 4:
                raise ParseBlockException('Wrong block size: {}, expected: {}'.format(
                                                    len(block), 4))
            rot = self.rotation_symbolic.index(block[3])
            for j in range(3):
                block[j] = self.slice_to_coord(int(block[j]), self._dimension)
            block[3] = 1
            blocks[i] = self.rotation_matrices[rot] + [block]
        self._blocks = blocks
        self._faces = None
        
    def randomize_and_apply(self):
        assert self._dimension
        self._set_cube_dimension()
        
        debug('random moves:', 20*self._dimension)
        for unused_i in xrange(20 * self._dimension):
            axis = self.rand.randrange(3)
            dir_ = self.rand.randrange(2)
            if self._dimension > 1 and self._dimension & 1:
                slice_ = self.rand.randrange(self._dimension - 1)
                if slice_ >= self._dimension // 2:
                    slice_ += 1
            else:
                slice_ = self.rand.randrange(self._dimension)
            self._rotate_slice(axis, slice_, dir_)
            
    def rotate_slice(self, move_data):
        self._rotate_slice(move_data.axis, move_data.slice, move_data.dir)
        
    def rotate_slice_back(self, move_data):
        self._rotate_slice(move_data.axis, move_data.slice, not move_data.dir)
        
        
