import gtk
import gtk.glade
import gobject
import gnome
import gnome.ui
import sys
import os
import shutil
import re
import subprocess
try:
	import nautilusburn
except ImportError:
	pass
else:
	pass
import traceback

import rdiff_interface
import version
import Dialogs
from LogHandler import LogHandler
from Dialogs import ErrorDialog, InfoDialog, MsgDialog
from Copy import RecursiveCopy, RecursiveCount
from mkisofs import Mkisofs
from seteditor import SetEditor
from filechooser import FileChooser
from statuswindow import StatusWindow

class GuiHandlers:
	def __init__(self, widgets):
		self.widgets = widgets
		self.seteditor = SetEditor()
		self.statuswin = StatusWindow()

	##########################
	## MAIN WINDOW HANDLERS ##
	##########################
	
	### HOME PAGE ###
	def on_button_home_dir_backup_clicked(self, widget):
		self.select_set("home")
		self.widgets.get_widget('cmb_backup_type').set_active(1) # CDRW
		self.widgets.get_widget('notebook1').set_current_page(1) # backup page
		for i in range(15):
			gtk.main_iteration()
		self.widgets.get_widget('button_do_backup').emit("clicked")

	### BACKUP PAGE ###
	def on_treeview_backup_sets_cursor_changed(self, widget):
		setid = widget.get_cursor()[0][0]
		self.widgets.get_widget('label_backup_set_info').set_text(rdiff_interface.backupsets[setid]['desc'])
		self.widgets.get_widget('label_backup_set_title').set_markup("<big><b>"+rdiff_interface.backupsets[setid]['name']+"</b></big>")
		for widget in [	'backup_ssh_user', \
						'backup_ssh_host', \
						'backup_ssh_path']:
			self.widgets.get_widget(widget).set_text('')
		self.widgets.get_widget('set_destination').child.set_text('')
		if rdiff_interface.backupsets[setid].has_key('default_dest'):
			dest = rdiff_interface.backupsets[setid]['default_dest']
			if dest[:7] == "sftp://":
				p = re.compile("sftp://([^@]+)@([^/]+)(.*)")
				matches = p.match(dest)
				if len(matches.groups()) < 3:
					user = ""
					host = ""
					path = ""
				else:
					user = matches.group(1)
					host = matches.group(2)
					path = matches.group(3)
				self.widgets.get_widget('cmb_backup_type').set_active(2)
				self.widgets.get_widget('backup_ssh_user').set_text(user)
				self.widgets.get_widget('backup_ssh_host').set_text(host)
				self.widgets.get_widget('backup_ssh_path').set_text(path)
				self.widgets.get_widget('button_do_backup').set_sensitive(True)
			elif dest[:7] == "cdrw://":
				device = dest.replace("cdrw://", "")
				self.widgets.get_widget('cmb_backup_type').set_active(1)
				found = False
				for row in self.widgets.get_widget('cmb_backup_burner').get_model():
					if row[1].get_device() == device:
						self.widgets.get_widget('cmb_backup_burner').set_active_iter(row.iter)
						found = True
				if found == False:
					self.widgets.get_widget('cmb_backup_burner').set_active(0)
				self.widgets.get_widget('button_do_backup').set_sensitive(True)
			else:
				if rdiff_interface.backupsets[setid]['removable'] == "True":
					self.widgets.get_widget('backup_removable').set_active(True)
				else:
					self.widgets.get_widget('backup_removable').set_active(False)
				self.widgets.get_widget('cmb_backup_type').set_active(0)
				self.widgets.get_widget('set_destination').child.set_text(dest)
		else:
			self.widgets.get_widget('cmb_backup_type').set_active(0)
		self.widgets.get_widget('backup_controls').set_sensitive(True)
		self.widgets.get_widget('button_delete_set').set_sensitive(True)

	def on_button_new_set_clicked(self, widget):
		self.seteditor.setexitnotify(self.refresh_set_list)
		self.seteditor.show()

	def on_button_delete_set_clicked(self, widget):
		set = rdiff_interface.backupsets[self.widgets.get_widget('treeview_backup_sets').get_cursor()[0][0]]['name']
		if set == _("home"):
			ErrorDialog(_("You cannot delete this set."), self.widgets.get_widget('window_main'))
			return True
			
		dlg = gtk.MessageDialog( \
				self.widgets.get_widget('window_main'), \
				gtk.DIALOG_MODAL, gtk.MESSAGE_WARNING, gtk.BUTTONS_YES_NO, \
				_("Are you sure you wish to delete the backup set '%s'?\nDeleting this backup set will not delete any previous backups, nor will it affect your ability to restore them.") % set)
		
		dlg.connect("response", self.on_delete_set_dialog_response, set)
		dlg.show()
		
	def on_delete_set_dialog_response(self, widget, response, set):
		if response == gtk.RESPONSE_YES:
			rdiff_interface.DeleteSet(set)
			self.refresh_set_list()
			self.widgets.get_widget('backup_controls').set_sensitive(False)
			self.widgets.get_widget('button_delete_set').set_sensitive(False)
			self.widgets.get_widget('label_backup_set_title').set_text("")
			if len(rdiff_interface.backupsets) == 0:
				self.widgets.get_widget('label_backup_set_info').set_text(_("There are no remaining backup sets.\nTo create a new backup set, click 'Add'."))
			else:
				self.widgets.get_widget('label_backup_set_info').set_text(_("Select a backup set from the list on the left."))
			widget.destroy()
		else:
			widget.destroy()
			
	def on_set_destination_changed(self, widget):
		if len(widget.child.get_text()) > 0:
			self.widgets.get_widget('button_do_backup').set_sensitive(True)
		else:
			self.widgets.get_widget('button_do_backup').set_sensitive(False)
	
	def on_edit1_activate(self, widget):
		selection = self.widgets.get_widget('treeview_backup_sets').get_selection()
		store, row = selection.get_selected()
		if row is not None:
			setid = int(store.get_string_from_iter(row))
			self.seteditor.setexitnotify(self.refresh_set_list)
			if self.seteditor.set_backupset_to_edit(setid):
				self.seteditor.show()
			else:
				ErrorDialog("Selected backup set cannot be edited.", self.widgets.get_widget('window_main'))

	def on_button_do_backup_clicked(self, widget):
		rdiff_interface.Refresh()
		if self.widgets.get_widget('chk_show_output_log').get_active():
			self.statuswin.show()
		self.widgets.get_widget('progressbar1').set_fraction(0.05)
		self.widgets.get_widget('lbl_status').set_text(_("Starting backup..."))
		self.widgets.get_widget('notebook1').set_sensitive(False)
		self.widgets.get_widget('menubar1').set_sensitive(False)
		self.widgets.get_widget('window_main').window.set_cursor(gtk.gdk.Cursor(gtk.gdk.WATCH))
		while gtk.events_pending():
			gtk.main_iteration()
		try:
			if self.widgets.get_widget('cmb_backup_type').get_active() == 0: # local
				self.do_local_backup()
			elif self.widgets.get_widget('cmb_backup_type').get_active() == 1: # cdr
				self.do_cdr_backup()
			elif self.widgets.get_widget('cmb_backup_type').get_active() == 2: # ssh
				self.do_ssh_backup()
		finally:
			self.widgets.get_widget('notebook1').set_sensitive(True)
			self.widgets.get_widget('menubar1').set_sensitive(True)
			self.widgets.get_widget('window_main').window.set_cursor(None)	

	def show_progress(self, progress=0):
		self.widgets.get_widget('progressbar1').set_fraction(progress/100)
		gtk.main_iteration()

	def do_cdr_backup(self):
		"""backs up to /tmp, does a mkisofs, then nautilusburns it"""
		set_id = self.widgets.get_widget('treeview_backup_sets').get_cursor()[0][0]
		set = rdiff_interface.backupsets[set_id]
		self.statuswin.addmsg(_("Starting backup of '%s' to CD\n") % set['name'])
		# do we have a burner to use?
		try:
			nautilusburn
		except NameError:
			error_string = _("No CD burner available, because you don't have the python module nautilusburn.")
			self.widgets.get_widget('lbl_backup_burner').set_text(error_string)
			self.widgets.get_widget('cmb_backup_burner').set_sensitive(False)

		if self.widgets.get_widget('cmb_backup_burner').get_active() == -1:
			ErrorDialog(_("There are no CD burners attached to this system."), self.widgets.get_widget('window_main'))
			self.statuswin.addmsg(_("Backup failed; no CD burners detected.\n"))
			return True
		drive = self.widgets.get_widget('cmb_backup_burner').get_model()[self.widgets.get_widget('cmb_backup_burner').get_active()][1]
		device_name = self.widgets.get_widget('cmb_backup_burner').get_model()[self.widgets.get_widget('cmb_backup_burner').get_active()][0]
		# check that we've got read permissions on everything we're backing up
		files_ok, filecount = self.check_backup_source(set)
		if not files_ok:
			return True
		# ok, do a backup to /tmp/pybackpack
		self.widgets.get_widget('lbl_status').set_text(_("Creating backup..."))
		self.widgets.get_widget('progressbar1').set_fraction(0.25)
		
		self.statuswin.addmsg(_("Creating temporary backup in /tmp"))
		gobject.idle_add(self.statuswin.scroll)
		dest_path = "/tmp/%s.cdimage/" % version.APPPATH
		try:
			shutil.rmtree(dest_path)
		except: # it doesn't exist, so don't worry
			pass
		try:
			os.makedirs(dest_path)
		except:
			ErrorDialog(_("An error occurred while setting up the temporary CD image directory '%s'.\nPlease ensure you have write access to this location.") % dest_path, self.widgets.get_widget('window_main'))
			self.widgets.get_widget('lbl_status').set_text(_("Ready."))
			self.widgets.get_widget('progressbar1').set_fraction(0)
			self.statuswin.addmsg(_("Backup failed; could not create CD image directory %s\n") % dest_path)
			return True
		stdout = LogHandler(self.statuswin, self.widgets, filecount, True)
		stderr = LogHandler(self.statuswin)
		try:
			rdiff_interface.BackupSet(set, dest_path, stdout, stderr)
		except:
			sys.stdout = sys.__stdout__
			sys.stderr = sys.__stderr__
			if len(stderr) > 0:
				self.widgets.get_widget('lbl_status').set_text(_("Ready."))
				self.widgets.get_widget('progressbar1').set_fraction(0)
				self.statuswin.addmsg(_("Backup failed; could not create temporary backup in %s\n") % dest_path)
				ErrorDialog(_("Backup failed: could not create temporary backup in %s") %
													dest_path, self.widgets.get_widget('window_main'))
				return True
				
		# we should have the completed backup in /tmp now, let's make an ISO
		self.widgets.get_widget('lbl_status').set_text(_("Creating CD image..."))
		self.widgets.get_widget('progressbar1').set_fraction(0.0)
		self.statuswin.addmsg(_("Creating CD image."))
		isomaker = Mkisofs()
		isomaker.set_progress_hook(self.show_progress)
		isomaker.create_iso(dest_path, "/tmp/%s.iso" % version.APPPATH)
		if isomaker.get_retval() != 0:
			self.widgets.get_widget('lbl_status').set_text(_("Ready."))
			self.widgets.get_widget('progressbar1').set_fraction(0)
			self.statuswin.addmsg(_("Backup failed; could not create CD image /tmp/%s.iso\n") % version.APPPATH)
			return True	

		# ISO is done, let's write that bad boy...
		self.widgets.get_widget('lbl_status').set_text(_("Preparing to burn CD..."))
		self.widgets.get_widget('progressbar1').set_fraction(1.0)
		InfoDialog(_("Please insert a blank CD into %s") % device_name, self.widgets.get_widget('window_main'))
		while Dialogs.last_response == None:
			gtk.main_iteration() # kill some time until the user clicks the button
		if Dialogs.last_response == gtk.RESPONSE_CANCEL:
			self.widgets.get_widget('lbl_status').set_text(_("Ready."))
			self.widgets.get_widget('progressbar1').set_fraction(0)
			self.statuswin.addmsg(_("Backup cancelled by user.\n"))
			return True #user clicked cancel
		self.widgets.get_widget('lbl_status').set_text(_("Burning CD..."))
		self.widgets.get_widget('progressbar1').set_fraction(0.75)
		self.statuswin.addmsg(_("Starting to burn CD."))
		cdburner = nautilusburn.Recorder()
		track = nautilusburn.DataTrack()
		track.filename = "/tmp/%s.iso" % version.APPPATH
		cdburner.connect('progress-changed', self.on_cdburn_progress_changed)
		self.widgets.get_widget('window_progress').set_transient_for(self.widgets.get_widget('window_main'))
		self.widgets.get_widget('window_progress').show()
		burn_result = cdburner.write_tracks(drive, [track], drive.get_max_speed_write(), nautilusburn.RECORDER_WRITE_EJECT)
		if burn_result != nautilusburn.RECORDER_RESULT_FINISHED:
			self.on_window_progress_response(gtk.RESPONSE_OK)
			ErrorDialog(_("An error occurred while burning the CD."), self.widgets.get_widget('window_main'))
			self.widgets.get_widget('lbl_status').set_text("Ready.")
			self.widgets.get_widget('progressbar1').set_fraction(0)
			self.statuswin.addmsg(_("Backup failed; CD burn failed.\n"))
			return True
		else:
			self.widgets.get_widget('lbl_status').set_text(_("Cleaning up temporary files..."))
			self.widgets.get_widget('progressbar1').set_fraction(0.9)
			self.statuswin.addmsg(_("Cleaning up temporary files."))
			try:
				shutil.rmtree(dest_path)
				for f in [	"/tmp/%s.iso" % (device, version.APPPATH), \
							"/tmp/%s.mkisofs.stdout" % version.APPPATH, \
							"/tmp/%s.mkisofs.stderr" % version.APPPATH]:
					os.unlink(f)
			except:
				pass
			self.widgets.get_widget('lbl_status').set_text(_("Backup finished."))
			self.widgets.get_widget('progressbar1').set_fraction(0)
			self.statuswin.addmsg(_("Backup completed successfully.\n"))
			MsgDialog(_("Backup completed successfully."), self.widgets.get_widget('window_main'))
			self.widgets.get_widget('lbl_status').set_text(_("Ready."))
			return False
		
			
	def do_local_backup(self):
		destination_path = self.widgets.get_widget('set_destination').child.get_text()
		set_id = self.widgets.get_widget('treeview_backup_sets').get_cursor()[0][0]
		set = rdiff_interface.backupsets[set_id]
		if len(destination_path) == 0:
			ErrorDialog(_("Please select a location to back up to."), self.widgets.get_widget('window_main'))
			return True
		self.statuswin.addmsg(_("Starting backup of '%(backup)s' to '%(destination)s'\n") %
								{'backup':set['name'], 'destination':destination_path})
		self.widgets.get_widget('lbl_status').set_text(_("Starting backup..."))
		if self.widgets.get_widget('backup_removable').get_active():
			InfoDialog(_("Please connect and mount the device for '%s'") % destination_path, self.widgets.get_widget('window_main'))
			while Dialogs.last_response == None:
				gtk.main_iteration() # kill some time until the user clicks the button
			if Dialogs.last_response == gtk.RESPONSE_CANCEL:
				self.widgets.get_widget('lbl_status').set_text(_("Ready."))
				self.widgets.get_widget('progressbar1').set_fraction(0)
				self.statuswin.addmsg(_("Backup cancelled by user.\n"))
				return True #user clicked cancel
		if not os.path.exists(destination_path):
			try:
				os.makedirs(destination_path)
			except:
				ErrorDialog(_("An error occurred while trying to create '%s'.\nPlease select a different backup destination.") % destination_path, self.widgets.get_widget('window_main'))
				self.statuswin.addmsg(_("Backup failed; couldn't write to '%s'.\n") % destination_path)
				self.statuswin.show()
				self.widgets.get_widget('lbl_status').set_text(_("Ready."))
				return True
		# is the destination OK?
		(status, reason) = rdiff_interface.CheckDestination(destination_path)
		if not status:
			if reason == "not_empty":
				dlg = gtk.MessageDialog( \
					self.widgets.get_widget('window_main'), \
					gtk.DIALOG_MODAL, gtk.MESSAGE_QUESTION, gtk.BUTTONS_YES_NO, \
					_("There are already files in the backup destination you have chosen, '%s'.\nIf you backup to this location, these files will be erased permanently.\n\nAre you sure you want to do this?") % destination_path)
				dlg.connect("response", self.on_overwrite_dialog_response, destination_path, self.on_button_do_backup_clicked)
				dlg.show()
				self.widgets.get_widget('lbl_status').set_text("Ready.")
				return True
			elif reason == "no_permission":
				ErrorDialog(_("You don't have permission to write to '%s'.\nPlease select a different backup destination.") % destination_path, self.widgets.get_widget('window_main'))
				self.statuswin.addmsg(_("Backup failed; no permission to write to '%s'.\n") % destination_path)
				self.widgets.get_widget('lbl_status').set_text(_("Ready."))
				return True
			elif reason == "is_home_dir":
				ErrorDialog(_("Cannot backup into your home directory; this would delete everything in it.\nPlease select a different location."), self.widgets.get_widget('window_main'))
				self.statuswin.addmsg(_("Backup failed (Bad destination).\n"))
				return True
		# check that we've got read permissions on everything we're backing up
		files_ok, filecount = self.check_backup_source(set)
		if not files_ok:
			return True
		self.widgets.get_widget('vbox2').set_sensitive(False)
		self.widgets.get_widget('menu_backup').set_sensitive(False)
		self.widgets.get_widget('menu_restore').set_sensitive(False)
		stdout = LogHandler(self.statuswin, self.widgets, filecount, True)
		stderr = LogHandler(self.statuswin)
		gobject.idle_add(self.statuswin.scroll)
		self.statuswin.addmsg(_("Running rdiff-backup"))
		try:
			rdiff_interface.BackupSet(set, destination_path, stdout, stderr)
			self.statuswin.addmsg(_("Backup completed.\n"))
			MsgDialog(_("Backup completed successfully."), self.widgets.get_widget('window_main'))
		except:
			sys.stdout = sys.__stdout__
			sys.stderr = sys.__stderr__
			if len(stderr) > 0:
				self.statuswin.addmsg(_("Backup to %s failed (rdiff-backup failed).\n") % set['name'])
				self.statuswin.show()
		self.widgets.get_widget('lbl_status').set_text(_("Ready."))
		gotit = False
		for path in self.widgets.get_widget('set_destination').get_model():
			if path[0] == self.widgets.get_widget('set_destination').child.get_text():
				gotit = True
				break
		if not gotit:
			self.widgets.get_widget('set_destination').get_model().append((self.widgets.get_widget('set_destination').child.get_text(),))
		self.widgets.get_widget('progressbar1').set_fraction(0)
		self.widgets.get_widget('vbox2').set_sensitive(True)
		self.widgets.get_widget('menu_backup').set_sensitive(True)
		self.widgets.get_widget('menu_restore').set_sensitive(True)

	def do_ssh_backup(self):
		user = self.widgets.get_widget('backup_ssh_user').get_text()
		host = self.widgets.get_widget('backup_ssh_host').get_text()
		path = self.widgets.get_widget('backup_ssh_path').get_text()
		destination_path = "%s@%s::%s" % (user, host, path)
		set_id = self.widgets.get_widget('treeview_backup_sets').get_cursor()[0][0]
		set = rdiff_interface.backupsets[set_id]
		
		error_string = ""
		if len(user) == 0:
			error_string += _("\nUser")
		if len(host) == 0:
			error_string += _("\nHost")
		if len(path) == 0:
			error_string += _("\nPath")
		if len(error_string) > 0:
			ErrorDialog(_("You did not enter enough information. Please check the \
following fields:%s") % error_string, self.widgets.get_widget('window_main'))
			return True

		self.statuswin.addmsg(_("Starting backup of '%(backup)s' to '%(hostname)s'") % 
								{'backup':set['name'], 'hostname':host})
		self.widgets.get_widget('lbl_status').set_text(_("Starting backup..."))

		# check that we've got read permissions on everything we're backing up
		files_ok, filecount = self.check_backup_source(set)
		if not files_ok:
			return True
		stdoutput = LogHandler(self.statuswin, self.widgets, filecount, True)
		stderror = LogHandler(self.statuswin)
		gobject.idle_add(self.statuswin.scroll)
		self.statuswin.addmsg(_("Running rdiff-backup."))
		rdiff_interface.Refresh()
		try:
			self.statuswin.addmsg(_("Backing up to %s") % destination_path)
			rdiff_interface.BackupSet(set, destination_path, stdoutput, stderror, False, True)
			inifile = os.path.join(rdiff_interface.setspath, set['path'], "set.ini")
			setfile = os.path.join(path, "rdiff-backup-data", "%s.set" % version.APPPATH)
			args = "scp %s %s@%s:%s" % (inifile, user, host, setfile)
			print args
			scp = subprocess.Popen(args, shell=True)
			while scp.poll() is None:
				gtk.main_iteration()
			if scp.poll() != 0:
				ErrorDialog(_("File backup to the remote host completed, but an error \
occurred whilst copying the %(appname)s backup set data file. All is not lost - you can manually copy \
this file from\n%(filesource)s\n to\n%(filepath)s (on host %(hostname)s)") % 
					{'appname':version.APPNAME, 'filesource':inifile, 'filepath':setfile, 'hostname':host},
					self.widgets.get_widget('window_main'))
				self.statuswin.addmsg(_("Copying data file failed, please copy %(filesource)s to %(filepath)s (on remote host %(hostname)s) manually.") % 
					{'filesource':inifile, 'filepath':setfile, 'hostname':host})
			self.statuswin.addmsg(_("Backup completed.\n"))
			MsgDialog(_("Backup completed successfully."), self.widgets.get_widget('window_main'))
		except:
			tb = traceback.format_exc()
			sys.stdout = sys.__stdout__
			sys.stderr = sys.__stderr__
			if len(stderror) > 0:
				self.statuswin.addmsg(_("An error occurred whilst backing up '%s'.") % set['name'])
				self.statuswin.show()
			else:
				ErrorDialog(_("An error occurred whilst backing up '%s'.") % set['name'], self.widgets.get_widget('window_main'))
			self.statuswin.addmsg(_("Backup failed (rdiff-backup failed)\n%s") % tb)
		self.widgets.get_widget('lbl_status').set_text(_("Ready."))
		self.widgets.get_widget('progressbar1').set_fraction(0)
	
	def check_backup_source(self, set):
		self.widgets.get_widget('lbl_status').set_text(_("Analysing backup source..."))
		self.widgets.get_widget('progressbar1').set_pulse_step(0.01)
		self.widgets.get_widget('progressbar1').pulse()
		self.statuswin.addmsg(_("Analysing backup source."))
		problems = []
		filecount = 0

		for path in set['filelist_inc']:
			if not os.path.exists(path):
				problems.append((path, _("Missing")))
			elif path == set['default_dest']:
				self.statuswin.addmsg(_("Destination directory in backup source. Omitting."))
			elif os.path.isdir(path):
				for dirname, dirs, files in os.walk(path):
					self.widgets.get_widget('progressbar1').pulse()
					for file in files:
						fullpath = os.path.join(dirname,file)
						if fullpath == set['default_dest']:
							self.statuswin.addmsg(_("NOTE: Destination directory in backup set. Omitting."))
						# We might be backing up a broken symlink, but who cares, we're not following them
						elif os.path.islink(fullpath):
							filecount += 1
						elif os.access(fullpath, os.R_OK):
							filecount += 1
						else:
							problems.append((fullpath, _("File")))
					for dir in dirs:
						fullpath = os.path.join(dirname,dir)
						if os.access(fullpath, os.R_OK|os.X_OK):
							filecount += 1
						else:
							problems.append((fullpath,_("Directory")))
					# Stop the GUI freezing up
					while (gtk.events_pending()):
						gtk.main_iteration()

			elif os.access(path, os.R_OK):
				filecount += 1
			else:
				problems.append((path, _("File")))

		if problems != []:
			ErrorDialog(_("Some files in the backup set are missing or not readable by your user.\n\
For details, see the status window."), self.widgets.get_widget('window_main'))
			self.widgets.get_widget('lbl_status').set_text("Ready.")
			self.widgets.get_widget('progressbar1').set_fraction(0)
			self.statuswin.addmsg(_("Permissions/file missing problems on certain files:"))
			for path, path_type in problems:
					self.statuswin.addmsg("%s: %s" % (path_type, path))
			self.statuswin.addmsg(_("Backup failed.\n"))
			return (False, 0)
		
		self.statuswin.addmsg(_("Check complete."))
		return (True, filecount)

	def on_window_progress_response(self, widget, response=None):
		self.widgets.get_widget('window_progress').hide()
		self.widgets.get_widget('prog_button').set_sensitive(False)
		self.widgets.get_widget('prog_status').set_markup("<i>"+_("Waiting for CD...")+"</i>")
		self.widgets.get_widget('prog_bar').set_fraction(0)
		self.widgets.get_widget('prog_bar').set_text("")
	
	def on_backup_local_dest_button_clicked(self, event):
		filechooser = FileChooser()
		filechooser.set_return_type(filechooser.RETURN_FILENAME)
		filechooser.set_exitnotify(self.filechosen_backup_dst)
		filechooser.set_title(_("Select a backup destination"))
		filechooser.show()

	def filechosen_backup_dst(self, filename):
		self.widgets.get_widget('set_destination').child.set_text(filename)

	def on_treeview_backup_sets_button_press_event(self, widget, event):
		if event.button == 3:
			self.widgets.get_widget('set_popup').popup(None, None, None, event.button, event.time)

	def on_cmb_backup_type_changed(self, widget):
		try:
			self.widgets.get_widget('notebook3').set_current_page(widget.get_active())
		except:
			pass
	

	def on_button_ssh_path_clicked(self, event):
		host = self.widgets.get_widget('backup_ssh_host').get_text()
		user = self.widgets.get_widget('backup_ssh_user').get_text()
		if len(host) == 0 or len(user) == 0:
			return True
		filechooser = FileChooser()
		filechooser.set_title(_("Select a default remote location to backup to"))
		filechooser.set_return_type(filechooser.RETURN_URI)
		filechooser.set_exitnotify(self.filechosen_new_set_ssh_path)
		filechooser.set_title(_("Select a default remote location to backup to."))
		filechooser.show()
		filechooser.set_current_folder_uri("sftp://%s@%s/" %(user, host))

	def filechosen_new_set_ssh_path(self, uri):
		host = self.widgets.get_widget('backup_ssh_host').get_text().replace(".", "\.")
		user = self.widgets.get_widget('backup_ssh_user').get_text()
		patt = re.compile("sftp:/[/]+%s@%s(.*)" % (user, host))
		match = patt.match(uri)
		if len(match.groups()) > 0:
			self.widgets.get_widget('backup_ssh_path').set_text(match.group(1))

	### RESTORE PAGE ###
	def on_restore_src_button_clicked(self, widget):
		filechooser = FileChooser()
		filechooser.set_return_type(filechooser.RETURN_URI)
		filechooser.set_exitnotify(self.filechosen_restore_src)
		filechooser.set_title(_("Select a location to restore from"))
		self.widgets.get_widget('notebook1').set_current_page(2)
		filechooser.show()

	def filechosen_restore_src(self, uri):
		self.widgets.get_widget('restore_src').child.set_text("")
		if uri[:7] == "sftp://":
			p = re.compile("sftp://([^@]+)@([^/]+)(.*)")
			matches = p.match(uri)
			if len(matches.groups()) < 3:
				user = ""
				host = ""
				path = ""
			else:
				user = matches.group(1)
				host = matches.group(2)
				path = matches.group(3)
			self.widgets.get_widget('radio_restore_ssh').set_active(True)
			self.widgets.get_widget('restore_src').child.set_text(path)
			self.widgets.get_widget('restore_ssh_user').set_text(user)
			self.widgets.get_widget('restore_ssh_host').set_text(host)
		elif uri[:7] == "file://":
			self.widgets.get_widget('radio_restore_local').set_active(True)
			self.widgets.get_widget('restore_src').child.set_text(uri[7:])
			self.widgets.get_widget('restore_ssh_user').set_text("username")
			self.widgets.get_widget('restore_ssh_host').set_text("host")
		else:
			ErrorDialog(_("Only sftp:// and file:// locations are supported."), self.widgets.get_widget('filechooserdialog1'))
	
	def on_restore_src_changed(self, widget):
		if self.widgets.get_widget('radio_restore_ssh').get_active():
			self.widgets.get_widget('lbl_restore_name').set_text(_("Click the 'refresh' button to read this backup set"))
			self.widgets.get_widget('lbl_restore_target').set_text('')
			self.widgets.get_widget('lbl_restore_desc').set_text('')
			self.widgets.get_widget('button_do_restore').set_sensitive(False)
			self.widgets.get_widget('cmb_restore_increment').get_model().clear()
			return True
		widget.child.set_text(widget.child.get_text().replace("/rdiff-backup-data", ""))
		set = rdiff_interface.ParseRestoreSrc(widget.child.get_text())
		if set is not None:
			self.widgets.get_widget('button_do_restore').set_sensitive(True)
			self.widgets.get_widget('lbl_restore_target').set_text(_("This data will be restored to %s") % os.path.join(os.environ['HOME'], "restored_files", set['name']))
			self.widgets.get_widget('lbl_restore_name').set_text(set['name'])
			self.widgets.get_widget('lbl_restore_desc').set_text(set['desc'])
			self.widgets.get_widget('cmb_restore_increment').set_model(set['increments'])
			self.widgets.get_widget('cmb_restore_increment').set_active(0)
			if set.has_key('readonly'):
				#self.widgets.get_widget('read_only_source').show()
				self.read_only_source = True
			else:
				#self.widgets.get_widget('read_only_source').hide()
				self.read_only_source = False
		else:
			self.widgets.get_widget('lbl_restore_target').set_text('')
			self.widgets.get_widget('lbl_restore_name').set_text('')
			self.widgets.get_widget('lbl_restore_desc').set_text('')
			self.widgets.get_widget('button_do_restore').set_sensitive(False)
			self.widgets.get_widget('cmb_restore_increment').set_model(gtk.ListStore(gobject.TYPE_STRING))
			#self.widgets.get_widget('read_only_source').hide()
			self.read_only_source = False
		
	def on_button_do_restore_clicked(self, widget=None):
		restore_failed = False
		ssh_restore = self.widgets.get_widget('radio_restore_ssh').get_active()
		if ssh_restore:
			user = self.widgets.get_widget('restore_ssh_user').get_text()
			host = self.widgets.get_widget('restore_ssh_host').get_text()
			path = self.widgets.get_widget('restore_src').child.get_text()
			restore_source = "%s@%s::%s" % (user, host, path)
		else:
			restore_source = self.widgets.get_widget('restore_src').child.get_text()
		self.statuswin.addmsg(_("Starting restore operation from '%s'\n") % restore_source)
		self.widgets.get_widget('lbl_status').set_text(_("Starting restore operation..."))
		self.widgets.get_widget('progressbar1').set_fraction(0)
		if self.widgets.get_widget('chk_rst_show_output_log').get_active():
			self.statuswin.show()
		rdiff_interface.Refresh()
		destination_path = os.path.join(os.environ['HOME'], "restored_files", self.widgets.get_widget('lbl_restore_name').get_text())
		if not os.path.exists(destination_path):
			try:
				os.makedirs(destination_path, 0755)
			except:
				ErrorDialog(_("An error occurred when trying to create '%s'.") % destination_path, self.widgets.get_widget('window_main'))
				self.statuswin.addmsg(_("Restore failed; could not create destination path '%s'\n") % destination_path)
				return True
		status, reason = rdiff_interface.CheckDestination(destination_path, False) # flag even if there's an rdiff-backup-data sub dir
		if not status:
			if reason == "no_permission":
				ErrorDialog(_("You don't have permission to write to '%s'.") + "\n" +
							_("Please change the permissions and try again.") % 
							destination_path, self.widgets.get_widget('window_main'))
			elif reason == "not_empty":
				dlg = gtk.MessageDialog( \
					self.widgets.get_widget('window_main'), \
					gtk.DIALOG_MODAL, gtk.MESSAGE_QUESTION, gtk.BUTTONS_YES_NO, \
					(_("There are already files in '%s'.") % destination_path) + "\n"
					+ _("If you restore your backup to this location, these files will be erased permanently.") + "\n\n"
					+ _("Are you sure you want to do this?"))
				dlg.connect("response", self.on_overwrite_dialog_response, destination_path, self.on_button_do_restore_clicked)
				dlg.show()
			return True
		# if we've reached this point, it's ok to do the restore
		self.widgets.get_widget('lbl_status').set_text(_("Checking restore source..."))
		self.widgets.get_widget('progressbar1').set_fraction(0.25)
		if self.read_only_source:
			self.total_count = RecursiveCount(restore_source)
			self.current_count = 0
			dest_path = "/tmp/%s.restore" % version.APPPATH
			self.widgets.get_widget('lbl_status').set_text(_("Copying from CD..."))
			self.widgets.get_widget('progressbar1').set_fraction(0.33)
			self.statuswin.addmsg(_("Copying from CD (%(count)d files) to %(destination)s.") %
										{'count':self.total_count, 'destination':dest_path})
			try:
				shutil.rmtree(dest_path)
			except:
				pass
			try:
				os.makedirs(dest_path)
				RecursiveCopy(restore_source, dest_path, self.on_recursive_copy_step)
			except:
				ErrorDialog(_("Couldn't copy CD from '%(cdpath)s' to '%(destination)s'.\nPlease check there is enough disk space and try again.") % 
					{'cdpath':restore_source, 'destination':dest_path}, self.widgets.get_widget('window_main'))
				self.statuswin.addmsg(_("Restore failed; could not copy CD to temporary location.\n"))
				return True
			self.statuswin.addmsg(_("CD copy finished."))
			source_path = dest_path
		self.widgets.get_widget('lbl_status').set_text(_("Restoring files..."))
		self.widgets.get_widget('progressbar1').set_fraction(0.5)
		self.widgets.get_widget('progressbar1').set_text("")
		self.widgets.get_widget('button_do_restore').set_sensitive(False)
		self.widgets.get_widget('menu_backup').set_sensitive(False)
		self.widgets.get_widget('menu_restore').set_sensitive(False)
		stdout = LogHandler(self.statuswin, None, 0, True)
		stderr = LogHandler(self.statuswin)
		increment = self.widgets.get_widget('cmb_restore_increment').get_model()[self.widgets.get_widget('cmb_restore_increment').get_active()][1]
		gobject.idle_add(self.statuswin.scroll)
		self.statuswin.addmsg(_("Restoring files."))
		while gtk.events_pending():
			gtk.main_iteration()
		try:
			rdiff_interface.RestoreSet(restore_source, destination_path, stdout, stderr, increment)
		except:
			sys.stdout = sys.__stdout__
			sys.stderr = sys.__stderr__
			if len(stderr) > 0:
				self.statuswin.addmsg(_("An error occurred whilst restoring from '%s'.") % restore_source)
				self.statuswin.show()
				restore_failed = True
		if self.read_only_source:
			self.widgets.get_widget('lbl_status').set_text(_("Cleaning up %s...") % dest_path)
			self.widgets.get_widget('progressbar1').set_fraction(0.75)
			self.statuswin.addmsg(_("Cleaning up temporary files from %s.") % dest_path)
			try:
				shutil.rmtree(dest_path)
			except:
				self.statuswin.addmsg(_("Cleaning up temporary files failed, please manually delete %s") % dest_path)
		self.widgets.get_widget('progressbar1').set_fraction(0)
		if not restore_failed:
			self.statuswin.addmsg(_("Restore succeeded.\n"))
			MsgDialog(_("Restore succeeded.\nThe restored files are in 'restored_files/%s' in your home directory.") % self.widgets.get_widget('lbl_restore_name').get_text(), self.widgets.get_widget('window_main'))
			if not ssh_restore:
				gotit = False
				for path in self.widgets.get_widget('restore_src').get_model():
					if path[0] == self.widgets.get_widget('restore_src').child.get_text():
						gotit = True
						break
				if not gotit:
					self.widgets.get_widget('restore_src').get_model().append((self.widgets.get_widget('restore_src').child.get_text(),))
		self.widgets.get_widget('button_do_restore').set_sensitive(True)
		self.widgets.get_widget('menu_backup').set_sensitive(True)
		self.widgets.get_widget('menu_restore').set_sensitive(True)
		self.widgets.get_widget('lbl_status').set_text(_("Ready."))
	
	def on_radio_restore_ssh_toggled(self, widget):
		if widget.get_active():
			self.widgets.get_widget('restore_ssh_user').set_sensitive(True)
			self.widgets.get_widget('restore_ssh_host').set_sensitive(True)
			self.widgets.get_widget('restore_ssh_refresh').set_sensitive(True)
		else:
			self.widgets.get_widget('restore_ssh_user').set_sensitive(False)
			self.widgets.get_widget('restore_ssh_host').set_sensitive(False)
			self.widgets.get_widget('restore_ssh_refresh').set_sensitive(False)
	
	def on_restore_ssh_refresh_clicked(self, widget):
		user = self.widgets.get_widget('restore_ssh_user').get_text()
		host = self.widgets.get_widget('restore_ssh_host').get_text()
		path = self.widgets.get_widget('restore_src').child.get_text()
		set = rdiff_interface.ParseRestoreSSHSrc(user, host, path)
		if set is not None:
			self.widgets.get_widget('button_do_restore').set_sensitive(True)
			self.widgets.get_widget('lbl_restore_target').set_text(_("This data will be restored to %s") % 
						os.path.join(os.environ['HOME'], "restored_files", set['name']))
			self.widgets.get_widget('lbl_restore_name').set_text(set['name'])
			self.widgets.get_widget('lbl_restore_desc').set_text(set['desc'])
			self.widgets.get_widget('cmb_restore_increment').set_model(set['increments'])
			self.widgets.get_widget('cmb_restore_increment').set_active(0)
			if set.has_key('readonly'):
				#self.widgets.get_widget('read_only_source').show()
				self.read_only_source = True
			else:
				#self.widgets.get_widget('read_only_source').hide()
				self.read_only_source = False
		else:
			ErrorDialog(_("An error occurred while reading the remote backup set information.\nPlease check the username, host name and path that you have entered and try again."), self.widgets.get_widget('window_main'))
			self.widgets.get_widget('lbl_restore_target').set_text('')
			self.widgets.get_widget('lbl_restore_name').set_text('')
			self.widgets.get_widget('lbl_restore_desc').set_text('')
			self.widgets.get_widget('button_do_restore').set_sensitive(False)
			self.widgets.get_widget('cmb_restore_increment').get_model().clear()
		
	##########################
	##     MENU HANDLERS    ##
	##########################
	def on_about1_activate(self, event):
		logo = None
		logofile = 'pybackpack_logo.png'
		for dir in [os.path.dirname(__file__), '/usr/share/pixmaps']:
			if os.path.exists(os.path.join(dir,logofile)):
				logo = gtk.gdk.pixbuf_new_from_file(os.path.join(dir, logofile))
				break 
		if logo is not None:
			dlg = gnome.ui.About(version.APPNAME, version.VERSION, version.COPYRIGHT, version.ABOUT, version.AUTHORS, None, None, logo)
		else:
			dlg = gnome.ui.About(version.APPNAME, version.VERSION, version.COPYRIGHT, version.ABOUT, version.AUTHORS)
		dlg.set_transient_for(self.widgets.get_widget('window_main'))
		dlg.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
		dlg.show()
	
	def on_view_output_log_activate(self, event):
		self.statuswin.show()

	##########################
	##     MISC HANDLERS    ##
	##########################
	def on_cdburn_progress_changed(self, recorder, fract, somethingelse):
		if fract > 0:
			self.widgets.get_widget('prog_status').set_markup("<i>"+_("Burning CD...")+"</i>")
			self.widgets.get_widget('prog_bar').set_fraction(fract)
			self.widgets.get_widget('prog_bar').set_text("%d%%" % (fract*100))
		else:
			self.widgets.get_widget('prog_status').set_markup("<i>"+_("Waiting for CD...")+"</i>")
			self.widgets.get_widget('prog_bar').set_fraction(0)
			self.widgets.get_widget('prog_bar').set_text("0%")

		if fract == 1:
			self.widgets.get_widget('prog_button').set_sensitive(True)
			self.widgets.get_widget('prog_status').set_markup("<i>"+_("Finished.")+"</i>")
		gtk.main_iteration()
	
	def on_recursive_copy_step(self, file):
		self.current_count += 1.0
		self.widgets.get_widget('progressbar1').set_fraction(self.current_count/self.total_count)
		self.widgets.get_widget('progressbar1').set_text("%d%%" % int((self.current_count/self.total_count)*100))
		self.widgets.get_widget('lbl_status').set_text(_("Copying from CD - current file:") + "%s" % os.path.basename(file))
		gtk.main_iteration()
	
	def on_overwrite_dialog_response(self, widget, response, path, func):
		if response == gtk.RESPONSE_YES:
			# user is totally sure they want to delete, so let's do it
			widget.destroy()
			shutil.rmtree(path)
			func() # go back to where we came from
		else:
			self.statuswin.addmsg(_("Restore aborted, no files were changed.\n"))
			widget.destroy()
		
	def find_cd_burners(self):

		"""Uses nautilusburn to detect CD burners and populate comboboxes"""

		try:
			nautilusburn
		except NameError:
			error_string = _("No CD burners available, because you do not have the python module nautilusburn.")
			self.widgets.get_widget('lbl_backup_burner').set_text(error_string)
			self.widgets.get_widget('cmb_backup_burner').set_sensitive(False)
		else:
			sel = nautilusburn.DriveSelection()
			self.widgets.get_widget('cmb_backup_burner').set_model(sel.get_model())
			if len(sel.get_model()) == 0:
				error_string = _("No CD burners were detected on your system.")
				self.widgets.get_widget('lbl_backup_burner').set_text(error_string)
				self.widgets.get_widget('cmb_backup_burner').set_sensitive(False)
			else:
				msg_string = _("Select which CD burner to use:")
				self.widgets.get_widget('lbl_backup_burner').set_text(msg_string)
				self.widgets.get_widget('cmb_backup_burner').set_active(0)
				self.widgets.get_widget('cmb_backup_burner').set_sensitive(True)
			
	def refresh_set_list(self, set_sel=None):
		rdiff_interface.FindSets()
		treeview = self.widgets.get_widget('treeview_backup_sets')
		setlist = treeview.get_model()
		setlist.clear()
		for set in rdiff_interface.backupsets:
			if set.has_key("default_dest"):
				if set['default_dest'][:7] == "sftp://":
					set_type = gtk.STOCK_NETWORK
				elif set['default_dest'][:7] == "cdrw://":
					set_type = gtk.STOCK_CDROM
				else:
					set_type = gtk.STOCK_DIRECTORY
			else:
				set_type = gtk.STOCK_DIRECTORY
			setlist.append([treeview.render_icon(set_type, gtk.ICON_SIZE_MENU, "TreeView"), 
							set['name']])
		if set_sel is not None:
			self.select_set(set_sel)
		
	def select_set(self, setname):
		"""Iterates through the set list and places the sets treeview cursor
				on the given set"""
		treeview = self.widgets.get_widget('treeview_backup_sets')
		setlist = treeview.get_model()
		for row in setlist:
			if row[1] == setname:
				treeview.set_cursor(setlist.get_string_from_iter(row.iter))
				break
		
	def gtk_main_quit(self, event):
		
		""" Write the backup and restore MRUs back to disk """
		
		for file, store in [("backup_mru", self.widgets.get_widget('set_destination').get_model()),\
			("restore_mru", self.widgets.get_widget('restore_src').get_model())]:
			outfile = open(os.path.join(os.environ['HOME'], ".%s" % version.\
					APPPATH, file), "w")
			for line in store:
				outfile.write("%s\n" % line[0])
			outfile.close()

		gtk.main_quit()

