/*
 * Copyright (C) 2003 Mihai RUSU (dizzy@rdsnet.ro)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "common/setup_before.h"
#include <stdio.h>
#ifdef HAVE_STDDEF_H
# include <stddef.h>
#else
# ifndef NULL
#  define NULL ((void *)0)
# endif
#endif
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include "compat/strchr.h"
#include "compat/strdup.h"
#include "compat/strcasecmp.h"
#include "compat/strncasecmp.h"
#include <ctype.h>
#ifdef HAVE_LIMITS_H
# include <limits.h>
#endif
#include "compat/char_bit.h"
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <errno.h>
#include "compat/strerror.h"
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#else
# ifdef WIN32
#  include <io.h>
# endif
#endif
#include "compat/pdir.h"
#include "common/eventlog.h"
#include "prefs.h"
#include "common/util.h"
#include "common/field_sizes.h"
#include "common/bnethash.h"
#include "common/introtate.h"
#define CLAN_INTERNAL_ACCESS
#define ACCOUNT_INTERNAL_ACCESS
#include "account.h"
#include "common/hashtable.h"
#include "storage.h"
#include "common/list.h"
#include "connection.h"
#include "watch.h"
#include "clan.h"
#undef ACCOUNT_INTERNAL_ACCESS
#undef CLAN_INTERNAL_ACCESS
#include "common/tag.h"
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#else
# ifdef HAVE_SYS_FILE_H
#  include <sys/file.h>
# endif
#endif
#include "tinycdb/cdb.h"
#include "common/setup_after.h"

/* cdb storage API functions */

static int cdbs_init(const char *);
static int cdbs_close(void);
static t_storage_info * cdb_create_account(char const *);
static t_storage_info * cdb_get_defacct(void);
static int cdb_free_info(t_storage_info *);
static int cdb_read_attrs(t_storage_info *, t_read_attr_func, void *);
static void * cdb_read_attr(t_storage_info *, const char *);
static int cdb_write_attrs(t_storage_info *, void *);
static int cdb_read_accounts(t_read_accounts_func, void *);
static void * cdb_read_account(t_read_account_func, const char *);
static int cdb_cmp_info(t_storage_info *, t_storage_info *);
static const char * cdb_escape_key(const char *);
static int cdb_load_clans(t_load_clans_func);
static int cdb_write_clan(void *);
static int cdb_remove_clan(int);
static int cdb_remove_clanmember(int);

/* storage struct populated with the functions above */

t_storage storage_cdb = {
    cdbs_init,
    cdbs_close,
    cdb_create_account,
    cdb_get_defacct,
    cdb_free_info,
    cdb_read_attrs,
    cdb_write_attrs,
    cdb_read_attr,
    cdb_read_accounts,
    cdb_read_account,
    cdb_cmp_info,
    cdb_escape_key,
    cdb_load_clans,
    cdb_write_clan,
    cdb_remove_clan,
    cdb_remove_clanmember
};

/* start of actual cdb file storage code */

static const char *cdbdir = NULL;
static const char *clansdir = NULL;
static const char *defacct = NULL;

//#define CDB_ON_DEMAND	1

static int cdbs_init(const char *path)
{
    char *tok, *copy, *tmp, *p;
    const char *dir = NULL;
    const char *clan = NULL;
    const char *def = NULL;

    if (path == NULL || path[0] == '\0') {
        eventlog(eventlog_level_error, __FUNCTION__, "got NULL or empty path");
        return -1;
    }

    if ((copy = strdup(path)) == NULL) {
        eventlog(eventlog_level_error, __FUNCTION__, "could not duplicate path");
        return -1;
    }

    tmp = copy;
    while((tok = strtok(tmp, ";")) != NULL) {
        tmp = NULL;
        if ((p = strchr(tok, '=')) == NULL) {
            eventlog(eventlog_level_error, __FUNCTION__, "invalid storage_path, no '=' present in token");
            free((void*)copy);
            return -1;
        }
        *p = '\0';
        if (strcasecmp(tok, "dir") == 0)
            dir = p + 1;
        else if (strcasecmp(tok, "clan") == 0)
	        clan = p + 1;
        else if (strcasecmp(tok, "default") == 0)
            def = p + 1;
        else eventlog(eventlog_level_warn, __FUNCTION__, "unknown token in storage_path : '%s'", tok);
    }

    if (def == NULL || clan ==NULL || dir == NULL) {
        eventlog(eventlog_level_error, __FUNCTION__, "invalid storage_path line for cdb file module (doesnt have a 'dir', a 'clan' and a 'default' token)");
        free((void*)copy);
        return -1;
    }

    if (cdbdir) cdbs_close();

    if ((cdbdir = strdup(dir)) == NULL) {
        eventlog(eventlog_level_error, __FUNCTION__, "not enough memory to store cdb accounts dir");
        free((void*)copy);
        return -1;
    }

    if ((clansdir = strdup(clan)) == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "not enough memory to store clans dir");
	free((void*)copy);
	return -1;
    }

    if ((defacct = strdup(def)) == NULL) {
        eventlog(eventlog_level_error, __FUNCTION__, "not enough memory to store default cdb account path");
        free((void*)cdbdir); cdbdir = NULL;
        free((void*)copy);
        return -1;
    }

    free((void*)copy);
    return 0;
}

static int cdbs_close(void)
{
    if (cdbdir) free((void*)cdbdir);
    cdbdir = NULL;

    if (clansdir) free((void*)clansdir);
    clansdir = NULL;

    if (defacct) free((void*)defacct);
    defacct = NULL;

    return 0;
}

static t_storage_info * cdb_create_account(const char * username)
{
    char *temp;

    if (cdbdir == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "file storage not initilized");
	return NULL;
    }

    if (prefs_get_savebyname())
    {
	char const * safename;

	if (!(safename = escape_fs_chars(username,strlen(username))))
	{
	    eventlog(eventlog_level_error, __FUNCTION__, "could not escape username");
	    return NULL;
	}
	if (!(temp = malloc(strlen(cdbdir)+1+strlen(safename)+1))) /* dir + / + name + NUL */
	{
	    eventlog(eventlog_level_error, __FUNCTION__, "could not allocate memory for temp");
	    free((void *)safename);
	    return NULL;
	}
	sprintf(temp,"%s/%s",cdbdir,safename);
	free((void *)safename); /* avoid warning */
    } else {
	if (!(temp = malloc(strlen(cdbdir)+1+8+1))) /* dir + / + uid + NUL */
	{
	    eventlog(eventlog_level_error, __FUNCTION__, "could not allocate memory for temp");
	    return NULL;
	}
	sprintf(temp,"%s/%06u",cdbdir,maxuserid+1); /* FIXME: hmm, maybe up the %06 to %08... */
    }

    return temp;
}

static int cdb_write_attrs(t_storage_info *info, void *attributes)
{
    FILE           *cdbfile;
    t_attribute * attr;
    char *        tempname;
    struct cdb_make cdbm;

    if (cdbdir == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "cdb storage not initilized");
	return -1;
    }

    if (info == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "got NULL info storage");
	return -1;
    }

    if (attributes == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "got NULL attributes");
	return -1;
    }

    if (!(tempname = malloc(strlen(cdbdir)+1+strlen(BNETD_ACCOUNT_TMP)+1))) {
	eventlog(eventlog_level_error, __FUNCTION__, "unable to allocate memory for tempname");
        return -1;
    }

    sprintf(tempname,"%s/%s", cdbdir, BNETD_ACCOUNT_TMP);

    if ((cdbfile = fopen(tempname, "w+b")) == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "unable to open file \"%s\" for writing ",tempname);
	free(tempname);
	return -1;
    }

    cdb_make_start(&cdbm, cdbfile);
    for (attr=(t_attribute *)attributes; attr; attr=attr->next) {
	if (attr->key && attr->val) {
	    if (strncmp("BNET\\CharacterDefault\\", attr->key, 20) == 0) {
		eventlog(eventlog_level_debug, __FUNCTION__, "skipping attribute key=\"%s\"",attr->key);
	    } else {
		eventlog(eventlog_level_debug, __FUNCTION__, "saving attribute key=\"%s\" val=\"%s\"",attr->key,attr->val);
		cdb_make_add(&cdbm, attr->key, strlen(attr->key), attr->val, strlen(attr->val));
	    }
	} else eventlog(eventlog_level_error, __FUNCTION__,"could not save attribute key=\"%s\"",attr->key);

    }

    cdb_make_finish(&cdbm);
    fclose(cdbfile);
   
#ifdef WIN32
   /* We are about to rename the temporary file
    * to replace the existing account.  In Windows,
    * we have to remove the previous file or the
    * rename function will fail saying the file
    * already exists.  This defeats the purpose of
    * the rename which was to make this an atomic
    * operation.  At least the race window is small.
    */
    if (access((const char *)info, 0) == 0) {
	if (remove((const char *)info)<0) {
	    eventlog(eventlog_level_error, __FUNCTION__, "could not delete account file \"%s\" (remove: %s)", (char *)info, strerror(errno));
            free(tempname);
	    return -1;
	}
    }
#endif
   
    if (rename(tempname,(const char *)info)<0) {
	eventlog(eventlog_level_error, __FUNCTION__, "could not rename account file to \"%s\" (rename: %s)", (char *)info, strerror(errno));
	free(tempname);
	return -1;
    }

    free(tempname);

    return 0;
}

#ifndef CDB_ON_DEMAND
/* code adapted from tinycdb-0.73/cdb.c */

static int fget(FILE * fd, unsigned char *b, cdbi_t len, cdbi_t *posp, cdbi_t limit)
{
    if (posp && limit - *posp < len) {
	eventlog(eventlog_level_error, __FUNCTION__, "invalid cdb database format");
	return -1;
    }

    if (fread(b, 1, len, fd) != len) {
	if (ferror(fd)) {
	    eventlog(eventlog_level_error, __FUNCTION__, "got error reading from db file");
	    return -1;
	}
	eventlog(eventlog_level_error, __FUNCTION__, "unable to read from cdb file, incomplete file");
	return -1;
    }

    if (posp) *posp += len;
    return 0;
}

static const char * fcpy(FILE *fd, cdbi_t len, cdbi_t *posp, cdbi_t limit, unsigned char * buf)
{
    static char *str;
    static unsigned strl;
    unsigned int res = 0, no = 0;

    if (strl < len + 1) {
	char *tmp;
	if ((tmp = malloc(len + 1)) == NULL) {
	    eventlog(eventlog_level_error, __FUNCTION__, "could not alloc memory to read from db");
	    return NULL;
	}

	if (str) free((void*)str);
	str = tmp;
	strl = len + 1;
    }

    while(len - res > 0) {
	if (len > 2048) no = 2048;
	else no = len;

	fget(fd, buf, no, posp, limit);
	memmove(str + res, buf, no);
	res += no;
    }

    if (res > strl - 1) {
	eventlog(eventlog_level_error, __FUNCTION__, "BUG, this should nto happen");
	return NULL;
    }

    str[res] = '\0';
    return str;
}

static int cdb_read_attrs(t_storage_info *info, t_read_attr_func cb, void *data)
{
    cdbi_t eod, klen, vlen;
    cdbi_t pos = 0;
    const char *key;
    const char *val;
    unsigned char buf[2048];
    FILE *f;

    if (cdbdir == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "cdb storage not initilized");
	return -1;
    }

    if (info == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "got NULL info storage");
	return -1;
    }

    if (cb == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "got NULL callback");
	return -1;
    }

    if ((f = fopen(info, "rb")) == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "got error opening file '%s'", (char *)info);
	return -1;
    }

    fget(f, buf, 2048, &pos, 2048);
    eod = cdb_unpack(buf);
    while(pos < eod) {
	fget(f, buf, 8, &pos, eod);
	klen = cdb_unpack(buf);
	vlen = cdb_unpack(buf + 4);
	if ((key = fcpy(f, klen, &pos, eod, buf)) == NULL) {
	    eventlog(eventlog_level_error, __FUNCTION__, "error reading attribute key");
	    fclose(f);
	    return -1;
	}

	if ((key = strdup(key)) == NULL) {
	    eventlog(eventlog_level_error, __FUNCTION__, "error duplicating attribute key");
	    fclose(f);
	    return -1;
	}

	if ((val = fcpy(f, vlen, &pos, eod, buf)) == NULL) {
	    eventlog(eventlog_level_error, __FUNCTION__, "error reading attribute val");
	    free((void *)key);
	    fclose(f);
	    return -1;
	}

//	eventlog(eventlog_level_trace, __FUNCTION__, "read atribute : '%s' -> '%s'", key, val);
	cb(key, val, data);
	free((void *)key);
    }

    fclose(f);
    return 0;
}
#else /* CDB_ON_DEMAND */
static int cdb_read_attrs(t_storage_info *info, t_read_attr_func cb, void *data)
{
    return 0;
}
#endif

static void * cdb_read_attr(t_storage_info *info, const char *key)
{
#ifdef CDB_ON_DEMAND
    FILE	*cdbfile;
    t_attribute	*attr;
    const char	*filename = (const char *)info;
    char	*val;
    unsigned	vlen = 1;

    if (!cdbdir) {
	eventlog(eventlog_level_error, __FUNCTION__, "cdb storage not initilized");
	return NULL;
    }

    if (key == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "got NULL key");
	return NULL;
    }

//    eventlog(eventlog_level_trace, __FUNCTION__, "reading key '%s'", key);
    if ((cdbfile = fopen(filename, "rb")) == NULL) {
//	eventlog(eventlog_level_debug, __FUNCTION__, "unable to open file \"%s\" for reading ",filename);
	return NULL;
    }

    if (cdb_seek(cdbfile, key, strlen(key), &vlen) <= 0) {
//	eventlog(eventlog_level_debug, __FUNCTION__, "could not find key '%s'", key);
	fclose(cdbfile);
	return NULL;
    }

    if ((attr = malloc(sizeof(t_attribute))) == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "not enough memory for attr result");
	fclose(cdbfile);
	return NULL;
    }

    if ((attr->key = strdup(key)) == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "not enough memory for attr key result");
	free((void*)attr);
	fclose(cdbfile);
	return NULL;
    }

    if ((val = malloc(vlen + 1)) == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "not enough memory for attr val result");
	free((void*)attr->key);
	free((void*)attr);
	fclose(cdbfile);
	return NULL;
    }

    cdb_bread(cdbfile, val, vlen);
	fclose(cdbfile);

    val[vlen] = '\0';

    attr->val = val;
    attr->dirty = 0;
//    eventlog(eventlog_level_trace, __FUNCTION__, "read key '%s' value '%s'", attr->key, attr->val);
    return attr;
#else
    return NULL;
#endif
}

static int cdb_free_info(t_storage_info *info)
{
    if (info) free((void*)info);
    return 0;
}

static t_storage_info * cdb_get_defacct(void)
{
    t_storage_info * info;

    if (defacct == NULL) {
        eventlog(eventlog_level_error, __FUNCTION__, "cdb storage not initilized");
	return NULL;
    }

    if ((info = strdup(defacct)) == NULL) {
        eventlog(eventlog_level_error, __FUNCTION__, "could not duplicate default account cdb filename");
	return NULL;
    }

    return info;
}

static int cdb_read_accounts(t_read_accounts_func cb, void *data)
{
    char const * dentry;
    char *       pathname;
    t_pdir *     accountdir;

    if (cdbdir == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "file storage not initilized");
	return -1;
    }

    if (cb == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "got NULL callback");
	return -1;
    }

    if (!(accountdir = p_opendir(cdbdir))) {
	eventlog(eventlog_level_error, __FUNCTION__, "unable to open user directory \"%s\" for reading (p_opendir: %s)",cdbdir,strerror(errno));
	return -1;
    }

    while ((dentry = p_readdir(accountdir))) {
	if (dentry[0]=='.') continue;

	if (!(pathname = malloc(strlen(cdbdir)+1+strlen(dentry)+1))) /* dir + / + file + NUL */
	 {
	   eventlog(eventlog_level_error, __FUNCTION__, "could not allocate memory for pathname");
	   continue;
	 }
	sprintf(pathname,"%s/%s", cdbdir, dentry);

	cb(pathname, data);
    }

    if (p_closedir(accountdir)<0)
	eventlog(eventlog_level_error, __FUNCTION__,"unable to close user directory \"%s\" (p_closedir: %s)", cdbdir,strerror(errno));

    return 0;
}

static void * cdb_read_account(t_read_account_func cb, const char * name)
{
    char *       pathname;

    if (cdbdir == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "file storage not initilized");
	return NULL;
    }

    if (cb == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "got NULL callback");
	return NULL;
    }

    if (!(pathname = malloc(strlen(cdbdir)+1+strlen(name)+1))) /* dir + / + file + NUL */
    {
	eventlog(eventlog_level_error, __FUNCTION__, "could not allocate memory for pathname");
	return NULL;
    }
    sprintf(pathname,"%s/%s", cdbdir, name);
    if (access(pathname,0)) /* if it doesn't exist */
    {
	free((void *) pathname);
	return NULL;
    }

    return cb(pathname);
}


static int cdb_cmp_info(t_storage_info *info1, t_storage_info *info2)
{
    return strcmp((const char *)info1, (const char *)info2);
}

static const char * cdb_escape_key(const char * key)
{
    return key;
}

static int cdb_load_clans(t_load_clans_func cb)
{
  char const	*dentry;
  t_pdir	*clandir;
  char		*pathname;
  int       clantag;
  t_clan	*clan;
  FILE		*fp;
  char		clanname[CLAN_NAME_MAX+1];
  char		motd[51];
  int		cid, creation_time;
  int		member_uid, member_join_time;
  char		member_status;
  t_clanmember	*member;
  
    if (cb == NULL) {
	eventlog(eventlog_level_error, __FUNCTION__, "get NULL callback");
	return -1;
    }

  if (!(clandir = p_opendir(clansdir)))
    {
      eventlog(eventlog_level_error,__FUNCTION__,"unable to open clan directory \"%s\" for reading (p_opendir: %s)",clansdir,strerror(errno));
      return -1;
    }
  eventlog(eventlog_level_trace,__FUNCTION__,"start reading clans");

  if (!(pathname = malloc(strlen(clansdir)+1+4+1)))
	{
	  eventlog(eventlog_level_error,__FUNCTION__,"could not allocate memory for pathname");
	  return -1;
	}

    while ((dentry = p_readdir(clandir)) != NULL)
    {
      if (dentry[0]=='.') continue;
    
      sprintf(pathname,"%s/%s",clansdir,dentry);

      if (strlen(dentry)>4)
	{
	  eventlog(eventlog_level_error,__FUNCTION__,"found too long clan filename in clandir \"%s\"",dentry);
	  continue;
	}
    
      clantag = str_to_clantag(dentry);

  if ((fp = fopen(pathname,"r"))==NULL)
    {
      eventlog(eventlog_level_error,__FUNCTION__,"can't open clanfile \"%s\"",pathname);
      continue;
    }

  if(!(clan = malloc(sizeof(t_clan))))
  {
    eventlog(eventlog_level_error,__FUNCTION__,"could not allocate memory for clan");
    free((void *)pathname);
    p_closedir(clandir);
    return -1;
  }

  clan->clantag=clantag;

  fscanf(fp,"\"%[^\"]\",\"%[^\"]\",%i,%i\n",clanname,motd,&cid,&creation_time);
  clan->clanname = strdup(clanname);
  clan->clan_motd = strdup(motd);
  clan->clanid = cid;
  clan->creation_time=(time_t)creation_time;
  clan->created = 1;
  clan->modified = 0;
  clan->channel_type = prefs_get_clan_channel_default_private();
 
  eventlog(eventlog_level_trace,__FUNCTION__,"name: %s motd: %s clanid: %i time: %i",clanname,motd,cid,creation_time);

  clan->members = list_create();

  while (fscanf(fp,"%i,%c,%i\n",&member_uid,&member_status,&member_join_time)==3)
    {
      if (!(member = malloc(sizeof(t_clanmember))))
	{
	  eventlog(eventlog_level_error,__FUNCTION__,"cannot allocate memory for clan member");
	  clan_remove_all_members(clan);
      if(clan->clanname)
	    free((void *)clan->clanname);
      if(clan->clan_motd)
        free((void *)clan->clan_motd);
	  free((void *)clan);
      fclose(fp);
      free((void *)pathname);
      p_closedir(clandir);
	  return -1;
	}
      if(!(member->memberacc   = accountlist_find_account_by_uid(member_uid)))
      {
	    eventlog(eventlog_level_error,__FUNCTION__,"cannot find uid %u", member_uid);
	    free((void *)member);
	    continue;
      }
      member->memberconn  = NULL;
      member->status    = member_status-'0';
      member->join_time = member_join_time;
      member->clan	= clan;

      if((member->status==CLAN_NEW)&&(time(NULL)-member->join_time>prefs_get_clan_newer_time()*3600))
      {
          member->status = CLAN_PEON;
          clan->modified = 1;
      }

      if (list_append_data(clan->members,member)<0)
	{
	  eventlog(eventlog_level_error,__FUNCTION__,"could not append item");
	  free((void *)member);
	  clan_remove_all_members(clan);
      if(clan->clanname)
	    free((void *)clan->clanname);
      if(clan->clan_motd)
	    free((void *)clan->clan_motd);
	  free((void *)clan);
      fclose(fp);
      free((void *)pathname);
      p_closedir(clandir);
	  return -1;
	}
      account_set_clanmember(member->memberacc, member);
      eventlog(eventlog_level_trace,__FUNCTION__,"added member: uid: %i status: %c join_time: %i",member_uid,member_status+'0',member_join_time);
    }

  fclose(fp);

      cb(clan);

    }

      free((void *)pathname);

  if (p_closedir(clandir)<0)
    {
      eventlog(eventlog_level_error,__FUNCTION__,"unable to close clan directory \"%s\" (p_closedir: %s)",clansdir,strerror(errno));
    }
  eventlog(eventlog_level_trace,__FUNCTION__,"finished reading clans");

  return 0;
}

static int cdb_write_clan(void * data)
{
  FILE			*fp;
  t_elem		*curr;
  t_clanmember	*member;
  char			*clanfile;
  t_clan        *clan=(t_clan *)data;
  if (!(clanfile = malloc(strlen(clansdir)+1+4+1)))
    {
      eventlog(eventlog_level_error,__FUNCTION__,"could not allocate memory for filename");
      return -1;
    }

  sprintf(clanfile,"%s/%c%c%c%c",clansdir,
	  clan->clantag>>24,(clan->clantag>>16)&0xff,
	  (clan->clantag>>8)&0xff,clan->clantag&0xff);

  if ((fp = fopen(clanfile,"w"))==NULL)
    {
      eventlog(eventlog_level_error,__FUNCTION__,"can't open clanfile \"%s\"",clanfile);
      free((void *)clanfile);
      return -1;
    }

  fprintf(fp,"\"%s\",\"%s\",%i,%i\n",clan->clanname,clan->clan_motd,clan->clanid,(int)clan->creation_time);

  LIST_TRAVERSE(clan->members,curr)
    {
      if (!(member = elem_get_data(curr)))
	{
	  eventlog(eventlog_level_error,__FUNCTION__,"got NULL elem in list");
	  continue;
	}
    if((member->status==CLAN_NEW)&&(time(NULL)-member->join_time>prefs_get_clan_newer_time()*3600))
      member->status = CLAN_PEON;
    fprintf(fp,"%i,%c,%u\n",account_get_uid(member->memberacc),member->status+'0',(unsigned)member->join_time);
    }

  fclose(fp);
  free((void *)clanfile);
  return 0;
}

static int cdb_remove_clan(int clantag)
{
    char * tempname;
    if (!(tempname = malloc(strlen(clansdir)+1+4+1)))
	{
	  eventlog(eventlog_level_error,__FUNCTION__,"could not allocate memory for pathname");
	  return -1;
	}
    sprintf(tempname, "%s/%c%c%c%c", clansdir, clantag>>24, (clantag>>16)&0xff, (clantag>>8)&0xff, clantag&0xff);
	if (remove((const char *)tempname)<0) {
	    eventlog(eventlog_level_error, __FUNCTION__, "could not delete clan file \"%s\" (remove: %s)", (char *)tempname, strerror(errno));
        free(tempname);
	    return -1;
	}
    free(tempname);
    return 0;
}

static int cdb_remove_clanmember(int uid)
{
    return 0;
}
