/* PureAdmin
 * Copyright (C) 2003 Isak Savo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Main: Initiates the GUI and spawns timeout-functions.
 *
 * Copyright (C) 2003 Isak Savo
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <errno.h>

#include "prefix.h"
#include "interface.h"
#include "support.h"
#include "gui_helper.h"
#include "globals.h"
#include "cfg.h"
#include "logfile.h"
#include "helper.h"
#include "famwrap.h"
#include "debugging.h"

void init_gui_activities (void);
void init_gui_online_users (void);
void init_gui_logwindow (void);
void get_srv_vars (void);

gboolean timeout_update_activity (gpointer data);
gboolean show_backup_messages (void);
gboolean show_startup_errors (void);
void activity_show_error_message (void);
guint timeout_id_activities;
guint timeout_logfile;;
gboolean prog_exiting = FALSE;
gboolean srv_vars_uninitialized = FALSE;
gboolean srv_comm_have_access = FALSE;
struct srv_externals_t srv_vars;

GtkWidget *main_window;
GtkWidget *dlg_usrman;

int main (int argc, char *argv[])
{
   gchar *msg;
   gchar *our_path;
   struct srv_externals_t uninitialized;
   memset (&uninitialized, 0, sizeof (struct srv_externals_t));
   

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif

  gtk_set_locale ();
  gtk_init (&argc, &argv);
  our_path = g_strdup_printf ("%s/%s/pixmaps", DATADIR, PACKAGE);
  add_pixmap_directory (our_path);
  g_free (our_path);

  init_dbg_console ();
  
  g_log_set_handler (NULL, G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL
                     | G_LOG_FLAG_RECURSION, (GLogFunc) pur_debug_handler, NULL);
  /* FIXME: Is there a glib function for this? */
  if (getuid () != 0)
    pur_log_wrn ("Warning, running without root access. This may not work!");
  pur_log_dbg ("PureAdmin Directories:");
  pur_log_dbg ("Prefix: %s", PREFIX);
  pur_log_dbg ("Datadir: %s", DATADIR);
  pur_log_dbg ("Locales: %s", LOCALEDIR);
  pur_log_dbg ("End PureAdmin Directories");
  get_srv_vars ();
  if (memcmp (&srv_vars, &uninitialized, sizeof (srv_vars)) == 0)
    srv_vars_uninitialized = TRUE;

  cfg_set_defaults ();
  cfg_read_settings ();
  
  main_window = create_main_window ();
  init_gui_activities ();
  init_gui_online_users ();
  init_gui_logwindow ();

#ifdef HAVE_LIBFAM
  pur_log_dbg ("HAVE_LIBFAM is defined");
  init_logfile ();
#else
  pur_log_dbg ("HAVE_LIBFAM is *not* defined!");
#endif

  gtk_widget_show (main_window);

  /* Update statusbar with status of PureFTPd and make the menu-widgets sensitive where appropiate */
  gui_update_server_status ();
  gui_update_menu_sensivity ();
  /* Automatic update of activities from server every third second */
  srv_comm_have_access = srv_try_get_activities ();
  if (srv_comm_have_access)
  {
     timeout_id_activities = g_timeout_add (3000, (GSourceFunc) timeout_update_activity, NULL);
     pur_log_dbg ("Access to server activites granted!");
  }
  else
  {
     activity_show_error_message ();
     pur_log_wrn ("Permission to read server activities not granted!");
  }
  pur_log_nfo ("PureAdmin started!");

  if (srv_vars_uninitialized)
  {
     pur_log_err ("Couldn't run: %s!", BR_BINDIR("/pureadminsearch"));
     msg = g_strdup_printf (_("PureAdmin was unable to fetch information about your system and "
			      "therefor may not work as expected.\n\n"
			      "The problem should be resolved if you reinstall PureAdmin."));
     gui_display_msgdialog (_("Failed to get system info"), msg, MSGDLG_TYPE_ERROR, main_window);
     g_free(msg);
  }
  gtk_main ();
  return 0;
}

void activity_show_error_message (void)
{
   GtkTreeIter iter;
   GtkWidget *tree_activities = NULL;
   GtkTreeModel *model = NULL;
   GtkTreeSelection *sel;
   //   GtkWidget *icon;
   GdkPixbuf *icon;
   tree_activities = lookup_widget (main_window, "tree_activity");
   model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree_activities));
   sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_activities));
   gtk_tree_selection_set_mode (sel, GTK_SELECTION_NONE);
   //icon = gtk_image_new_from_stock ("gtk-error", GTK_ICON_SIZE_MENU);
   icon = gtk_widget_render_icon (tree_activities, "gtk-dialog-error",GTK_ICON_SIZE_MENU, "");
   gtk_list_store_append (GTK_LIST_STORE (model), &iter);
   gtk_list_store_set (GTK_LIST_STORE (model), &iter,
		       COL_ACT_ICON, icon,
		       COL_ACT_TEXT, _("Unable to retreive information about server activities."),
		       COL_ACT_ID, 666,
		       -1);
   

}
void init_gui_activities (void)
{
   GtkListStore *model;
   GtkCellRenderer *renderer;
   GtkTreeViewColumn *column;
   GtkTreeView *tree_activity;

   pur_log_dbg ("Initializing gui_activities");
   /* Set-up treeview */
   tree_activity = GTK_TREE_VIEW (lookup_widget (main_window, "tree_activity"));
   model  = gtk_list_store_new (N_ACT_COLUMNS, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_INT);
   
   gtk_tree_view_set_model (GTK_TREE_VIEW (tree_activity), GTK_TREE_MODEL (model));
   //   g_object_unref (G_OBJECT (model));
   gtk_tree_selection_set_mode (gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_activity)),
				GTK_SELECTION_SINGLE);

   
   renderer = gtk_cell_renderer_pixbuf_new();
   column = gtk_tree_view_column_new_with_attributes (NULL, renderer,
						      "pixbuf", COL_ACT_ICON,
						      NULL);
   gtk_tree_view_append_column (GTK_TREE_VIEW(tree_activity), column);
   
   renderer = gtk_cell_renderer_text_new();
   column = gtk_tree_view_column_new_with_attributes (NULL, renderer,
						      "text", COL_ACT_TEXT,
						      NULL);
   gtk_tree_view_append_column (GTK_TREE_VIEW(tree_activity), column);
}

void init_gui_online_users (void)
{
   gint i;
   gchar *titles[] =  {
     "", "User", "Host", "Connections"
   }; 
   GtkListStore *model;
   GtkCellRenderer *renderer;
   GtkTreeViewColumn *column;
   GtkTreeView *tree_users;
   
   /* Set-up treeview */
   pur_log_dbg ("Initializing gui_online_users");

   tree_users = GTK_TREE_VIEW (lookup_widget (main_window, "tree_users"));
   model  = gtk_list_store_new (N_USR_COLUMNS, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT);
   
   gtk_tree_view_set_model (GTK_TREE_VIEW (tree_users), GTK_TREE_MODEL (model));
   g_object_unref (G_OBJECT (model));
   gtk_tree_selection_set_mode (gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_users)),
				GTK_SELECTION_SINGLE);

   
   renderer = gtk_cell_renderer_pixbuf_new();
   column = gtk_tree_view_column_new_with_attributes (titles[0], renderer,
						      "pixbuf", COL_USR_ICON,
						      NULL);
   gtk_tree_view_append_column (GTK_TREE_VIEW(tree_users), column);
   
   for (i = 1; i < N_USR_COLUMNS; i++)
   {
      renderer = gtk_cell_renderer_text_new();
      column = gtk_tree_view_column_new_with_attributes (titles[i], renderer,
							 "text", i,
							 NULL);
      gtk_tree_view_append_column (GTK_TREE_VIEW(tree_users), column);
   }
   
   /* Hook-up some signals */
   /*   selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (main_treeview));
   g_signal_connect ((gpointer) selection, "changed",
                    G_CALLBACK (on_list_selection_changed),
                    NULL);*/
}

void init_gui_logwindow (void)
{
   GtkWidget *tmp, *lbl;
   GtkTextBuffer *text;
   pur_log_dbg ("Initializing gui_logwindow");
   tmp = lookup_widget (main_window, "logview");
   
   text = gtk_text_view_get_buffer (GTK_TEXT_VIEW (tmp));

   logview_tags[LOG_INFO] = gtk_text_buffer_create_tag (text, "info",
							"foreground", "blue", NULL);
   logview_tags[LOG_WARNING] = gtk_text_buffer_create_tag (text, "warning",
							   "foreground", "#2e9344", NULL);
   logview_tags[LOG_ERROR] = gtk_text_buffer_create_tag (text, "error",
							 "foreground", "red", NULL);
   logview_tags[LOG_NOTICE] = gtk_text_buffer_create_tag (text, "notice",
							  "foreground", "#a427a9", NULL);
   logview_tags[LOG_DEBUG] =  gtk_text_buffer_create_tag (text, "debug",
							  "foreground", "#899f22", NULL);
   
   lbl = lookup_widget (main_window, "lbl_logfilename");
#ifdef HAVE_LIBFAM
   gtk_label_set_text (GTK_LABEL (lbl), cfg.logfile);
#else
   /* Disable logview controls and print a message explaining why! */
   log_display_error_text (0);
#endif
}

void get_srv_vars (void)
{
   FILE *p;
#define BUFSIZE 1200
   gchar buf[BUFSIZE], *ptr;
   gchar *search_prog;
   
   memset (&srv_vars, 0, sizeof (struct srv_externals_t));
   search_prog = g_strdup_printf ("%s/pureadminsearch", BINDIR);
   if (g_file_test (search_prog, G_FILE_TEST_IS_EXECUTABLE) == FALSE)
   {
      pur_log_wrn ("pureadminsearch not executable!");
      g_free (search_prog);
      return;
   }
   if ((p = popen (search_prog, "r")) == NULL)
   {
      pur_log_wrn ("popen pureadminsearch: %s", strerror(errno));
      return;
   }
   while (fgets (buf, BUFSIZE, p))
   {
      g_strchomp (buf);
      ptr = strchr (buf, ':') + 1;
      if (*ptr == '\0' || strncmp (ptr, "NOT FOUND", 9) == 0)
	continue;
      if (strncmp (buf, "pure-pw:", 8) == 0)
	strncpy (srv_vars.cmd_pure_pw, ptr, FILEPATH_MAX);
      else if (strncmp (buf, "pidfile:", 8) == 0)
	strncpy (srv_vars.fl_pidfile, ptr, FILEPATH_MAX);
      else if (strncmp (buf, "config:", 7) == 0)
	strncpy (srv_vars.fl_config, ptr, FILEPATH_MAX);
      else if (strncmp (buf, "ftpgroup:", 9) == 0)
	srv_vars.id_ftpgid = atoi (ptr);
      else if (strncmp (buf, "ftpuser:", 8) == 0)
	srv_vars.id_ftpuid = atoi (ptr);
      else if (strncmp (buf, "ftpuserhome:", 12) == 0)
	strncpy (srv_vars.dir_usrhome, ptr, FILEPATH_MAX);
      else if (strncmp (buf, "pwfile:", 7) == 0)
	strncpy (srv_vars.fl_pwfile, ptr, FILEPATH_MAX);
      else if (strncmp (buf, "pdbfile:", 8) == 0)
	strncpy (srv_vars.fl_pdbfile, ptr, FILEPATH_MAX);
      else if (strncmp (buf, "groupadd:", 9) == 0)
	strncpy (srv_vars.cmd_groupadd, ptr, FILEPATH_MAX);
      else if (strncmp (buf, "useradd:", 8) == 0)
	strncpy (srv_vars.cmd_useradd, ptr, FILEPATH_MAX);
      else if (strncmp (buf, "processid:", 10) == 0)
	srv_vars.pureftpd_pid = atoi (ptr);
      else if (strncmp (buf, "status:", 7) == 0)
	srv_vars.pureftpd_running = (strncmp (ptr, "running", 7) == 0) ? TRUE : FALSE;
      else if (strncmp (buf, "backups:", 8) == 0)
	strncpy (srv_vars.backup, ptr, FILEPATH_MAX);
      else if (strncmp (buf, "startupscript:", 14) == 0)
	strncpy (srv_vars.startupscript, ptr, FILEPATH_MAX);
      else if (strncmp (buf, "pure-ftpwho:", 12) == 0)
	strncpy (srv_vars.cmd_pure_ftpwho, ptr, FILEPATH_MAX);

   }
   pclose (p);
}

