require 'etc'
require 'facter'
require 'puppet/type/parsedtype'
require 'puppet/type/state'

module Puppet
    newtype(:sshkey, Puppet::Type::ParsedType) do
        newstate(:type) do
            desc "The encryption type used.  Probably ssh-dss or ssh-rsa."
        end

        newstate(:key) do
            desc "The key itself; generally a long string of hex digits."
        end

        # FIXME This should automagically check for aliases to the hosts, just
        # to see if we can automatically glean any aliases.
        newstate(:alias) do
            desc "Any alias the host might have.  Multiple values must be
                specified as an array.  Note that this state has the same name
                as one of the metaparams; using this state to set aliases will
                make those aliases available in your Puppet scripts."

            # We actually want to return the whole array here, not just the first
            # value.
            def should
                if defined? @should
                    return @should
                else
                    return nil
                end
            end

            validate do |value|
                if value =~ /\s/
                    raise Puppet::Error, "Aliases cannot include whitespace"
                end
                if value =~ /,/
                    raise Puppet::Error, "Aliases cannot include whitespace"
                end
            end

            # Make a puppet alias in addition.
            munge do |value|
                # Add the :alias metaparam in addition to the state
                @parent.newmetaparam(@parent.class.metaparamclass(:alias), value)
                value
            end
        end

        newparam(:name) do
            desc "The host name."

            isnamevar
        end

        @doc = "Installs and manages ssh host keys.  At this point, this type
            only knows how to install keys into /etc/ssh/ssh_known_hosts, and
            it cannot manage user authorized keys yet."

        @instances = []

        # FIXME This should be configurable.
        @path = "/etc/ssh/ssh_known_hosts"
        @fields = [:name, :type, :key]

        @filetype = Puppet::FileType.filetype(:flat)
#        case Facter["operatingsystem"].value
#        when "Solaris":
#            @filetype = Puppet::FileType::SunOS
#        else
#            @filetype = Puppet::CronType::Default
#        end

        # Parse a host file
        #
        # This method also stores existing comments, and it stores all host
        # jobs in order, mostly so that comments are retained in the order
        # they were written and in proximity to the same jobs.
        def self.parse(text)
            count = 0
            hash = {}
            text.chomp.split("\n").each { |line|
                case line
                when /^#/, /^\s*$/:
                    # add comments and blank lines to the list as they are
                    @instances << line 
                else
                    hash = {}
                    fields().zip(line.split(" ")).each { |param, value|
                        hash[param] = value
                    }

                    if hash[:name] =~ /,/
                        names = hash[:name].split(",")
                        hash[:name] = names.shift
                        hash[:alias] = names
                    end

                    if hash[:alias] == ""
                        hash.delete(:alias)
                    end

                    hash2obj(hash)

                    hash.clear
                    count += 1
                end
            }
        end

        # Convert the current object into a host-style string.
        def to_record
            name = self[:name]
            if @states.include?(:alias)
                name += "," + @states[:alias].value.join(",")
            end
            [name, @states[:type].value, @states[:key].value].join(" ")
        end
    end
end

# $Id: sshkey.rb 1279 2006-06-14 15:30:25Z luke $
