/*
   drvwmf.cpp : This file is part of pstoedit
	 Backend for Windows Meta File (WMF/EMF)

   Copyright (C) 1996,1997 Jens Weber, rz47b7_AT_PostAG.DE
   Copyright (C) 1998 Thorsten Behrens and Bjoern Petersen
   Copyright (C) 1998 - 2005 Wolfgang Glunz
   Copyright (C) 2000 Thorsten Behrens

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include "drvwmf.h"

#include I_string_h
#include I_fstream
#include I_stdio
#include I_stdlib

#include <math.h>

#ifndef M_PI
#define M_PI		3.14159265358979323846
#endif



//
// recognized options for WMF/EMF driver:
// ======================================
//
// m :  maps all fonts in the document to Arial (should be available on every Windows installation)
// n :  emulate narrow fonts by shrinking fonts horizontally (sometimes does not look that good, but 
//      it's the only chance, when requested font weight is not available. And this is quite common 
//      for off-the-shelf Windows installations)
// b :  DON'T draw two white border pixel (upper left and lower right corner). They are normally 
//      drawn to keep content always within bounding box (is sometimes clipped otherwise, i.e. Windows 
//      doesn't respect pen thickness or rotated text extents).
//      This could be done more smarter for EMF, have to figure out...
// l :  prune lineends from text output (lineends are mapped to '#' in isolatin1)
//
// NOTE: old option 'e' (to generate enhanced metafile) removed, use driver emf instead!
//       old option 'p' (to generate palette) removed, wasn't documented and therefore not used,
//       and seems pretty unnecessary nowadays... 
//
// NOTE: there's now experimental image support for WMF. Current deficiencies: as the WMF/EMF format
//       cannot account for transparencies, the whole bounding box of an image is white. this
//       might cause trouble if the image is rotated/sheared, because that enlarges the bounding
//       beyond the actual image. 
//


// windows structure for standard metafile
// placeable header (aka Aldus Metafile)
const long PLACEABLEKEY = 0x9AC6CDD7L;
const int PLACEABLESIZE = 22;
// see also http://premium.microsoft.com/msdn/library/techart/html/emfdcode.htm
// regarding alignment (Wo Gl)
#pragma pack(2)
struct PLACEABLEHEADER {
	DWORD32 key;
	WORD hmf;
	WORD left;
	WORD top;
	WORD right;
	WORD bottom;
	WORD inch;
	DWORD32 reserved;
	WORD checksum;
};

static const char description[] = "generated by WMF/EMF backend of pstoedit\0input\0\0";


inline float drvWMF::scale() const
{
	const float scalef =  20.0f;
	if (options->OpenOfficeMode)
		return 1.0f;
	else
		return scalef;
}


inline long drvWMF::transx(float x) const
{
	if (options->OpenOfficeMode)
		return l_transX(x);
	else
		return (long)(scale() * (x));
}

inline long drvWMF::transy(float y) const
{
	if (options->OpenOfficeMode)
		return l_transY(y);
	else
		return (long)(scale() * (currentDeviceHeight - y));
}

void drvWMF::initMetaDC(HDC hdc){


#ifdef OLDCODE
// temporary - testing
			// set bounding box
//          SetWindowOrgEx(hdc, minX, minY, NIL);
			SetWindowOrgEx(hdc, minX, maxY, NIL);

//			SetViewportOrgEx(hdc, 0, 0, NIL);

			SetWindowExtEx(hdc, maxX - minX, maxY - minY, NIL);
			SetViewportExtEx(hdc, maxX - minX, maxY - minY, NIL);
//          SetViewportExtEx(hdc,
//                           (long)((float)(maxX - minX) * (float)GetDeviceCaps(desktopDC,HORZRES) / 
//                                  (float)GetDeviceCaps(desktopDC,HORZSIZE) + .5),
//                           (long)((float)(maxY - minY) * (float)GetDeviceCaps(desktopDC,VERTRES) /
//                                  (float)GetDeviceCaps(desktopDC,VERTSIZE) + .5), NIL);

#else
		
			// set reasonable mapping mode (don't want distortion)
			//SetMapMode(hdc, MM_ISOTROPIC);
			SetMapMode(hdc, MM_ANISOTROPIC);


	if (!options->OpenOfficeMode) {
		if (1){
			const int result = SetMapMode(hdc, MM_ANISOTROPIC);
			assert(result != 0);
		}
		SIZE oldWinExt;
		{
			const BOOL result = SetWindowExtEx(hdc, 1440, 1440, &oldWinExt);
//      const BOOL result = SetWindowExtEx(pd->hdcEMF,144,144,&oldWinExt);
			assert(result != 0);
		}
		SIZE oldViewPortExt;
		{
			const int result = SetViewportExtEx(hdc,
												GetDeviceCaps(metaDC, LOGPIXELSX),
												GetDeviceCaps(metaDC, LOGPIXELSY),
												&oldViewPortExt);
			assert(result != 0);
		}
	}
#endif


}

drvWMF::derivedConstructor(drvWMF):
constructBase, enhanced(false)
{
	// do some consistency checking
	if (! ( (sizeof(PLACEABLEHEADER) == PLACEABLESIZE) || (sizeof(PLACEABLEHEADER) == (PLACEABLESIZE+2)) ) ) {
		errf <<
			"WARNING: structure size mismatch. Please contact author. Expecting :"
			<< PLACEABLESIZE << " found " << sizeof(PLACEABLEHEADER) << endl;
		errf << "sizeof(WORD)    = " << sizeof(WORD) << endl;	// MSVC:  2
		errf << "sizeof(DWORD32)   = " << sizeof(DWORD32) << endl;	// MSVC:  4
		errf << "sizeof(RECT)    = " << sizeof(RECT) << endl;	// MSVC: 16
		errf << "sizeof(HANDLE)  = " << sizeof(HANDLE) << endl;	// MSVC:  4
 		ctorOK = false;
 		return;
		// now the write below writes PLACEABLESIZE and no longer sizeof(PLACEABLEHEADER)
		// this should be independent then of the packing
		// the 2 bytes saved are those at the end of the struct
	}

	// desktopDC = GetDC(GetDesktopWindow());

	if (options->OpenOfficeMode) {
		desktopDC = GetDC(GetDesktopWindow());
	} else {
		desktopDC = NIL;
	}

	// which output format?
	if (strcmp(Pdriverdesc->symbolicname, "emf") == 0) {
		enhanced = true;
	}

	if (enhanced) {
#if 1
		const BBox & psBBox = getCurrentBBox();

		minX = transx(psBBox.ll.x_);
		minY = transy(psBBox.ur.y_);
		maxX = transx(psBBox.ur.x_);
		maxY = transy(psBBox.ll.y_);

//		const RECT bbox = { minX, minY, maxX, maxY };
		if (Verbose()) errf << "calculated Bounding Box: " << minX << " " << minY << " " << maxX << " " << maxY << endl;
		// cout << "PostScript Bounding Box: " << psBBox.ll.x_  << " " << psBBox.ll.y_ << " " << psBBox.ur.x_ << " " << psBBox.ur.y_ << endl;
#endif

		// 
		// enhanced-Metafile (memory based) for W95/98 or NT
		// if -nb is set, then narrowbox = false , -nb means no bounding box 
		if (options->winbb) {
			if (Verbose()) errf << " Windows will calculate BB " << endl;
			metaDC = CreateEnhMetaFile(desktopDC, NIL, NIL, NIL);
		} else {
	// under non Windows systems we cannot use PlayEnhMetafile
			if (Verbose()) errf << " not creating with bounding box " << endl;
			// metaDC = CreateEnhMetaFile(desktopDC, outFileName.value(), &bbox , description);
			metaDC = CreateEnhMetaFile(desktopDC, outFileName.value(), NIL, description);
		}

		if (!metaDC) {
			errf << "ERROR: could not create enhanced metafile" << endl;
			ctorOK = false;
			return;
		} 
		
		initMetaDC(metaDC);

	} else {
		// take temp file for standard WMF, to prepend placeable header
		tempName = full_qualified_tempnam("wmftemp");

		// standard metafile for Win16
		metaDC = CreateMetaFile(tempName.value());
		if (!metaDC) {
			errf << "ERROR: could not open temporary metafile: " << tempName << endl;
			ctorOK = false;
			return;
		}

		if ((outFile = fopen(outFileName.value(), "wb")) == 0L) {
			errf << "ERROR: cannot open final metafile " << outFileName << endl;
			ctorOK = false;
			return;
		}
	}

	y_offset = 0;
	x_offset = 0;

	// in maxY and maxX are the highest values of the drawing
	maxY = 0;
	maxX = 0;
	maxStatus = 0;				// 0, until first value set in maxY and maxX

	// in minX und minY are the lowest values of the drawing
	minX = 0;
	minY = 0;
	minStatus = 0;				// 0, until first value set in minX and minY

	// setup pen for drawing functions
	const POINT PenWidth = { 0, 0 };
	penData.lopnStyle = PS_SOLID;	// solid pen
	penData.lopnWidth = PenWidth;	// width of pen
	penData.lopnColor = RGB(0, 0, 0);	// color of pen: black
	coloredPen = 0L;

	// setup brush for drawing functions
	brushData.lbStyle = BS_SOLID;	// solid brush
	brushData.lbColor = RGB(0, 0, 0);	// color of brush (black)
	brushData.lbHatch = 0L;		// no pattern
	coloredBrush = 0L;

	// set default font
	if (options->mapToArial) {
		const char *const defaultfontname = "Arial";
		setCurrentFontName(defaultfontname, false /* is standard font */ );
	} else {
		const char *const defaultfontname = "System";
		setCurrentFontName(defaultfontname, false /* is standard font */ );
	}
	myFont = 0L;
	oldFont = 0L;

	// set reasonable text drawing mode
	SetBkMode(metaDC, TRANSPARENT);

	// set Postscript-like text alignment
	SetTextAlign(metaDC, TA_BASELINE);
}

static void writeErrorCause(const char * mess)  
{
#ifdef _WIN32
	DWORD ec = GetLastError(); 
	LPVOID lpMsgBuf; 
	FormatMessage( 
		FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM, 
		NULL, 
		ec, 
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
		(LPTSTR) &lpMsgBuf,	
		0,	
		NULL	
	);	
	cerr << "Error Code for "  << mess << " ec: " << ec << " " << (char*) lpMsgBuf << endl; 
	LocalFree( lpMsgBuf ); 
#endif
}


drvWMF::~drvWMF()
{
	const BBox & psBBox = getCurrentBBox();

	minX = transx(psBBox.ll.x_);
	minY = transy(psBBox.ur.y_);
	maxX = transx(psBBox.ur.x_);
	maxY = transy(psBBox.ll.y_);

	if (Verbose()) errf << "original PostScript Bounding Box   : " << psBBox.ll.x_  << " " << psBBox.ll.y_ << " " << psBBox.ur.x_ << " " << psBBox.ur.y_ << endl;
	if (Verbose()) errf << "transformed PostScript Bounding Box: " << minX << " " << minY << " " << maxX << " " << maxY << endl;


	// is the output within signed integer boundaries?
	if (minX < -20000 ||
		minX > 20000 ||
		minY < -20000 ||
		minY > 20000 || maxX < -20000 || maxX > 20000 || maxY < -20000 || maxY > 20000) {
		// issue warning, coordinate overflow possible !
		errf << "Possible coordinate overflow, reduce scale! " << endl;
		errf << "  Origin: " << minX << " , " << minY << endl;
		errf << "  Size: " << maxX - minX << " , " << maxY - minY << endl;
	}
	// set dummy pixel, if not EMF
//  if( !enhanced && drawBoundingBox )
	if (options->drawBoundingBox) {
		// standard metafile does not care much about user space.
		// so we have to draw two white dummy points at the pic's boundaries
		if (Verbose()) {
			// in verbose mode use a non with pixel
			SetPixel(metaDC, minX, minY, RGB(0, 120, 0));
			SetPixel(metaDC, maxX, maxY, RGB(0, 120, 120));
		} else {
			SetPixel(metaDC, minX, minY, RGB(255, 255, 255));
			SetPixel(metaDC, maxX, maxY, RGB(255, 255, 255));
		}
	}
	// free any allocated objects
	if (coloredPen) {
		SelectObject(metaDC, oldColoredPen);
		DeleteObject(coloredPen);
	}

	if (coloredBrush) {
		SelectObject(metaDC, oldColoredBrush);
		DeleteObject(coloredBrush);
	}

	if (myFont) {
		SelectObject(metaDC, oldFont);
		DeleteObject(myFont);
		myFont = 0L;
	}
	// close and destroy metafile
	if (enhanced) {
		// close memory-based metafile (we're done drawing)
		HENHMETAFILE hMeta = CloseEnhMetaFile(metaDC);

#if 0
		// query bounding informations
		ENHMETAHEADER enhHeader;
		GetEnhMetaFileHeader(hMeta, sizeof(ENHMETAHEADER), &enhHeader);

//
// under construction. should somehow be possible to abandon dummy pixel drawing for
// EMF, by setting bounding box and friends appropriately
// 
		errf << "bbox: " << minX << " " << minY << " " << maxX << " " << maxY << endl;

		const RECT dimension = { 0, 0,
			enhHeader.rclFrame.right - enhHeader.rclFrame.left,
			enhHeader.rclFrame.bottom - enhHeader.rclFrame.top
		};
		const RECT dimension = { 0, 0,
			(long) (((float) (maxX - minX) * (float) GetDeviceCaps(desktopDC,
																   HORZRES) /
					 (float) GetDeviceCaps(desktopDC,
										   HORZSIZE)) * 10.0 + .5),
			(long) (((float) (maxY - minY) * (float) GetDeviceCaps(desktopDC,
																   VERTRES) /
					 (float) GetDeviceCaps(desktopDC,
										   VERTSIZE)) * 10.0 + .5)
		};
#endif


		// create our final metafile (metaDC is reused)
		// need to have two metafiles, because we know the output dimension just now
		if (options->winbb) {
			// const RECT bbox = { minX, minY, maxX, maxY };
			if (Verbose()) cout << "creating final metafile" << endl;
			// cout << "creating with bounding box 2: " << minX << "," << minY<< "," << maxX<< "," << maxY << endl;
			// don't need a BB here - Windows will calculate it by itself (that is the whole purpose of the later replay)
			metaDC = CreateEnhMetaFile(desktopDC, outFileName.value(), NIL, description);
			initMetaDC(metaDC);
		}
		if (metaDC) {

			if (options->winbb) {
				const RECT bbox = { minX, minY, maxX, maxY };
//          const RECT  bbox = {minX, minY, maxX-minX, maxY-minY};
//          const RECT  bbox = {minX, maxY, maxX-minX, maxY-minY};
//          const RECT  bbox = {minX, maxY, maxX+2*minX, -minY-maxY};

			// replay recorded metafile (hMeta -> metaDC)
				if (Verbose()) errf << "Info: replaying hMeta to metaDC with bounding box : " << minX << "," << minY<< "," << maxX<< "," << maxY << endl;

				if (!PlayEnhMetaFile(metaDC, hMeta, &bbox)) {
					writeErrorCause("PlayEnhMetaFile");
					errf << "ERROR: cannot replay created metafile" << endl;
				} else {
					if (Verbose())	errf << "Info: replayed metafile" << endl;
				}
			}
			// close and commit metafile
			DeleteEnhMetaFile(CloseEnhMetaFile(metaDC));
		} else {
			errf << "ERROR: could not open metafile for replay: " << outFileName << endl;
		}

		DeleteEnhMetaFile(hMeta);
	} else {
		DeleteMetaFile(CloseMetaFile(metaDC));

		// add placeable header to standard metafile
		const int BUFSIZE = 1024;
		char buf[BUFSIZE];
		PLACEABLEHEADER pHd;
		FILE *inFile;
		WORD checksum;
		WORD *checksumField;

		if ((inFile = fopen(tempName.value(), "rb")) != 0L) {
			if (outFile != 0L) {
				// setup header
				pHd.key = PLACEABLEKEY;
				pHd.hmf = 0;
				pHd.left = (WORD) minX;
				pHd.top = (WORD) minY;
				pHd.right = (WORD) maxX;
				pHd.bottom = (WORD) maxY;
				pHd.inch = 12;
				pHd.reserved = 0;

				// calculate checksum
				checksum = 0;
				checksumField = (WORD *) & pHd;
				for (unsigned int i = 0; i < 10; i++)
					checksum ^= checksumField[i];

				pHd.checksum = checksum;

				// write header
				if (fwrite(&pHd, PLACEABLESIZE , 1, outFile) != 1) {
					errf << "Can't write final metafile" << endl;
				}
				// append metafile data
				do {
					size_t read = fread(buf, 1, BUFSIZE, inFile);

					if (read > 0) {
						if (fwrite(buf, 1, read, outFile) != read) {
							errf << "Can't write final metafile" << endl;
						}
					}
				}
				while (!feof(inFile));

				fclose(outFile);
			}

			fclose(inFile);
		}
		remove(tempName.value());
		// cout << "tmp name " << tempName.value() << endl;

	}

	// delete desktop DC (might need it above)
	ReleaseDC(GetDesktopWindow(), desktopDC);
}


// the attributes for pens and brushes are set here.
// new Pen/Brush is selected into the DC
void drvWMF::setDrawAttr()
{
	// specify in RGB values
	penData.lopnColor =
		RGB((BYTE) (edgeR() * 255 + .5), (BYTE) (edgeG() * 255 + .5), (BYTE) (edgeB() * 255 + .5));
	brushData.lbColor =
		RGB((BYTE) (fillR() * 255 + .5), (BYTE) (fillG() * 255 + .5), (BYTE) (fillB() * 255 + .5));

	switch (currentLineType()) {
	case dashed:
		penData.lopnStyle = PS_DASH;
		break;

	case dotted:
		penData.lopnStyle = PS_DOT;
		break;

	case dashdot:
		penData.lopnStyle = PS_DASHDOT;
		break;

	case dashdotdot:
		penData.lopnStyle = PS_DASHDOTDOT;
		break;

	case solid:
		penData.lopnStyle = PS_SOLID;
		break;
	}

	const POINT PenWidth = { (int) (currentLineWidth() + .5), 0 };	// rounded int
	penData.lopnWidth = PenWidth;

	if (coloredPen) {
		SelectObject(metaDC, oldColoredPen);
		DeleteObject(coloredPen);
		coloredPen = 0L;
	}

	coloredPen = CreatePenIndirect(&penData);
	if (!coloredPen) {
		errf << "ERROR: setDrawAttr: could not create pen !" << endl;
	} else {
		oldColoredPen = (HPEN) SelectObject(metaDC, coloredPen);
	}

	if (coloredBrush) {
		SelectObject(metaDC, oldColoredBrush);
		DeleteObject(coloredBrush);
		coloredBrush = 0L;
	}

	coloredBrush = CreateBrushIndirect(&brushData);
	if (!coloredBrush) {
		errf << "ERROR: setDrawAttr: could not create brush !" << endl;
	} else {
		oldColoredBrush = (HBRUSH) SelectObject(metaDC, coloredBrush);
	}
}


int drvWMF::fetchFont(const TextInfo & textinfo, short int textHeight, short int textAngle)
{
	theFontRec.lfHeight = -textHeight ;
	theFontRec.lfWidth = 0;		// optimal fit
	theFontRec.lfEscapement = textAngle;
	theFontRec.lfOrientation = textAngle;

	theFontRec.lfWeight = FW_DONTCARE;	// default: don't care

	if (strstr(textinfo.currentFontWeight.value(), "Regular"))
		theFontRec.lfWeight = FW_NORMAL;	// other values don't work  

	if (strstr(textinfo.currentFontWeight.value(), "Medium"))
		theFontRec.lfWeight = FW_NORMAL;	// other values don't work  

	if (strstr(textinfo.currentFontWeight.value(), "Normal"))
		theFontRec.lfWeight = FW_NORMAL;

	if (options->emulateNarrowFonts) {
		if (strstr(textinfo.currentFontWeight.value(), "Thin") ||
			strstr(textinfo.currentFontName.value(), "Thin") ||
			strstr(textinfo.currentFontFullName.value(), "Thin")) {
			theFontRec.lfWidth = textHeight / 3;	// narrow font emulation (trial and error value for Arial font)
		}

		if (strstr(textinfo.currentFontWeight.value(), "Extralight") ||
			strstr(textinfo.currentFontName.value(), "Extralight") ||
			strstr(textinfo.currentFontFullName.value(), "Extralight")) {
			theFontRec.lfWidth = textHeight / 4;	// narrow font emulation (trial and error value for Arial font)
		}

		if (strstr(textinfo.currentFontWeight.value(), "Ultralight") ||
			strstr(textinfo.currentFontName.value(), "Ultralight") ||
			strstr(textinfo.currentFontFullName.value(), "Ultralight")) {
			theFontRec.lfWidth = textHeight / 4;	// narrow font emulation (trial and error value for Arial font)
		}

		if (strstr(textinfo.currentFontWeight.value(), "Light") ||
			strstr(textinfo.currentFontName.value(), "Light") ||
			strstr(textinfo.currentFontFullName.value(), "Light") ||
			strstr(textinfo.currentFontWeight.value(), "Condensed") ||
			strstr(textinfo.currentFontName.value(), "Condensed") ||
			strstr(textinfo.currentFontFullName.value(), "Condensed")) {
			theFontRec.lfWidth = textHeight / 3;	// narrow font emulation (trial and error value for Arial font)
		}
	}

	if (strstr(textinfo.currentFontWeight.value(), "Semibold") ||
		strstr(textinfo.currentFontName.value(), "Semibold") ||
		strstr(textinfo.currentFontFullName.value(), "Semibold"))
		theFontRec.lfWeight = FW_BOLD;	// other values don't work

	if (strstr(textinfo.currentFontWeight.value(), "Demibold") ||
		strstr(textinfo.currentFontName.value(), "Demibold") ||
		strstr(textinfo.currentFontFullName.value(), "Demibold"))
		theFontRec.lfWeight = FW_BOLD;	// other values don't work

	if (strstr(textinfo.currentFontWeight.value(), "Bold") ||
		strstr(textinfo.currentFontName.value(), "Bold") ||
		strstr(textinfo.currentFontFullName.value(), "Bold"))
		theFontRec.lfWeight = FW_BOLD;

	if (strstr(textinfo.currentFontWeight.value(), "Extrabold") ||
		strstr(textinfo.currentFontName.value(), "Extrabold") ||
		strstr(textinfo.currentFontFullName.value(), "Extrabold"))
		theFontRec.lfWeight = FW_BOLD;	// other values don't work

	if (strstr(textinfo.currentFontWeight.value(), "Ultrabold") ||
		strstr(textinfo.currentFontName.value(), "Ultrabold") ||
		strstr(textinfo.currentFontFullName.value(), "Ultrabold"))
		theFontRec.lfWeight = FW_BOLD;	// other values don't work

	if (strstr(textinfo.currentFontWeight.value(), "Heavy") ||
		strstr(textinfo.currentFontName.value(), "Heavy") ||
		strstr(textinfo.currentFontFullName.value(), "Heavy"))
		theFontRec.lfWeight = FW_BOLD;	// other values don't work

	if (strstr(textinfo.currentFontWeight.value(), "Black") ||
		strstr(textinfo.currentFontName.value(), "Black") ||
		strstr(textinfo.currentFontFullName.value(), "Black"))
		theFontRec.lfWeight = FW_BOLD;	// other values don't work

	if ((strstr(textinfo.currentFontName.value(), "Italic") != NIL) ||
		(strstr(textinfo.currentFontName.value(), "Oblique") != NIL) ||
		(strstr(textinfo.currentFontFullName.value(), "Italic") != NIL) ||
		(strstr(textinfo.currentFontFullName.value(), "Oblique") != NIL)) {
		theFontRec.lfItalic = TRUE;
	} else {
		theFontRec.lfItalic = 0;
	}

	theFontRec.lfUnderline = 0;
	theFontRec.lfStrikeOut = 0;
	theFontRec.lfOutPrecision = OUT_DEFAULT_PRECIS;
	theFontRec.lfClipPrecision = CLIP_DEFAULT_PRECIS;
	theFontRec.lfQuality = PROOF_QUALITY;
	theFontRec.lfPitchAndFamily = VARIABLE_PITCH | FF_DONTCARE;	// let every font be possible

	if ((strstr(textinfo.currentFontFullName.value(), "Symbol") != NIL) ||
		(strstr(textinfo.currentFontFullName.value(), "symbol") != NIL)) {
		theFontRec.lfCharSet = SYMBOL_CHARSET;
		strcpy(theFontRec.lfFaceName, "symbol");
	} else if ((strstr(textinfo.currentFontFamilyName.value(), "Computer Modern") != NIL) ) {
		// special handling for TeX Fonts - fix supplied by James F. O'Brien (job at cs.berkeley.edu)
        theFontRec.lfWeight = FW_NORMAL;
  	    theFontRec.lfItalic = 0;
  	    theFontRec.lfUnderline = 0;
	    theFontRec.lfCharSet = ANSI_CHARSET;
  	    strcpy(theFontRec.lfFaceName,  textinfo.currentFontName.value());
	} else 	{
		theFontRec.lfCharSet = ANSI_CHARSET;

		if (options->mapToArial)
			strcpy(theFontRec.lfFaceName, "Arial");
		else /* formerly we used currentFontFamilyName but FontName seems to be better */
			strcpy(theFontRec.lfFaceName, textinfo.currentFontName.value());
	}

	if (myFont) {
		SelectObject(metaDC, oldFont);
		DeleteObject(myFont);
		myFont = 0L;
	}

	myFont = CreateFontIndirect(&theFontRec);
	if (!myFont) {
		errf << "ERROR: fetchFont: could not create font !" << endl;
	} else {
		oldFont = (HFONT) SelectObject(metaDC, myFont);
	}

	return 0;
}


// fill point array and draw
void drvWMF::drawPoly(POINT * aptlPoints, int *aptlNumPts, polyType type)
{
	POINT lastStart = { 0, 0 };
	bool lastWasMoveTo = FALSE;
	bool lastWasClosePath = FALSE;
	unsigned int numOfPolies = 0;
	unsigned int numOfPts = 0;
	unsigned int p = 0;

	const unsigned int numOfElements = numberOfElementsInPath();
	for (unsigned int n = 0; n < numOfElements &&
		 p < 2 * numberOfElementsInPath() && numOfPolies < 2 * numOfElements; n++) {
		const basedrawingelement & elem = pathElement(n);

		if (elem.getType() != closepath) {
			aptlPoints[p].x = transx(elem.getPoint(0).x_);
			aptlPoints[p].y = transy(elem.getPoint(0).y_);
		} else {
			aptlPoints[p].x = transx(0);
			aptlPoints[p].y = transy(0);
		}

		if (elem.getType() == moveto || elem.getType() == lineto || elem.getType() == curveto) {
			const long lineWidth = (int) (currentLineWidth() + .5);	// rounded int

			// set drawing sizes
			if (minStatus) {
				if (aptlPoints[p].x - lineWidth < minX)
					minX = aptlPoints[p].x - lineWidth;

				if (aptlPoints[p].y - lineWidth < minY)
					minY = aptlPoints[p].y - lineWidth;
			} else {
				minX = aptlPoints[p].x - lineWidth;
				minY = aptlPoints[p].y - lineWidth;
				minStatus = 1;
			}

			if (maxStatus) {
				if (aptlPoints[p].x + lineWidth > maxX)
					maxX = aptlPoints[p].x + lineWidth;

				if (aptlPoints[p].y + lineWidth > maxY)
					maxY = aptlPoints[p].y + lineWidth;
			} else {
				maxX = aptlPoints[p].x + lineWidth;
				maxY = aptlPoints[p].y + lineWidth;
				maxStatus = 1;
			}
		}

		switch (elem.getType()) {
		case moveto:
			if (type == TYPE_LINES) {
				if (!MoveToEx(metaDC, aptlPoints[p].x, aptlPoints[p].y, NIL)) {
					errf << "ERROR: MoveTo: " << aptlPoints[p].x << "," << aptlPoints[p].y << endl;
				}
			} else {
				// ignore every path that is no area!
				if (numOfPts > 2) {
					// if path not closed -> do it manually!
					if (lastWasClosePath == FALSE) {
						if (p >= 2 * numberOfElementsInPath()) {
							errf << "ERROR: MoveTo: Out of array mem!" << endl;
						}
						// move this point one ahead
						aptlPoints[p + 1] = aptlPoints[p];

						// insert line to startpoint
						aptlPoints[p] = lastStart;

						p++;
						numOfPts++;
					}
					// store number of points for old subpath
					aptlNumPts[numOfPolies] = numOfPts;

					// one polygon more (for PolyPolygon)
					numOfPolies++;
				} else if (numOfPts == 2) {
					// we have a line here, so draw it!
					if (!MoveToEx(metaDC, aptlPoints[p - 2].x, aptlPoints[p - 2].y, NIL)) {
						errf << "ERROR: MoveTo: " << aptlPoints[p -
																2].
							x << "," << aptlPoints[p - 2].y << endl;
					}

					if (!LineTo(metaDC, aptlPoints[p - 1].x, aptlPoints[p - 1].y)) {
						errf << "ERROR: LineTo: " << aptlPoints[p -
																1].
							x << "," << aptlPoints[p - 1].y << endl;
					}
				}
				// all lower numbers do not represent a line/polygon. Ignore them

				// start this subpath newly
				numOfPts = 0;

				// set flag to remove lone moveto's
				lastWasMoveTo = TRUE;

				// clear flag to connect filled segments
				lastWasClosePath = FALSE;
			}

			// save last starting position in case we've to close path
			lastStart = aptlPoints[p];

			break;

		case curveto:
		case lineto:
			if (type == TYPE_LINES) {
				if (!LineTo(metaDC, aptlPoints[p].x, aptlPoints[p].y)) {
					errf << "ERROR: LineTo: " << aptlPoints[p].x << "," << aptlPoints[p].y << endl;
				}
			} else {
				// clear flag to remove lone moveto's
				lastWasMoveTo = FALSE;

				// clear flag to connect filled segments
				lastWasClosePath = FALSE;

				// do nothing, point is already in array
			}
			break;

		case closepath:
			if (type == TYPE_LINES) {
				// close the thingy
				if (!LineTo(metaDC, lastStart.x, lastStart.y)) {
					errf << "ERROR: LineTo: " << lastStart.x << "," << lastStart.y << endl;
				}
			} else {
				// clear flag to remove lone moveto's
				lastWasMoveTo = FALSE;

				// set flag to connect filled segments
				lastWasClosePath = TRUE;

				// insert line to startpoint
				// (overwrite current point - closepath did not inserted points)
				aptlPoints[p] = lastStart;
			}
			break;

		default:
			errf << "\t\tFatal: unexpected case in drvwmf (line " << __LINE__ << ")" << endl;
			abort();
			break;
		}

		// next free point
		p++;

		// next point in subpath
		numOfPts++;
	}

	if (type != TYPE_LINES) {
		// remove last moveTo, as it's lonely
		if (lastWasMoveTo != FALSE)
			numOfPts--;

		// ignore every path that is no area!
		if (numOfPts > 2) {
			// if path not closed -> do it manually!
			if (lastWasClosePath == FALSE) {
				if (p >= 2 * numberOfElementsInPath()) {
					errf << "ERROR: MoveTo: Out of array mem!" << endl;
				}
				// move this point one ahead
				aptlPoints[p + 1] = aptlPoints[p];

				// insert line to startpoint
				aptlPoints[p] = lastStart;

				p++;
				numOfPts++;
			}
			// store number of points for old subpath
			aptlNumPts[numOfPolies] = numOfPts;

			// one polygon more (for PolyPolygon), as either we closed it above,
			// or closepath did it and started no new
			numOfPolies++;
		} else if (numOfPts == 2) {
			// we have a line here, so draw it!
			if (!MoveToEx(metaDC, aptlPoints[p - 2].x, aptlPoints[p - 2].y, NIL)) {
				errf << "ERROR: MoveTo: " << aptlPoints[p -
														2].x << "," << aptlPoints[p - 2].y << endl;
			}

			if (!LineTo(metaDC, aptlPoints[p - 1].x, aptlPoints[p - 1].y)) {
				errf << "ERROR: LineTo: " << aptlPoints[p -
														1].x << "," << aptlPoints[p - 1].y << endl;
			}
		}
		// all lower numbers do not represent a line/polygon. Ignore them

		// anything to do?
		if (numOfPolies > 0) {
			if (!SetPolyFillMode(metaDC, (currentShowType() == fill) ? WINDING : ALTERNATE)) {
				errf << "ERROR: could not set fill mode" << endl;
			}

			if (!PolyPolygon(metaDC, aptlPoints, aptlNumPts, numOfPolies)) {
				DWORD ec = GetLastError();
				errf << "ERROR: Polygon could not be drawn: (" << ec << ")" << endl;
				errf << "Number of Points: " << p << endl;
				for (unsigned int i = 0; i < p; i++) {
					errf << aptlPoints[i].x << "," << aptlPoints[i].y << endl;
				}
			}
		}
	}
}



void drvWMF::close_page()
{
	// no function in drvwmf
}


void drvWMF::open_page()
{
	// no function in drvwmf
}


void drvWMF::show_path()
{
	POINT *aptlPoints = NIL;
	int *aptlNumPts = NIL;


	// get us twice the number of elements in path,
	// as maybe every subpath is closed and consists of ONE point!
	if ((aptlPoints = new POINT[2 * numberOfElementsInPath()]) == NIL) {
		errf << "ERROR: Cannot allocate memory for point-array" << endl;
		return;
	}
	// get us twice the number of elements in path,
	// as maybe every subpath is closed and consists of ONE point!
	if ((aptlNumPts = new int[2 * numberOfElementsInPath()]) == NIL) {
		errf << "ERROR: Cannot allocate memory for pointNum-array" << endl;
		return;
	}
	// update pen/brush
	setDrawAttr();

	// determine type: fill or line
	switch (currentShowType()) {
	case drvbase::stroke:
		// draw lines
		drawPoly(aptlPoints, aptlNumPts, TYPE_LINES);
		break;

	case drvbase::fill:
	case drvbase::eofill:
		// draw filled polygon
		drawPoly(aptlPoints, aptlNumPts, TYPE_FILL);
		break;

	default:
		// cannot happen
		errf << "unexpected ShowType " << (int) currentShowType();
		break;
	}

	delete[]aptlPoints;
	delete[]aptlNumPts;
}


void drvWMF::show_text(const TextInfo & textinfo)
{
	// set text color
	SetTextColor(metaDC, RGB((BYTE) (textinfo.currentR * 255 + .5),
							 (BYTE) (textinfo.currentG * 255 + .5),
							 (BYTE) (textinfo.currentB * 255 + .5)));

	const short int textHeight = (short int) (textinfo.currentFontSize*scale() + .5) ;	// rounded int
	const short int textAngle = (short int) (10 * textinfo.currentFontAngle + .5);	// Windows needs 10th of degree

	// any need to change font handle?
	if (fontchanged())
		fetchFont(textinfo, textHeight, textAngle);

	const long x1 = transx(textinfo.x);
	const long y1 = transy(textinfo.y);

	const long x2 = transx(textinfo.x_end);
	const long y2 = transy(textinfo.y_end);

	if (Verbose()) cout << "placing text : " << textinfo.thetext << " at " << textinfo.x << "," << textinfo.y<< " in EMF coords: " << x1 << "," << y1 << endl;
	// calculate bounding box
	//
	// NOTE: cannot do that exactly (with ascent and descent height),
	// because DC queries are not permitted for metafile DCs.
	//
	const int xOff = abs((int)
						 (sin(textinfo.currentFontAngle * M_PI / 180) * textHeight + .5));
	const int yOff = abs((int)
						 (cos(textinfo.currentFontAngle * M_PI / 180) * textHeight + .5));

	const int xMin = (int) min(x1 - xOff, x2 - xOff);
	const int xMax = (int) max(x1 + xOff, x2 + xOff);
	const int yMin = (int) min(y1 - yOff, y2 - yOff);
	const int yMax = (int) max(y1 + yOff, y2 + yOff);

	if (minStatus) {
		if (xMin < minX)
			minX = xMin;

		if (yMin < minY)
			minY = yMin;
	} else {
		minX = xMin;
		minY = yMin;
		minStatus = 1;
	}

	if (maxStatus) {
		if (xMax > maxX)
			maxX = xMax;

		if (yMax > maxY)
			maxY = yMax;
	} else {
		maxX = xMax;
		maxY = yMax;
		maxStatus = 1;
	}

	unsigned int textLen = strlen(textinfo.thetext.value());
	if (options->pruneLineEnds) {
		/* check for '#' at lineend */
		if (textLen > 0 && textinfo.thetext.value()[textLen - 1] == '#') {
			/* write one character less */
			textLen--;
		}
	}

#if defined(_WIN32)
	TextOut(metaDC, x1, y1, textinfo.thetext.value(), textLen);
	// TextOut(metaDC, 10, 10, "hello",4);
#else

	if (options->notforWindows) {
		// the user is aware of generating non-portable output
		TextOut(metaDC, x1, y1, textinfo.thetext.value(), textLen);
	} else {
		// we are not running Windows - so we need an emulation - see note below
		const long textdistance = (long) sqrt((double)((x1-x2)*(x1-x2) + (y1-y2)*(y1-y2)));
		const int letterspace = (textLen > 1) ? (textdistance / (textLen-1)) : 0 ;
		// if there is just one char in the text, then the inter letter spacing is 0 anyway
		int * pxDistance = new int[textLen];
		for (unsigned int letter = 0; letter < textLen; letter++) {
			pxDistance[letter] = letterspace;
		}
		const UINT fuOptions = 0;
		ExtTextOut (metaDC, x1, y1, fuOptions, 0, textinfo.thetext.value(), textLen, pxDistance);
		delete [] pxDistance;

		static bool warningwritten = false;
		if (textLen > 1 && !warningwritten) {
			warningwritten = true;
			errf << "Warning: Inter letter spacing is approximated by pstoedit because of problems in libemf. Use -pta option if results are not OK." << endl;
		}
	}
	
#endif

#if 0

ExtTextOut (hdc, xStart, yStart, iOptions, &rect, pString, iCount, pxDistance) ;
The fifth argument is a pointer to a rectangle structure. This is either a 
clipping rectangle, (if iOptions is set to ETO_CLIPPED, or a background 
rectangle to be filled with the current background color, 
if iOptions is set to ETO_OPAQUE. You can specify both options or neither. 

The last argument is an array of integers that specify the spacing between 
consecutive characters in the string. This allows a program to tighten or 
loosen intercharacter spacing, which is sometimes required for justifying 
a single word of text in a narrow column. 
The argument can be set to NULL for default character spacing. 

Note by Glunz: Even if the spacing argument can be set to NULL according to 
the description above, this creates a problem when libemf is used.
Libemf passes this NULL to the generated EMF file which then causes the
file to be no longer usable under newer versions of Windows.

It seems so that the Windows-API calculates this inter letter spacing automatically
before writing the record to the EMF file. But this needs access to font information
which is not easily done in libemf. 
Hence one work around is to assume a fixed width font and approximate the 
inter letter spacing by dividing the distance (P_end - P_start) by 
the number of characters in the string.

#endif




}


void drvWMF::show_rectangle(const float llx, const float lly, const float urx, const float ury)
{
	if (1) {
		show_path();
	} else {
// this code is disabled - see note below.
// needs to be fixed. 

	RECT localRect;

	// update pen/brush
	setDrawAttr();

	localRect.left = transx(llx);
	localRect.top = transy(lly);
	localRect.right = transx(urx);
	localRect.bottom = transy(ury);

	// calculate bounding box
	//
	const int xMin = (int) min(localRect.left, localRect.right);
	const int xMax = (int) max(localRect.left, localRect.right);
	const int yMin = (int) min(localRect.top, localRect.bottom);
	const int yMax = (int) max(localRect.top, localRect.bottom);

	if (minStatus) {
		if (xMin < minX)
			minX = xMin;

		if (yMin < minY)
			minY = yMin;
	} else {
		minX = xMin;
		minY = yMin;
		minStatus = 1;
	}

	if (maxStatus) {
		if (xMax > maxX)
			maxX = xMax;

		if (yMax > maxY)
			maxY = yMax;
	} else {
		maxX = xMax;
		maxY = yMax;
		maxStatus = 1;
	}

	if (0 && currentShowType() == drvbase::stroke) {
		// wogl - this code is disabled. I don't know why this was this way. 
		// one cannot use a RECT as Point * and a polyline needs 4 points to make a RECT.
		// strange ....
		Polyline(metaDC, (POINT *) & localRect, 2);
		// but also using a Rectangle isn't correct. 
	} else {
		Rectangle(metaDC, transx(llx), transy(lly), transx(urx), transy(ury));
	}
	}
}


void drvWMF::show_image(const PSImage & image)
{
	// first retrieve bounding box
	Point lowerLeft, upperRight;
	image.getBoundingBox(lowerLeft, upperRight);

	// not only bounding box must account for scale,
	// but also transformation matrix!

	// scale bounding box
	lowerLeft.x_ *= getScale();
	lowerLeft.y_ *= getScale();
	upperRight.x_ *= getScale();
	upperRight.y_ *= getScale();

	const long width  = abs(i_transX(upperRight.x_) - i_transX(lowerLeft.x_));
	const long height = abs(i_transY(upperRight.y_) - i_transY(lowerLeft.y_));

	if (Verbose()) {
		errf << "image.Width:" << image.width << " image.Height: " << image.height << endl;
		errf << "Width:" << width << " Height: " << height << endl;
	}
	// calculate bounding box
	//
	const int xMin = (int) min(transx(upperRight.x_), transx(lowerLeft.x_));
	const int xMax = (int) max(transx(upperRight.x_), transx(lowerLeft.x_));
	const int yMin = (int) min(transy(upperRight.y_), transy(lowerLeft.y_));
	const int yMax = (int) max(transy(upperRight.y_), transy(lowerLeft.y_));

	if (minStatus) {
		if (xMin < minX)
			minX = xMin;

		if (yMin < minY)
			minY = yMin;
	} else {
		minX = xMin;
		minY = yMin;
		minStatus = 1;
	}

	if (maxStatus) {
		if (xMax > maxX)
			maxX = xMax;

		if (yMax > maxY)
			maxY = yMax;
	} else {
		maxX = xMax;
		maxY = yMax;
		maxStatus = 1;
	}

	// calc long-padded size of scanline 
	const long scanlineLen = ((width * 3) + 3) & ~3L;

	// now lets get some mem
	unsigned char *const output = new unsigned char[scanlineLen * height];

	for (long i = 0; i < scanlineLen * height; i++)
		output[i] = 255;		// default is background (white)    

	if (!output) {
		errf << "ERROR: Cannot allocate memory for image" << endl;
		return;
	}
	// setup inverse transformation matrix (scaled, too!)
	const float matrixScale(image.normalizedImageCurrentMatrix[0] *
							image.normalizedImageCurrentMatrix[3] -
							image.normalizedImageCurrentMatrix[2] *
							image.normalizedImageCurrentMatrix[1]);
	const float inverseMatrix[] = {
		image.normalizedImageCurrentMatrix[3] / matrixScale / getScale(),
		-image.normalizedImageCurrentMatrix[1] / matrixScale / getScale(),
		-image.normalizedImageCurrentMatrix[2] / matrixScale / getScale(),
		image.normalizedImageCurrentMatrix[0] / matrixScale / getScale(),
		(image.normalizedImageCurrentMatrix[2] *
		 image.normalizedImageCurrentMatrix[5] -
		 image.normalizedImageCurrentMatrix[4] *
		 image.normalizedImageCurrentMatrix[3]) / matrixScale,
		(image.normalizedImageCurrentMatrix[4] *
		 image.normalizedImageCurrentMatrix[1] -
		 image.normalizedImageCurrentMatrix[0] *
		 image.normalizedImageCurrentMatrix[5]) / matrixScale
	};

	// now transform image
	for (long y = 0; y < height; y++) {
		// buffer current output scanline (saves us some multiplications)
		unsigned char *const currOutput = &output[scanlineLen * y];

		for (long x = 0; x < width; x++) {
			// now transform from device coordinate space to image space

			// apply transformation
			const Point currPoint = Point(x + lowerLeft.x_,
										  y + lowerLeft.y_).transform(inverseMatrix);

			// round to integers
			const long sourceX = (long) (currPoint.x_ + .5);
			const long sourceY = (long) (currPoint.y_ + .5);

			// is the pixel out of bounds? If yes, no further processing necessary
			if (sourceX >= 0L && (unsigned long) sourceX < image.width &&
				sourceY >= 0L && (unsigned long) sourceY < image.height) {
				// okay, fetch source pixel value into 
				// RGB triplet

				unsigned char r(255), g(255), b(255), c, m, y, k;

				// how many components?
				switch (image.ncomp) {
				case 1:
					r = g = b = image.getComponent(sourceX, sourceY, 0);
					break;

				case 3:
					r = image.getComponent(sourceX, sourceY, 0);
					g = image.getComponent(sourceX, sourceY, 1);
					b = image.getComponent(sourceX, sourceY, 2);
					break;

				case 4:
					c = image.getComponent(sourceX, sourceY, 0);
					m = image.getComponent(sourceX, sourceY, 1);
					y = image.getComponent(sourceX, sourceY, 2);
					k = image.getComponent(sourceX, sourceY, 3);

					// account for key
					c += k;
					m += k;
					y += k;

					// convert color
					r = 255 - c;
					g = 255 - m;
					b = 255 - y;
					break;

				default:
					errf << "\t\tFatal: unexpected case in drvwmf (line "
						<< __LINE__ << ")" << endl;
					abort();
					return;
				}

				// set color triple
				currOutput[3 * x] = b;
				currOutput[3 * x + 1] = g;
				currOutput[3 * x + 2] = r;
			}
		}
	}

	// draw Windows DI bitmap
	BITMAPINFO bmi;

			/* setup BITMAPINFO */
		bmi.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
		bmi.bmiHeader.biWidth = width;
		bmi.bmiHeader.biHeight = height;
		bmi.bmiHeader.biPlanes = 1;
		bmi.bmiHeader.biBitCount = 24;	// always truecolor output
		bmi.bmiHeader.biClrUsed = 0;
		bmi.bmiHeader.biCompression = BI_RGB;
		bmi.bmiHeader.biSizeImage = 0;	/* size not important */
		bmi.bmiHeader.biXPelsPerMeter = 0;	/* size not important */
		bmi.bmiHeader.biYPelsPerMeter = 0;	/* size not important */
		bmi.bmiHeader.biClrImportant = 0;

		if (!SetDIBitsToDevice(metaDC,
							  transx(lowerLeft.x_),
							  transy(upperRight.y_),
							  width, height, 0, 0, 0, height, output, &bmi, DIB_RGB_COLORS)) {
			errf << "ERROR: Cannot draw bitmap" << endl;
		}
	delete[]output;
}



#if defined(_WIN32)
//
// Wmf works only under Windows since the libemf has only a broken emulation of wmf.
// Under libemf, a createMetaFile effectively creates an enhMetaFile but that confuses almost all
// programs which expect to read an WMF with an aldus placeable header
// 
static DriverDescriptionT < drvWMF > D_wmf("wmf", "Windows metafile", "","wmf", true,	// backend supports subpathes
										   // if subpathes are supported, the backend must deal with
										   // sequences of the following form
										   // moveto (start of subpath)
										   // lineto (a line segment)
										   // lineto 
										   // moveto (start of a new subpath)
										   // lineto (a line segment)
										   // lineto 
										   //
										   // If this argument is set to false each subpath is drawn 
										   // individually which might not necessarily represent
										   // the original drawing.
										   false,	// backend does not support curves (at least for WMF - have to take least common denominator here)
										   true,	// backend supports elements which are filled and have edges 
										   true,	// backend supports text
										   DriverDescription::memoryeps,	// no support for PNG file images
										   DriverDescription::noopen,	// we open output file ourselves
										   false,	// if format supports multiple pages in one file (DEFINETELY not) 
										   false  /*clipping */ 
										   );
#endif

static DriverDescriptionT < drvWMF > D_emf("emf", "Enhanced Windows metafile", "","emf", true,	// backend supports subpathes
										   // if subpathes are supported, the backend must deal with
										   // sequences of the following form
										   // moveto (start of subpath)
										   // lineto (a line segment)
										   // lineto 
										   // moveto (start of a new subpath)
										   // lineto (a line segment)
										   // lineto 
										   //
										   // If this argument is set to false each subpath is drawn 
										   // individually which might not necessarily represent
										   // the original drawing.
										   false,	// backend does not support curves (not yet)
										   true,	// backend supports elements which are filled and have edges 
										   true,	// backend supports text
										   DriverDescription::memoryeps,	// no support for PNG file images
										   DriverDescription::noopen,	// we open output file ourselves
										   false,	// if format supports multiple pages in one file (DEFINETELY not) 
										   false  /*clipping */
										   );
 
