/*
 * main.cxx
 *
 * PWLib application source file for Voxilla
 *
 * A H.323 "net telephone" application.
 *
 * Copyright (c) 1993-1998 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Portable Windows Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of this code were written with the assistance of funding from
 * Vovida Networks, Inc. http://www.vovida.com.
 *
 * Portions are Copyright (C) 1993 Free Software Foundation, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *      Derek J Smithies, Indranet Technologies Ltd (derek@indranet.co.nz)
 *          ______________________________________.
 *
 * $Log: main.cxx,v $
 * Revision 1.50  2004/05/03 13:25:47  rjongbloed
 * Converted everything to be codec plug in freindly
 * Removed GSM and G.729 as now plug ins are "the way"!
 *
 * Revision 1.49  2004/01/31 07:28:28  rjongbloed
 * Changed #warning to #error to be compatible with MSVC
 *
 * Revision 1.48  2003/05/15 04:02:15  dereksmithies
 * Add linux specific code to detect incoming ringing signal - User sets the
 * cadence of the incoming ringing signal.
 *
 * Revision 1.47  2003/03/05 07:01:33  robertj
 * Major update changing to a service, thanks Pietro Ravasio
 *
 * Revision 1.46  2003/02/26 02:49:07  robertj
 * Fixed race condition when makeing calls.
 * Fixed caller hanging up issue, thanks Pietro Ravasio
 *
 * Revision 1.45  2003/01/06 06:23:39  robertj
 * Increased maximum possible jitter configuration to 10 seconds.
 * Fixed ability to set jitter size (condition wrong way round).
 * Added abort ability and removed 100% CPU problem diring DTMF digit
 *   gathering and awaiting connect.
 *
 * Revision 1.44  2002/12/11 10:48:36  rogerh
 * delete listener when exiting
 * Add NAT support with the --translate option (copied from ohphone)
 *
 * Revision 1.43  2002/11/10 08:12:42  robertj
 * Moved constants for "well known" ports to better place (OPAL change).
 *
 * Revision 1.42  2002/10/31 00:55:13  robertj
 * Enhanced jitter buffer system so operates dynamically between minimum and
 *   maximum values. Altered API to assure app writers note the change!
 * Removed some redundent code.
 *
 * Revision 1.41  2002/07/02 03:24:09  dereks
 * Add LOTS of PTrace statements..  Improve check on LID ringing.
 *
 * Revision 1.40  2002/06/17 23:26:13  dereks
 * Add fast start disable (-F) and H245 Tunnelling disable (-T) options.
 *
 * Revision 1.39  2002/03/20 06:09:39  robertj
 * Added options for setting volume, thanks Jason Spence
 * Bug fix for collecting digits time out, thanks David Rowe
 *
 * Revision 1.38  2002/01/27 14:37:10  rogerh
 * Add --listenport option
 *
 * Revision 1.37  2001/11/19 06:42:34  robertj
 * Added ability to collect digits before making H.323 connection, thanks David Rowe.
 *
 * Revision 1.36  2001/06/20 07:36:57  robertj
 * Added dial plan parameter, thanks Chih-Wei Huang
 *
 * Revision 1.35  2001/06/19 22:48:55  robertj
 * Gatekeeper supprt, multiple aliases and prefixes, thanks Chih-Wei Huang
 *
 * Revision 1.34  2001/06/06 00:16:15  robertj
 * Removed code to extract e164 address and use OpenH323 function instead.
 *
 * Revision 1.33  2001/06/05 03:14:41  robertj
 * Upgraded H.225 ASN to v4 and H.245 ASN to v7.
 *
 * Revision 1.32  2001/03/20 23:42:55  robertj
 * Used the new PTrace::Initialise function for starting trace code.
 *
 * Revision 1.31  2001/01/20 10:23:27  craigs
 * Added another fix to ring timer thanks to Chih-Wei Huang
 *
 * Revision 1.30  2001/01/12 22:32:55  craigs 
 * Added fix to ring timer thanks to Chih-Wei Huang
 * 
 * Revision 1.29  2001/01/01 22:57:47  craigs
 * Added correct handling of incoming ring detection
 *
 * Revision 1.28  2000/11/28 01:40:37  robertj
 * Added country code to command line options.
 *
 * Revision 1.27  2000/10/19 03:39:17  dereks
 * Fixed PTRACE report
 *
 * Revision 1.26  2000/10/19 03:36:30  dereks
 * Added argument to set jitter buffer length in pstngw.
 * Use -j delay    (or --jitter delay) where delay is 20<= 150(def) <= 10 000 ms
 *
 * Revision 1.25  2000/10/16 08:50:09  robertj
 * Added single function to add all UserInput capability types.
 *
 * Revision 1.24  2000/09/23 07:20:46  robertj
 * Fixed problem with being able to distinguish between sw and hw codecs in LID channel.
 *
 * Revision 1.23  2000/09/22 01:35:53  robertj
 * Added support for handling LID's that only do symmetric codecs.
 *
 * Revision 1.22  2000/08/21 04:39:54  dereks
 * add parameter to set number of audio packets in an ethernet frame
 *
 * Revision 1.21  2000/06/22 17:40:21  craigs
 * Fixed problem with ringing causing multiple connections
 *
 * Revision 1.20  2000/06/19 00:32:22  robertj
 * Changed functionf or adding all lid capabilities to not assume it is to an endpoint.
 *
 * Revision 1.19  2000/05/30 10:19:28  robertj
 * Added function to add capabilities given a LID.
 * Improved LID capabilities so cannot create one that is not explicitly supported.
 *
 * Revision 1.18  2000/05/25 13:25:47  robertj
 * Fixed incorrect "save" parameter specification.
 *
 * Revision 1.17  2000/05/25 12:06:17  robertj
 * Added PConfigArgs class so can save program arguments to config files.
 *
 * Revision 1.16  2000/05/23 12:57:37  robertj
 * Added ability to change IP Type Of Service code from applications.
 *
 * Revision 1.15  2000/05/23 11:32:37  robertj
 * Rewrite of capability table to combine 2 structures into one and move functionality into that class
 *    allowing some normalisation of usage across several applications.
 * Changed H323Connection so gets a copy of capabilities instead of using endponts, allows adjustments
 *    to be done depending on the remote client application.
 *
 * Revision 1.14  2000/04/19 02:07:29  robertj
 * Fixed reuse of variable.
 *
 * Revision 1.13  2000/04/05 04:06:05  robertj
 * Added ability to change order of codecs.
 * Added transfer of caller ID to H.323
 *
 * Revision 1.12  2000/03/27 18:21:03  robertj
 * Added IP access control and more improvements in calling tone translation into H.323 signals.
 *
 * Revision 1.11  2000/03/24 03:41:45  robertj
 * Fixed some GNU C++ warnings.
 *
 * Revision 1.10  2000/03/23 23:33:16  robertj
 * Added more calling tone detection functionality.
 *
 * Revision 1.9  2000/03/23 03:03:46  craigs
 * Added interface option, and improved semantics of forced dialling
 *
 * Revision 1.8  2000/03/23 02:42:33  robertj
 * Added detection of PSTN hang up when making outgoing call
 *
 * Revision 1.7  2000/02/17 22:12:48  craigs
 * Changed error to warning when no line interface devices
 *
 * Revision 1.6  2000/02/17 06:27:30  robertj
 * Added #ifdef for IxJack support code.
 *
 * Revision 1.5  2000/02/16 03:30:01  robertj
 * Fixed some references to VPB code that should be enclosed in #ifdef.
 *
 * Revision 1.4  2000/01/07 08:28:09  robertj
 * Additions and changes to line interface device base class.
 *
 * Revision 1.3  2000/01/04 00:16:25  craigs
 * Changed for new AnswerCall timing
 *
 * Revision 1.2  1999/12/23 23:02:35  robertj
 * File reorganision for separating RTP from H.323 and creation of LID for VPB support.
 *
 * Revision 1.1  1999/12/22 09:49:36  craigs
 * Initial version
 *
 *
 */

#include <ptlib.h>

#include <h323pdu.h>
#include <lid.h>

#ifdef HAS_IXJ
#include <ixjlid.h>
#ifndef __GNUC__
#pragma message("Including IxJ support.")
#endif
#endif

#ifdef HAS_VPB
#include <vpblid.h>
#ifndef __GNUC__
#pragma message("Including VPB support.")
#endif
#endif

#include "main.h"
#include "version.h"


#if !defined(HAS_IXJ) && !defined(HAS_VPB)
#error Must specify a Line Interface Device! Use either HAS_IXJ or HAS_VPB.
#endif


PCREATE_PROCESS(PSTNGw);


#define new PNEW


///////////////////////////////////////////////////////////////

PSTNGw::PSTNGw()
  : PServiceProcess("OpenH323 Project", "PSTNGw",
                    MAJOR_VERSION, MINOR_VERSION, BUILD_TYPE, BUILD_NUMBER)
{
}


void PSTNGw::Main()
{
  PConfig myconf;
  
#if PTRACING
  PTrace::Initialise(myconf.GetInteger("Trace" , "Trace Level"),
                     myconf.GetBoolean("Trace" , "Trace Output") ? (const char *)myconf.GetString("Trace" , "Trace File" , "NULL") : NULL,
                     PTrace::Blocks | PTrace::Timestamp | PTrace::Thread | PTrace::FileAndLine);
#endif

  GatewayEndPoint endpoint;
  if (endpoint.Initialise(myconf)) {
    PSYSTEMLOG(Info, "PSTNGw\tEndpoint succesfully initialised");

    exitMain.Wait(); //exits *ONLY* when a signal is received (OnStop calls exitMain.Signal()

    PSYSTEMLOG(Info, "PSTNGw\tMain received Signal, acknowledging and exiting");
    exitMain.Acknowledge();
  }
  else {
    PSYSTEMLOG(Error, "PSTNGw\tFailed initialising endpoint");
    exitMain.Acknowledge();
  }
}


void PSTNGw::OnStop()
{
  PSYSTEMLOG(Debug, "PSTNGw\tOnStop called, calling PserviceProcess::OnStop()");

  PServiceProcess::OnStop();

  PSYSTEMLOG(Debug, "PSTNGw\tPServiceProcess::OnStop returned, sending Signal() to exitMain");
  exitMain.Signal();
}


BOOL PSTNGw::OnStart()
{
  return TRUE;
}


void PSTNGw::OnControl()
{
  //void function, needed only for tray icon
}


///////////////////////////////////////////////////////////////

GatewayEndPoint::GatewayEndPoint()
  : natRouter(0)
{
  terminalType = e_GatewayOnly;
}


GatewayEndPoint::~GatewayEndPoint()
{
  ClearAllCalls();
}


BOOL GatewayEndPoint::Initialise(const PConfig & myconf)
{
  PINDEX i;

  autoDial         = myconf.GetString("H.323" , "Default Dialnumber" , "NULL");
  forceDial        = myconf.GetBoolean("H.323" , "Force Dial");
  convertTones     = myconf.GetBoolean("H.323" , "Calling Tones");
  disableFastStart = myconf.GetBoolean("H.323" , "Faststart Disable");
  disableH245Tunneling = myconf.GetBoolean("H.323" , "Disable H245tunnelling");
  disableH245inSetup = myconf.GetBoolean("H.323" , "Disable H245setup");

  if (!myconf.GetString("Audio" , "Jitter Delay" , "").IsEmpty()) {
    unsigned minJitter;
    unsigned maxJitter;
    PStringArray delays = myconf.GetString("Audio" , "Jitter Delay" , "").Tokenise(",-");
    if (delays.GetSize() == 2) {
      minJitter = delays[0].AsUnsigned();
      maxJitter = delays[1].AsUnsigned();
    }
    else {
      maxJitter = delays[0].AsUnsigned();
      minJitter = PMIN(GetMinAudioJitterDelay(), maxJitter);
    }
    if (minJitter >= 20 && minJitter <= maxJitter && maxJitter <= 10000)
      SetAudioJitterDelay(minJitter, maxJitter);
    else {
      PSYSTEMLOG(Error, "PSTNGw\tJitter should be between 20 milliseconds and 10 seconds.\n"
                        "Using default value of " << GetMinAudioJitterDelay() <<
                        " to " << GetMaxAudioJitterDelay() << " ms.");
    }
  }
  PSYSTEMLOG(Info, "PSTNGw\t Jitter buffer is "
                << GetMinAudioJitterDelay() << " to " << GetMaxAudioJitterDelay() << " ms");


  if (!myconf.GetString("Network" , "TOS" , "").IsEmpty()) // Questo e' un test esaustivo perche':
    // chiave non presente -> GetString ritorna dflt (impostato a "") e IsEmpty = TRUE
    // chiave presente e vuota -> IsEmpty ritorna TRUE
    // SE INVECE chiave presente e = 0 => TOS=0 (valore possibile, d'altronde)  
    SetRtpIpTypeofService(myconf.GetString("Network" , "TOS" , "").AsUnsigned());

  if (forceDial && autoDial.IsEmpty()) {
    PSYSTEMLOG(Fatal, "PSTNGw\tMust specify autodial number if forcedial option used");
    return FALSE;
  }

  // get local username
  if (!myconf.GetString("H.323" , "Endpoint Name" , "").IsEmpty()) {
    PStringArray aliases = myconf.GetString("H.323" , "Endpoint Name" , "").Tokenise("-,"); 
    SetLocalUserName(aliases[0]);
    for (i = 1; i < aliases.GetSize(); i++)
      AddAliasName(aliases[i]);
  } 
  else
    SetLocalUserName("OpenH323 PSTN Gateway");

  accessControl.LoadHostsAccess();

  H323ListenerTCP * listener;

  WORD listenPort = H323EndPoint::DefaultTcpPort;
  if (!myconf.GetString("Network" , "Listenport" , "").IsEmpty())
    listenPort = (WORD)myconf.GetString("Network" , "Listenport" , "").AsInteger();

  if (myconf.GetString("Network" , "Interface Ip" , "").IsEmpty()) {
    PIPSocket::Address interfaceAddress(INADDR_ANY);
    listener  = new H323ListenerTCP(*this, interfaceAddress, listenPort);
  }
  else {
    PIPSocket::Address interfaceAddress(myconf.GetString("Network" , "Interface Ip" , ""));
    listener  = new H323ListenerTCP(*this, interfaceAddress, listenPort);
  }

  if (!StartListener(listener)) {
    PSYSTEMLOG(Fatal, "PSTNGw\tCould not open H.323 listener port on "
                  << listener->GetListenerPort());
    delete listener;
    return FALSE;
  }

  PSYSTEMLOG(Info, "PSTNGw\tListening on port " << listener->GetListenerPort());

  if (!myconf.GetString("Network" , "Masquerade Ip", "").IsEmpty()) {
    natRouter = myconf.GetString("Network" , "Masquerade Ip", "");
    PSYSTEMLOG(Info, "PSTNGw\tMasquerading as address " << natRouter);
  }


#ifdef HAS_IXJ
  BOOL usingIXJ = FALSE;
  if (!myconf.GetString("Devices" , "Quicknet Dev" , "").IsEmpty()) {
    PStringArray devices = myconf.GetString("Devices" , "Quicknet Dev" , "").Tokenise(",-");
  // PStringArray devices = args.GetOptionString('q').Lines();
    for (i = 0; i < devices.GetSize(); i++) {
      OpalIxJDevice * lid = new OpalIxJDevice;
      if (lid->Open(devices[i])) {
        lids.Append(lid);
        usingIXJ = TRUE;
      }
    else {
        PSYSTEMLOG(Fatal, "PSTNGw\terror: cannot open ixj device " << devices[i]);
        return FALSE;
      }
      lid->EnableAudio(1, TRUE);
    }
  }
#endif

#ifdef HAS_VPB
  if (!myconf.GetString("Devices" , "Voicetronix Dev" , "").IsEmpty()) {
    PStringArray devices = myconf.GetString("Devices" , "Voicetronix Dev" , "").Tokenise(",-");
    for (i = 0; i < devices.GetSize(); i++) {
      OpalVpbDevice * lid = new OpalVpbDevice;
      if (lid->Open(devices[i]))
        lids.Append(lid);
    else {
        PSYSTEMLOG(Fatal, "PSTNGw\terror: cannot open VPB device " << device);
        return FALSE;
      }
    }
  }
#endif

  OpalLineInterfaceDevice::AECLevels aecLevel;
  if (!myconf.GetString("Devices" , "AEC Level" , "").IsEmpty())
    aecLevel = (OpalLineInterfaceDevice::AECLevels)myconf.GetString("Devices" , "AEC Level" , "").AsUnsigned();
  else
    aecLevel = OpalLineInterfaceDevice::AECMedium;
  
  PString country = myconf.GetString("Devices", "Country Code" , "");

  BOOL noLinesAvailable = TRUE;
    
#ifdef P_LINUX
#warning linux specific check on ringing

  PStringArray incomingCadence = myconf.GetString("Devices", "Incoming Ringing", "").Tokenise(",-");
  IXJ_FILTER_CADENCE cadence;
  memset(&cadence, 0, sizeof(cadence));
  cadence.enable    = 1;
  cadence.filter    = 4;
  unsigned int *res = &cadence.on1;
  for (PINDEX j = 0; (j < 7) && (j < incomingCadence.GetSize()); j++) {
    cerr << j << "  time (cs) " << incomingCadence[j].AsInteger()/10 << endl;
    res[j] = incomingCadence[j].AsInteger()/10;
  }

  for (i = 0; i < lids.GetSize(); i++) {
    OpalLineInterfaceDevice *temp = &(lids[i]);
    int osHandle = ((OpalIxJDevice *)temp)->GetOSHandle();
    ::ioctl(osHandle, IXJCTL_FILTER_CADENCE, &cadence);
  }
#endif

  PStringArray redirections = myconf.GetString("H.323" , "Incoming Redirections" , "").Tokenise(",-");
  unsigned thisLine = 0; // we need this to go throug the PString Array of redirections
  
  for (i = 0; i < lids.GetSize(); i++) {
    for (unsigned l = 0; l < lids[i].GetLineCount(); l++) {
      if (lids[i].IsLineTerminal(l))
        continue;
      
      lids[i].SetLineOnHook(l);
      PSYSTEMLOG(Debug, "PSTNGw\tInitialise SetLineOnHook true");

      //if (!lids[i].IsLinePresent(l))
      //  PSYSTEMLOG(Info, "No line on " << lids[i] << '-' << l);
      //else {
        noLinesAvailable = FALSE;
      
        if (!myconf.GetString("Devices" , "Voicetronix Playgain" , "").IsEmpty())
          lids[i].SetPlayVolume(l, myconf.GetString("Devices" , "Voicetronix Playgain" , "").AsUnsigned());
      
        if (!myconf.GetString("Devices" , "Voicetronix Recordgain" , "").IsEmpty())
          lids[i].SetRecordVolume(l, myconf.GetString("Devices" , "Voicetronix Recordgain" , "").AsUnsigned());
      
        lids[i].SetAEC(l, aecLevel);
        if (!country.IsEmpty())
          lids[i].SetCountryCodeName(country);
      
        PString h323Address;
      
        PSYSTEMLOG(Info, "PSTNGw\tUsing line " << lids[i] << '-' << l);
        if (!redirections[thisLine].IsEmpty()) {
          h323Address = redirections[thisLine];
          PSYSTEMLOG(Info, "PSTNGw\tIncoming calls redirected to \"" << h323Address << '"');
        }
        else
          PSYSTEMLOG(Info, "PSTNGw\tIncoming calls ignored");
      
        H323_LIDCapability::AddAllCapabilities(lids[i], capabilities, 0, 0);
        lineMonitors.Append(new LineMonitor(*this, lids[i], l, h323Address, 
        myconf.GetBoolean("H.323" , "Collect Number"),
        myconf.GetString("IVR" , "Intro Message", "")));
        thisLine++;
      //}
    }
  }
  
  if (noLinesAvailable) {
    PSYSTEMLOG(Fatal, "PSTNGw\tNo usable lines available");
    return FALSE;
  }
  
  AddAllCapabilities(0, 0, OpalGSM0610);
  AddAllCapabilities(0, 0, OpalG711uLaw);
  AddAllCapabilities(0, 0, OpalG711ALaw);
  
  H323_UserInputCapability::AddAllCapabilities(capabilities, 0, 1);
  
  capabilities.Remove(myconf.GetString("Audio" , "Disabled Codecs" , "").Tokenise(",-"));
  capabilities.Reorder(myconf.GetString("Audio" , "Preferred Codecs" , "").Tokenise(",-"));
  
  if (!myconf.GetString("Audio" , "Num Audioframes", "").IsEmpty()) {
    int txFrames = myconf.GetString("Audio" , "Num Audioframes" , "").AsInteger();
    txFrames     = PMAX(1, PMIN(20,txFrames));
    PSYSTEMLOG(Info, "PSTNGw\tAudio frames of data per packet : "<<txFrames);
    for (i = 0; i < capabilities.GetSize(); i++) 
      capabilities[i].SetTxFramesInPacket(txFrames);
  }
  
  // Gatekeeper registration
  if (!myconf.GetString("Gatekeeper" , "Gatekeeper Prefix" , "").IsEmpty())
    SupportedPrefix = myconf.GetString("Gatekeeper" , "Gatekeeper Prefix" , "").Tokenise(",-");
  
  if (!myconf.GetBoolean("Gatekeeper" , "Disable Gatekeeper")) {
    if (UseGatekeeper(myconf.GetString("Gatekeeper" , "Gatekeeper Host" , ""),
                      myconf.GetString("Gatekeeper" , "Gatekeeper Idname" , ""),
                      myconf.GetString("Network" , "Interface Ip", "")))
      PSYSTEMLOG(Info, "PSTNGw\tGatekeeper set: " << *gatekeeper);
    else {
      if (myconf.GetBoolean("Gatekeeper" , "Force Gatekeeper")) {
        PSYSTEMLOG(Fatal, "PSTNGw\tError registering with gatekeeper");
        return FALSE;
      }
      PSYSTEMLOG(Error, "PSTNGw\tError registering with gatekeeper");
    }
  }
  else
  {
    PSYSTEMLOG(Info, "PSTNGw\tGatekeeper disabled");
  }
  
  PSYSTEMLOG(Info, "PSTNGw\tCodecs (in preference order):\n" << setprecision(2) << capabilities);
  PSYSTEMLOG(Warning, "PSTNGw\tWaiting for incoming calls for gateway \"" << GetLocalUserName() << '"');
  
  return TRUE;
}
  
void GatewayEndPoint::SetEndpointTypeInfo( H225_EndpointType & info ) const
{
  H323EndPoint::SetEndpointTypeInfo(info);
  info.m_gateway.IncludeOptionalField(H225_GatewayInfo::e_protocol);
  info.m_gateway.m_protocol.SetSize(1);
  H225_SupportedProtocols &protocol=info.m_gateway.m_protocol[0];
  protocol.SetTag(H225_SupportedProtocols::e_voice);
  PINDEX as=SupportedPrefix.GetSize();
  ((H225_VoiceCaps &)protocol).m_supportedPrefixes.SetSize(as);
  for (PINDEX p=0; p<as; p++)
    H323SetAliasAddress(SupportedPrefix[p], ((H225_VoiceCaps &)protocol).m_supportedPrefixes[p].m_prefix);
}

//
// Rewrite E164 number according to the dial plan
// Format:
//  prefix_match=strip,prefix[,suffix]
//
//         strip    number of digits to strip
//         prefix   prefix to insert
//         suffix   suffix to append
//
// Example:
//  09=1,886      0953378875 --> 8869378875
//  07=2,,0       07123      --> 1230
//
PString GatewayEndPoint::RewriteE164(const PString &dialedDigits)
{
  static PStringToString DialPlan(PConfig("DialPlan").GetAllKeyValues());
  return RewriteE164(dialedDigits, DialPlan);
}


PString GatewayEndPoint::RewriteE164(const PString &dialedDigits, const PStringToString &myplan)
{
  for (PINDEX i=dialedDigits.GetLength(); i>0; i--) {
    PString key = dialedDigits.Left(i);
    if (myplan.Contains(key)) {
      PStringArray rule = myplan[key].Tokenise(",");
      PString result = rule[1] + dialedDigits.Mid(rule[0].AsInteger());
      if (!rule[2])
        result += rule[2];
      PSYSTEMLOG(Info, "PSTNGw\tPrefix " << key << " matched, rewrite " << dialedDigits << " to " << result << endl);
      return result;
    }
  }
  // no match
  return dialedDigits;
}

H323Connection * GatewayEndPoint::CreateConnection(unsigned callReference)
{
 return new GatewayConnection(*this, callReference);
}


BOOL GatewayEndPoint::OnIncomingCall(H323Connection & connection,
                                    const H323SignalPDU & setupPDU,
                                    H323SignalPDU &)
{
  H323TransportAddress address = connection.GetControlChannel().GetRemoteAddress();
  PIPSocket::Address ip;
  WORD port;
  if (address.GetIpAndPort(ip, port) && !accessControl.IsAllowed(ip)) {
    PSYSTEMLOG(Warning, "PSTNGw\tIncoming H.323 call from " << ip << " is denied.");
    return FALSE;
  }

  GatewayConnection & myConnection = (GatewayConnection &)connection;

  // Locate a free line
  PINDEX i;
  for (i = 0; i < lineMonitors.GetSize(); i++) {
    if (lineMonitors[i].IsAvailable()) {
      myConnection.line = &lineMonitors[i];
      break;
    }
  }

  if (myConnection.line == NULL) {
    PSYSTEMLOG(Warning, "PSTNGw\tNo available lines for incoming H.323 call.");
    return FALSE;
  }

  myConnection.line->AttachConnection(connection);

  OpalLineInterfaceDevice & device = myConnection.line->GetDevice();
  unsigned line = myConnection.line->GetLineNumber();

  // Sieze the line
  device.SetLineOffHook(line);
  PSYSTEMLOG(Info, "PSTNGw\tSetting line " << device << '-' << line << " off hook");

  PString number;

  if (forceDial || !setupPDU.GetDestinationE164(number))
    number = autoDial;

  if (!number) {
    // Wait for dial tone
    PSYSTEMLOG(Debug, "PSTNGw\tWaiting for DialTone for 5000msec");
    if (device.WaitForTone(line, OpalLineInterfaceDevice::DialTone, 5000)) {
      PSYSTEMLOG(Info, "PSTNGw\tDial tone detected.");
    } else {
      PSYSTEMLOG(Info, "PSTNGw\tNo dial tone detected.");
    }

    PString dialed(RewriteE164(number));
    PSYSTEMLOG(Info, "PSTNGw\tDialling: " << dialed);
    device.PlayDTMF(line, dialed);
  }

  return TRUE;
}


void GatewayEndPoint::TranslateTCPAddress(PIPSocket::Address & localAddr,
                                          const PIPSocket::Address & remoteAddr)
{
   /*
    * We are using NAT and the remote address is outside our LAN, so
    * replace the local address with the IP address of the NAT box.
    */
  if (natRouter.IsValid() && IsLocalAddress(localAddr) && !IsLocalAddress(remoteAddr))
    localAddr = natRouter;
}


H323Connection::AnswerCallResponse GatewayEndPoint::OnAnswerCall(H323Connection & connection,
                                                                 const PString & /*callerName*/,
                                                                 const H323SignalPDU & /*setupPDU*/,
                                                                 H323SignalPDU & /*connectPDU*/)
{
  //Moved myConnection declaration on top of everything else (if(!converTones), too)  

  GatewayConnection & myConnection = (GatewayConnection &)connection;
  OpalLineInterfaceDevice & device = myConnection.line->GetDevice();
  unsigned line = myConnection.line->GetLineNumber();
  
  if (!convertTones) {
    PSYSTEMLOG(Debug, "PSTNGw\tOn Answer Call: converTones FALSE, returning AnswerCallNow");
    return H323Connection::AnswerCallNow;
  }  

  PSYSTEMLOG(Debug, "PSTNGw\tNumber dialled. trying to translate PSTN events into H.323 OnAnswerCall ones");
  // Very bad implementation: line already dialed (ringing on the remote side) but not ringing in H.323 application (waiting)
  // hence if no ring is detected (ie: bad tone definition), phone rings, H.323 app waits, then disconnected
  switch (device.WaitForToneDetect(line, 3000)) {
    case OpalLineInterfaceDevice::RingTone :
      PSYSTEMLOG(Info, "PSTNGw\tRing tone detected.");
      return H323Connection::AnswerCallNow; // Temporary code, not completed yet

    case OpalLineInterfaceDevice::BusyTone :
      PSYSTEMLOG(Info, "PSTNGw\tBusy tone detected, call being refused.");
      break;

    default :
      PSYSTEMLOG(Info, "PSTNGw\tNo ring/busy tone detected.");
  }

  return H323Connection::AnswerCallDenied; //Connection aborted, release complete PDU
}


BOOL GatewayEndPoint::OpenAudioChannel(H323Connection & connection,
                                      BOOL /*isEncoding*/,
                                      unsigned,
                                      H323AudioCodec & codec)
{
  GatewayConnection & myConnection = (GatewayConnection &)connection;
  if (myConnection.line == NULL)
    return FALSE;

  codec.SetSilenceDetectionMode(H323AudioCodec::NoSilenceDetection);

  return codec.AttachChannel(new OpalLineChannel(myConnection.line->GetDevice(),
                                                 myConnection.line->GetLineNumber(),
                                                 codec));
}


void GatewayEndPoint::OnConnectionCleared(H323Connection & connection,
                                         const PString & /*token*/)
{
  GatewayConnection & myConnection = (GatewayConnection &)connection;
  if (myConnection.line == NULL)
    return;

  OpalLineInterfaceDevice & device = myConnection.line->GetDevice();
  unsigned lineNumber = myConnection.line->GetLineNumber();

  PSYSTEMLOG(Info, "PSTNGw\tConnection cleared. Setting line " << device << '-' << lineNumber << " on hook");
  device.SetLineOnHook(lineNumber);
}


///////////////////////////////////////////////////////////////

GatewayConnection::GatewayConnection(GatewayEndPoint & _ep, unsigned callReference)
  : H323Connection(_ep, callReference)
{
  line = NULL;
}


void GatewayConnection::OnUserInputString(const PString & value)
{
  if (line == NULL)
    return;

  OpalLineInterfaceDevice & device = line->GetDevice();
  unsigned lineNumber = line->GetLineNumber();

  PSYSTEMLOG(Debug, "PSTNGw\tLine " << device << '-' << lineNumber << " user input: " << value);
  device.PlayDTMF(lineNumber, value);
}


BOOL GatewayConnection::OnStartLogicalChannel(H323Channel & channel)
{
  PStringStream message;
  message << "Line " << line->GetDevice() << '-' << line->GetLineNumber()
       << " started logical channel: ";

  switch (channel.GetDirection()) {
    case H323Channel::IsTransmitter :
      message << "sending ";
      break;

    case H323Channel::IsReceiver :
      message << "receiving ";
      break;

    default :
      break;
  }

  message << channel.GetCapability();
  PSYSTEMLOG(Debug, "PSTNGw\tConnection\t" << message);

  return TRUE;
}


///////////////////////////////////////////////////////////////

LineMonitor::LineMonitor(GatewayEndPoint & ep,
                         OpalLineInterfaceDevice & dev,
                         unsigned line,
                         const PString & addr,
              BOOL collectNum,
             PString introMsg)
  : PThread(10000, NoAutoDeleteThread),
    endpoint(ep),
    device(dev),
    lineNumber(line),
    h323Address(addr),
    collectNumber(collectNum),
    introMessage(introMsg)

{
  Resume();
}


LineMonitor::~LineMonitor()
{
  shutdown.Signal();
  WaitForTermination();
}

static BOOL IsLIDRinging(OpalLineInterfaceDevice & lid, PINDEX line, PTimer & timer)
{
  if (timer.IsRunning())
    return TRUE;

  if (lid.IsLineOffHook(line)) {    // The Phone cannot be ringing, cause it is off hook.
    PSYSTEMLOG(Debug, "PSTNGw\tIsLIDRinging called on a off hook line (" << line << ")");
    return FALSE;
  }

  if (!lid.IsLineRinging(line))
    return FALSE;

  timer = 3000;

  return TRUE;
}


void LineMonitor::Main()
{
  PTimer ringTimer;
  
  PSYSTEMLOG(Debug, "PSTNGW\tLine monitor thread for " << device << '-' << lineNumber << " started");
  
  // TODO: sposta gestione disconnessione da HandleCall a qui (IsLineDisconnected)  
  
  while (!shutdown.Wait(500)) {
    
    // if the line is ringing, start a new call
    PSYSTEMLOG(Debug, "PSTNGW\tLine Monitor for " << device << '-' << lineNumber << ", start of loop");
    if (IsLIDRinging(device, lineNumber, ringTimer)) {
      PSYSTEMLOG(Debug, "PSTNGW\tLine Monitor, detected line " << lineNumber << " on device " << device << " is ringing");
      
      // ignore incoming calls when no IP address to dial
      if (h323Address.IsEmpty()) {
        PString callerID;
        if (device.GetCallerID(lineNumber, callerID))
          PSYSTEMLOG(Info, "PSTNGw\tCaller ID: " << callerID);

        PSYSTEMLOG(Debug, "PSTNGw\twaiting for ringing to stop.");
        while (!shutdown.Wait(500) && IsLIDRinging(device, lineNumber, ringTimer))
          ;
        PSYSTEMLOG(Debug, "PSTNGW\tRinging stopped on device " << device << "-" << lineNumber);
      }
      else { // do not ignore incoming calls
        PSYSTEMLOG(Info, "PSTNGW\tLine Monitor. Call address " << h323Address);
        
        // if still onhook, wait for the ringing to stop
        if (HandleCall()) { 
          PSYSTEMLOG(Info, "PSTNGw\tCall from " << device << '-' << lineNumber << " to " << h323Address << " ended.");
        } 
        else {
          PSYSTEMLOG(Info, "PSTNGw\tCall from " << device <<  '-' << lineNumber << " to " << h323Address << " failed.");
          if(!callToken) {
            PSYSTEMLOG(Debug, "PSTNGW\t HandleCall returned FALSE and in LineMonitor I've found that callToken is still busy");
            // callToken = PString();   //Release Call Token if busy (really bad, but Clearing connection on line 1143 seems to not to work 
            // Cfr.   0:17.730         LineMonitor:080aea00       h323ep.cxx(1518)  H323    Clearing connection ip$localhost/2665 reason=EndedByLocalUser
          }

          while (!shutdown.Wait(500) && IsLIDRinging(device, lineNumber, ringTimer))
            ;
          PSYSTEMLOG(Debug, "PSTNGW\tLine Monitor, Ringing stopped on device" << device << '-' << lineNumber);
        }
      }
    }
    
    // if a call is active, but the line has been disconnected, hangup the call
    // if (!callToken && device.IsLineDisconnected(lineNumber)) {
    if (!callToken) {
      PSYSTEMLOG(Debug, "PSTNGW\tLine Monitor on an active call on line " << lineNumber << ", (may be already on hook) checking if disconnected");
      if (device.IsLineDisconnected(lineNumber)) {
        PSYSTEMLOG(Debug, "PSTNGW\tLine Monitor, Line " << device << '-' << lineNumber << " disconnected");
        endpoint.ClearCall(callToken);
        callToken = PString();
      }
    }
  }
  
  PSYSTEMLOG(Debug, "PSTNGW\tLine monitor thread for " << device << '-' << lineNumber << " stopped");
}


BOOL LineMonitor::HandleCall()
{
  PString   callerID;
  PString   exth323Address;
  const int max_digits = 20; // era 10
  char      number[max_digits];
  
  PSYSTEMLOG(Debug, "PSTNGw\t Line Monitor HandleCall called on line " << lineNumber );
  
  *number = 0;
  
  // TODO: aggiungi un line monitor anche qui in modo che chiuda *TUTTO* (ocio ai thread!)
  // se rielva un busy (chiusa cornetta, es in caso di collect e chiusura aspetta il timeout)
  
  if (device.GetCallerID(lineNumber, callerID))
    PSYSTEMLOG(Info, "PSTNGw\tCaller is " << callerID);
  
  if (collectNumber) {
    // DR - collect number to dial before we make H323 call
    // - take lid port off-hook
    // - generate dial tone from lid
    // - collect digits until # pressed or 10 second timeout
    // - every new digit resets timer for another 10 seconds
    
    device.SetLineOffHook(lineNumber);
    
    //while (!device.IsLineDisconnected(lineNumber)) {  
    // Aggiunta: play di un avviso di benvenuto (istruzioni all'uso)
    // TODO: interrompi play al caught di un DTMF, la play_async non va bene
    // perche' continua  
    
    BOOL PlayAudio = 0; // Is PlayAudio implemented?
    
    if(!introMessage.IsEmpty())  
      if(device.PlayAudio(lineNumber, introMessage))
        PlayAudio = 1;
      else {
        PSYSTEMLOG(Debug, "PSTNGw\tPlayAudio returned false (not implemented?), using PlayTone");
        device.PlayTone(lineNumber, OpalLineInterfaceDevice::DialTone);
      }      
      else  
        device.PlayTone(lineNumber, OpalLineInterfaceDevice::DialTone);
      
      char      digit;
      int       num_digits = 0;
      BOOL    stopped = 0;  //Have I stopped audio/tone generation?
      
      // wait 15 seconds for the first digit
      PTimer collectTimer(15000);
      
      do {
        digit = device.ReadDTMF(lineNumber);
        if (digit) {
          number[num_digits++] = digit;
          PSYSTEMLOG(Debug, "PSTNGw\tdigit: " << digit);
          collectTimer = 10000;
        }
        if (device.IsLineDisconnected(lineNumber)) {
          PSYSTEMLOG(Info, "PSTNGw\tCaller hung up while collecting, returning FALSE (as timeout)");
          return FALSE;
        }
        if (shutdown.Wait(50))
          return FALSE;
        // Stop dial tone at first digit
        if(num_digits == 1 && !stopped) {
          if(PlayAudio)
            stopped = device.StopAudio(lineNumber);
          else
            stopped = device.StopTone(lineNumber);
        }
      } while((digit != '#') && (num_digits < max_digits) && collectTimer.IsRunning());
      //} while((digit != '#') && (num_digits < max_digits) && collectTimer.IsRunning() && !device.IsLineDisconnected(lineNumber)); //evaluated every 50msec (shutdown.Wait(50))
      
      // if exited from previous do-while without entering a digit, be sure to stop Threads for Tone and Audio playing
      // (otherwise a new Thread is launched while another one is running=>assert from vpbdial.cpp (Tone)
      if(!stopped) {
        if(PlayAudio)
          device.StopAudio(lineNumber);
        else
          device.StopTone(lineNumber);
      }
      
      if (num_digits) {
        if (digit == '#')
          num_digits--;
        number[num_digits] = 0;
        PSYSTEMLOG(Debug, "PSTNGw\tnum_digits: " << num_digits << " number: " << number);
      }
      else {
        device.SetLineOnHook(lineNumber);
        return FALSE;
      } 
  } // if(collectNumber)
  
  // if we have collected a number concatenate with ip host
  if (*number) {
    exth323Address = PString(number) + PString("@") + h323Address;
  }
  else {
    exth323Address = h323Address;    
  }
  PSYSTEMLOG(Debug, "PSTNGw\tMake call to h323 Address " << exth323Address);
  // now make call
  
  {
    GatewayConnection * myConnection = (GatewayConnection *)endpoint.MakeCallLocked(exth323Address, callToken);
    if (myConnection == NULL)
      return FALSE;
    
    myConnection->line = this;
    myConnection->SetLocalPartyName(callerID);
    myConnection->Unlock();
  }
  
  PTimer ringTimer(10000); /* There is only one ring signal, yet we check multiple times for it ringing*/
                           /*This timer means 1 ring signal works for 10 seconds */
  
  while (!endpoint.IsConnectionEstablished(callToken)) {
    //while ((!endpoint.IsConnectionEstablished(callToken)) && !device.IsLineDisconnected(lineNumber)) {
    // If H.323 call has failed then ignore the call
    if (!endpoint.HasConnection(callToken)) {
      PSYSTEMLOG(Warning, "PSTNGw\tEndpoint does not have connection for " << callToken);
      return FALSE;
    }
    
    if (!collectNumber) {
      // If the ringing stops before establishment, clear the call and exit
      if (!IsLIDRinging(device, lineNumber, ringTimer)) {
        PSYSTEMLOG(Warning, "PSTNGw\tLid has stopped  ringing, so clear the call we created. ");
        //endpoint.ClearCall(callToken);
        endpoint.ClearCallSynchronous(callToken);
        callToken = PString();
        return FALSE;
      } 
    }
    
    if (device.IsLineDisconnected(lineNumber)) {
      PSYSTEMLOG(Info, "PSTNGw\tCaller hung up while alerting, Clearing H.323 Call and returning FALSE");
      endpoint.ClearCallSynchronous(callToken);
      callToken = PString();
      return FALSE;
    }
    
    if (shutdown.Wait(50))
      return FALSE;
  }
  
  if (!collectNumber) {
    PSYSTEMLOG(Debug, "PSTNGw\tLine " << device << '-' << lineNumber << " going off hook");
    // Have H.323 call answered, so answer the PSTN line
    device.SetLineOffHook(lineNumber);
  } 
  
  // Now wait for either party to hang up
  while (endpoint.HasConnection(callToken) && !device.IsLineDisconnected(lineNumber)) {
    if (shutdown.Wait(200))
      return TRUE;
  }

  PSYSTEMLOG(Info, "PSTNGw\tLine " << device << '-' << lineNumber << " disconnected.");
  endpoint.ClearCallSynchronous(callToken);
  callToken = PString();
  return TRUE;
}


// End of File ///////////////////////////////////////////////////////////////






