/*
 * testoptplugin.cpp - Psi plugin for testing save-load options
 * Copyright (C) 2009  Pavel Titkov
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * You can also redistribute and/or modify this program under the
 * terms of the Psi License, specified in the accompanied COPYING
 * file, as published by the Psi Project; either dated January 1st,
 * 2005, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <QtCore>

#include "psiplugin.h"
#include "optionaccessor.h"
#include "optionaccessinghost.h"

#include <QtGui>

#define constSpeed  "spead"
#define constFirstName "firstName"

class TestOptPlugin: public QObject, public PsiPlugin, public OptionAccessor
{
	Q_OBJECT
	Q_INTERFACES(PsiPlugin OptionAccessor)

public:
	TestOptPlugin();

	// PsiPlugin
	virtual QString name() const;
	virtual QString shortName() const;
	virtual QString version() const;
	virtual QWidget* options();
	virtual bool enable();
	virtual bool disable();

	// OptionAccessor
	virtual void setOptionAccessingHost(OptionAccessingHost* host);
	virtual void optionChanged(const QString& option);

	virtual void applyOptions();
	virtual void restoreOptions();

private:
	bool enabled;
	OptionAccessingHost* psiOptions;

	int speed;
	QString firstName;

	QLineEdit *editFirstName;
	QSpinBox *editSpeed;
};

Q_EXPORT_PLUGIN(TestOptPlugin);

TestOptPlugin::TestOptPlugin()
	: enabled(false), psiOptions(0)
{
	speed = 54;
	firstName = "Bob";

	editFirstName = 0;
	editSpeed = 0;
}

//-- PsiPlugin ------------------------------------------------------

QString TestOptPlugin::name() const
{
	return "TestOpt Plugin";
}

QString TestOptPlugin::shortName() const
{
	return "testopt";
}

QString TestOptPlugin::version() const
{
	return "0.1";
}

QWidget* TestOptPlugin::options()
{
	if (!enabled) {
		return 0;
	}
	QWidget *optionsWid = new QWidget();

	editFirstName = new QLineEdit(optionsWid);
	editSpeed = new QSpinBox(optionsWid);

	editFirstName->setText(firstName);
	editSpeed->setValue(speed);

	QGridLayout *layout = new QGridLayout(optionsWid);
	layout->addWidget(new QLabel("Speed:", optionsWid), 0, 0);
	layout->addWidget(editSpeed, 0, 1);
	layout->addWidget(new QLabel("First Name:", optionsWid), 1, 0);
	layout->addWidget(editFirstName, 1, 1);
	return optionsWid;
}

bool TestOptPlugin::enable()
{
	if (psiOptions) {
		enabled = true;

		//Read default values
		QVariant vFirstName(firstName);
		vFirstName = psiOptions->getPluginOption(constFirstName);
		if (!vFirstName.isNull()) {
			firstName = vFirstName.toString();
		}

		QVariant vSpeead(speed);
		vSpeead = psiOptions->getPluginOption(constSpeed);
		if (!vSpeead.isNull()) {
			speed = vSpeead.toInt();
		}
	}

	return enabled;
}

bool TestOptPlugin::disable()
{
	enabled = false;
	return true;
}

//-- OptionAccessor -------------------------------------------------

void TestOptPlugin::setOptionAccessingHost(OptionAccessingHost* host)
{
	psiOptions = host;
}

void TestOptPlugin::optionChanged(const QString& option)
{
	Q_UNUSED(option);
}

void TestOptPlugin::applyOptions() {
	if (!editFirstName || !editSpeed) {
		return;
	}
	QVariant vFirstName(editFirstName->text());
	psiOptions->setPluginOption(constFirstName, vFirstName);
	firstName = vFirstName.toString();

	QVariant vSpeead(editSpeed->value());
	psiOptions->setPluginOption(constSpeed, vSpeead);
	speed = vSpeead.toInt();
}

void TestOptPlugin::restoreOptions() {
	if (!editFirstName || !editSpeed) {
		return;
	}
	QVariant vFirstName(editFirstName->text());
	vFirstName = psiOptions->getPluginOption(constFirstName);
	if (!vFirstName.isNull()) {
		editFirstName->setText(vFirstName.toString());
	}

	QVariant vSpeead(editSpeed->value());
	vSpeead = psiOptions->getPluginOption(constSpeed);
	if (!vSpeead.isNull()) {
		editSpeed->setValue(vSpeead.toInt());
	}
}

#include "testoptplugin.moc"
