#include "psiplugin.h"
#include "optionaccessor.h"
#include "optionaccessinghost.h"
#include "shortcutaccessor.h"
#include "shortcutaccessinghost.h"

#include "screenshot.h"
#include <QtGui>

#define constUrl "url"
#define constShortCut "shortCut"
#define constFormat "format"
#define constFileName "fileName"

class ScreenshotPlugin : public QObject, public PsiPlugin, public OptionAccessor, public ShortcutAccessor
{
	Q_OBJECT
	Q_INTERFACES(PsiPlugin OptionAccessor ShortcutAccessor)

public:
	ScreenshotPlugin();
	~ScreenshotPlugin();

	virtual QString name() const;
	virtual QString shortName() const;
	virtual QString version() const;
	virtual QWidget* options();
	virtual bool enable();
	virtual bool disable();

	// OptionAccessor
	virtual void setOptionAccessingHost(OptionAccessingHost* host);
	virtual void optionChanged(const QString& option);

	// ShortcutsAccessor
	virtual void setShortcutAccessingHost(ShortcutAccessingHost* host);
	virtual void setShortcuts();

	virtual void applyOptions();
	virtual void restoreOptions();

private:
	bool enabled_;
	Screenshot *screenshot;
	OptionAccessingHost* psiOptions;
	ShortcutAccessingHost* psiShortcuts;

	QString url;
	QString shortCut;
	QString format;
	QString fileName;

	QLineEdit *editUrl;
	QLineEdit *editShortCut;
	QLineEdit *editFormat;
	QLineEdit *editFileName;
};

Q_EXPORT_PLUGIN(ScreenshotPlugin);

ScreenshotPlugin::ScreenshotPlugin()
{
	enabled_ = false;
	screenshot = 0;
	psiOptions = 0;

	url = "";
	editUrl = 0;

	shortCut = "Alt+Ctrl+p";
	editShortCut = 0;

	format = "png";
	editFormat = 0;

	fileName = "pic-yyyyMMdd-hhmmss";
	editFileName= 0;
}

ScreenshotPlugin::~ScreenshotPlugin()
{
	disable();
}

QString ScreenshotPlugin::name() const
{
	return "Screenshot Plugin";
}

QString ScreenshotPlugin::shortName() const
{
	return "Screenshot";
}

QString ScreenshotPlugin::version() const
{
	return "1.3";
}

QWidget* ScreenshotPlugin::options()
{
	if (!enabled_) {
		return 0;
	}
	QWidget *optionsWid = new QWidget();

	editUrl= new QLineEdit(optionsWid);
	editShortCut= new QLineEdit(optionsWid);
	editFormat = new QLineEdit(optionsWid);
	editFileName= new QLineEdit(optionsWid);

	editUrl->setText(url);
	editShortCut->setText(shortCut);
	editFormat->setText(format);
	editFileName->setText(fileName);

	QGridLayout *layout = new QGridLayout(optionsWid);
	layout->addWidget(new QLabel(tr("ShortCut:"), optionsWid), 0, 0);
	layout->addWidget(editShortCut, 0, 1);
	layout->addWidget(new QLabel(tr("Format:"), optionsWid), 1, 0);
	layout->addWidget(editFormat, 1, 1);
	layout->addWidget(new QLabel(tr("File Name:"), optionsWid), 2, 0);
	layout->addWidget(editFileName, 2, 1);
	layout->addWidget(new QLabel("Url:", optionsWid), 3, 0);
	layout->addWidget(editUrl, 3, 1);
	return optionsWid;
}

bool ScreenshotPlugin::enable()
{
	if (!screenshot) {

		//Read default values
		QVariant vUrl(url);
		vUrl= psiOptions->getPluginOption(constUrl);
		if (!vUrl.isNull()) {
			url= vUrl.toString();
		}

		QVariant vShortCut(shortCut);
		vShortCut= psiOptions->getPluginOption(constShortCut);
		if (!vShortCut.isNull()) {
			shortCut = vShortCut.toString();
		}

		QVariant vFormat(format);
		vFormat= psiOptions->getPluginOption(constFormat);
		if (!vFormat.isNull()) {
			format = vFormat.toString();
		}

		QVariant vFileName(fileName);
		vFileName= psiOptions->getPluginOption(constFileName);
		if (!vFileName.isNull()) {
			fileName = vFileName.toString();
		}

		screenshot = new Screenshot(url, fileName, format);
	}
	psiShortcuts->connectShortcut(QKeySequence(shortCut), screenshot, SLOT(shootScreen()));
	enabled_ = true;
	return true;
}

bool ScreenshotPlugin::disable()
{
	if (!screenshot) {
		psiShortcuts->disconnectShortcut(QKeySequence(shortCut), screenshot, SLOT(shootScreen()));
		delete screenshot;
		screenshot = 0;
	}
	enabled_ = false;
	return true;
}

void ScreenshotPlugin::setOptionAccessingHost(OptionAccessingHost* host)
{
	psiOptions = host;
}

void ScreenshotPlugin::optionChanged(const QString& option)
{
	Q_UNUSED(option);
}

void ScreenshotPlugin::setShortcutAccessingHost(ShortcutAccessingHost* host)
{
	psiShortcuts = host;
}

void ScreenshotPlugin::setShortcuts()
{
	if (screenshot) {
		psiShortcuts->connectShortcut(QKeySequence(shortCut), screenshot, SLOT(shootScreen()));
	}
}

void ScreenshotPlugin::applyOptions() {
	if (!editUrl || !editShortCut || !editFormat || !editFileName) {
		return;
	}
	QVariant vUrl(editUrl->text());
	psiOptions->setPluginOption(constUrl, vUrl);
	url = vUrl.toString();

	QVariant vShortCut(editShortCut->text());
	psiOptions->setPluginOption(constShortCut, vShortCut);
	psiShortcuts->disconnectShortcut(QKeySequence(shortCut), screenshot, SLOT(shootScreen()));
	shortCut = vShortCut.toString();

	QVariant vFormat(editFormat->text());
	psiOptions->setPluginOption(constFormat, vFormat);
	format = vFormat.toString();

	QVariant vFileName(editFileName->text());
	psiOptions->setPluginOption(constFileName, vFileName);
	fileName = vFileName.toString();

	screenshot->setUrl(url);
	screenshot->setFormat(format);
	screenshot->setFileNameFormat(fileName);
	setShortcuts();
}

void ScreenshotPlugin::restoreOptions() {
	if (!editUrl || !editShortCut || !editFormat || !editFileName) {
		return;
	}
	QVariant vUrl(editUrl->text());
	vUrl = psiOptions->getPluginOption(constUrl);
	editUrl->setText(vUrl.toString());

	QVariant vShortCut(editShortCut->text());
	vShortCut = psiOptions->getPluginOption(constShortCut);
	if (!vShortCut.isNull()) {
		editShortCut->setText(vShortCut.toString());
	}

	QVariant vFormat(editFormat->text());
	vFormat= psiOptions->getPluginOption(constFormat);
	if (!vFormat.isNull()) {
		editFormat->setText(vFormat.toString());
	}

	QVariant vFileName(editFileName->text());
	vFileName= psiOptions->getPluginOption(constFileName);
	if (!vFileName.isNull()) {
		editFileName->setText(vFileName.toString());
	}
}

#include "screenshotplugin.moc"
