/*
    ProjectWindowController.m

    Implementation of the ProjectWindowController class for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "ProjectWindowController.h"

#import <AppKit/NSImage.h>
#import <AppKit/NSWindow.h>
#import <AppKit/NSWindow+Toolbar.h>
#import <AppKit/NSToolbar.h>
#import <AppKit/NSToolbarItem.h>
#import <AppKit/NSBox.h>

#import <Foundation/NSBundle.h>
#import <Foundation/NSException.h>
#import <Foundation/NSNotification.h>

#import "ProjectDocument.h"
#import "ProjectModule.h"
#import "ProjectType.h"
#import "FileManager.h"

@implementation ProjectWindowController

+ (NSArray *) toolbarItemIdentifiersForProjectCapabilities: (int) capabilities
{
  NSMutableArray * array = [NSMutableArray arrayWithCapacity: 7];

  [array addObject: @"Attributes"];
  if (capabilities & FilesProjectCapability)
    [array addObject: @"Files"];
  if (capabilities & FrameworksProjectCapability)
    [array addObject: @"Frameworks"];
  if (capabilities & BuildProjectCapability)
    [array addObject: @"Build"];
  if (capabilities & RunProjectCapability)
    [array addObject: @"Run"];
  if (capabilities & DebugProjectCapability)
    [array addObject: @"Debug"];
  if (capabilities & SubprojectsProjectCapability)
    [array addObject: @"Subprojects"];

  return [[array copy] autorelease];
}

- initWithWindowNibName: (NSString *) nibName
          ownerDocument: (ProjectDocument *) anOwner
{
  // we must do this before we load the nib
  owner = anOwner;

  return [self initWithWindowNibName: nibName];
}

- (void) dealloc
{
  TEST_RELEASE(itemIdentifiers);
  TEST_RELEASE(currentModule);

  [super dealloc];
}

- (void) awakeFromNib
{
  NSToolbar * toolbar;

  toolbar = [[[NSToolbar alloc]
    initWithIdentifier: [NSString stringWithFormat: @"%@-Toolbar",
    [[[owner projectType] class] projectTypeID]]]
    autorelease];

  [toolbar setDelegate: self];

  [[self window] setToolbar: toolbar];
  [toolbar setVisible: YES];
  [[self window] setMiniwindowImage: [NSImage imageNamed: @"pmproj"]];

  [self selectView: AttributesTab];
  [toolbar setSelectedItemIdentifier: @"Attributes"];
}

- (void) switchView: sender
{
  [self selectView: [sender tag]];
}

/// Returns the identifier of the currently selected tab.
- (ProjectWindowTab) currentTab
{
  return currentTab;
}

/**
 * Returns the currently visible project module. To some stuff (e.g. the
 * debugger's inspectors) this information is better than the current tab.
 */
- (ProjectModule *) currentModule
{
  return currentModule;
}

- (void) selectView: (ProjectWindowTab) tag
{
  currentTab = tag;

  switch (tag)
    {
      case AttributesTab:
        ASSIGN(currentModule, [owner projectAttributes]);
        break;
      case FilesTab:
        ASSIGN(currentModule, [owner fileManager]);
        break;
      case FrameworksTab:
        ASSIGN(currentModule, [owner frameworksManager]);
        break;
      case BuildTab:
        ASSIGN(currentModule, [owner builder]);
        break;
      case RunTab:
        ASSIGN(currentModule, [owner launcher]);
        break;
      case DebugTab:
        ASSIGN(currentModule, [owner debugger]);
        break;
      case SubprojectsTab:
        ASSIGN(currentModule, [owner subprojectsManager]);
        break;
    }

  [[NSNotificationCenter defaultCenter]
    postNotificationName: CurrentProjectModuleDidChangeNotification
                  object: self
                userInfo: [NSDictionary dictionaryWithObject: currentModule
                                                      forKey: @"Module"]];

  [box setContentView: [currentModule view]];
}

- (void) viewAttributes: sender
{
  [self selectView: AttributesTab];
  [[[self window] toolbar] setSelectedItemIdentifier: @"Attributes"];
}

- (void) viewFiles: sender
{
  [self selectView: FilesTab];
  [[[self window] toolbar] setSelectedItemIdentifier: @"Files"];
}

- (void) viewFrameworks: sender
{
  [self selectView: FrameworksTab];
  [[[self window] toolbar] setSelectedItemIdentifier: @"Frameworks"];
}

- (void) viewBuild: sender
{
  [self selectView: BuildTab];
  [[[self window] toolbar] setSelectedItemIdentifier: @"Build"];
}

- (void) viewLaunch: sender
{
  [self selectView: RunTab];
  [[[self window] toolbar] setSelectedItemIdentifier: @"Run"];
}

- (void) viewDebug: sender
{
  [self selectView: DebugTab];
  [[[self window] toolbar] setSelectedItemIdentifier: @"Debug"];
}

- (void) viewSubprojects: sender
{
  [self selectView: SubprojectsTab];
  [[[self window] toolbar] setSelectedItemIdentifier: @"Subprojects"];
}

- (NSToolbarItem*)toolbar: (NSToolbar*)toolbar
    itemForItemIdentifier: (NSString*)itemIdentifier
willBeInsertedIntoToolbar: (BOOL)flag
{
  NSToolbarItem * item;

  item = [[[NSToolbarItem alloc]
    initWithItemIdentifier: itemIdentifier]
    autorelease];

  [item setLabel: _(itemIdentifier)];
  [item setImage: [NSImage imageNamed: itemIdentifier]];
  [item setTarget: self];
  [item setAction: @selector(switchView:)];

  if ([itemIdentifier isEqualToString: @"Attributes"])
    {
      [item setTag: AttributesTab];
      [item setToolTip: _(@"Project Attributes")];
      [item setImage: [NSImage imageNamed: @"Tab_Attributes"]];
    }
  else if ([itemIdentifier isEqualToString: @"Files"])
    {
      [item setTag: FilesTab];
      [item setToolTip: _(@"Project Files Management")];
      [item setImage: [NSImage imageNamed: @"Tab_Files"]];
    }
  else if ([itemIdentifier isEqualToString: @"Frameworks"])
    {
      [item setTag: FrameworksTab];
      [item setToolTip: _(@"Project Frameworks Management")];
      [item setImage: [NSImage imageNamed: @"Tab_Frameworks"]];
    }
  else if ([itemIdentifier isEqualToString: @"Build"])
    {
      [item setTag: BuildTab];
      [item setToolTip: _(@"Project Building")];
      [item setImage: [NSImage imageNamed: @"Tab_Build"]];
    }
  else if ([itemIdentifier isEqualToString: @"Run"])
    {
      [item setTag: RunTab];
      [item setToolTip: _(@"Project Running")];
      [item setImage: [NSImage imageNamed: @"Tab_Run"]];
    }
  else if ([itemIdentifier isEqualToString: @"Debug"])
    {
      [item setTag: DebugTab];
      [item setToolTip: _(@"Project Debugging")];
      [item setImage: [NSImage imageNamed: @"Tab_Debug"]];
    }
  else if ([itemIdentifier isEqualToString: @"Subprojects"])
    {
      [item setTag: SubprojectsTab];
      [item setToolTip: _(@"Subprojects Management")];
      [item setImage: [NSImage imageNamed: @"Tab_Subprojects"]];
    }

  return item;
}

- (NSArray*) toolbarAllowedItemIdentifiers: (NSToolbar*)toolbar
{
  return [self toolbarDefaultItemIdentifiers: toolbar];
}

- (NSArray*) toolbarDefaultItemIdentifiers: (NSToolbar*)toolbar
{
  if (itemIdentifiers == nil)
    {
      ASSIGN(itemIdentifiers, [[self class]
        toolbarItemIdentifiersForProjectCapabilities: [[[owner projectType]
        class] projectCapabilities]]);
    }

  return itemIdentifiers;
}

- (NSArray *) toolbarSelectableItemIdentifiers: (NSToolbar *)toolbar
{
  return [self toolbarDefaultItemIdentifiers: toolbar];
}

@end
