/*
    ProjectDocument.h

    Interface declaration of the ProjectDocument class for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import <AppKit/NSDocument.h>
#import <AppKit/NSMenu.h>

@class NSString, NSArray, NSMutableArray, NSMutableDictionary;

@class ProjectModule, ProjectWindowController;

@protocol ProjectType;

extern NSString * const ProjectFilesDidChangeNotification,
                * const ProjectSubprojectsDidChangeNotification,
                * const ProjectFrameworksDidChangeNotification;

@interface ProjectDocument : NSDocument <NSMenuValidation>
{
  NSString * projectDirectory;
  NSString * projectName;
  NSString * projectTypeID;
  id <ProjectType> projectType;

  NSMutableArray * files;
  NSMutableDictionary * subprojects;
  NSMutableArray * frameworks;

  ProjectModule * projectAttributes,
                * fileManager,
                * frameworksManager,
                * launcher,
                * builder,
                * debugger,
                * subprojectsManager;

  ProjectWindowController * wc;
  int projectCapabilities;
}

- (NSString *) projectName;
- (NSString *) projectDirectory;
- (void) setProjectName: (NSString *) projectName;

- (NSString *) projectTypeID;
- (id <ProjectType>) projectType;

- (ProjectModule *) projectAttributes;
- (ProjectModule *) fileManager;
- (ProjectModule *) frameworksManager;
- (ProjectModule *) builder;
- (ProjectModule *) launcher;
- (ProjectModule *) debugger;
- (ProjectModule *) subprojectsManager;

- (BOOL) openFile: (NSString *) fileName
       inCategory: (NSString *) category;

/* project file/category management */
- (NSArray *) subcategoriesInCategory: (NSString *) category;
- (NSArray *) filesInCategory: (NSString *) category;
- (NSArray *) allFilesUnderCategory: (NSString *) category;
- (NSArray *) allCategoriesUnderCategory: (NSString *) category;

- (BOOL) addFile: (NSString *) filePath
      toCategory: (NSString *) category
            link: (BOOL) linkFlag;
- (BOOL) addCategory: (NSString *) categoryName
          toCategory: (NSString *) category;

- (BOOL) removeFile: (NSString *) fileName
       fromCategory: (NSString *) category
             delete: (BOOL) flag;
- (BOOL) removeCategory: (NSString *) categoryName
           fromCategory: (NSString *) category
            deleteFiles: (BOOL) flag;

- (BOOL) renameFile: (NSString *) fileName
         inCategory: (NSString *) category
             toName: (NSString *) newName;
- (BOOL) renameCategory: (NSString *) oldCategoryName
             inCategory: (NSString *) category
                 toName: (NSString *) newCategoryName;

- (BOOL) copyFile: (NSString *) fileName
       inCategory: (NSString *) sourceCategory
       toCategory: (NSString *) destinationCategory;
- (BOOL) moveFile: (NSString *) fileName
       inCategory: (NSString *) sourceCategory
       toCategory: (NSString *) destinationCategory;
- (BOOL) linkFile: (NSString *) fileName
       inCategory: (NSString *) sourceCategory
       toCategory: (NSString *) destinationCategory;

- (NSImage *) iconForCategory: (NSString *) category;

- (BOOL) validateMenuItem: (id <NSMenuItem>) anItem;

// Subproject management
- (BOOL) addSubproject: (NSString *) projectPath;
- (BOOL) removeSubproject: (NSString *) subprojectName;
- (BOOL) openSubproject: (NSString *) subprojectName;
- (NSDictionary *) subprojects;

// Frameworks management
- (BOOL) addFramework: (NSString *) aFramework;
- (void) removeFramework: (NSString *) aFramework;
- (BOOL) renameFramework: (NSString *) oldName toName: (NSString *) newName;
- (NSArray *) frameworks;
- (NSArray *) fixedFrameworks;

- (BOOL) addFiles: (NSArray *) filenames
       toCategory: (NSString *) category
             link: (BOOL) linkFlag;

// menu actions
- (void) addFiles: sender;
- (void) newFileFromTemplate: sender;
- (void) deleteFiles: sender;

- (void) newCategory: sender;
- (void) deleteCategory: sender;

- (void) openSubprojectAction: sender;
- (void) addSubprojectAction: sender;
- (void) removeSubprojectAction: sender;
- (void) newSubprojectAction: sender;

@end
