/*
  libreiserfs - a library for manipulating reiserfs partitions
  Copyright (C) 2001-2004 Yury Umanets <torque@ukrpost.net>.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.
                                                                                                 
  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.
                                                                                                 
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>

#include <reiserfs/reiserfs.h>
#include <reiserfs/debug.h>

#define N_(String) (String)
#if ENABLE_NLS
# include <libintl.h>
# define _(String) dgettext (PACKAGE, String)
#else
# define _(String) (String)
#endif

#define get_st_blocks(size) ((size + 511) / 512)

static int reiserfs_tree_internal_insert(reiserfs_block_t *node, 
					 uint32_t pos, struct key *key,
					 reiserfs_disk_child_t *child);

static reiserfs_block_t *reiserfs_tree_node_alloc(reiserfs_tree_t *tree, 
						  uint32_t level) 
{
	blk_t blk;
	reiserfs_block_t *node;

	if (!(blk = reiserfs_fs_bitmap_find_free_block(tree->fs, 1))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Couldn't find free block."));
		return 0;
	}
	
	if (!(node = reiserfs_block_alloc(reiserfs_tree_dal(tree), blk, 0)))
		return 0;
	    
	set_blkh_level(GET_BLOCK_HEAD(node), level);
	set_blkh_nr_items(GET_BLOCK_HEAD(node), 0);
	
	set_blkh_free_space(GET_BLOCK_HEAD(node),
			    reiserfs_fs_block_size(tree->fs) - 
			    BLKH_SIZE);
	
	return node;
}

dal_t *reiserfs_tree_dal(reiserfs_tree_t *tree) {
    
	ASSERT(tree != NULL, return NULL);
	return tree->fs->host_dal;
}

blk_t reiserfs_tree_root(reiserfs_tree_t *tree) {
    
	ASSERT(tree != NULL, return 0);
	return get_sb_root_block(tree->fs->super);
}

void reiserfs_tree_set_root(reiserfs_tree_t *tree, blk_t root) {
    
	ASSERT(tree != NULL, return);
    
	set_sb_root_block(tree->fs->super, root);
	mark_super_dirty(tree->fs);
}

uint32_t reiserfs_tree_height(reiserfs_tree_t *tree) {

	ASSERT(tree != NULL, return 0);
	return get_sb_tree_height(tree->fs->super);
}

void reiserfs_tree_set_height(reiserfs_tree_t *tree, uint32_t height) {
    
	ASSERT(tree != NULL, return);
	ASSERT(height < MAX_HEIGHT, return);
    
	set_sb_tree_height(tree->fs->super, height);
	mark_super_dirty(tree->fs);
}

reiserfs_tree_t *reiserfs_tree_open(reiserfs_fs_t *fs) {
	reiserfs_tree_t *tree;
    
	ASSERT(fs != NULL, return NULL);
    
	if (!(tree = (reiserfs_tree_t *)libreiserfs_calloc(sizeof(*tree), 0)))
		return NULL;
	
	tree->fs = fs;
	return tree;
}

static inline void make_empty_direntry(reiserfs_de_head_t *deh, int format, 
				       uint32_t dirid, uint32_t objid,
				       uint32_t par_dirid, uint32_t par_objid)
{
	size_t dir_size;

	dir_size = (size_t)(format == FS_FORMAT_3_6 ?
			    EMPTY_DIR_V2_SIZE : EMPTY_DIR_V1_SIZE);
    
	memset(deh, 0, dir_size);
    
	/* Direntry header of "." */
	set_de_offset(&deh[0], DOT_OFFSET);
	set_de_dirid(&deh[0], dirid);
	set_de_objectid(&deh[0], objid);

	if (format == FS_FORMAT_3_6) {
		set_de_location(&deh[0], (EMPTY_DIR_V2_SIZE -
					  ROUND_UP(strlen ("."))));
	} else {
		set_de_location(&deh[0], (EMPTY_DIR_V1_SIZE - strlen(".")));
	}
	
	set_de_state(&deh[0], 0);
	mark_de_visible(&deh[0]);
  
	/* Direntry header of ".." */
	set_de_offset(&deh[1], DOT_DOT_OFFSET);
    
	/* Key of ".." for the directory */
	set_de_dirid(&deh[1], par_dirid);
	set_de_objectid(&deh[1], par_objid);
    
	if (format == FS_FORMAT_3_6) {
		set_de_location(&deh[1], (get_de_location(&deh[0]) -
					  ROUND_UP(strlen(".."))));
	} else {
		set_de_location(&deh[1], (get_de_location(&deh[0]) -
					  strlen("..")));
	}
	
	set_de_state(&deh[1], 0);
	mark_de_visible(&deh[1]);

	memcpy((void *)deh + get_de_location(&deh[0]), ".", 1);
	memcpy((void *)deh + get_de_location(&deh[1]), "..", 2);
}

static void make_empty_dir(void *body, int format, size_t blocksize, 
			   uint32_t dirid, uint32_t objid,
			   uint32_t par_dirid, uint32_t par_objid) 
{
	reiserfs_item_head_t *ih;
	reiserfs_sd_v1_t *sd_v1;
	reiserfs_sd_v2_t *sd_v2;

	ASSERT(body != NULL, return);
    
	/* First item is stat data item of the directory */
	ih = (reiserfs_item_head_t *)body;
	set_key_dirid(&ih->ih_key, dirid);
	set_key_objectid(&ih->ih_key, objid);

	if (format == FS_FORMAT_3_6) {
		set_ih_item_format(ih, ITEM_FORMAT_2);
		set_key_v2_offset(&ih->ih_key, SD_OFFSET);
		set_key_v2_type(&ih->ih_key, KEY_TYPE_SD);
	} else {
		set_ih_item_format(ih, ITEM_FORMAT_1);
		set_key_v1_offset(&ih->ih_key, SD_OFFSET);
		set_key_v1_type(&ih->ih_key, KEY_UNIQ_SD);
	}	
	set_ih_item_len(ih, (format == FS_FORMAT_3_6 ?  SD_V2_SIZE : SD_V1_SIZE));
	set_ih_item_location(ih, blocksize - (format == FS_FORMAT_3_6 ? SD_V2_SIZE : 
					      SD_V1_SIZE));

	set_ih_free_space(ih, MAX_US_INT);

	/* Fill new stat data */
	if (format == FS_FORMAT_3_6) {
		sd_v2 = (reiserfs_sd_v2_t *)(body + get_ih_item_location(ih) -
					     BLKH_SIZE);
		
		set_sd_v2_mode(sd_v2, S_IFDIR + 0755);
		set_sd_v2_nlink(sd_v2, 3);
		set_sd_v2_uid(sd_v2, getuid());
		set_sd_v2_gid(sd_v2, getgid());
		
		set_sd_v2_size(sd_v2, EMPTY_DIR_V2_SIZE);
		
		set_sd_v2_atime(sd_v2, time(NULL));
		set_sd_v2_ctime(sd_v2, time(NULL));
		set_sd_v2_mtime(sd_v2, time(NULL));
		
		set_sd_v2_blocks(sd_v2, get_st_blocks(EMPTY_DIR_V2_SIZE));
		set_sd_v2_rdev(sd_v2, 0);
	} else {
		sd_v1 = (reiserfs_sd_v1_t *)(body + get_ih_item_location(ih) -
					     BLKH_SIZE);
		
		set_sd_v1_mode(sd_v1, S_IFDIR + 0755);
		set_sd_v1_nlink(sd_v1, 3);
		set_sd_v1_uid(sd_v1, getuid());
		set_sd_v1_gid(sd_v1, getgid());
		
		set_sd_v1_size(sd_v1, EMPTY_DIR_V1_SIZE);
		
		set_sd_v1_atime(sd_v1, time(NULL));
		set_sd_v1_ctime(sd_v1, time(NULL));
		set_sd_v1_mtime(sd_v1, time(NULL));
		
		set_sd_v1_blocks(sd_v1, get_st_blocks(EMPTY_DIR_V1_SIZE));
	}

	/* Second item is directory item, containing "." and ".." */
	ih++;
	set_key_dirid(&ih->ih_key, dirid);
	set_key_objectid(&ih->ih_key, objid);
    
	if (format == FS_FORMAT_3_6) {
		set_ih_item_format(ih, ITEM_FORMAT_2);
		set_key_v2_offset(&ih->ih_key, DOT_OFFSET);
		set_key_v2_type(&ih->ih_key, KEY_TYPE_DR);
	} else {
		set_ih_item_format(ih, ITEM_FORMAT_1);
		set_key_v1_offset(&ih->ih_key, DOT_OFFSET);
		set_key_v1_type(&ih->ih_key, KEY_UNIQ_DR);
	}	
	
	set_ih_item_len(ih, (format == FS_FORMAT_3_6 ?
			     EMPTY_DIR_V2_SIZE : EMPTY_DIR_V1_SIZE));
	
	set_ih_item_location(ih, get_ih_item_location(ih - 1) -
			     get_ih_item_len(ih));
	
	set_ih_entry_count(ih, 2);

	/* Compose item */
	make_empty_direntry((reiserfs_de_head_t *)(body + get_ih_item_location(ih) - 
						   BLKH_SIZE), format, dirid,
			    objid, 0, dirid);
}

reiserfs_tree_t *reiserfs_tree_create(reiserfs_fs_t *fs) {
	int format;
	blk_t root_blk;
	size_t blocksize;
	reiserfs_tree_t *tree;
	reiserfs_block_t *root;
	reiserfs_block_head_t *blkh;
    
	ASSERT(fs != NULL, return NULL);
    
	if (!(tree = (reiserfs_tree_t *)libreiserfs_calloc(sizeof(*tree), 0)))
		return NULL;

	tree->fs = fs;
    
	if (!(root = reiserfs_tree_node_alloc(tree, 2)))
		goto error_free_tree;
    
	blocksize = get_sb_block_size(fs->super);
	format = get_sb_format(fs->super);
    
	/* Block head */
	blkh = (reiserfs_block_head_t *)root->data;
	set_blkh_level(blkh, LEAF_LEVEL);
	set_blkh_nr_items(blkh, 2);
        
	set_blkh_free_space(blkh, blocksize - BLKH_SIZE - 2 * IH_SIZE - 
			    (format == FS_FORMAT_3_6 ? SD_V2_SIZE : SD_V1_SIZE) - 
			    (format == FS_FORMAT_3_6 ? EMPTY_DIR_V2_SIZE : EMPTY_DIR_V1_SIZE));

	make_empty_dir(root->data + BLKH_SIZE, format, 
		       blocksize, ROOT_DIR_ID, ROOT_OBJ_ID, 0, ROOT_DIR_ID);
	
	if (!reiserfs_block_write(reiserfs_tree_dal(tree), root))
		goto error_free_root;
	
	root_blk = reiserfs_block_location(root);
	reiserfs_fs_bitmap_use_block(tree->fs, root_blk);
    
	reiserfs_object_use(fs, ROOT_DIR_ID);
	reiserfs_object_use(fs, ROOT_OBJ_ID);
    
	reiserfs_tree_set_height(tree, 2);
	reiserfs_tree_set_root(tree, root_blk);
    
	reiserfs_block_free(root);
	return tree;

error_free_root:
	reiserfs_block_free(root);    
error_free_tree:
	libreiserfs_free(tree);    
error:
	return NULL;    
}

static int reiserfs_tree_node_lookup(reiserfs_tree_t *tree, blk_t blk, 
				     reiserfs_comp_func_t comp_func,
				     struct key *key, int for_leaf, 
				     reiserfs_path_t *path) 
{
	reiserfs_block_t *node;
	uint32_t level, is_found = 0, pos = 0;
    
	ASSERT(tree != NULL, return 0);
	ASSERT(key != NULL, return 0);
    
	if (!comp_func)
		return 0;
    
	if (path)
		reiserfs_path_clear(path);
    
	while (1) {
	
		if (!(node = reiserfs_block_read(reiserfs_tree_dal(tree), blk)))
			reiserfs_block_reading_failed(blk, return 0);
		
		if ((level = get_blkh_level((reiserfs_block_head_t *)node->data)) > 
		    (uint32_t)reiserfs_tree_height(tree) - 1)
		{
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
						    _("Invalid node level. Found %d, "
						      "expected less than %d."), 
						    level, reiserfs_tree_height(tree));
			return 0;
		}
		
		if (!for_leaf && is_leaf_node(node))
			return 0;
			
		is_found = reiserfs_tools_fast_search(key, GET_ITEM_HEAD(node, 0), 
						      get_blkh_nr_items(GET_BLOCK_HEAD(node)),
						      (is_leaf_node(node) ? IH_SIZE : FULL_KEY_SIZE),
						      comp_func, &pos);
		
		if (path) {
			unsigned int ppos = is_found && is_internal_node(node) ?
				pos + 1 : pos;
			
			reiserfs_path_node_t *pnode = reiserfs_path_node_create(
				reiserfs_path_last(path), node, ppos);
				
			reiserfs_path_inc(path, pnode);
		}
		
		if (is_leaf_node(node))
			return is_found;
			
		if (level == 2 && !for_leaf)
			return 1;
			
		if (is_found)
			pos++;
		
		blk = get_dc_child_blocknr(GET_DISK_CHILD(node, pos));
	}
	
	return 0;
}

reiserfs_path_node_t *reiserfs_tree_lookup_internal(reiserfs_tree_t *tree, blk_t from,
						    reiserfs_comp_func_t comp_func,
						    struct key *key, reiserfs_path_t *path) 
{
	if (tree && reiserfs_tree_height(tree) < 2)
		return NULL;

	return (reiserfs_tree_node_lookup(tree, from, comp_func, key, 0, path) ? 
		reiserfs_path_last(path) : NULL);
}

reiserfs_path_node_t *reiserfs_tree_lookup_leaf(reiserfs_tree_t *tree, blk_t from,
						reiserfs_comp_func_t comp_func,
						struct key *key, reiserfs_path_t *path) 
{
	if (tree && reiserfs_tree_height(tree) < 2)
		return NULL;
    
	return (reiserfs_tree_node_lookup(tree, from, comp_func, key, 1, path) ? 
		reiserfs_path_last(path) : NULL);
}

static long reiserfs_tree_node_traverse(reiserfs_tree_t *tree, blk_t blk, void *data,
					reiserfs_edge_traverse_func_t before_node_func,
					reiserfs_node_func_t node_func,
					reiserfs_chld_func_t chld_func,
					reiserfs_edge_traverse_func_t after_node_func)
{
	uint32_t block_level;
	long call_result = 0;
	reiserfs_block_t *node;

	ASSERT(node_func != NULL, return 0);

	if (!(node = reiserfs_block_read(reiserfs_tree_dal(tree), blk)))
		reiserfs_block_writing_failed(blk, goto error);
    
	if (!is_leaf_node(node) && !is_internal_node(node)) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Invalid node detected (%lu). "
					      "Unknown type."), blk);
		goto error_free_node;
	}
    
	if (before_node_func && !(call_result = before_node_func(node, data)))
		goto error_free_node;
	
	if (!(call_result = node_func(node, data)))
		goto error_free_node;
	
	if (is_internal_node(node)) {
		uint32_t i;
		
		for (i = 0; i <= get_blkh_nr_items(GET_BLOCK_HEAD(node)); i++) {
			blk_t chld_blk = get_dc_child_blocknr(GET_DISK_CHILD(node, i));
			
			if (!(call_result = reiserfs_tree_node_traverse(tree, chld_blk, 
									data, before_node_func,
									node_func, chld_func,
									after_node_func)))
			{
				goto error_free_node;
			}
			
			if (chld_func && !chld_func(node, i, call_result, data)) 
				goto error_free_node;
		}
	}

	if (after_node_func && !(call_result = after_node_func(node, data)))
		goto error_free_node;
	
	reiserfs_block_free(node);
	return call_result;
    
error_free_node:
	reiserfs_block_free(node);    
error:
	return call_result;
}

long reiserfs_tree_simple_traverse(reiserfs_tree_t *tree, void *data,
				   reiserfs_node_func_t node_func)
{
	if (reiserfs_tree_root(tree) < 2)
		return 1;
    
	return reiserfs_tree_node_traverse(tree, reiserfs_tree_root(tree), data, 
					   NULL, node_func, NULL, NULL);
}
    
long reiserfs_tree_traverse(reiserfs_tree_t *tree, void *data,
			    reiserfs_edge_traverse_func_t before_node_func,
			    reiserfs_node_func_t node_func, 
			    reiserfs_chld_func_t chld_func,
			    reiserfs_edge_traverse_func_t after_node_func)
{
	if (reiserfs_tree_height(tree) < 2)
		return 1;
	
	return reiserfs_tree_node_traverse(tree, reiserfs_tree_root(tree), data,
					   before_node_func, node_func, chld_func,
					   after_node_func);
}


void reiserfs_tree_free(reiserfs_tree_t *tree) {
	if (!tree)
		return;
    	libreiserfs_free(tree);
}

struct check_hint {
	reiserfs_tree_t *tree;
	reiserfs_gauge_t *gauge;
};

typedef struct check_hint check_hint_t;

static int tree_check_blknr(reiserfs_tree_t *tree, uint32_t blk) {
	reiserfs_bitmap_t *bitmap = tree->fs->bitmap;
	
	blk_t first_bm = (DEFAULT_SUPER_OFFSET / 
			dal_block_size(tree->fs->host_dal)) + 1;
	
	if (blk <= first_bm) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Tree cannot contain nodes in "
					      "range 0-%llu."), first_bm);
		return 0;
	}
	
	if (!reiserfs_bitmap_test_block(bitmap, blk)) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Node %llu is not marked used in "
					      "block allocator bitmap."), blk);
		return 0;
	}

	return 1;
}

static long tree_check_node_func(reiserfs_block_t *node, void *data) {
	check_hint_t *hint = (check_hint_t *)data;

	reiserfs_tree_t *tree = hint->tree;
	reiserfs_gauge_t *gauge = hint->gauge;
	blk_t blk = reiserfs_block_location(node);
	reiserfs_bitmap_t *bitmap = tree->fs->bitmap;

	if (!tree_check_blknr(tree, blk))
		return 0;

	if (is_internal_node(node)) {
		/* here we do nothing, as internal nodes will be check
		 * automatically by traverse function. */
	} else if (is_leaf_node(node)) {
		uint32_t i;
		reiserfs_item_head_t *item;

		for (i = 0; i < get_blkh_nr_items(GET_BLOCK_HEAD(node)); i++) {
			item = GET_ITEM_HEAD(node, i);

			if (is_indirect_ih(item)) {
				uint32_t j, *blocks;

				blocks = (uint32_t *)GET_ITEM_BODY(node, item);

				for (j = 0; j < IH_UNFM_NUM(item); j++) {
					uint32_t ind_blk = blocks[i];
					
					if (!tree_check_blknr(tree, ind_blk))
						return 0;
				}
			} else if (is_direct_ih(item)) {
			} else if (is_direntry_ih(item)) {
			} else if (is_stat_data_ih(item)) {
			} else {
				libreiserfs_exception_throw(EXCEPTION_ERROR,
							    EXCEPTION_CANCEL, 
							    _("Invalid item type "
							      "has beem detected "
							      "in node %llu."), blk);
				return 0;
			}
		}
	} else {		
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Invalid node type has beed "
					      "detected. Node %llu."), blk);
		return 0;
	}
		
	if (gauge)
		libreiserfs_gauge_touch(gauge);
	
	return 1;
}

long reiserfs_tree_check(reiserfs_tree_t *tree, reiserfs_gauge_t *gauge) {
	long res;

	check_hint_t hint = {
		.tree = tree,
		.gauge = gauge
	};

	if (gauge) {
		libreiserfs_gauge_reset(gauge);
		libreiserfs_gauge_set_name(gauge, _("checking"));
	}
	
	res = reiserfs_tree_simple_traverse(tree, &hint, 
					    tree_check_node_func);

	if (gauge)
		libreiserfs_gauge_done(gauge);

	return res;
}
