/*
  libreiserfs - a library for manipulating reiserfs partitions
  Copyright (C) 2001-2004 Yury Umanets <torque@ukrpost.net>.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.
                                                                                                 
  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.
                                                                                                 
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <dal/file_dal.h>
#include <reiserfs/reiserfs.h>
#include <progs/tools.h>

int main(int argc, char *argv[]) {
	dal_t *dal;
	long long pos = 0;
	unsigned long long buffer_size = 0;
	reiserfs_fs_t *fs;
	reiserfs_file_t *file;

	uint64_t readed;
	char *buffer;

	if (argc < 3) {
		fprintf(stderr, "Usage: %s DEV FILE [ pos ] \n", argv[0]);
		return 0xff;
	}

	if (!(dal = file_dal_open(argv[1], DEFAULT_BLOCK_SIZE, O_RDONLY))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    "Couldn't create device abstraction "
					    "for %s.", argv[1]);
		return 0xfe;    
	}
    
	if (!(fs = reiserfs_fs_open_fast(dal, dal))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    "Couldn't open filesystem on %s.", argv[1]);
		goto error_free_dal;
	}
    
	if (!(file = reiserfs_file_open(fs, argv[2], O_RDONLY))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    "Couldn't open file %s.", argv[2]); 
		goto error_free_fs;
	}
    
	if (argc == 4) {
		if ((pos = progs_str2long(argv[3], -1)) == -1) {
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
						    "Invalid position %s.", argv[3]); 
			goto error_free_file;
		}
	}
	
	if (!reiserfs_file_seek(file, pos)) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    "Couldn't seek file at %llu position.", pos); 
		goto error_free_file;
	}
    
	buffer_size = reiserfs_file_size(file) - reiserfs_file_offset(file);
    
	if (!(buffer = libreiserfs_malloc(buffer_size + 1)))
		goto error_free_file;
	
	memset(buffer, 0, buffer_size + 1);
	readed = reiserfs_file_read(file, buffer, buffer_size);
    
	if (readed != buffer_size) {
		libreiserfs_exception_throw(EXCEPTION_WARNING, EXCEPTION_IGNORE, 
					    "Couldn't read %llu bytes from file %s. "
					    "Readed %llu bytes.", reiserfs_file_size(file),
					    argv[2], readed); 
	}

	fwrite(buffer, readed, 1, stdout);
	fflush(stdout);
	
	libreiserfs_free(buffer);
	reiserfs_file_close(file);
    
	reiserfs_fs_close(fs);
	file_dal_close(dal);
    
	return 0;

error_free_buffer:
	libreiserfs_free(buffer);
error_free_file:
	reiserfs_file_close(file);    
error_free_fs:
	reiserfs_fs_close(fs);
error_free_dal:
	file_dal_close(dal);
error:
	return 0xff;    
}

