/*****
*
* Copyright (C) 2001, 2002 Jeremie Brebec <flagg@ifrance.com>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Jeremie Brebec <flagg@ifrance.com>
*
*****/


#include <stdio.h>
#include <stdlib.h>

#include <libprelude/prelude-log.h>

#include "rules.h"
#include "rules-type.h"
#include "rules-operations.h"




/*
 * Copy a variable.
 */
var_t *copy_var(var_t *dst, const var_t *src)
{
	generic_type_t *type;

	dst->id = src->id;

	/* don't copy variable if this node is a leaf */
	if ( src->id == ID_LEAF_NODE || src->id == ID_IGNORE_NODE ) 
		dst->set = NULL;
	else {
		type = signature_engine_get_type_by_id(src->id);
		if ( ! type )
			return NULL;

		dst->set = type->copy(src->set);
	}

	return dst;
}




/*
 * create rules
 */
rules_t *make_new_rules(rule_t *rule, rules_t *next)
{
	rules_t *new;

        new = malloc(sizeof(rules_t));
	if ( ! new ) {
		log(LOG_ERR, "memory exhausted.\n");
		return NULL;
	}
        
	new->rule = rule;
	new->next = next;

	return new;
}




/*
 * create a rule
 */
rule_t *make_new_rule(int id, void *set)
{
	rule_t *new;

        new = malloc(sizeof(rule_t));
	if ( ! new ) {
		log(LOG_ERR, "memory exhausted.\n");
		return NULL;
	}

	if ( set ) {
		new->test = malloc(sizeof(test_t));
		if ( ! new->test ) {
                        log(LOG_ERR, "memory exhausted.\n");
			return NULL;
		}
		
		new->test->var.id = id;
		new->test->var.set = set;
		new->test->next = NULL;
		new->test->inversed = 0;
	} else
		new->test = NULL;

	new->data = NULL;
	new->leaf_match = NULL;

	return new;
}



unsigned int signature_engine_get_new_data_id(void) 
{
        static unsigned int data_id = 0;

        return data_id++;
}



/*
 * create an empty rule with a data
 */
rule_t *make_data_rule(int id, void *data)
{
	rule_t *rule;

        rule = malloc(sizeof(rule_t));
	if ( ! rule ) {
		log(LOG_ERR, "memory exhausted.\n");
		return NULL;
	}

	rule->test = NULL;
	rule->data = NULL;
	rule->leaf_match = NULL;
	
	add_rule_data(rule, id, data);

	return rule;
}




/*
 * Delete a rule
 */
void delete_rule(rule_t *rule)
{
	test_t *test, *next;
        generic_type_t *type;
        
	if ( ! rule )
		return;

	/*
         * delete test
         */
	test = rule->test;
	while ( test ) {                
                type = signature_engine_get_type_by_id(test->var.id);
		if ( type )
			type->delete(test->var.set);

                next = test->next;
		free(test);
		test = next;
	}

	/*
         * delete rule
         */
	free(rule);
}
		



void delete_rules(rules_t *rules)
{
	if ( ! rules )
		return;

	delete_rule(rules->rule);
	delete_rules(rules->next);
	free(rules);
}




static l_leaf_match_t *copy_leaf(l_leaf_match_t *leaf_match) 
{
        l_leaf_match_t *tmp;
        l_leaf_match_t **last_leaf, *current_leaf;

        last_leaf = &leaf_match;
        
        for ( current_leaf = leaf_match; current_leaf != NULL;
              current_leaf = current_leaf->next ) {

                tmp = malloc(sizeof(*tmp));
		if ( ! tmp ) {
                        log(LOG_ERR, "memory exhausted.\n");
			return NULL;
		}

                tmp->var.id = current_leaf->var.id;
                tmp->var.set = current_leaf->var.set;
                
                tmp->inversed = current_leaf->inversed;
                tmp->leaf_match = current_leaf->leaf_match;

                *last_leaf = tmp;
                last_leaf = &tmp->next;
        }

        *last_leaf = NULL;
        
        return leaf_match;
}




/*
 * copy a rule
 */
rule_t *copy_rule(rule_t *rule)
{
	rule_t *new_rule;
	test_t **last_test, *current_test;

	new_rule = malloc(sizeof(rule_t));
	if (! new_rule) {
                log(LOG_ERR, "memory exhausted.\n");
		return NULL;
	}

	/*
	 * warning: "data", and "leaf_match" are linked together, not copied...
	 * data are shared by all (tree, rules, rule)
	 */
	new_rule->data = rule->data;
	new_rule->leaf_match = copy_leaf(rule->leaf_match);

	/*
	 * copy tests
	 */
	last_test = &new_rule->test;

	for ( current_test = rule->test; current_test != NULL;
              current_test = current_test->next ) {
		test_t *tmp;

                tmp = malloc(sizeof(test_t));
		if ( ! tmp ) {
                        log(LOG_ERR, "memory exhausted.\n");
			return NULL;
		}

		copy_var(&tmp->var, &current_test->var);
		tmp->inversed = current_test->inversed;
		*last_test = tmp; /* link the last node to tmp */
		last_test = &tmp->next;
	}
        
	*last_test = NULL;

	return new_rule;
}




/*
 * copy rules
 */
rules_t *copy_rules(rules_t *rules)
{
	if ( ! rules )
		return NULL;
	else
		return make_new_rules(copy_rule(rules->rule), copy_rules(rules->next));
} 




/*
 * "or" operation on a rule
 * warning: rules & rule are joined and should not be used after this operation
 */
rules_t *rule_or(rules_t *rules, rule_t *rule)
{
	return make_new_rules(rule, rules);
}




/*
 * "or" operation on rules
 * warning: rules1 & rules2 are joined and should not be used after this operation
 */
rules_t *rules_or(rules_t *rules1, rules_t *rules2)
{
	if ( ! rules1 )
		return rules2;

	if ( rules1 == rules2 )
		return rules1;

	/* find the last rule with a recursive call */
	if ( rules1->next != NULL ) 
	        rules_or(rules1->next, rules2);
	else 
		rules1->next = rules2;
		
	return rules1;
}




/*
 * "and" operation on a rule
 * warning: rules & rule are joined and should not be used after this operation
 */
rules_t *rule_and(rules_t *rules, rule_t *rule)
{
	test_t **test;
	data_t **data;
	l_leaf_match_t **leaf_match;

	if ( ! rules )
		return NULL;

	if ( ! rule )
		return rules;

	if ( ! rules->rule ) {
		rules->rule = rule;
		return rules;
	}

	/* find the end of rules->rule->test 
	 * and add the test of the new rule */
	test = &(rules->rule->test);
	while ( *test != NULL )
		test = & (*test)->next;
	*test = rule->test;

	/* try to link data and leaf_match, avoid circular definition */
	data = &rules->rule->data;
	while ( *data ) {
		data_t *data_tmp = rule->data;

                while ( data_tmp && data_tmp!= *data ) 
			data_tmp = data_tmp->next;

                if ( data_tmp )
			break;
                
		data = & (*data)->next;
	}
        
	if (! *data)
		*data = rule->data;

	leaf_match = &rules->rule->leaf_match;

	while ( *leaf_match ) {
		l_leaf_match_t *leaf_match_tmp = rule->leaf_match;

                while ( leaf_match_tmp && leaf_match_tmp != *leaf_match )
			leaf_match_tmp = leaf_match_tmp->next;
                
		if ( leaf_match_tmp )
			break;
                
		leaf_match = & (*leaf_match)->next;
	}
        
	if (! *leaf_match)
		*leaf_match = rule->leaf_match;

	/* recursive call with a copy of the rule */
	if (rules->next)
		rule_and(rules->next, copy_rule(rule));

	free(rule);

	return rules;
}




/*
 * "and" operation on rules
 * warning: rules & rule are joined and should not be used after this operation
 */
rules_t *rules_and(rules_t *rules1, rules_t*rules2)
{
	rules_t *rules;

	if ( ! rules1 ) 
		return rules2;

	if ( ! rules2 )
		return rules1;

	/* (a or b) and (c or d) = ( a and (c or d)) or (b and (c or d)) */
        rules = rules2->next ? copy_rules(rules1) : rules1;

        return rules_or(rule_and(rules, rules2->rule), 
                        rules_and(rules1, rules2->next));
} 




/* 
 * not operation on a rule
 */
rules_t *rule_not(rule_t *rule)
{
	test_t *test;
        rules_t *rules;
        l_leaf_match_t *leaf_match;
	
	if ( ! rule )
		return NULL;

	rules = NULL;

	test = rule->test;
	while ( test ) {
                test_t *next = test->next;
		rule_t *new_rule = make_new_rule(0, NULL);

		new_rule->test = test;
		new_rule->test->inversed = (test->inversed ? 0 : 1);
		new_rule->test->next = NULL;
		new_rule->data = rule->data;

		rules = make_new_rules(new_rule, rules);
		test = next;
	}

	leaf_match = rule->leaf_match;
	while ( leaf_match ) {
		l_leaf_match_t *next = leaf_match->next;
                rule_t *new_rule = make_new_rule(0, NULL);

		new_rule->leaf_match = leaf_match;
		new_rule->leaf_match->next = NULL;
		new_rule->leaf_match->inversed = (leaf_match->inversed ? 0 : 1);
		new_rule->data = rule->data;

		rules = make_new_rules(new_rule, rules);
		leaf_match = next;
	}

	free(rule);

	return rules;
}




/*
 * not operation on rules
 */
rules_t *rules_not(rules_t *rules)
{
	if ( ! rules )
		return NULL;

	return rules_and(rule_not(rules->rule), rules_not(rules->next));
}




/*
 *
 */
int add_rule_leaf_match(rule_t *rule, int rid, void *rdata, leaf_match_f_t leaf_match)
{
	l_leaf_match_t *leaf_match_tmp;

	if ( ! rule )
		return 0;

	leaf_match_tmp = malloc(sizeof(l_leaf_match_t));
	if (! leaf_match_tmp) {
                log(LOG_ERR, "memory exhausted.\n");
		return -1;
	}

	leaf_match_tmp->leaf_match = leaf_match;
	leaf_match_tmp->inversed = 0;
	leaf_match_tmp->next = rule->leaf_match;
	leaf_match_tmp->var.id = rid;
	leaf_match_tmp->var.set = rdata;
	rule->leaf_match = leaf_match_tmp;
#if 0
	if ( ! rule->leaf_match )
		rule->leaf_match = leaf_match_tmp;
	else
		add_leaf_match_by_id(&rule->leaf_match, leaf_match_tmp);
#endif
	return 0;
}




/*
 * add data to a rule
 */
int add_rule_data(rule_t *rule, int id, void *data)
{
	data_t *data_tmp;

	if (! rule)
		return -1;

	data_tmp = malloc( sizeof(data_t) );
	if (! data_tmp) {
		log(LOG_ERR, "memory exhausted.\n");
		return -1;
	}

	data_tmp->data = data;
	data_tmp->id = id;

	data_tmp->next = rule->data;
	rule->data = data_tmp;

	return 0;
}



/*
 * get data by id
 */
void *signature_engine_get_data_by_id(data_t *data, int id)
{
	while ( data ) {
		if (data->id == id)
			return data->data;
                
		data = data->next;
	}
        
	return NULL;
}




/*
 * get data from a rule by id
 */
void *signature_engine_get_rule_data(rule_t *rule, int id)
{
	data_t *data_tmp;

	if (! rule)
		return NULL;

	data_tmp = rule->data;
	while ( data_tmp ) {
		if (data_tmp->id == id)
			return data_tmp->data;
		
		data_tmp = data_tmp->next;
	}

	return NULL;
}




/*
 * validate_rule
 *
 * sort rule by priority
 */
static int _r_validate_test(test_t **test)
{
	generic_type_t *type;
	test_t *current, **last, **last_min;
	int min_priority = 99999; /* +oo (todo?) */

	current = *test;
	last = test;
	last_min = test;

	if ( ! current )
		return 0;

	while (current) {

		type = signature_engine_get_type_by_id(current->var.id);
		if (! type) 
			return -1; /* unknow type */

		/* sort the rule */
		if (type->priority < min_priority) {
			last_min = last;
			min_priority = type->priority;
		}

		last = &current->next;
		current = current->next;
	}

	current = *last_min;

	*last_min = (*last_min)->next;
	current->next = *test;
	*test = current;

	/* recursive call with the n+1 elements */
	return _r_validate_test(&current->next);
}




int validate_rule(rule_t *rule)
{
	return (rule == NULL) ? -1 : _r_validate_test(&rule->test);
}




int validate_rules(rules_t *rules)
{
	if ( ! rules )
		return 0;

	if ( validate_rule(rules->rule) < 0 )
		return -1;
	else
		return validate_rules(rules->next);
}




/*
 * check rule integrity
 */
int check_rule_integrity(rule_t *rule)
{
	test_t *test;
	generic_type_t *type;
	int c_priority = -1;

	if ( ! rule ) 
		return -1;

	/* check if the rule is sorted by priority */
	for ( test = rule->test; test != NULL; test = test->next ) {

		type = signature_engine_get_type_by_id(test->var.id);
		if (type->priority < c_priority)
			return -1;

		c_priority = type->priority;
	}

	return 0;
}




/*
 * copy nodes
 */
rules_node_t *copy_rules_node(rules_node_t *node)
{
	rules_node_t *new_node;

	if (! node)
		return NULL;

	new_node = malloc(sizeof(rules_node_t));
	if (! new_node) {
		log(LOG_ERR, "memory exhausted.\n");
		return NULL;
	}

	new_node->ref = 0;

	/*
	 * copy node var
	 */
	copy_var(&new_node->var, &node->var);

	/*
	 * copy node run functions
	 */
	/* only link them */

	new_node->run = node->run;

	/*
	 * copy match_packet
	 */
	new_node->match_packet = node->match_packet;

	/*
	 * copy brother & child
	 */

	new_node->brother = node->brother;
	if (node->brother) node->brother->ref++;

	new_node->child = node->child;
	if (node->child) node->child->ref++;

	return new_node;

}




/*
 * delete nodes
 * data, run and leaf_match cant be deleted
 */
void delete_rules_node(rules_node_t *node)
{
	generic_type_t *type;

	if (! node)
		return;

	if (--node->ref)
		return;

	/*
	 * delete node var
	 */
	type = signature_engine_get_type_by_id(node->var.id);
	if ( type )
		type->delete(node->var.set);

	/*
	 * delete brother and child
	 */
	delete_rules_node(node->brother);
	delete_rules_node(node->child);

	free(node);
}




/*
 * make an empty tree
 */
rules_node_t *make_rules_root(void)
{
	rules_node_t *root;

	root = malloc(sizeof(rules_node_t));
	if (! root) {
		log(LOG_ERR, "memory exhausted.\n");
		return NULL;
	}

	root->var.id = ID_ROOT_NODE;
	root->brother = NULL;
	root->child = NULL;
	root->ref = 1;

	return root;
}
