/*****
*
* Copyright (C) 1998, 1999, 2000, 2002 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <limits.h> /* for NAME_MAX */
#include <sys/time.h>

#include "packet.h"

#include <libprelude/common.h>
#include <libprelude/list.h>
#include <libprelude/prelude-log.h>
#include <libprelude/plugin-common.h>
#include <libprelude/plugin-common-prv.h>

#include "plugin-protocol.h"
#include "plugin-detect.h"

#define HASH_SIZE 256

typedef struct port {
        uint16_t port;
        struct port *next;
} port_t;


struct port_list {
        port_t *port_list[HASH_SIZE];
};


static LIST_HEAD(protocol_plugins);



/*
 * Subscribe plugins to it's specified protocol,
 * if the plugin need to subscribe to > 1 protocol :
 * we need to copy it.
 */
static int subscribe(plugin_container_t *pc)
{       
        plugin_protocol_t *plugin = (plugin_protocol_t *) pc->plugin;
        
        log(LOG_INFO, "- %s subscribed for \"%s\" protocol handling.\n", plugin->name, plugin->proto);
        
        INIT_LIST_HEAD(&plugin->detect_plugin_list);
        return plugin_add(pc, &protocol_plugins, plugin->proto);
}




static void unsubscribe(plugin_container_t *pc) 
{
        plugin_protocol_t *plugin = (plugin_protocol_t *) pc->plugin;
        
        log(LOG_INFO, "- %s un-subscribed from \"%s\" protocol handling.\n", plugin->name, plugin->proto);
        plugin_del(pc);
}





/*
 * Start all Protocol Plugins
 */
int protocol_plugins_run(packet_container_t *packet, unsigned char *data, int len)
{
        int ret = 0;
        struct list_head *tmp;
        plugin_container_t *pc;
         
        list_for_each(tmp, &protocol_plugins) {
                pc = list_entry(tmp, plugin_container_t, ext_list);
                
                plugin_run_with_return_value(pc, plugin_protocol_t, run, ret, packet, data, len);
                if ( ret > 0 ) {
                        detect_plugins_run(packet, &((plugin_protocol_t *)pc->plugin)->detect_plugin_list, NULL);
                        break;
                }
        }

        return ret;
}



plugin_protocol_t *protocol_plugins_search(const char *proto) 
{
        struct list_head *tmp;
        plugin_container_t *pc;
        plugin_protocol_t *plugin;
        
        list_for_each(tmp, &protocol_plugins) {         
                pc = list_entry(tmp, plugin_container_t, ext_list);

                plugin = (plugin_protocol_t *) pc->plugin;
                
                if ( strcmp(plugin->proto, proto) == 0 )
                        return (plugin_protocol_t *) pc->plugin;
        }

        return NULL;
}




/*
 * Open the plugin directory (dirname),
 * and try to load all plugins located int it.
 */
int protocol_plugins_init(const char *dirname, int argc, char **argv)
{
        int ret;

	ret = access(dirname, F_OK);
	if ( ret < 0 ) {
		if ( errno == ENOENT )
			return 0;
		log(LOG_ERR, "can't access %s.\n", dirname);
		return -1;
	}

        ret = plugin_load_from_dir(dirname, argc, argv, subscribe, unsubscribe);
        if ( ret < 0 ) {
                log(LOG_ERR, "couldn't load plugin subsystem.\n");
                return -1;
        }

        return ret;
}




/*
 *
 *
 */
static inline int get_port_key(uint16_t dport) 
{
        return dport % HASH_SIZE;
}




int protocol_plugin_is_port_ok(port_list_t *plist, uint16_t dport) 
{
        port_t *port;

        port = plist->port_list[get_port_key(dport)];
        while ( port ) {
                if ( port->port == dport ) 
                        return 0;

                port = port->next;
        }

        return -1;
}
	



int protocol_plugin_add_port_to_list(port_list_t *plist, uint16_t port)
{
        port_t *new;
        int ret, key;
        
        ret = protocol_plugin_is_port_ok(plist, port);
        if ( ret == 0 )
                /*
                 * port already in list.
                 */
                return 0;

        new = malloc(sizeof(*new));
        if ( ! new ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        key = get_port_key(port);
        
        new->port = port;
        new->next = plist->port_list[key];
        plist->port_list[key] = new;
        	
	return 0;
}



int protocol_plugin_add_string_port_to_list(port_list_t *plist, const char *pstring) 
{
	int ret = -1;
	char *ptr, *s;

	s = ptr = strdup(pstring);
	
        while ( (ptr = strtok(ptr, " ")) ) {

                ret = protocol_plugin_add_port_to_list(plist, atoi(ptr));
		if ( ret < 0 ) 
			break;
		
		ptr = NULL;
        }

        free(s);

        return ret;
}



port_list_t *protocol_plugin_port_list_new(void) 
{
        return calloc(1, sizeof(port_list_t));
}



void protocol_plugin_port_list_free(port_list_t *plist)
{
        int i;
        port_t *tmp, *next;
        
        for ( i = 0; i < HASH_SIZE; i++ ) {

                for ( tmp = plist->port_list[i]; tmp != NULL; tmp = next ) {
                        next = tmp->next;
                        free(tmp);
                }
        }
        
        free(plist);
}






