/*****
*
* Copyright (C) 1998 - 2003 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/poll.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <inttypes.h>

#include "packet.h"

#include <libprelude/common.h>
#include <libprelude/prelude-log.h>
#include <libprelude/timer.h>
#include <libprelude/prelude-list.h>
#include <libprelude/prelude-io.h>
#include <libprelude/prelude-message.h>
#include <libprelude/prelude-client.h>
#include <libprelude/prelude-client-mgr.h>
#include <libprelude/idmef-tree.h>
#include <libprelude/sensor.h>


/* 
 * This is needed to decide if we should enable a workaround for a poll()
 * bug present on FreeBSD 4.x.
 *
 * See comment in do_capture_from_multiple_devices() later in this file
 * for more information.
 */
#ifdef __FreeBSD__
 #include <osreldate.h>
#endif


/*
 * NetBSD doesn't define ENOTSUP.
 */
#ifndef ENOTSUP
 #define ENOTSUP EOPNOTSUPP
#endif

#include "pcap.h"
#include "bpf/net/bpf.h" /* pcap linktype definition */

#include "pconfig.h"
#include "capture.h"
#include "packet-decode.h"


#define PCAP_READ_TIMEOUT 500



typedef struct {
        char *bpf;
        pcap_t *pd;
        const char *interface;
        bpf_u_int32 netmask;
        pcap_handler p_handler;
        struct bpf_program *fcode;
        pcap_dumper_t *file_dumper; 
        unsigned int packet_counter;
} capdev_t;


static int interfaces_nbr = 0;
static capdev_t **devices = NULL;
static struct timeval capstart, capend;
static struct rusage rusage_before_capture;
                                      

static void read_packet(u_char *user, const struct pcap_pkthdr *h, const u_char *p) 
{
        capdev_t *dev = (capdev_t *) user;
        
        /*
         * We use our own packet counter to workaround a pcap bug
         * (which doesn't count packet when reading from a file - which confuse
         * the user).
         */
        dev->packet_counter++;

        /*
         * In case we capture to a file,
         * pcap store the file pointer as userdata. we store this pointer
         * in dev->dumper. We'll pass it even if we're not capturing to a file
         * cause we don't use userdata ourselve.
         */
        dev->p_handler((unsigned char *)dev->file_dumper, h, p);
}




/*
 * Search for the datalink handler corresponding
 * to the datalink of type 'type'.
 *
 * Return NULL if no handler was found.
 */
static pcap_handler search_datalink_handler(int type)
{
        int i;
        struct datalink_handler {
                void (*handler)(u_char *user, struct pcap_pkthdr *h, u_char *p);
                int type;
        } p[] = {
                {       capture_null,     DLT_NULL      },
                {	capture_ethernet, DLT_EN10MB	},
                {	capture_slip,     DLT_SLIP      },
                {       capture_slip,    DLT_SLIP_BSDOS },
                {       capture_ppp,      DLT_PPP       },
                {       capture_fddi,     DLT_FDDI      },
                {       capture_raw,      DLT_RAW       },
                {       capture_null,     DLT_LOOP      },
                {       capture_sll,      DLT_LINUX_SLL },
                
#if defined(DLT_CIP) && (DLT_CIP == 16)
                {       capture_clip,       16          },
#endif
                {       capture_clip,       18          },
                {       capture_clip,       19          },
                {       NULL, 0                         }, 
        };
        
        for ( i = 0; p[i].handler; i++ ) {
                if ( type == p[i].type ) 
                        return (pcap_handler) p[i].handler;
        }
        
        log(LOG_ERR, "Unknow datalink type %x.\n", type);
        
        return NULL;
        
}



/*
 * Setup bpf rule 'bpf' for device identified by 'dev'.
 */
static int setup_bpf(capdev_t *dev, const char *bpf)
{
        int ret;
        
        if ( dev->fcode ) {
                /*
                 * an old BPF rule is already set.
                 */
                pcap_freecode(dev->fcode);
                free(dev->fcode);
                dev->fcode = NULL;
        }
        
        if ( ! bpf ) 
                return 0;

        dev->fcode = malloc(sizeof(*dev->fcode));
        if ( ! dev->fcode ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        ret = pcap_compile(dev->pd, dev->fcode, bpf, 1, dev->netmask);
	if ( ret < 0 ) {
                /*
                 * we return 0 here because PCAP won't support BPF on
                 * some link layer.
                 */
                log(LOG_ERR, "pcap: %s.\n", pcap_geterr(dev->pd));
                free(dev->fcode); dev->fcode = NULL;
                return 0;
	}
        
	ret = pcap_setfilter(dev->pd, dev->fcode);
        if ( ret < 0 ) {
		log(LOG_ERR, "pcap: %s.\n", pcap_geterr(dev->pd));
                free(dev->fcode); dev->fcode = NULL;
                return -1;
	}
        
	return 0;
}



static void *pktalloc(size_t size) 
{        
        return malloc(size);
}



static void pktfree(void *ptr) 
{
        free(ptr);
}



#if defined(__FreeBSD__) && defined(BIOCIMMEDIATE) 

static void freebsd_poll_workaround(capdev_t *dev) 
{
        int ret, on = 1;
        
	/*
         * When linked against libc_r, polling on a BPF device won't return until the
         * buffer for the device is full. This is a FreeBSD bug, and we use this as a
         * workarround.
         *
         * In case the operation is not supported we do not want to abort.
         */
                        
         ret = ioctl(pcap_fileno(dev->pd), BIOCIMMEDIATE, &on);
         if ( ret < 0 && errno != ENOTSUP ) 
         	log(LOG_ERR, "warning: couldn't set BIOCIMMEDIATE for pcap devices.\n");
}

#endif


/*
 * Setup pcap to capture from device 'device'.
 */
static capdev_t *setup_capture_from_device(const char *device, int snaplen, int promisc)
{
        int ret;
        capdev_t *dev;
        char err[PCAP_ERRBUF_SIZE];
	bpf_u_int32 localnet, netmask;

        dev = malloc(sizeof(capdev_t));
        if ( ! dev ) {
                log(LOG_ERR, "memory exhausted.\n");
                return NULL;
        }

        dev->bpf = NULL;
        dev->fcode = NULL;
        
	dev->pd = pcap_open_live(device, snaplen, promisc, PCAP_READ_TIMEOUT, err);
	if ( ! dev->pd ) {
		log(LOG_ERR,"pcap_open_live: %s.\n", err);
                free(dev);
                return NULL;
	}

        ret = pcap_lookupnet(device, &localnet, &netmask, err);
	if ( ret < 0)
                /*
                 * don't return here, the device may not have an address.
                 */
                log(LOG_INFO,"- %s: Listening in stealth mode.\n", err);

#if defined(__FreeBSD__) && defined(BIOCIMMEDIATE)
        freebsd_poll_workaround(dev);
#endif
        
	return dev;
}



/*
 * Setup pcap to capture from file 'filename'.
 */
static capdev_t *setup_capture_from_file(const char *filename)
{
        capdev_t *dev;
	char error[PCAP_ERRBUF_SIZE];

        dev = malloc(sizeof(capdev_t));
        if ( ! dev ) {
                log(LOG_ERR, "memory exhausted.\n");
                return NULL;
        }

        dev->bpf = NULL;
        dev->fcode = NULL;
        
	dev->pd = pcap_open_offline(filename, error);
	if ( ! dev->pd ) {
                log(LOG_ERR, "pcap_open_offline: %s.\n", error);
                free(dev);
                return NULL;
	}

	return dev;
}



/*
 * Poll our fd array for data,
 * return 0 on timeout, 1 on data availlable, -1 on error.
 */
inline static int do_poll(struct pollfd *pfd, unsigned int nfds, int timeout) 
{
        int ret;

        do {
                ret = poll(pfd, nfds, timeout);
                
        } while ( ret < 0 && errno == EINTR );

        if ( ret < 0 )
                log(LOG_ERR, "poll returned an error.\n");
        
        return ret;
}



inline static int handle_device_event(capdev_t *dev) 
{
        int ret;

        errno = 0;

        ret = pcap_dispatch(dev->pd, 1, read_packet, (unsigned char *) dev);        
        if ( ret < 0 ) {
                log(LOG_ERR, "pcap error: %s.\n", pcap_geterr(dev->pd));
                return ret;
        }

        /*
         * we want to always return 1 when reading from a device.
         * we want zero to be returned in case of end of file if we're
         * capturing from a file. Or 1 if there is a packet.
         *
         * pcap_fileno will return -1 if we're reading from a file.
         */
        return ( pcap_fileno(dev->pd) < 0 ) ? ret : 1;
}



static void setup_fd_set(struct pollfd *pfd, int needed_events) 
{
        int i, fd;
        
        for ( i = 0; i < interfaces_nbr; i++ ) {
                pfd[i].events = needed_events;

                fd = pcap_fileno(devices[i]->pd);
                                
                if ( fd < 0 ) 
                        /*
                         * we're capturing from a file.
                         */
                        pfd[i].fd = fileno(pcap_file(devices[i]->pd));
                else 
                        /*
                         * we're capturing from a device.
                         */
                        pfd[i].fd = fd;
        }
}



/*
 * Start capturing from multiple devices,
 * only called if more than 1 capture device were configured.
 */
static int do_capture_from_multiple_devices(void)
{
        capdev_t *dev;
        struct pollfd pfd[interfaces_nbr];
        int ret = 0, i, needed_events, events;

	/* 
	 * This is a workaround for a bug present on FreeBSD 4.x.
   	 *
	 * If you use poll(2) on BPF descriptors setting POLLIN in events table
	 * poll() returns immediately with revents table that has POLLIN set
	 * for each BPF descriptor regardless if there are really awaiting data
	 * or not. 
   	 *	   
	 * This causes prelude-nids to block in read() on each BPF device until
	 * a packet arrives, thus effectively breaking up sniffing on multiple
	 * interfaces. 
	 *  
	 * The workaround is to use POLLRDNORM instead.
	 *  
	 * This bug is fixed in FreeBSD 5.0.
	 */
	
#if defined(__FreeBSD__) && (__FreeBSD_version < 500000)
	needed_events = POLLRDNORM;
#else
	needed_events = POLLIN;
#endif

        setup_fd_set(pfd, needed_events);                
  
        while ( 1 ) {
                events = do_poll(pfd, interfaces_nbr, -1);
                if ( events < 0 ) 
                        break;
                
		for ( i = 0; i < interfaces_nbr && events != 0; i++ ) {
                        dev = devices[i];
                        
			if ( pfd[i].revents & needed_events ) {
                                events--;
                                
                                ret = handle_device_event(dev);
                                if ( ret <= 0 )
                                        return ret;
                        }

                        else if ( pfd[i].revents & POLLERR || pfd[i].revents & POLLHUP ) {
                                log(LOG_ERR, "Network is probably down on interface %s.\n", dev->interface);
                                return -1;
                        }
		}
	}
        
        return ret;
}





static int do_capture_from_single_devices(void)
{
        int ret;
            
        ret = pcap_loop(devices[0]->pd, -1, read_packet, (unsigned char *) devices[0]);                
        if ( ret < 0 ) {
                log(LOG_ERR, "Network is probably down on interface %s.\n", devices[0]->interface);
                return -1;
        }
        
        return ret;
}




/*
 * Gather stats from device 'pd'.
 */
static unsigned int pcap_dump_stat_from_device(capdev_t *dev) 
{
        int ret;
        double percent;
        struct pcap_stat stat;

        log(LOG_INFO, "[%s]: %u packets analyzed (prelude-nids counted).\n", dev->interface, dev->packet_counter);
        
        ret = pcap_stats(dev->pd, &stat);
        if ( ret < 0 ) {
                log(LOG_ERR, "[%s]: pcap_stats: %s.\n", dev->interface, pcap_geterr(dev->pd));
                return dev->packet_counter;
        }
        
        percent = (double) 100 / (double) (stat.ps_recv) * (stat.ps_drop);
        
        log(LOG_INFO, "[%s]: %u packets received by filter (pcap counted).\n", dev->interface, stat.ps_recv);
        log(LOG_INFO, "[%s]: %u (%.2f%%) packets dropped by the kernel (pcap counted).\n", dev->interface, stat.ps_drop, percent);

        percent = (double) 100 / (double) (stat.ps_recv + stat.ps_drop) * stat.ps_drop;
        log(LOG_INFO, "[%s]: %u (%.2f%%) packets dropped by the kernel (snort counted (buggy)).\n", dev->interface, stat.ps_drop, percent);
        
        return dev->packet_counter;
}



/*
 * Dump some process stats, gathered using getrusage.
 */
static void dump_system_stats(uint64_t *packet_counter) 
{
	double total_time;
        struct rusage rusage;
        
        getrusage(RUSAGE_SELF, &rusage);

        rusage.ru_utime.tv_sec  -= rusage_before_capture.ru_utime.tv_sec;
        rusage.ru_stime.tv_sec  -= rusage_before_capture.ru_stime.tv_sec;
        rusage.ru_utime.tv_usec -= rusage_before_capture.ru_utime.tv_usec;
        rusage.ru_stime.tv_usec -= rusage_before_capture.ru_stime.tv_usec;
        
        total_time = (double) rusage.ru_utime.tv_sec;
        total_time += (double) rusage.ru_stime.tv_sec;        
	total_time += ((double) rusage.ru_utime.tv_usec +
                       (double) rusage.ru_stime.tv_usec) * 1e-6;
        
        log(LOG_INFO, "It took %fs CPU  time to process %llu packets.\n", total_time, *packet_counter);

        /*
         * Not accurate if the counter *by* device (uint) woukd overflow.
         */
        log(LOG_INFO, "\n*** System stats (not accurate if > 2e%d-1 packet) ***\n\n",
            sizeof(unsigned int) * (sizeof(unsigned char) * 8));
        
	log(LOG_INFO, "Average cpu time by packet : %fs, %fms, %fus.\n",
            (total_time / (double) *packet_counter),
            (total_time / (double) *packet_counter) * 1e3,
            (total_time / (double) *packet_counter) * 1e6);
                       
        log(LOG_INFO, "Page reclaims = %ld\n", rusage.ru_minflt);
        log(LOG_INFO, "Page faults = %ld\n", rusage.ru_majflt);
        log(LOG_INFO, "Swap = %ld\n\n", rusage.ru_nswap);
}




/*
 * Add device identified by 'dev', to the devices and poll array.
 */
static int add_device(capdev_t *dev) 
{
        devices = prelude_realloc(devices, ++interfaces_nbr * sizeof(capdev_t *));
        if ( ! devices ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }
                
        devices[interfaces_nbr - 1] = dev;
                
        return 0;
}




static void client_list_to_bpf(struct list_head *clist, char *buf, size_t size)
{
        int i = 0;
        const char *daddr;
        struct list_head *tmp;
        prelude_client_t *client;
        
        list_for_each(tmp, clist) {
                client = prelude_list_get_object(tmp, prelude_client_t);

                /*
                 * this client is dead (no connection).
                 */
                if ( prelude_client_is_alive(client) < 0 )
                        continue;

                daddr = prelude_client_get_daddr(client);
                
                /*
                 * special case.
                 */
                if ( strcmp(daddr, "unix") == 0 )
                        continue;
                
                i += snprintf(buf + i, size - i, "! (host %s and port %d",
                              daddr, prelude_client_get_dport(client));

                if ( prelude_client_get_saddr(client) ) 
                        i += snprintf(buf + i, size - i, " and host %s and port %d)",
                                      prelude_client_get_saddr(client), prelude_client_get_sport(client));
                else
                        i += snprintf(buf + i, size - i, ")");
        }
}



static int setup_global_bpf(char *gbpf)
{
        int i = 0, ret;
        char buf[1024];
        
        for ( i = 0; i < interfaces_nbr; i++ ) {
            
                if ( devices[i]->bpf ) {
                        snprintf(buf, sizeof(buf), "(%s) and (%s)", gbpf, devices[i]->bpf);
                        ret = setup_bpf(devices[i], buf);
                } else 
                        ret = setup_bpf(devices[i], gbpf);
                
                if ( ret < 0 )
                        return -1;
        }
        
        return 0;
}




static void setup_global_bpf_cb(struct list_head *clist) 
{
        char buf[1024];

        buf[0] = '\0';
        
        client_list_to_bpf(clist, buf, sizeof(buf));
        if ( *buf == '\0' )
                return;
        
        setup_global_bpf(buf);
}




/*
 * Shared setup capture function.
 */
static int capture_setup_generic(capdev_t *dev, const char *bpf) 
{
        int ret;

        dev->packet_counter = 0;
        
        dev->p_handler = search_datalink_handler(pcap_datalink(dev->pd));
        if ( ! dev->p_handler )
                return -1;

        ret = setup_bpf(dev, bpf);
        if ( ret < 0 )
                return -1;

        if ( bpf )
                dev->bpf = strdup(bpf);
        
        ret = add_device(dev);
        
        return ret;
}





/**
 * @device: Device to get address for.
 * @buf: Where to store the network address/mask for the device.
 * @size: Size of the buffer.
 *
 * Get network address/mask where @device is on,
 * and store it to @buf.
 *
 * Return 0 on success, -1 if an error occured.
 */
int capture_get_device_address(const char *device, char *buf, size_t size) 
{
        int ret;
        uint8_t *addr, *mask;
        char err[PCAP_ERRBUF_SIZE];
        bpf_u_int32 localnet, netmask;

        ret = pcap_lookupnet(device, &localnet, &netmask, err);
	if ( ret < 0 ) 
                return -1;

        addr = (uint8_t *) &localnet;
        mask = (uint8_t *) &netmask;
        
        snprintf(buf, size, "%d.%d.%d.%d/%d.%d.%d.%d",
                 addr[0] & 0xff, addr[1] & 0xff, addr[2] & 0xff, addr[3] & 0xff,
                 mask[0] & 0xff, mask[1] & 0xff, mask[2] & 0xff, mask[3] & 0xff);        

        return 0;
}




/**
 * capture_from_device:
 * @device: The device to capture packet from.
 * @bpf: An optionnal Berkeley Packet Filter rule.
 * @snaplen: Maximum number of byte per packet.
 * @promisc: Tell if the device should be opened in promiscuous mode.
 * @outfile: A file to capture packet to.
 *
 * Initialize the capture on @device, with the optionnal @bpf rule,
 * capturing a maximum of @snaplen byte per packet, and eventually outputing
 * the captured data to @outfile.
 *
 * @outfile will get an extension (the device name), in order to not confuse
 * libpcap (which associate a capture file with the type of link for the device).
 *
 * Returns: 0 on success, -1 otherwise.
 */
int capture_from_device(const char *device, const char *bpf, int snaplen, int promisc, const char *outfile) 
{
        int ret;
        capdev_t *dev;
        
        dev = setup_capture_from_device(device, snaplen, promisc);
        if ( ! dev )
                return -1;
        
        dev->interface = device;

        ret = capture_setup_generic(dev, bpf);
        if ( ret < 0 )
                return -1;
        
        if ( outfile ) {
                char buf[256];
                
                /*
                 * Add an extension to outfile specifying the type of device,
                 * cause pcap associate a file dumper to a device.
                 */
                snprintf(buf, sizeof(buf), "%s.%s", outfile, device);
                
                dev->file_dumper = pcap_dump_open(dev->pd, buf);
                if ( ! dev->file_dumper ) {
                        log(LOG_ERR, "%s", pcap_geterr(dev->pd));
                        return -1;
                }

                dev->p_handler = pcap_dump;
        } else
                pcap_set_alloc_func(dev->pd, pktalloc, pktfree);

        
        return 0;
}




/**
 * capture_from_file:
 * @filename: The filename to read the packet from.
 * @bpf: An optionnal Berkeley Packet Filter rule.
 *
 * Initialize the capture on specified filename, with
 * an optionnal bpf rule.
 *
 * Returns: 0 on success, -1 otherwise.
 */
int capture_from_file(const char *filename, const char *bpf) 
{
        int ret;
        capdev_t *dev;

        dev = setup_capture_from_file(filename);
        if ( ! dev )
                return -1;

        dev->interface = filename;

        ret = capture_setup_generic(dev, bpf);
        if ( ret < 0 )
                return -1;
        
        pcap_set_alloc_func(dev->pd, pktalloc, pktfree);
        
        return ret;
}




/**
 * capture_stats:
 *
 * Dump packet capture statistic.
 */
void capture_stats(void)
{
        int i;
        double total;
        unsigned int devstat;
        uint64_t global_counter = 0;
        
        if ( ! capend.tv_sec && ! capend.tv_usec )
                gettimeofday(&capend, NULL);
        
        log(LOG_INFO, "\n*** Capture stats (not accurate if > 2e%d-1 packet) ***\n\n",
            sizeof(devstat) * (sizeof(unsigned char) * 8));
        
        for ( i = 0; i < interfaces_nbr; i++ ) {
                devstat = pcap_dump_stat_from_device(devices[i]);
                global_counter += (uint64_t) devstat;
        }
        
        log(LOG_INFO, "[all]: %llu packets received by filter.\n\n", global_counter);
        
        total = (double) capend.tv_sec - capstart.tv_sec;
        total += ((double) capend.tv_usec + (double) capend.tv_usec) * 1e-6;
        log(LOG_INFO, "It took %fs real time to process %llu packets.\n", total, global_counter);
        
        dump_system_stats(&global_counter);
}




/**
 * capture_start:
 *
 * Start the capture on previously configured device.
 * This function will not return unless an error occur.
 *
 * Returns: -1 on error.
 */
int capture_start(void)
{
	int ret;
        char buf[1024];
        struct list_head *clist;
        
        buf[0] = '\0';
        
        clist = prelude_sensor_get_client_list();

        client_list_to_bpf(clist, buf, sizeof(buf));
        if ( *buf != '\0' ) {
        
                ret = setup_global_bpf(buf);
                if ( ret < 0 )
                        return -1;
        }
        
        prelude_sensor_notify_mgr_connection(setup_global_bpf_cb);

        capend.tv_sec = 0;
        capend.tv_usec = 0;
        getrusage(RUSAGE_SELF, &rusage_before_capture);
        
        gettimeofday(&capstart, NULL);

#if defined(__OpenBSD__)
        /*
         * OpenBSD pthread are broken: certain system call never get
         * interrupted in order to wake up other thread. Using
         * do_capture_from_multiple_devices() make sure we call poll()
         * before pcap_read, working around the problem.
         */
        ret = do_capture_from_multiple_devices();
#else
        if ( interfaces_nbr > 1 )
                ret = do_capture_from_multiple_devices();
        else
                ret = do_capture_from_single_devices();
#endif   
        gettimeofday(&capend, NULL);

        return ret;
}



/**
 * capture_stop:
 *
 * Stop packet capture.
 */
void capture_stop(void)
{
        int i;
        
        for (i = 0; i < interfaces_nbr; i++)
                pcap_close(devices[i]->pd);
}




