/* $Cambridge: hermes/src/prayer/servers/prayer_login.c,v 1.6 2008/06/12 10:31:42 dpc22 Exp $ */
/************************************************
 *    Prayer - a Webmail Interface              *
 ************************************************/

/* Copyright (c) University of Cambridge 2000 - 2008 */
/* See the file NOTICE for conditions of use and distribution. */

#include "prayer_shared.h"
#include "prayer.h"
#include "prayer_login.h"
#include "prayer_server.h"

/* prayer_login_generate() ***********************************************
 *
 * Generate login screen
 *   prayer: Global state
 *  request: HTTP request that initiated this action
 *     user: Default username for login scren.
 *     form: Whether to include the login form.
 ************************************************************************/

static void
prayer_login_generate(struct prayer *prayer,
                      struct request *request, char *user, BOOL form)
{
    struct config *config = prayer->config;
    struct buffer *b = request->write_buffer;
    FILE *file;
    int c;
    char *url_prefix = prayer_url_prefix(prayer, request->pool);
    char *hostname;

    if (!prayer->use_ssl && config->ssl_required) {
        response_error(request, 403);   /* Forbidden */
        return;
    }

    /* XXX Want some colour substitutions here? */
    if (config->login_prefix_path &&
        ((file = fopen(config->login_prefix_path, "r")) != NULL)) {
        /* Config provides template for start of login page */
        while ((c = fgetc(file)) != EOF) {
            if (c == '\n')
                bputs(b, "" CRLF);
            else
                bputc(b, c);
        }
        fclose(file);
    } else {
        /* Old behaviour */
        if (config->login_banner)
            html_common_start(config, b, config->login_banner);
        else
            html_common_start(config, b, "Webmail Service Login");

        if (config->login_banner)
            bprintf(b,
                    "<h1 align=\"center\">%s</h1>" CRLF,
                    config->login_banner);
        else
            bputs(b,
                  "<h1 align=\"center\">Webmail Service Login</h1>" CRLF);
    }

    /* Add plaintext warning if not SSL */
    if ((prayer->use_ssl == NIL) && (config->ssl_default_port > 0)) {
        bputs(b, "<table>" CRLF);
        bputs(b, "<tr>" CRLF);
        bputs(b, "<td><strong>Warning:</strong></td>");
        bputs(b, "<td>This is an insecure HTTP session.</td></tr>" CRLF);
        bputs(b, "<tr>" CRLF);
        bputs(b, "<td>&nbsp;</td>" CRLF);
        bputs(b, "<td>It is strongly recommended that you use ");
        bputs(b, "SSL encryption if your browser supports it." CRLF);
        bputs(b, "</td></tr>" CRLF);

        bputs(b, "<tr>" CRLF);
        bputs(b, "<td>&nbsp;</td>" CRLF);
        bputs(b, "<td><strong>" CRLF);

        hostname = config->hostname_service ? config->hostname_service
                                            : config->hostname;
        if (config->ssl_default_port != 443)
            bprintf(b, "<a href=\"https://%s:%lu\">",
                    hostname, config->ssl_default_port);
        else
            bprintf(b, "<a href=\"https://%s\">", hostname);
        bputs(b, "Click Here</a> for an SSL enabled login session." CRLF);
        bputs(b, "</strong></td></tr>" CRLF);

        /* Link to insecure login if the form was not provided */
        if (!form) {
            bputs(b, "<tr>" CRLF);
            bputs(b, "<td>&nbsp;</td>" CRLF);
            bputs(b, "<td>" CRLF);
            bputs(b, "For an insecure login session, ");
            bprintf(b, "<a href=\"%s/login/\">click here</a>.", url_prefix);
            bputs(b, "</td></tr>" CRLF);
        }

        bputs(b, "</table>" CRLF);
    }

    if (form) {
        /* Actual login form */

        bprintf(b, ("<form method=\"post\" accept-charset=\"UTF-8\""
                    " enctype=\"multipart/form-data\""
                    " action=\"%s%s\">"CRLF), url_prefix, request->url_path);

        bputs(b, "<table cellpadding=\"5\">" CRLF);
        bputs(b, "<tr>" CRLF);

        bputs(b, "  <td>Username</td>" CRLF);

        bputs(b, "  <td>");
        bprintf(b, "<input name=\"username\" value=\"");
        if (user)
            html_quote_string(b, user);
        bputs(b, "\" size=\"8\" maxlength=\"8\" />");
        bputs(b, "</td>" CRLF);
        bputs(b, "</tr>" CRLF);

        bputs(b, "<tr>" CRLF);
        bputs(b, "<td>Password</td>" CRLF);
        bputs(b,
                "<td><input type=\"password\" name=\"password\" size=\"16\" /></td>"
                CRLF);
        bputs(b, "</tr>" CRLF);

        bputs(b, "<tr>" CRLF);
        bputs(b, "  <td>&nbsp;</td>" CRLF);
        bputs(b, "  <td>" CRLF);
        bputs(b,
                "    <input type=\"submit\" name=\"login\" value=\"Login\" />"
                CRLF);
        bputs(b, "  </td>" CRLF);
        bputs(b, "</tr>" CRLF);
        bputs(b, "</table>" CRLF);

        bputs(b, "</form>" CRLF);
    }

    if (config->motd_path) {
        /* MOTD */
        if ((file = fopen(config->motd_path, "r")) != NULL) {
            while ((c = fgetc(file)) != EOF) {
                if (c == '\n')
                    bputs(b, "" CRLF);
                else
                    bputc(b, c);
            }
            fclose(file);
        }
    }

    if (config->login_suffix_path &&
        ((file = fopen(config->login_suffix_path, "r")) != NULL)) {
        /* Config provides template for end of login page */

        while ((c = fgetc(file)) != EOF) {
            if (c == '\n')
                bputs(b, "" CRLF);
            else
                bputc(b, c);
        }
        fclose(file);
    } else {
        /* Old behaviour */
        html_common_end(b);
    }

    /* Send out the response */
    response_html(request, 200);
}

/* ====================================================================== */

/* prayer_login_process_request() ****************************************
 *
 * Process login request (static support function)
 *   prayer: Global state
 * username: Username to send to IMAP server
 * password: Password to send to IMAP server
 *  statusp: Returns Server status code
 *   valuep: Returns Descritive text or URL, depending on status code
 *
 * Connection/protocol error if either *statusp and *valuep NIL on return
 ************************************************************************/

static void
prayer_login_process_request(struct prayer *prayer,
                             char *username, char *password,
                             char **statusp, char **valuep)
{
    struct config *config = prayer->config;
    struct request *request = prayer->request;
    struct pool *pool = request->pool;
    struct user_agent *user_agent = request->user_agent;
    struct iostream *iostream;
    char *socketname;
    int sockfd;
    struct buffer *b = buffer_create(pool, 0L);
    int c;
    char *line, *status, *value;
    char *ua_options;

    *statusp = NIL;
    *valuep = NIL;

    if (!(username && username[0])) {
        *statusp = "NO";
        *valuep = "No username supplied";
        return;
    }

    if (!(password && password[0])) {
        *statusp = "NO";
        *valuep = "No password supplied";
        return;
    }

    socketname = pool_printf(pool, "%s/%s",
                             config->socket_dir, config->init_socket_name);

    if ((sockfd = os_connect_unix_socket(socketname)) < 0)
        return;

    if ((iostream = iostream_create(pool, sockfd, 0)) == NIL) {
        log_misc("[process_login_request()] iostream_create() failed");
        return;
    }

    iostream_set_timeout(iostream, config->session_timeout);

    ua_options = user_agent_options(user_agent);

    ioprintf(iostream, "%s %s %s %d %d %s" CRLF,
             string_canon_encode(pool, username),
             string_canon_encode(pool, password),
             string_canon_encode(pool, ua_options),
             prayer->port, prayer->use_ssl, ipaddr_text(prayer->ipaddr));

    ioflush(iostream);

    while (((c = iogetc(iostream)) != EOF) && (c != '\015')
           && (c != '\012'))
        bputc(b, c);

    if (c == EOF) {
        log_misc("[process_login_request()] iogetc() got end of file");
        return;
    }

    line = buffer_fetch(b, 0, buffer_size(b), NIL);

    status = string_get_token(&line);
    value = string_next_token(&line);

    if (!(status && value))
        log_misc("[process_login_request()] Invalid response from server");

    *statusp = status;
    *valuep = value;
    return;
}

/* ====================================================================== */

/* prayer_login_process() ************************************************
 *
 * Process login request (static internal fn).
 *   prayer: Global state
 *  request: HTTP request that initiated this action
 ************************************************************************/

static void
prayer_login_process(struct prayer *prayer, struct request *request)
{
    struct config *config = request->config;
    struct user_agent *user_agent = request->user_agent;
    struct buffer *b = request->write_buffer;
    char *username;
    char *password;
    char *status;
    char *value;
    char *url_prefix = prayer_url_prefix(prayer, request->pool);

    /* Decode information from post request */
    request_decode_form(request);

    username = assoc_lookup(request->form, "username");
    password = assoc_lookup(request->form, "password");

    if ((config->referer_log_invalid || config->referer_block_invalid) &&
        !request_test_referer(request, config->hostname_service) &&
        !request_test_referer(request, config->hostname)) {

        if (config->referer_log_invalid)
            log_misc("[prayer_login_process()] Invalid Referer: %s",
                     assoc_lookup(request->hdrs, "referer"));

        if (config->referer_block_invalid) {
            html_common_start(config, b, "Security Alert");
            bputs(b, "<h2>Security Alert</h2>"CRLF);

            bprintf(b, ("<p>Login request did not come from "
                        "<a href = \"%s\">%s<a/></p>. "CRLF),
                    url_prefix, url_prefix);

            html_common_end(b);
            response_html(request, 200);
            return;
        }
    }

    prayer_login_process_request(prayer, username, password, &status,
                                 &value);

    if (!(status && value)) {
        html_common_start(config, b, "System Error");
        bprintf(b,
                "<h1>Couldn't connect to Webmail session server</h1>"
                CRLF);
        if (username) {
            bprintf(b, "<a href=\"%s/login/", url_prefix);
            html_quote_string(b, username);
            bprintf(b, "\">Try again</a> later" CRLF);
        } else
            bprintf(b, "<a href=\"%s\">Try again</a> later" CRLF,
                    url_prefix);

        html_common_end(b);
        response_html(request, 200);
        return;
    }

    if (!strcmp(status, "OK")) {
        response_redirect(request, value);
        return;
    }

    html_common_start(config, b, "Login Failed");
    bprintf(b, "<h1>Login failed</h1>" CRLF);

    bputs(b, "<hr />");

    bputs(b, "<b>Status:</b> ");
    html_common_quote_string(b, value);
    bputs(b, "" CRLF);
    bputs(b, "<hr />");

    if (username) {
        bprintf(b, "Please <a href=\"%s/login/", url_prefix);
        html_quote_string(b, username);
        bprintf(b, "\">try again</a>" CRLF);
    } else
        bprintf(b, "Please <a href=\"%s\">try again</a>" CRLF,
                url_prefix);

    html_common_end(b);
    response_html(request, 200);
}

/* ====================================================================== */

/* prayer_login() ********************************************************
 *
 * Process login URLs: POST => login request, GET => display login screen
 *    prayer: Global state
 *  username: Default login name.
 ************************************************************************/

void prayer_login(struct prayer *prayer, char *username)
{
    struct request *request = prayer->request;

    if (request->method == POST)
        prayer_login_process(prayer, request);
    else
        prayer_login_generate(prayer, request, username, T);
}

/* prayer_login_preamble() ***********************************************
 *
 * Handle pre-login actions for insecure connections, either a warning
 * banner or a redirect to the secure server, depending on configuration.
 *    prayer: Global state
 ************************************************************************/

void prayer_login_preamble(struct prayer *prayer)
{
    struct config *config = prayer->config;
    struct request *request = prayer->request;
    char *hostname, *url;

    /* Normal login if we are secure */
    if (prayer->use_ssl) {
        prayer_login(prayer, NIL);
        return;
    }

    /* Redirect to the secure login url */
    if (config->ssl_redirect) {
        hostname = config->hostname_service ? config->hostname_service
                                            : config->hostname;
        if (config->ssl_default_port != 443)
            url = pool_printf(request->pool, "https://%s:%lu",
                    hostname, config->ssl_default_port);
        else
            url = pool_printf(request->pool, "https://%s", hostname);
        response_redirect(prayer->request, url);
        return;
    }

    if (config->ssl_encouraged) {
        /* Produce a pre-login warning page */
        prayer_login_generate(prayer, request, NIL, NIL);
    } else {
        /* Configuration allows plaintext logins (e.g: testrig on magenta) */
        prayer_login(prayer, NIL);
    }
}
