/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *            Copyright (C) 2005 SUSE Linux Products GmbH                  *
 *                                                                         *
 *              Author(s): Holger Macht <hmacht@suse.de>                   *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include <signal.h>

#include "main_loop.h"
#include "powerlib.h"
#include "globals.h"
#include "cpufreq_management.h"
#include "event_management.h"
#include "apm.h"

using namespace Powersave::Globals;

/**** class Callback ****/
Callback::Callback()
{
	id = -1;
	interval = -1;
	function = NULL;
}

Callback::Callback(int interval_, gboolean (*function_)(gpointer data))
{
	id = -1;
	interval = interval_;
	function = function_;
}

/**** class MainLoop ****/

MainLoop::MainLoop() {
	/* set up a new main loop with glib and the default context */
	_gmain = g_main_loop_new(NULL, FALSE);

	DBusConnection *dbus_connection = _dbus_server.openSystemConnection();

	if ( dbus_connection == NULL ) {
		pDebug(DBG_ERR,
		       "Error setting up connection to DBus system bus. Aborting...");
		exit(1);
	} else {
		dbus_connection_setup_with_g_main(dbus_connection, NULL);
	}

	if (!hwEvent_connect()) {
		pDebug(DBG_ERR, "Cannot access acpi/apm event file\n");

		/* only try to reconnect if we have acpi. If /dev/apm_bios
		   is not there on startup, it won't appear later */
		APM_Interface *p = dynamic_cast< APM_Interface* >(pm);

		if (p == NULL) {
			addCallback(3000, (GSourceFunc) hwEvent_reconnect);
		}
	}

	// wait about 10 minutes for hal to appear
	int waiting = 0;
	while (!DBus_Server::serviceHasOwner("org.freedesktop.Hal")) {
		sleep(3);
		waiting++;
		if (waiting > 200) {
			pDebug(DBG_ERR, "Haldaemon did not appear. Aborting...");
			kill(getpid(), SIGTERM);
		}
	}

	setupCallbacks();
}

void MainLoop::run()
{
	pDebug(DBG_INFO, "Starting main loop");
	g_main_loop_run(_gmain);
}

void MainLoop::quit()
{
	g_main_loop_quit(_gmain);
}

gboolean MainLoop::hwEvent_callback(GIOChannel *io_channel,
				    GIOCondition io_condition,
				    gpointer data)
{
	int io_channel_fd = g_io_channel_unix_get_fd(io_channel);

	if (io_condition & (G_IO_ERR | G_IO_HUP)) {
		pDebug(DBG_ERR, "hw event socket broke away trying to reconnect...");

		close(io_channel_fd);

		main_loop->addCallback(3000, (GSourceFunc) hwEvent_reconnect);

		return false;
	}

	pm->handleHWEventRequest(io_channel_fd);

	return true;
}

gboolean MainLoop::hwEvent_reconnect(gpointer data)
{
	if (hwEvent_connect()) {
		pDebug(DBG_ERR, "Successfully reconnected to acpi/apm event file\n");
		return false;
	}
	
	return true;
}

gboolean MainLoop::hwEvent_connect()
{
	int hwEvent_fd = -1;
	GIOChannel *hwEvent_channel;
	int hwEvent_watch = -1;

	/* set up the event interface. /proc/acpi/event or /dev/apm_bios */
	if ((hwEvent_fd = pm->openHWEventFD()) < 0) {
		return false;
	} else {
		hwEvent_channel = g_io_channel_unix_new(hwEvent_fd);
		hwEvent_watch = g_io_add_watch(hwEvent_channel,
					       GIOCondition(G_IO_IN | G_IO_ERR | G_IO_HUP),
					       hwEvent_callback, NULL);
		return true;
	}
}

gboolean MainLoop::checkThrottling_callback(gpointer data)
{
	if (config_obj->current_scheme->ALLOW_THROTTLING) {
		pm->checkThrottling();
		return true;
	}

	return false;
}

gboolean MainLoop::checkEventTimeouts_callback(gpointer data)
{
	return pm->_eM->checkEventTimeouts();
}

gboolean MainLoop::adjustSpeeds_callback(gpointer data)
{
	if (cpufreq->hasControlMode(CPUFREQ_USERSPACE)) {
		cpufreq->adjustSpeeds();
		return true;
	}

	return false;
}

void MainLoop::setupCallbacks()
{
	_callback_functions.push_back(
		Callback(config_obj->current_scheme->POLL_INTERVAL,
			 checkThrottling_callback));

	_callback_functions.push_back(
		Callback(3000 ,checkEventTimeouts_callback));

	_callback_functions.push_back(
		Callback(config_obj->current_scheme->POLL_INTERVAL,
			 adjustSpeeds_callback));

	updateCallbacks();
}

void MainLoop::updateCallbacks()
{
	for (std::list<Callback>::iterator it = _callback_functions.begin();
	     it != _callback_functions.end(); ++it) {

		if (g_main_context_find_source_by_id(NULL, it->id) == NULL) {
			it->id = g_timeout_add(it->interval, it->function, NULL);
		}
	}
}

int MainLoop::addCallback(int interval, gboolean (*function)(gpointer data))
{
	return g_timeout_add(interval, function, NULL);
}
