/*
 * Copyright 2013-2014 Canonical Ltd.
 *
 * Authors:
 * Michael Frey: michael.frey@canonical.com
 * Matthew Fischer: matthew.fischer@canonical.com
 * Seth Forshee: seth.forshee@canonical.com
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <string.h>

#include <glib.h>
#include <gio/gio.h>

#include "powerd-internal.h"
#include "log.h"

static GDBusProxy* create_uscreen_proxy()
{
    GError *error = NULL;
    GDBusProxy *uscreen_proxy = NULL;
    uscreen_proxy = g_dbus_proxy_new_for_bus_sync(G_BUS_TYPE_SYSTEM,
            G_DBUS_PROXY_FLAGS_NONE,
            NULL,
            "com.canonical.Unity.Screen",
            "/com/canonical/Unity/Screen",
            "com.canonical.Unity.Screen",
            NULL,
            &error);

    if (error != NULL) {
        powerd_warn("could not connect to Unity.Screen: %s", error->message);
        g_error_free(error);
        return NULL;
    }

    return uscreen_proxy;
}

static void request_keep_display_on(GDBusConnection *connection,
        const gchar *name,
        const gchar *name_owner,
        gpointer user_data)
{
    GDBusProxy *uscreen_proxy = NULL;
    GVariant *ret = NULL;
    GError *error = NULL;
    int request_id;

    powerd_debug("request_keep_display_on");

    uscreen_proxy = create_uscreen_proxy();
    if (uscreen_proxy == NULL)
    {
        powerd_warn("failed creating unity screen proxy to keep display on");
        return;
    }

    ret = g_dbus_proxy_call_sync(uscreen_proxy,
            "keepDisplayOn",
            NULL,
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            &error);

    if (ret == NULL) {
        powerd_warn("keepDisplayOn failed: %s", error->message);
        g_error_free(error);
        return;
    }

    g_variant_get(ret, "(i)", &request_id);
    powerd_debug("keepDisplayOn request succeeded - id: %d", request_id);

    g_variant_unref(ret);
    g_object_unref(uscreen_proxy);
}

gboolean turn_display_on(gboolean on, enum DisplayStateChangeReason reason)
{
    GDBusProxy *uscreen_proxy = NULL;

    powerd_debug("turn_display_on(%d)", on);

    if (on)
        powerd_hal_signal_activity();

    uscreen_proxy = create_uscreen_proxy();
    if (uscreen_proxy == NULL)
        return FALSE;

    g_dbus_proxy_call(uscreen_proxy,
            "setScreenPowerMode",
            g_variant_new("(si)", (on ? "on" : "off"), reason),
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            NULL,
            NULL);

    g_object_unref(uscreen_proxy);

    return TRUE;
}

void keep_display_on()
{
    powerd_debug("issuing keep_display_on request");

    g_bus_watch_name(G_BUS_TYPE_SYSTEM,
            "com.canonical.Unity.Screen",
            G_BUS_NAME_WATCHER_FLAGS_NONE,
            request_keep_display_on,
            NULL,
            NULL,
            NULL);
}
