/*
 * Copyright 2013 Canonical Ltd.
 *
 * Authors:
 * Michael Frey: michael.frey@canonical.com
 * Matthew Fischer: matthew.fischer@canonical.com
 * Seth Forshee: seth.forshee@canonical.com
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <errno.h>
#include <glib.h>
#include <glib-object.h>
#include <android/hardware/hardware.h>
#include <android/hardware/lights.h>

#include "powerd-internal.h"
#include "device-config.h"
#include "log.h"

struct light_device_t *light_dev;

enum bl_state {
    BL_OFF,
    BL_ON,
    BL_AUTO,

    NUM_BL_STATES
};

/*
 * Defaults are Android default settings. They may not work if we
 * can't read the per-device values, but they're as good as
 * anything else.
 */
int min_user_brightness = 10;
int max_brightness = 255;
int default_brightness = 102;
int dim_brightness = 10;
int current_brightness = -1;

static int user_brightness;

int powerd_get_brightness(void)
{
    return current_brightness;
}

int powerd_get_max_brightness(void)
{
    return max_brightness;
}

int powerd_set_brightness(int brightness)
{
    struct light_state_t state;
    int scaled_brightness;
    int ret;

    if (!light_dev)
        return -ENODEV;
    if (brightness < 0 || brightness > max_brightness)
        return -EINVAL;
    if (brightness == current_brightness)
        return 0;

    /* Scale brightness to range of 0 to 255 */
    scaled_brightness = (brightness * 255) / max_brightness;
    if (scaled_brightness > 255)
        scaled_brightness = 255;

    state.flashMode = LIGHT_FLASH_NONE; 
    state.brightnessMode = BRIGHTNESS_MODE_USER;
    /* color is ARGB, docs specifiy that alpha should be 0xff,
     * although it also instructs callers to ignore it. */
    state.color = (int)((0xffU << 24) | (scaled_brightness << 16) |
                        (scaled_brightness << 8) | scaled_brightness);

    ret = light_dev->set_light(light_dev, &state);
    if (!ret) {
        powerd_debug("light_dev: set_light to brightness %i succ", brightness);
        current_brightness = brightness;
    } else {
        powerd_debug("light_dev: set_light to brightness %i failed", brightness);
    }

    return ret;
}

int powerd_set_user_brightness(int brightness)
{
    return powerd_set_brightness(brightness < min_user_brightness ?
                                 min_user_brightness : brightness);
}

int powerd_backlight_init(void)
{
    GValue v = G_VALUE_INIT;
    const struct hw_module_t *hwmod = NULL;

    if (!device_config_get("screenBrightnessDim", &v)) {
        if (G_VALUE_HOLDS_UINT(&v))
            dim_brightness = (int)g_value_get_uint(&v);
        g_value_unset(&v);
    }
    if (!device_config_get("screenBrightnessSettingMinimum", &v)) {
        if (G_VALUE_HOLDS_UINT(&v))
            min_user_brightness = (int)g_value_get_uint(&v);
        g_value_unset(&v);
    }
    if (!device_config_get("screenBrightnessSettingMaximum", &v)) {
        if (G_VALUE_HOLDS_UINT(&v))
            max_brightness = (int)g_value_get_uint(&v);
        g_value_unset(&v);
    }
    if (!device_config_get("screenBrightnessSettingDefault", &v)) {
        if (G_VALUE_HOLDS_UINT(&v))
            default_brightness = (int)g_value_get_uint(&v);
        g_value_unset(&v);
    }

    /* Sanity check default brightness value */
    if (default_brightness < min_user_brightness)
        default_brightness = min_user_brightness;
    else if (default_brightness > max_brightness)
        default_brightness = max_brightness;

    if (hw_get_module(LIGHTS_HARDWARE_MODULE_ID, &hwmod)) {
        powerd_warn("Could not get lights module");
        return -ENODEV;
    }
    if (hwmod->methods->open(hwmod, LIGHT_ID_BACKLIGHT,
                             (hw_device_t **)&light_dev)) {
        powerd_warn("Could not open backlight, brightness adjustment will be disabled");
        light_dev = NULL;
        return -ENODEV;
    }

    /* XXX: May want to eliminate this when integration with shell
     * is complete and just wait for it to apply user settings.
     * Unfortunately the HAL provides no way for us to read the
     * current brightness when we start. */
    powerd_set_brightness(default_brightness);
    user_brightness = default_brightness;

    return 0;
}

void powerd_backlight_deinit(void)
{
    light_dev = NULL;
}

static void set_user_brightness(int brightness)
{
    int max = powerd_get_max_brightness();
    if (brightness > max)
        brightness = max;
    user_brightness = brightness;
}

static void set_backlight(enum bl_state state)
{
    static enum bl_state current_state = BL_ON;
    static int restore_brightness = 0;

    if (state >= NUM_BL_STATES) {
        powerd_warn("Unknown backlight state %d\n", state);
        return;
    }

    if (state != BL_AUTO)
        powerd_autobrightness_disable();

    /*
     * When autobrightness is enabled, it takes a second to
     * settle on a brightness level after enabling. This delay
     * becomes very noticeable when going from the dim to bright
     * state. To avoid this lag, save off the current brightness
     * and restore it when transitioning to AUTO mode again.
     */
    if (current_state != BL_OFF && state == BL_OFF)
        restore_brightness = powerd_get_brightness();

    switch (state) {
    case BL_OFF:
        powerd_set_brightness(0);
        break;
    case BL_ON:
        powerd_set_user_brightness(user_brightness);
        break;
    case BL_AUTO:
        if (current_state == BL_OFF && restore_brightness > 0)
            powerd_set_brightness(restore_brightness);
        powerd_autobrightness_enable();
        break;
    default:
        /* default case is to satisfy static analysis tools, should
         * never actually get here */
        powerd_error("Unknwown backlight state %d\n", state);
        return;
    }
    current_state = state;
}


/** dbus interfaces **/

gboolean handle_get_brightness_params(PowerdSource *obj,
                                      GDBusMethodInvocation *invocation)
{
    gboolean ab_available = powerd_autobrightness_available();
    g_dbus_method_invocation_return_value(invocation,
                                          g_variant_new("((iiiib))",
                                                        dim_brightness,
                                                        min_user_brightness,
                                                        max_brightness,
                                                        default_brightness,
                                                        ab_available));
    return TRUE;
}

gboolean handle_user_autobrightness_enable(PowerdSource *obj,
                                           GDBusMethodInvocation *invocation,
                                           gboolean enable)
{
    if (powerd_autobrightness_available() && enable)
        set_backlight(BL_AUTO);
    else
        set_backlight(BL_ON);

    g_dbus_method_invocation_return_value(invocation, NULL);
    return TRUE;
}

gboolean handle_set_user_brightness(PowerdSource *obj,
                                    GDBusMethodInvocation *invocation,
                                    gint brightness)
{
    if (brightness == 0)
    {
        set_backlight(BL_OFF);
    }
    else
    {
        set_user_brightness(brightness);
        set_backlight(BL_ON);
    }
    g_dbus_method_invocation_return_value(invocation, NULL);
    return TRUE;
}
