/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <errno.h>
#include <math.h>
#include <inttypes.h>
#include <glib.h>

#include "powerd-internal.h"
#include "device-config.h"
#include "log.h"
#include "spline.h"

/*
 * Autobrightness Control Algorithm
 *
 * Automatic backlight adjustments should respond quickly to changes in
 * the ambient brightness without constantly adjusting for minor or
 * transient changes in ambient brightness. The following mechanisms are
 * employed to maintain this balance.
 *
 * EXPONENTIAL SMOOTHING
 *
 * Rather than processing incoming lux values directly, an exponential
 * moving average is calculated using an algorithm based off of the
 * formulation known as "Brown's simple exponential smoothing" [1]:
 *
 *   S(n) = a * x(n) + (1 - a) * S(n - 1)
 *        = S(n - 1) + a * (x(n) - S(n - 1))
 *
 * where 0 <= a <= 1. To make smoothing time-based, a is calculated
 * based on the amount of time spent at a given lux value.
 *
 *   delta_t = t(n) - t(n - 1)
 *   a(n) = delta_t / (C + delta_t)
 *
 * This gives decreasing weight to older samples. Smaller values of C
 * will cause faster decay of old samples, giving a faster response
 * time, while larger values of C create slower decay and give more
 * filtering of transient events.
 *
 * This code usese two moving averages, one which responds slowly to
 * changes and another which responds quickly. The slow average is used
 * to detect the trend with less sensitivity to transient events, while
 * the fast average is used for picking the brightness once we've
 * decided to change it.
 *
 * HYSTERESIS
 *
 * To avoid making constant adjustments to screen brightness in response
 * to small changes in ambient lighting conditions, the change in lux is
 * required to exceed some threshold before changing screen brightness.
 *
 * DEBOUNCING
 *
 * Changes in ambient brightness which exceed the hysteresis threshold
 * must do so for some minimum amount of time before any changes in
 * screen brightness will be made. This "debounces" the incoming lux
 * values and also helps to filter out transient changes in ambient
 * conditions.
 *
 * [1] http://en.wikipedia.org/wiki/Exponential_smoothing
 */

#define SMOOTHING_FACTOR_SLOW 2000.0f
#define SMOOTHING_FACTOR_FAST 200.0f
#define HYSTERESIS_FACTOR 0.10f
#define DEBOUNCE_MS 4000

enum ab_state {
    AB_STATE_DISABLED,
    AB_STATE_INITIALIZING,
    AB_STATE_IDLE,      /* Idle, nothing to do */
    AB_STATE_DEBOUNCE,  /* Debouncing lux change */
};

struct ab_status {
    enum ab_state state;
    guint source_id;
    double last_lux, applied_lux;
    double average_slow, average_fast;
    gint64 last_lux_ms;
    struct ab_spline *spline;
};

static gboolean ab_supported = FALSE;
static struct ab_status ab_status;
struct spline *ab_spline;

static gint64 get_ms(void)
{
    return g_get_monotonic_time() / 1000;
}

static double aggregate_lux(double old_average, double new_lux,
                            double smoothing_factor, double time_delta)
{
    double alpha;
    if (time_delta < 0.0f)
        time_delta = 0.0f;
    alpha = time_delta / (smoothing_factor + time_delta);
    return old_average + alpha * (new_lux - old_average);
}

static enum ab_state process_state_debounce(guint *delay_ms)
{
    gint64 now;
    double time_delta;
    double slow_lux, fast_lux;
    double hysteresis, slow_delta, fast_delta;
    enum ab_state next;
   
    next = AB_STATE_IDLE;
    *delay_ms = 0;

    now = get_ms();
    time_delta = (double)(now - ab_status.last_lux_ms);
    slow_lux = aggregate_lux(ab_status.average_slow, ab_status.last_lux,
                             SMOOTHING_FACTOR_SLOW, time_delta);
    fast_lux = aggregate_lux(ab_status.average_fast, ab_status.last_lux,
                             SMOOTHING_FACTOR_FAST, time_delta);
    powerd_debug("%" PRId64 " slow avg %f fast avg %f last %f",
                 now, slow_lux, fast_lux, ab_status.last_lux);

    /*
     * Require that both slow and fast averages exceed hysteresis,
     * and in the same direction.
     */
    hysteresis = ab_status.applied_lux * HYSTERESIS_FACTOR;
    if (hysteresis < 2)
        hysteresis = 2;
    slow_delta = slow_lux - ab_status.applied_lux;
    fast_delta = fast_lux - ab_status.applied_lux;
    if ((slow_delta >= hysteresis && fast_delta >= hysteresis) ||
        (-slow_delta >= hysteresis && -fast_delta >= hysteresis)) {
        int brightness = (int)(spline_interpolate(ab_spline, fast_lux) + 0.5);
        powerd_debug("set brightness %d", brightness);
        powerd_set_brightness(brightness);
        ab_status.applied_lux = fast_lux;
    }

    hysteresis = ab_status.last_lux * HYSTERESIS_FACTOR;
    if (hysteresis < 2)
        hysteresis = 2;
    if (fabs(fast_lux - ab_status.last_lux) >= hysteresis) {
        /*
         * Average should settle near the last reported lux. If
         * it hasn't made it there yet, continue debouncing.
         */
        next = AB_STATE_DEBOUNCE;
        *delay_ms = DEBOUNCE_MS;
    }

    return next;
}

static gboolean ab_process(gpointer unused)
{
    guint delay = 0;

    ab_status.source_id = 0;

    switch (ab_status.state) {
    case AB_STATE_DISABLED:
    case AB_STATE_INITIALIZING:
    case AB_STATE_IDLE:
        /* Nothing to do */
        break;
    case AB_STATE_DEBOUNCE:
        ab_status.state = process_state_debounce(&delay);
        break;
    default:
        powerd_warn("Unexpected autobrightness state %d\n", ab_status.state);
        ab_status.state = AB_STATE_IDLE;
        break;
    }

    if (delay != 0 && ab_status.state != AB_STATE_IDLE)
        ab_status.source_id = g_timeout_add(delay, ab_process, NULL);
    return FALSE;
}

static gboolean handle_new_lux(gpointer data)
{
    gint64 now;
    double lux;

    if (!ab_supported)
        return FALSE;

    now = get_ms();
    lux = *(double *)data;
    g_free(data);

    if (ab_status.state == AB_STATE_DISABLED)
        return FALSE;

    if (ab_status.state == AB_STATE_INITIALIZING) {
        int brightness = (int)(spline_interpolate(ab_spline, lux) + 0.5);
        powerd_debug("set brightness %d", brightness);
        powerd_set_brightness(brightness);

        ab_status.average_slow = lux;
        ab_status.average_fast = lux;
        ab_status.applied_lux = lux;
        ab_status.state = AB_STATE_IDLE;
    } else {
        double time_delta;
        /* Ignore duplicates */
        if (lux == ab_status.last_lux)
            return FALSE;
        time_delta = (double)(now - ab_status.last_lux_ms);
        ab_status.average_slow = aggregate_lux(ab_status.average_slow, lux,
                                               SMOOTHING_FACTOR_SLOW,
                                               time_delta);
        ab_status.average_fast = aggregate_lux(ab_status.average_fast, lux,
                                               SMOOTHING_FACTOR_FAST,
                                               time_delta);
    }
    ab_status.last_lux = lux;
    ab_status.last_lux_ms = now;

    if (ab_status.state == AB_STATE_IDLE) {
        ab_status.state = AB_STATE_DEBOUNCE;
        ab_status.source_id = g_timeout_add(DEBOUNCE_MS, ab_process, NULL);
    }

    return FALSE;
}

void powerd_new_als_event(double lux)
{
    double *p_lux = g_memdup(&lux, sizeof(lux));
    g_timeout_add(0, handle_new_lux, p_lux);
}

/* Must be run from main loop */
void powerd_autobrightness_enable(void)
{
    /* Must not leave disabled state if we can't map lux to brightness */
    if (!ab_supported)
        return;

    if (ab_status.state == AB_STATE_DISABLED) {
        ab_status.state = AB_STATE_INITIALIZING;
        powerd_sensors_als_enable();
    }
}

/* Must be run from main loop */
void powerd_autobrightness_disable(void)
{
    if (ab_status.state != AB_STATE_DISABLED) {
        ab_status.state = AB_STATE_DISABLED;
        powerd_sensors_als_disable();
        if (ab_status.source_id != 0)
            g_source_remove(ab_status.source_id);
    }
}

int powerd_autobrightness_init(void)
{
    GValue v = G_VALUE_INIT;
    GArray *levels = NULL, *lux = NULL;
    double (*mappings)[2] = NULL;
    int i;
    int ret = -ENODEV;

    if (device_config_get("automatic_brightness_available", &v)) {
        powerd_info("Could not determine platform autobrightness capability, disabling");
        goto error;
    }

    if (!G_VALUE_HOLDS_BOOLEAN(&v) || !g_value_get_boolean(&v)) {
        powerd_info("Platform does not support autobrightness, disabling");
        goto error;
    }
    g_value_unset(&v);

    if (device_config_get("autoBrightnessLevels", &v)) {
        powerd_info("No device-specific autobrightness lux table defined, disabling");
        goto error;
    }
    if (G_VALUE_HOLDS_BOXED(&v))
        lux = g_value_dup_boxed(&v);
    g_value_unset(&v);
    if (!lux || lux->len == 0) {
        powerd_info("Invalid autobrighness lux levels, disabling");
        goto error;
    }

    if (device_config_get("autoBrightnessLcdBacklightValues", &v)) {
        powerd_info("No device-specific autobrightness backlight levels defined, disabling");
        goto error;
    }
    if (G_VALUE_HOLDS_BOXED(&v))
        levels = g_value_dup_boxed(&v);
    g_value_unset(&v);
    if (!levels || levels->len == 0) {
        powerd_info("Invalid autobrighness backlight levels, disabling");
        goto error;
    }

    /* We should have one more backlight level than lux values */
    if (levels->len != lux->len + 1) {
        powerd_info("Invalid lux->brightness mappings, autobrightness disabled");
        goto error;
    }

    mappings = g_malloc(2 * sizeof(double) * levels->len);
    for (i = 0; i < levels->len; i++) {
        mappings[i][0] = (double)((i == 0) ? 0 : g_array_index(lux, guint, i-1));
        mappings[i][1] = (double)g_array_index(levels, guint, i);
    }

    ab_status.state = AB_STATE_DISABLED;
    ab_spline = spline_new(mappings, levels->len);
    if (!ab_spline) {
        ret = -ENOMEM;
        goto error;
    }
    g_free(mappings);
    ab_supported = TRUE;
    return 0;

error:
    if (levels)
        g_array_unref(levels);
    if (lux)
        g_array_unref(lux);
    if (mappings)
        g_free(mappings);
    return ret;
}

void powerd_autobrightness_deinit(void)
{
    powerd_autobrightness_disable();
    ab_supported = FALSE;
    if (ab_spline)
        spline_free(ab_spline);
}
