/***************************************************************************
 *
 * pm-cmd.c : DBUS test program
 *
 * Copyright (C) 2005 Richard Hughes, <richard@hughsie.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 **************************************************************************/

#include <glib.h>
#include <string.h>
#include <dbus/dbus-glib.h>
#include "pm-main.h"

/** Prints program usage.
 *
 */
static void print_usage (void)
{
	g_print ("\nusage : PowerManagerCmd [options]\n");
	g_print (
		"\n"
		"    --help             Show this information and exit\n"
		"\n"
		" options\n"
		"    --isActive         returns TRUE if service active\n"
		"    --shutdown         shuts the computer down\n"
		"                       (runs /usr/sbin/pm-shutdown)\n"
		"    --suspend          suspends the computer\n"
		"                       (runs /usr/sbin/pm-suspend)\n"
		"    --hibernate        hibernates the computer\n"
		"                       (runs /usr/sbin/pm-hibernate)\n"
		"    --restart          restarts the computer\n"
		"                       (runs /usr/sbin/pm-restart)\n"
		"    --hdparm           sets /dev/hda timeout to 10 seconds\n"
		"\n");
}

static void
process_error (GError *error)
{
		if (error->domain == DBUS_GERROR && error->code == DBUS_GERROR_REMOTE_EXCEPTION)
			g_printerr ("Caught remote method exception %s: %s\n",
						dbus_g_error_get_name (error),
						error->message);
		else
			g_printerr ("Error: %s\n", error->message);
		g_error_free (error);
}

int
main (int argc, char **argv)
{
	GMainLoop *loop;
	DBusGConnection *system_connection;
	DBusGProxy *pm_proxy;
	GError *error = NULL;

	/* initialise threads */
	g_type_init ();
	if (!g_thread_supported ())
		g_thread_init (NULL);
	dbus_g_thread_init ();

	/* Create a new event loop to run in */
	loop = g_main_loop_new (NULL, FALSE);

	/* Get a connection to the system connection */
	system_connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
	if (system_connection == NULL) {
		g_printerr ("Failed to open connection to bus: %s\n", error->message);
		g_error_free (error);
		return 0;
	}

	pm_proxy = dbus_g_proxy_new_for_name (system_connection,
							PM_DBUS_SERVICE,
							PM_DBUS_PATH,
							PM_DBUS_INTERFACE);

	gboolean isOkay = FALSE;

	int a;
	gboolean boolret;
	for (a=1; a < argc; a++) {
		if (strcmp (argv[a], "--help") == 0) {
			print_usage ();
			return 1;
		} else if (strcmp (argv[a], "--isActive") == 0) {
			isOkay = TRUE;
			if (!dbus_g_proxy_call (pm_proxy, "isActive", &error, 
					G_TYPE_INVALID,
					G_TYPE_BOOLEAN, &boolret, G_TYPE_INVALID))
				process_error (error);
			g_print (boolret ? "TRUE\n" : "FALSE\n");
		} else if (strcmp (argv[a], "--hdparm") == 0) {
			isOkay = TRUE;
			char *device = "/dev/hda";
			int timeout = 60;
			if (!dbus_g_proxy_call (pm_proxy, "hdparm", &error, 
					G_TYPE_INT, timeout, G_TYPE_STRING, device, G_TYPE_INVALID,
					G_TYPE_BOOLEAN, &boolret, G_TYPE_INVALID))
				process_error (error);
			g_print (boolret ? "TRUE\n" : "FALSE\n");
		} else if (strcmp (argv[a], "--shutdown") == 0 || 
			   strcmp (argv[a], "--restart") == 0 ||
			   strcmp (argv[a], "--hibernate") == 0 ||
			   strcmp (argv[a], "--suspend") == 0) {
			isOkay = TRUE;
			if (!dbus_g_proxy_call (pm_proxy, argv[a]+2, &error, 
					G_TYPE_INVALID,
					G_TYPE_BOOLEAN, &boolret, G_TYPE_INVALID))
				process_error (error);
			g_print (boolret ? "TRUE\n" : "FALSE\n");
		}
	}
	if (!isOkay)
		print_usage ();

	g_object_unref (pm_proxy);
	return 0;
}
