/*
 * Potamus: an audio player
 * Copyright (C) 2004, 2005, 2006, 2007, 2009 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <jack/jack.h>
#include <jack/ringbuffer.h>
#include <time.h>
#include <glib.h>
#include "buffer.h"
#include "format.h"
#include "output.h"
#include "output-jack.h"

// FIXME: flush the output buffer on a seek.

#define NUM_PORTS 2

typedef struct output_jack {
	jack_client_t *client;
	jack_ringbuffer_t *ringbuf;
	jack_port_t **ports;
	sample_format fmt;
} output_jack;

static int oj_play(struct output *o, buffer *buf, sample_format *fmt) {
	output_jack *a = (output_jack *) o->data;

	while (buf->used > 0) {
		// Move data into the ringbuf (as is).
		size_t n = jack_ringbuffer_write(a->ringbuf,
		                                 (const char *) buf->data,
		                                 buf->used);
		buffer_remove(buf, n);

		if (n == 0) {
			// The ringbuf is full -- wait a bit before trying again.
			struct timespec ts = {.tv_sec = 0, .tv_nsec = 1000000};
			nanosleep(&ts, NULL);
		}
	}

	return 0;
}

static int oj_process_callback(jack_nframes_t nframes, void *arg) {
	output_jack *a = (output_jack *) arg;

	float *outs[NUM_PORTS];
	for (int i = 0; i < NUM_PORTS; i++) {
		outs[i] = jack_port_get_buffer(a->ports[i], nframes);
	}

	jack_nframes_t outspace = nframes;
	size_t inread = 0;

	jack_ringbuffer_data_t vec[2];
	jack_ringbuffer_get_read_vector(a->ringbuf, vec);

	// Unpack data from the ringbuf into the port buffers.
	int e = 0;
	const float *inp = (const float *) vec[e].buf;
	size_t left = vec[e].len;
	while (left > 0 && e < 2 && outspace > 0) {
		for (int i = 0; i < NUM_PORTS; i++) {
			*(outs[i]++) = *inp++;
			inread += sizeof(float);
			left -= sizeof(float);
			if (left == 0) {
				++e;
				inp = (const float *) vec[e].buf;
				left = vec[e].len;
			}
		}
		--outspace;
	}

	jack_ringbuffer_read_advance(a->ringbuf, inread);

	// Fill any remaining space in the port buffers with zeros.
	while (outspace > 0) {
		for (int i = 0; i < NUM_PORTS; i++) {
			*(outs[i]++) = 0.0;
		}
		--outspace;
	}

	return 0;
}

static void oj_release(struct output *o) {
	// Doesn't need to do anything.
}

static void oj_close(struct output *o) {
	output_jack *a = (output_jack *) o->data;

	if (jack_deactivate(a->client) != 0)
		g_error("jack_deactivate failed");
	for (int i = 0; i < NUM_PORTS; i++) {
		if (jack_port_unregister(a->client, a->ports[i]) != 0)
			g_error("jack_port_unregister failed");
	}
	free(a->ports);
	if (jack_client_close(a->client) != 0)
		g_error("jack_client_close failed");
	jack_ringbuffer_free(a->ringbuf);

	free(a);
	free(o);
}

output *output_new_jack(sample_format **force_fmt) {
	output *o = output_alloc();

	output_jack *a = malloc(sizeof *a);
	if (a == NULL)
		g_error("out of memory");
	a->client = NULL;
	a->ringbuf = NULL;
	o->data = a;

	o->play = oj_play;
	o->release = oj_release;
	o->close = oj_close;

	a->client = jack_client_open("potamus", JackNoStartServer, NULL);
	if (a->client == NULL)
		goto out;

	if (jack_set_process_callback(a->client, oj_process_callback, a) != 0)
		g_error("jack_set_process_callback failed");

	a->ports = malloc(NUM_PORTS * sizeof *a->ports);
	for (int i = 0; i < NUM_PORTS; i++) {
		char name[10];
		snprintf(name, sizeof name, "output_%d", i + 1);

		a->ports[i] = jack_port_register(a->client, name,
		                                 JACK_DEFAULT_AUDIO_TYPE,
		                                 JackPortIsOutput, 0);
		if (a->ports[i] == NULL)
			g_error("jack_port_register failed");
	}

	a->fmt.bits = BITS_FLOAT;
	a->fmt.channels = NUM_PORTS;
	a->fmt.byte_format = END_NATIVE;
	a->fmt.rate = jack_get_sample_rate(a->client);
	*force_fmt = &a->fmt;

	// Allocate a half-second ringbuffer.
	const int ringbuf_size = a->fmt.rate * a->fmt.channels * sizeof(float)
	                         / 2;
	a->ringbuf = jack_ringbuffer_create(ringbuf_size);

	if (jack_activate(a->client) != 0)
		g_error("jack_activate failed");

	// Connect to hardware ports automatically.
	// (This is arguably broken behaviour, but it's what the bio2jack
	// backend used to do, and it's generally what the user actually
	// wants.)
	const char **hw_ports = jack_get_ports(a->client, NULL,
	                                       JACK_DEFAULT_AUDIO_TYPE,
	                                       JackPortIsInput | JackPortIsPhysical);
	const char **p = hw_ports;
	for (int i = 0; i < NUM_PORTS; i++) {
		if (*p != NULL)
			jack_connect(a->client, jack_port_name(a->ports[i]), *p++);
	}
	free(hw_ports);

	return o;

out:
	if (a->ringbuf != NULL)
		jack_ringbuffer_free(a->ringbuf);
	if (a->client != NULL)
		jack_client_close(a->client);
	free(a);
	free(o);
	return NULL;
}
