/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

public class Postler.Viewer : Gtk.VBox {
    Gtk.InfoBar infobar;
    Postler.Content content;
    Postler.Attachments attachments;
    Gtk.Toolbar findbar;
    Gtk.Entry find_entry;

    bool allow_external_images;

    public Viewer (Postler.Content content) {
        this.content = content;

        infobar = new Gtk.InfoBar ();
        pack_start (infobar, false, false, 0);
        var infobox = infobar.get_content_area () as Gtk.Box;
        var image = new Gtk.Image.from_stock (STOCK_IMAGE, Gtk.IconSize.BUTTON);
        infobox.pack_start (image, false, false, 0);
        var infolabel = new Gtk.Label (_("External images were blocked."));
        var attrlist = new Pango.AttrList ();
        attrlist.insert (Pango.attr_weight_new (Pango.Weight.BOLD));
        infolabel.attributes = attrlist;
        infobox.pack_start (infolabel, true, true, 0);
        infobar.add_button (_("Load external _Images"), Gtk.ResponseType.OK);
        infobar.response.connect (infobar_response);
        infobar.set_no_show_all (true);
        content.resource_request_starting.connect (resource_request);

        var scrolled = new Postler.ScrolledWindow (content);
        pack_start (scrolled, true, true, 0);

        attachments = new Postler.Attachments ();
        attachments.icon_size = 24;
        pack_start (attachments, false, false, 0);

        notify_message_parts (content, null);
        content.notify["message-parts"].connect (notify_message_parts);
        content.notify["current-part"].connect (notify_current_part);
        attachments.selection_changed.connect (part_selected);

        findbar = new Gtk.Toolbar ();
        pack_start (findbar, false, false, 0);
        find_entry = new Gtk.Entry ();
        var toolitem = new Gtk.ToolItem ();
        toolitem.add (find_entry);
        toolitem.show_all ();
        findbar.insert (toolitem, -1);
        findbar.set_no_show_all (true);
        key_press_event.connect (key_pressed);
        find_entry.changed.connect (find_entry_changed);
        find_entry.activate.connect (find_entry_changed);
        find_entry.focus_out_event.connect (find_entry_unfocus);
    }

    void resource_request (WebKit.WebFrame frame, WebKit.WebResource resouce,
        WebKit.NetworkRequest request, WebKit.NetworkResponse? response) {
        if (!allow_external_images && request.uri.has_prefix ("http")) {
            request.uri = "about:blank";
            infobar.set_no_show_all (false);
            infobar.show_all ();
        }
        if (request.uri.has_prefix ("cid:")) {
            string content_id = request.uri.substring (4, -1);
            foreach (var part in content.message_parts) {
                if (part.content_id == content_id) {
                    request.uri = "data:" + part.mime_type + ";base64," + part.body.str;
                    attachments.remove_part (part);
                    break;
                }
            }
        }
    }

    void infobar_response (Gtk.InfoBar infobar, int response) {
        infobar.hide ();
        allow_external_images = true;
        content.display_part (content.current_part);
    }

    void notify_message_parts (GLib.Object object, GLib.ParamSpec? pspec) {
        infobar.hide ();
        allow_external_images = false;

        if (content.message_parts.length () < 2) {
            attachments.hide ();
            return;
        }

        (attachments.model as Gtk.ListStore).clear ();
        foreach (var part in content.message_parts) {
            if (part.mime_type.has_prefix ("multipart/"))
                continue;
            if (part == content.html_part || part == content.text_part)
                continue;

            attachments.add_part (part);
            if (part == content.current_part)
                attachments.select (content.current_part);
        }
    }

    void notify_current_part () {
        attachments.select (content.current_part);
    }

    void part_selected () {
        MessagePart? part = attachments.get_selected_part ();
        if (part != null)
            content.display_part (part);
    }

    bool key_pressed (Gdk.EventKey event) {
        uint character = Gdk.unicode_to_keyval (event.keyval);
        if (character == (event.keyval | 0x01000000))
            return false;

        string text = "%c".printf ((char)character);
        find_entry.text = text;
        findbar.show ();
        find_entry.grab_focus ();
        content.search_text (text, false, true, true);
        return true;
    }

    void find_entry_changed () {
        content.search_text (find_entry.text, false, true, true);
    }

    bool find_entry_unfocus (Gdk.EventFocus event) {
        find_entry.text = "";
        findbar.hide ();
        return false;
    }
}

