/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

public class Postler.MessagePart : Object {
    public StringBuilder? body;
    public string? mime_type;
    public string? charset;
    public bool plain_text;
    public string? content_disposition;
    public string? content_id;
    public string? filename;
    public MessagePart (string mime_type) {
        body = new StringBuilder ();
        this.mime_type = mime_type;
        plain_text = mime_type == "text/plain";
        filename = null;
    }

    public bool is_empty () {
        bool empty = true;
        if (!(body == null)) {
            var b_str = body.str.strip ();
            string[] b_str_parts = b_str.split ("<br>");
            long str_length = 0;
            foreach (var current_line in b_str_parts) {
                string[] split_split = current_line.strip().split("\n");
                foreach (var between_linebreaks in split_split)
                str_length = between_linebreaks.strip().length;
            }
            if (str_length != 0)
                empty = false;
        }
        return empty;
    }

    public string get_plain_text () {
        string body_plain = body.str;
        if ((body != null) && (mime_type != null)) {
            try {
                string plain_text = "";
                int std_in;
                int std_out;
                string[] argv = { "lynx", "-force_html", "-width=900", "-dump",
                                  "-display_charset=utf-8", "-stdin" };
                Process.spawn_async_with_pipes (null, argv, null,
                    SpawnFlags.SEARCH_PATH, null, null,
                    out std_in, out std_out, null);
                IOChannel ioc_stdout = new IOChannel.unix_new (std_out);
                IOChannel ioc_stdin = new IOChannel.unix_new (std_in);
                char[] content_chars = (char[]) (body.str.data);
                var status = ioc_stdin.write_chars (content_chars, null);
                ioc_stdin.shutdown (true);
                if (status != IOStatus.ERROR)
                    status = ioc_stdout.read_to_end (out plain_text, null);
                if (status != IOStatus.ERROR)
                    body_plain = plain_text;
            } catch (Error error) {
                GLib.message (_("Error converting HTML to text: %s"), error.message);
            }
        }
        return body_plain;
    }
}

struct Postler.EmoticonMapping {
    public string token;
    public string icon_name;
}

public class Postler.Content : WebKit.WebView {
    AccountInfo? selected_account;
    string charset;
    string content_encoding;
    string date;
    string carbon_copy;
    string blind_copy;
    string reply;
    string organization;
    string x_mailer;

    public string default_charset { get; set; default = "ISO-8859-1"; }

    public string? last_location { get; set; }
    public string? message_id { get; set; }
    public string? sender { get; set; }
    public string? recipient { get; set; }
    public string? subject { get; set; }
    public string? reply_to { get; set; }
    public string? reply_to_all { get; set; }
    public string list_post { get; set; }
    public string list_unsubscribe { get; set; }

    public unowned List<MessagePart> message_parts { public get; set; }
    public MessagePart? current_part { get; private set; }
    public int current_part_index { get; private set; }
    public MessagePart? html_part { get; private set; }
    public MessagePart? text_part { get; private set; }

    const string style_sheet = """
        .headers {
            display: block;
            background-color: Window !important; color: WindowText !important;
            font: 9pt sans-serif !important;
        }
        .headers .extra_headers {
            display: none;
        }
        .headers:hover .extra_headers {
            display: inline;
        }
        .body {
            background-color: Window !important; color: WindowText !important;
            font: 10pt sans-serif !important;
        }
        /* Quotations */
        blockquote {
            margin: 1em;
            padding: 0.5em;
            white-space: pre;
            color: GrayText;
        }
        /* Plain text */
        .plain_text {
            white-space: pre-wrap;
            font-family: monospace;
        }
        /* Signature */
        .signature {
            color: GrayText;
        }
        .signature:before {
            content: "——";
        }
        /* Addresses not underlined, but underlined when hovering */
        a[href] {
            text-decoration: underline !important;
        }
        a[href]:hover {
            text-decoration: none !important;
        }
        /* Dynamically sized and visible statusbar */
        a[href^=http]:hover:after {
            content: attr(href);
            position: fixed; left: 1pt; bottom: 1pt;
            padding: 0 1pt !important;
            max-width: 95%; overflow: hidden;
            white-space: nowrap; text-overflow: ellipsis;
            font:10pt sans-serif !important; text-shadow: 0 0 5pt ButtonHighlight;
            background-color: ButtonFace !important; color: ButtonText !important;
            outline: ButtonFace solid thick; z-index: 9999;
        }
        /* Contain images to the size of the view */
        img {
            max-height:100%; max-width:100%;
        }
        pre { white-space: pre-wrap; }
        """;

    public Content () {
        settings.set ("enable-scripts", false, "enable-plugins", false);
        if (settings.get_class ().find_property ("enable-private-browsing") != null) {
            settings.set ("enable-private-browsing", true);
        }
        navigation_policy_decision_requested.connect (navigation_decision);
        create_web_view.connect (new_window_created);
        populate_popup.connect (populate_menu);
    }

    internal static string format_x_mailer (string x_mailer) {
        string mailer = x_mailer.chomp ();
        if (mailer.has_prefix ("Mozilla/") && ") " in mailer)
            return mailer.rchr (-1, ' ').chug ().replace ("/", " ");

        int bracket = 0;
        while (mailer[bracket] != '(' && mailer[bracket] != '\0')
            bracket++;
        if (mailer[bracket] == '\0')
            return mailer;
        return mailer.substring (0, bracket).chomp ().replace ("/", " ");
    }

    static string html_escape (string unescaped) {
        return unescaped.replace ("\"", "&quot;")
            .replace ("<", "&lt;").replace (">", "&gt;");
    }

    internal static string linkify_address (string addresses, string? arguments)
        requires (arguments == null || !arguments.contains ("<")) {
        var linkified = new StringBuilder ();

        string edit_icon = "";
        var screen = Gdk.Screen.get_default ();
        if (screen != null) {
            var icon_theme = Gtk.IconTheme.get_for_screen (screen);
            if (icon_theme.has_icon ("not-starred"))
                edit_icon = "<img src=\""
                    + icon_name_to_data_uri ("not-starred") + "\">";
            else
                edit_icon = "<b>←</b>";
        }

        foreach (string address in addresses.split (",")) {
            if (address == "")
                continue;
            if (!address.contains ("@")) {
                linkified.append (html_escape (address));
                continue;
            }
            string[] parsed = Postler.Messages.parse_address (address);
            string name = html_escape (parsed[0]);
            string quoted = html_escape (address);
            if (linkified.len > 0)
               linkified.append (", ");
            linkified.append ("<a href=\"" + quoted + (arguments != null ? arguments : "") +
                "\" title=\"" + quoted + "\">" + name + "</a>");
            if (edit_icon != "")
                linkified.append ((" <a href=\"contact:%s:%s\">%s</a>").printf (
                                  name, html_escape (parsed[1]), edit_icon));
        }
        return linkified.str;
    }

    void populate_menu (Gtk.Menu menu) {
        if (editable || can_copy_clipboard ())
            return;

        menu.hide_all ();

        var event = Gtk.get_current_event ();
        var result = get_hit_test_result ((Gdk.EventButton?)event);
        Gtk.MenuItem menuitem;

        if (result.link_uri != null) {
            menuitem = new Gtk.ImageMenuItem.with_mnemonic (_("Copy _Address"));
            var icon = new Gtk.Image.from_stock (Gtk.STOCK_COPY, Gtk.IconSize.MENU);
            (menuitem as Gtk.ImageMenuItem).set_image (icon);
            menuitem.activate.connect ((menuitem) => {
                var clipboard = get_clipboard (Gdk.SELECTION_CLIPBOARD);
                string address = result.link_uri;
                if ("@" in address)
                    clipboard.set_text (address.split ("?")[0], -1);
                else
                    clipboard.set_text (address, -1);
            });
            menuitem.show ();
            menu.append (menuitem);
            return;
        }

        menuitem = new Gtk.ImageMenuItem.from_stock (Gtk.STOCK_COPY, null);
        menuitem.activate.connect ((menuitem) => {
            copy_clipboard (); });
        menuitem.show ();
        menuitem.sensitive = can_copy_clipboard ();
        menu.append (menuitem);
        menuitem = new Gtk.MenuItem.with_mnemonic (_("Copy _Filename"));
        menuitem.activate.connect ((menuitem) => {
            var clipboard = get_clipboard (Gdk.SELECTION_CLIPBOARD);
            clipboard.set_text (last_location, -1);
        });
        menuitem.show ();
        menu.append (menuitem);

        menuitem = new Gtk.SeparatorMenuItem ();
        menuitem.show ();
        menu.append (menuitem);
        menuitem = new Gtk.ImageMenuItem.from_stock (Gtk.STOCK_ZOOM_IN, null);
        menuitem.label = _("_Enlarge Text");
        menuitem.activate.connect ((menuitem) => {
            zoom_in ();
        });
        menuitem.show ();
        menu.append (menuitem);
        menuitem = new Gtk.ImageMenuItem.from_stock (Gtk.STOCK_ZOOM_OUT, null);
        menuitem.label = _("Sh_rink Text");
        menuitem.activate.connect ((menuitem) => {
            zoom_out ();
        });
        menuitem.show ();
        menu.append (menuitem);

        if (get_view_source_mode ())
            return;

        menuitem = new Gtk.SeparatorMenuItem ();
        menuitem.show ();
        menu.append (menuitem);
        menuitem = new Gtk.MenuItem.with_mnemonic (_("View _Source"));
        menuitem.activate.connect ((menuitem) => {
            Postler.App.spawn_module ("source", last_location); });
        menuitem.show ();
        menuitem.sensitive = last_location != null;
        menu.append (menuitem);
        menuitem = new Gtk.ImageMenuItem.from_stock (Gtk.STOCK_PRINT, null);
        menuitem.activate.connect ((menuitem) => {
            get_main_frame ().print (); });
        menuitem.show ();
        menuitem.sensitive = last_location != null;
        menu.append (menuitem);
    }

    public virtual signal void clear () {
        last_location = null;
        reply_to = null;
        reply_to_all = null;
        message_parts = null;
        notify_property ("message-parts");
        load_string ("", "text/plain", "UTF-8", "about:blank");
    }

    const EmoticonMapping[] emoticons = {
        { ":-D", "face-smile-big" },
        { ":-(", "face-sad" },
        { ":'-(", "face-crying" },
        { "O:-)", "face-angel" },
        { ":-|", "face-plain" },
        { ":-*", "face-kiss" },
        { ":-)", "face-smile" },
        { ";-)", "face-wink" },
        { ":-O", "face-surprise" },
        { "8-)", "face-glasses" },
        { ":-P", "face-monkey" },
        { ":)", "face-smile" },
        { "xD", "face-smile-big" },
        { ";)", "face-wink" },
        { "^_^", "face-smile" },
        { "~_^", "face-wink" },
        { "^_~", "face-wink" }
    };

    static string? icon_name_to_data_uri (string icon_name) {
        Gdk.Pixbuf? pixbuf = null;
        var toplevels = Gtk.Window.list_toplevels ();
        if (toplevels.nth_data (0) != null)
            pixbuf = toplevels.nth_data (0).render_icon (icon_name,
                Gtk.IconSize.MENU, null);
        if (pixbuf == null) {
            try {
                pixbuf = Gtk.IconTheme.get_for_screen (
                    Gdk.Screen.get_default ()).load_icon (icon_name,
                        Gtk.IconSize.MENU, Gtk.IconLookupFlags.FORCE_SIZE);
            } catch (GLib.Error error) { }
        }
        if (pixbuf == null) {
            return null;
        }
        string buffer;
        size_t buffer_size;
        try {
            if (!Gdk.pixbuf_save_to_buffer (pixbuf, out buffer,
                out buffer_size, "png", null, null)) {
                return null;
            }
        }
        catch (GLib.Error error) {
            return null;
        }

        string encoded = GLib.base64_encode (buffer, buffer_size);
        return "data:image/png;base64," + encoded;
    }

    static bool evaluate_emoticon (GLib.MatchInfo     match_info,
                                   GLib.StringBuilder result,
                                   EmoticonMapping    emoticon) {
        string? data_uri = icon_name_to_data_uri (emoticon.icon_name);
        if (data_uri == null) {
            result.append (" " + emoticon.token);
            return true;
        }

        result.append (" <img src=\"" + data_uri + "\" " +
                       "title=\"" + emoticon.token + "\"/>");
        return false;
    }

    const string[] link_formats = {
        "(https?:\\/\\/[^\\s\\<]+)",
        "([a-zA-Z0-9.\\-]+@[a-zA-Z0-9.\\-]+[a-zA-Z0-9.]+)"
    };

    static bool evaluate_hex (GLib.MatchInfo     match_info,
                              GLib.StringBuilder result) {
        string match = "0x" + match_info.fetch (2);
        result.append_printf ("%c", (int)match.to_ulong (null));
        return false;
    }

    internal static string quoted_printable_decode (string quoted) {
        /* =20 is not hex-encoded, to faciliate quote handling */
        string unquoted = quoted.replace ("=20", " ");
        try {
            var regex = new GLib.Regex ("([=]([0-9A-F][0-9A-F]))");
            return regex.replace_eval (unquoted, -1, 0, 0, evaluate_hex);
        }
        catch (GLib.RegexError error) {
            GLib.critical (_("Failed to decode string \"%s\": %s"),
                           unquoted, error.message);
        }
        return unquoted;
    }

    static string parse_encoded (string quoted, out string charset) {
        return Postler.Messages.parse_encoded (quoted, out charset);
    }

    string format_header (string header, string data) {
        if (data != "")
            return "<b>%s</b> %s<br>".printf (header, data);
        return "";
    }

    internal static string format_date (DateTime the_time) {
        var now = new DateTime.now_local ();
        if (the_time.get_day_of_month () == now.get_day_of_month ())
            return _("Today") + the_time.format (" %X");
        if (the_time.get_day_of_year () == now.get_day_of_year () - 1)
            return _("Yesterday") + the_time.format (" %X");
        if (the_time.get_year () == now.get_year ()) {
            /* i18n: strftime format for full month name and day number */
            return the_time.format (_("%B %e") + (" %X"));
        }
        /* i18n: strftime format for full month name, day number and year */
        return the_time.format (_("%B %e, %Y") + (" %X"));
    }

    internal static string format_timestamp (int64 timestamp, int offset=0) {
        var the_time = new DateTime.from_unix_local (timestamp).add_minutes (offset);
        return format_date (the_time);
    }

    internal static DateTime date_from_string (string date) {
        var parsed = new Soup.Date.from_string (date);
        TimeVal time = new TimeVal ();
        int offset = 0;
        if (parsed != null) {
            parsed.to_timeval (time);
            offset = parsed.offset;
        }
        return new DateTime.from_timeval_local (time).add_minutes (offset);
    }

    public void prepare_reply (string? location, AccountInfo account_info,
        string recipient, bool quote, int part, string? excerpt) {

        var body = new StringBuilder ();

        if (location != null) {
            ulong now = new Soup.Date.from_now (0).to_time_t ();
            string date = GLib.Time.local ((time_t)now).format ("%x %X");
            string sender = Postler.Messages.parse_address (recipient)[0];

            body.append_printf (_("On %s, %s wrote:"), date, sender);
            body.append_c ('\n');
            var contents = File.new_for_path (location);
            try {
                parse_message(location);
                bool last_line_empty = true;
                StringBuilder b_temp = new StringBuilder ();
                if (excerpt != null)
                    b_temp.append (excerpt);
                else if (mime_type_is_text ((message_parts.nth_data (part)).mime_type)) {
                    b_temp.append ((message_parts.nth_data (part)).get_plain_text ());
                } else {
                    foreach (var bpart in message_parts) {
                        if (mime_type_is_text (bpart.mime_type)) {
                            if (!bpart.is_empty ()) {
                                b_temp.append_c ('\n');
                                b_temp.append (bpart.get_plain_text ());
                                break;
                            }
                        }
                    }
                }
                string [] lines = b_temp.str.split ("\n");
                foreach (var line in lines) {
                    /* Skip effectively empty lines */
                    int position = 0;
                    do {
                        if (line[position] != '>' && line[position] != ' ')
                            break;
                        position++;
                    } while (true);
                    string effective = line.substring (position);
                    if (effective == "") {
                        if (last_line_empty)
                            continue;
                        last_line_empty = true;
                        body.append ("> \n");
                        continue;
                    }
                    /* Skip non-helpful decorative lines */
                    if (effective == "-----BEGIN PGP SIGNED MESSAGE-----"
                     || effective.has_prefix ("Hash: SHA")
                     || effective == "-----END PGP SIGNATURE-----"
                     || effective == "-----BEGIN PGP SIGNATURE-----")
                        continue;
                    last_line_empty = false;
                    string[] words_in_line = line.split (" ");
                    long len = 0;
                    body.append ((quote ? "> " : ""));
                    for (int i = 0; i < words_in_line.length; i++)
                    {
                        len += words_in_line[i].length + 1;
                        if (len < (quote ? 70 : 72)) {
                            body.append (words_in_line[i]);
                            body.append_c (' ');
                        } else {
                            body.append_c ('\n');
                            body.append ((quote ? "> " : ""));
                            body.append (words_in_line[i]);
                            body.append_c (' ');
                            len = 0;
                        }
                    }
                    body.append ("\n");
                }
            } catch (GLib.Error error) {
                display_error (_("Failed to quote message \"%s\": %s").printf (
                    contents.get_path (), error.message));
            }
        }

        /* If the original message was top-post, we also top-post.
           Presumably more quotes than none at the bottom indicates that. */
        bool top_post = false;
        ssize_t position = body.len;
        uint line_breaks = 0;
        uint quote_lines = 0;
        do {
            if (body.str[position] == ' ' && body.str[position - 1] == '>') {
                quote_lines++;
                if (quote_lines > 5) {
                    top_post = true;
                    break;
                }
                position--;
            } else if (body.str[position] == '\n') {
                line_breaks++;
                if (line_breaks > 5)
                    break;
            }
            position--;
        } while (position > 0 && position < body.len);

        unowned string? signature = account_info.signature;
        if (signature != null && top_post)
            body.prepend ("\n\n" + signature + "\n");
        else if (signature != null)
            body.append ("\n\n" + signature);
        load_string (body.str, "text/plain", "UTF-8", "about:blank");
    }

    static void parse_content_type (string? content_type, ref string charset,
        ref string boundary, ref string mime_type, ref string filename) {
        if (content_type == null)
            return;

        string[] parts = content_type.replace (" = ", "=").split_set ("; \t");
        filename = null;
        for (int i = 0; i < parts.length; i++) {
            string part = parts[i];
            unowned string? next = parts[i + 1];
            if ("\"" in part && next != null && "\"" in next) {
                part += parts[i + 1];
                i++;
            }

            if (part.has_prefix ("charset="))
                charset = part.substring (8, part.length - 8).replace ("\"", "");
            else if (part.has_prefix ("name=") || part.has_prefix ("NAME="))
                filename = part.substring (5, part.length - 5).replace ("\"", "");
            else if (part.has_prefix ("filename="))
                filename = part.substring (9, part.length - 9).replace ("\"", "");
            else if (part.down ().has_prefix ("boundary="))
                boundary = part.substring (9, part.length - 9).replace ("\"", "");
            else if (part != "" && !part.contains ("="))
                mime_type = ascii_strdown (part);
        }

        if (mime_type == "application/octet-stream" && filename != null) {
            uchar[] data = {};
            bool uncertain;
            mime_type = g_content_type_guess (filename, data, out uncertain);
        }
    }

    bool mime_type_is_text (string mime_type) {
        return g_content_type_is_a (mime_type, "text/plain");
    }

    public void display_source (string location) {
        last_location = location;
        subject = _("Source Code: %s").printf (location);

        string line;
        string content_type = null;
        string mime_type = "text/plain";
        string charset = null;
        string[] parts;

        var contents = File.new_for_path (location);
        GLib.StringBuilder body = new GLib.StringBuilder ();

        try {
            var stream = new DataInputStream (contents.read (null));
            while ((line = stream.read_line (null, null)) != null) {
                parts = line.split (":", 2);
                if (parts != null && parts[0] != null) {
                    string field = ascii_strdown (parts[0]);
                    if (field == "content-type")
                        content_type = parts[1].strip ();
                    else if (field == "subject")
                        subject = _("Source Code: %s").printf (parse_encoded (parts[1], out charset));
                }
                body.append (line + "\n");
            }
        } catch (GLib.Error error) {
            display_error (_("Failed to view source: %s").printf (error.message));
            return;
        }

        string? boundary = null;
        string? fname = null;
        parse_content_type (content_type, ref charset, ref boundary,
                            ref mime_type, ref fname);
        set_view_source_mode (true);
        load_string (body.str, "text/plain", charset, "about:blank");
    }

    string selected_address () {
        return selected_account != null ? (selected_account.address ?? "") : "";
    }

    public string? choose_from () {
        /* See if recipient is among the accounts, otherwise pick a fallback */
        if (recipient != null) {
            string from = Postler.Messages.parse_address (recipient)[1];
            var accounts = new Accounts ();
            foreach (var info in accounts.get_infos ())
                if (info.address != null && from in info.address)
                    return from;
        }
        if (selected_account == null || selected_account.address == null)
            return null;

        return selected_account.address.split (",")[0];
    }

    void parse_message (string location) throws GLib.FileError {
        last_location = location;
        charset = null;
        subject = _("(No subject)");

        message_parts = new List<MessagePart> ();
        var contents = File.new_for_path (location);
        try {
            var stream = new DataInputStream (contents.read (null));
            string line;
            string content_type = null;
            string[] parts;

            message_id = null;
            content_encoding = "";
            string from = _("Unknown");
            date = _("(No date)");
            recipient = "";
            carbon_copy = "";
            blind_copy = "";
            reply = "";
            list_post = "";
            list_unsubscribe = "";
            organization = "";
            x_mailer = "";
            string previous_line = "";
            while ((line = stream.read_line (null, null)) != null) {
                if (line == "")
                    break;
                if (line[0] == '\t' || line[0] == ' ')
                    line = previous_line + " " + line.chug ();
                previous_line = line;

                parts = line.split (":", 2);
                if (parts == null || parts[0] == null)
                    continue;

                string field = ascii_strdown (parts[0]);
                if (field == "content-type")
                    content_type = parts[1].strip ();
                else if (field == "content-transfer-encoding")
                    content_encoding = parts[1].strip ();
                else if (field == "message-id")
                    message_id = parts[1].strip ();
                else if (field == "subject")
                    subject = parse_encoded (parts[1], out charset);
                else if (field == "from") {
                    string from_charset = null;
                    from = parse_encoded (parts[1], out from_charset);
                }
                else if (field == "date") {
                    date = format_date (date_from_string (parts[1]));
                }
                else if (field == "to")
                    recipient = parts[1].strip ();
                else if (field == "cc") {
                    string cc_charset = null;
                    carbon_copy = parse_encoded (parts[1], out cc_charset);
                }
                else if (field == "bcc") {
                    string bcc_charset = null;
                    blind_copy = parse_encoded (parts[1], out bcc_charset);
                }
                else if (field == "reply-to") {
                    string reply_charset = null;
                    reply = parse_encoded (parts[1], out reply_charset);
                }
                else if (field == "list-post") {
                    string list_post_charset = null;
                    list_post = parse_encoded (parts[1], out list_post_charset);
                    list_post = Postler.Messages.parse_address (list_post)[1];
                }
                else if (field == "list-unsubscribe") {
                    string list_unsubscribe_charset = null;
                    list_unsubscribe = parse_encoded (parts[1], out list_unsubscribe_charset);
                    list_unsubscribe = Postler.Messages.parse_address (list_unsubscribe)[1];
                }
                else if (field == "organization")
                    organization = parts[1];
                else if (field == "x-mailer" || field == "user-agent")
                    x_mailer = format_x_mailer (parts[1]);
            }

            if (charset == null)
                charset = default_charset;

            /* Some mailing list systems override Reply-To so that it doesn't
               point to the author but the list.
               Also Reply-To may equal From, which is at best confusing. */
            if (reply != "") {
                string canonical = Postler.Messages.parse_address (reply)[1];
                if (canonical == Postler.Messages.parse_address (recipient)[1])
                    reply = "";
                else if (list_post != ""
                  && canonical == list_post)
                    reply = "";
                else if (canonical == Postler.Messages.parse_address (from)[1])
                    reply = "";
            }

            reply_to = reply != "" ? reply : from;
            if ("," in recipient && carbon_copy != "")
                reply_to_all = reply_to + "," + recipient + "," + carbon_copy;
            else if ("," in recipient)
                reply_to_all = reply_to + "," + recipient;
            else if (carbon_copy != "")
                reply_to_all = reply_to + "," + carbon_copy;
            else
                reply_to_all = null;

            /* Linkify From, To and Reply-To */
            /* FIXME: Use raw subject for argument? */
            /* TODO: Show addressbook icons beside addresses */
            string arguments = "?subject=Re: " + subject;
            string? chosen_from = choose_from ();
            if (chosen_from != null)
                arguments += "?from=" + html_escape (chosen_from);
            sender = linkify_address (from, arguments);
            if (recipient != "") {
                /* Show recipient only if isn't unique */
                if ("," in recipient || "," in selected_address ()
                                     || !recipient.contains (selected_address ()))
                    recipient = linkify_address (recipient, arguments);
                else
                    recipient = "";
            }
            if (carbon_copy != "")
                carbon_copy = linkify_address (carbon_copy, arguments);
            if (blind_copy != "")
                blind_copy = linkify_address (blind_copy, arguments);
            if (reply != "")
                reply = linkify_address (reply, arguments);
            if (list_unsubscribe != "")
                list_unsubscribe = linkify_address (list_unsubscribe, null);

            parse_body (stream, content_type);
        } catch (GLib.Error contents_error) {
            throw new GLib.FileError.FAILED (_("Failed to read message: %s").
                                             printf (contents_error.message));
        }
        notify_property ("message-parts");
    }

    void parse_body (DataInputStream stream, string? content_type) throws GLib.Error {

        try {
            string? boundary = null;
            string mime_type = "text/plain";
            string? fname = null;
            parse_content_type (content_type, ref charset, ref boundary,
                                ref mime_type, ref fname);

            uint multipart = mime_type.has_prefix ("multipart/") ? 1 : 0;
            if (multipart > 0 && boundary != null)
                boundary = boundary.replace ("\"", " ").strip ();
            else
                multipart = 0;
            if (!(mime_type_is_text (mime_type) || multipart > 0)) {
                throw new GLib.FileError.FAILED (
                    _("This message of type \"%s\" can't be displayed.").
                    printf (mime_type));
            }

            MessagePart? message_part = null;
            /* Message body starts here */
            if (multipart == 0) {
                message_part = new MessagePart (mime_type);
                message_parts.append (message_part);
            }

            string inner_boundary = "";
            string line;
            string previous_line = "";
            while ((line = stream.read_line (null, null)) != null) {
                if (multipart > 0) {
                    if (line.has_prefix ("--")) {
                        if (line == "--" + inner_boundary) {
                            message_part = new MessagePart ("text/plain");
                            message_parts.append (message_part);
                            multipart = 2;
                            continue;
                        } else if (line == "--" + boundary) {
                            inner_boundary = "";
                            message_part = new MessagePart ("text/plain");
                            message_parts.append (message_part);
                            multipart = 2;
                            continue;
                        } else if (line == "--" + boundary + "--") {
                            /* Final boundary, we can skip it */
                            continue;
                        }
                    }
                    else if (multipart == 2) {
                        if (line == "") {
                            multipart++;
                            continue;
                        }

                        if (line[0] == '\t' || line[0] == ' ')
                        line = previous_line + " " + line.chug ();
                        previous_line = line;

                        string[] parts = line.split (":", 2);
                        string field = ascii_strdown (parts[0] ?? "");
                        if (field == "content-type") {
                            string ctype = charset;
                            string mtype = "text/plain";
                            fname = null;
                            parse_content_type (parts[1].strip (), ref ctype,
                                ref inner_boundary, ref mtype, ref fname);
                            message_part.filename = fname;
                            message_part.mime_type = mtype;
                            message_part.charset = ctype;
                            message_part.plain_text = mtype == "text/plain";
                        }
                        else if (field == "content-transfer-encoding") {
                            content_encoding = ascii_strdown (parts[1].strip ());
                        }
                        else if (field == "content-disposition" ) {
                            string content_disposition = parts[1].strip ();
                            string cset = charset;
                            string mtype = "text/plain";
                            fname = null;
                            parse_content_type (content_disposition, ref cset,
                                ref inner_boundary, ref mtype, ref fname);
                            message_part.filename = fname;
                            if (content_disposition.has_prefix ("attachment"))
                                message_part.content_disposition = content_disposition;
                            else if (content_disposition.has_prefix ("inline"))
                                message_part.content_disposition = content_disposition;
                        }
                        else if (field == "content-id") {
                            string content_id = parts[1].strip ();
                            if (content_id[0] == '<' && content_id.has_suffix (">"))
                                message_part.content_id = content_id.slice (1, -1);
                        }
                        continue;
                    }
                    else if (multipart == 1)
                        continue;
                }

                if (content_encoding == "quoted-printable")
                    line = quoted_printable_decode (line);
                else if (content_encoding == "base64"
                      && mime_type_is_text (message_part.mime_type))
                    line = line != "" ? (string)GLib.Base64.decode (line) : "";
                try {
                    string cset = message_part.charset ?? charset;
                    if (cset != "UTF-8")
                        line = GLib.convert (line, -1, "UTF-8", cset, null);
                }
                catch (GLib.ConvertError error) { }
                if (message_part.plain_text)
                    line = line.replace ("<", "&lt;");
                if (content_encoding == "base64") {
                    if (message_part.plain_text)
                        line = line.replace ("\n", "<br>");
                    message_part.body.append (line);
                } else {
                    message_part.body.append (line);
                    if (content_encoding == "quoted-printable" && line.has_suffix ("="))
                        message_part.body.truncate (message_part.body.len - 1);
                    else if (message_part.plain_text)
                        message_part.body.append ("<br>");
                    else if (mime_type_is_text (message_part.mime_type))
                        message_part.body.append ("\n");
                    else
                        message_part.body.append (" ");
                }
            }
        } catch (GLib.Error error) {
            throw error;
        }
    }

    public bool display (string location, AccountInfo? account_info=null) {
        selected_account = account_info;
        try {
            parse_message (location);

            /* Look for an HTML part, or otherwise plain text */
            html_part = null;
            text_part = null;
            foreach (var part in message_parts) {
                /* Ignore empty parts inserted by faulty clients */
                if (part.body.str.strip () == "")
                    continue;
                /* Select part, merge "inline" parts, commonly signatures */
                if (part.mime_type == "text/html") {
                    if (html_part == null)
                        html_part = part;
                    else if (part.content_disposition == "inline") {
                        html_part.body.append (part.body.str);
                        message_parts.remove (part);
                    }
                }
                if (part.mime_type == "text/plain") {
                    if (html_part != null && part.content_disposition == "inline") {
                        html_part.body.append (part.body.str);
                        message_parts.remove (part);
                    }
                    if (text_part == null)
                        text_part = part;
                    else if (part.content_disposition == "inline") {
                        text_part.body.append (part.body.str);
                        message_parts.remove (part);
                    }
                }
            }
            notify_property ("message-parts");
            if (html_part != null)
                display_part (html_part);
            else if (text_part != null)
                display_part (text_part);
            else {
                text_part = message_parts.nth_data (0);
                display_part (text_part);
            }
        } catch (GLib.Error error) {
            display_error  (error.message);
        }
        return false;
    }

    string render_plain_text (string body) {
        bool in_signature = false;
        int in_quote = 0;

        var new_body = new StringBuilder ("<span class=\"plain_text\">");
        foreach (string line in body.split ("<br>")) {
            /* Looks like a signature */
            if (in_signature && line[0] == '\0') {
                new_body.append ("</span>");
                in_signature = false;
            } else if (!in_signature && line[0] == '-' && line[1] == '-'
                    && line[2] == ' ' && line[3] == '\0') {
                new_body.append ("<span class=\"signature\">");
                in_signature = true;
                line = "";
            }
            /* Looks like quoting */
            if (in_quote > 0 || line.has_prefix ("> ") || line.has_prefix (">>")) {
                /* Determine level of nesting */
                int quote_mark = 0;
                int position = 0;
                do {
                    if (line[position] == '>')
                        quote_mark++;
                    else if (line[position] != ' ')
                        break;
                    position++;
                } while (true);

                if (quote_mark < in_quote)
                    new_body.append ("</blockquote>");
                else if (quote_mark > in_quote) {
                    for (int i = in_quote; i < quote_mark; i++)
                        new_body.append ("<blockquote>");
                }
                in_quote = quote_mark;
            }

            if (in_quote > 0 && line[0] == '>' && line[1] == ' ') {
                line = line.substring (2);
                int position = 0;
                do {
                    if (line[position] != '>' && line[position] != ' ')
                        break;
                    position++;
                } while (true);
                line = line.substring (position);
            }
            else if (in_quote > 0 && line[0] == '>' && line[1] == '\0')
                line = line.substring (1);
            new_body.append (line + "<br>");
        }
        new_body.append ("</span>");
        return new_body.str;
    }

    public void display_part (MessagePart message_part) {
        if (current_part != message_part) {
            current_part = message_part;
            current_part_index = message_parts.index (message_part);
        }

        string body_chunk;
        if (message_part == html_part)
            body_chunk = message_part.body.str;
        else if (message_part == text_part) {
            body_chunk = render_plain_text (message_part.body.str);
        }
        else {
            body_chunk = """
                <b>%s</b>
                <a href="message-part:open:%d">%s</a>
                <a href="message-part:save:%d">%s</a>
                """.
                printf (
                message_part.filename ?? message_part.mime_type,
                message_parts.position (message_parts.find (message_part)),
                _("Open file"),
                message_parts.position (message_parts.find (message_part)),
                _("Save As..."));
            if (mime_type_is_text (message_part.mime_type))
                body_chunk += "<p><span class=\"plain_text\">%s</span>".
                    printf (message_part.body.str);
            else if (message_part.mime_type.has_prefix ("image/"))
                body_chunk += "<p><img src=\"data:image/jpg;base64,%s\">".
                    printf (message_part.body.str);
        }

        try {
            /* Linkify */
            if (message_part.plain_text) {
                foreach (var link_format in link_formats) {
                    var regex = new GLib.Regex (link_format);
                    body_chunk = regex.replace (body_chunk, -1, 0,
                        "<a href=\"\\1\">\\1</a>");
                }
            }

            /* Emoticons */
            /* foreach (var emoticon in emoticons) { */
            for (int i = 0; i < emoticons.length; i++) {
                var emoticon = emoticons[i];
                try {
                    var escaped = GLib.Regex.escape_string (" " + emoticon.token);
                    var regex = new GLib.Regex (escaped);
                    body_chunk = regex.replace_eval (body_chunk, -1, 0, 0,
                        (match_info, result) => { evaluate_emoticon (match_info, result, emoticon); });
                }
                catch (GLib.RegexError error) { }
            }

            load_string ("""
                <style text="text/css">%s</style>
                <span class="headers">
                <div style="float: right;">%s</div>
                <b>%s</b> %s<br>
                %s
                %s
                %s <span style="float: right;"><a href="#">%s</a></span>
                <b>%s</b> %s <br>
                <span class="extra_headers">
                %s
                %s
                %s
                %s
                </span>
                </span>
                <p class="body" style="%s">%s</p>
                """.
                printf (style_sheet,
                        date,
                        _("From:"), sender,
                        format_header (_("To:"), recipient),
                        format_header (_("Copy:"), carbon_copy),
                        format_header (_("Blind Copy:"), blind_copy),
                        reply != "" || organization != "" || x_mailer != ""
                            ? "%s »".printf (_("More")) : "",
                        _("Subject:"), subject,
                        /* TODO: Sender:? */
                        format_header (_("Reply To:"), reply),
                        format_header (_("Organization:"), organization),
                        format_header (_("Application:"), x_mailer),
                        format_header (_("Unsubscribe:"), list_unsubscribe),
                        message_part.plain_text ? "font-family: Monospace;" : "",
                        body_chunk),
                "text/html", "UTF-8", "about:blank");
        } catch (GLib.Error contents_error) {
            display_error (_("Failed to display message part \"%s\": %s").printf (
                message_part.mime_type, contents_error.message));
        }
    }

    void display_error (string message) {
        load_string ("""
            <title>%s</title>
            <body>
            <h1>%s</h1>
            <p>%s</p>
            </body>
            """.
            printf (_("Error"), _("Error"), message),
            "text/html", "UTF-8", "about:blank");
        GLib.critical (message);
    }

    bool navigation_decision (WebKit.WebFrame web_frame,
        WebKit.NetworkRequest request, WebKit.WebNavigationAction action,
        WebKit.WebPolicyDecision decision) {
        string? uri = request.get_uri ();

        if (uri == null || uri == "about:blank") {
            decision.use ();
            return true;
        }

        if (uri != null && uri.has_prefix ("about:"))
            return false;

        decision.ignore ();
        if (uri.has_prefix ("contact:")) {
            string[] parts = Soup.URI.decode (uri).split (":");
            new Dexter.Dexter ().edit_contact (parts[1], parts[2]);
            return true;
        }
        else if (uri.has_prefix ("message-part:")) {
            int index;
            if (uri.has_prefix ("message-part:open:"))
                index = uri.substring (18, -1).to_int ();
            else if (uri.has_prefix ("message-part:save:"))
                index = uri.substring (18, -1).to_int ();
            else {
                /* FIXME: Show error dialogue */
                GLib.message (_("Erroneous attachment: %s"), uri);
                return false;
            }
            var message_part = message_parts.nth_data (index);
            string filename;
            bool should_open = true;
            if (uri.has_prefix ("message-part:open:")) {
                filename = "/tmp/%sXXXXXX".printf (message_part.filename ?? "");
                if (FileUtils.mkstemp (filename) == -1) {
                    /* FIXME: Show error dialogue */
                    GLib.message (_("Failed to create temporary file: %s"), filename);
                    return false;
                }
            } else if (uri.has_prefix ("message-part:save:")) {
                var filechooser = new Gtk.FileChooserDialog (_("Save As..."),
                    null, Gtk.FileChooserAction.SAVE,
                    Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
                    Gtk.STOCK_SAVE, Gtk.ResponseType.ACCEPT);
                filechooser.do_overwrite_confirmation = true;
                string? folder = Environment.get_user_special_dir (UserDirectory.DOWNLOAD);
                if (folder == null)
                    folder = Environment.get_home_dir ();
                filechooser.set_current_folder (folder);
                filechooser.set_current_name (message_part.filename ?? "");
                if (filechooser.run () == Gtk.ResponseType.ACCEPT)
                    filename = filechooser.get_filename ();
                else
                    filename = null;
                filechooser.destroy ();
                if (filename == null)
                    return true;
                should_open = false;
            } else
                assert_not_reached ();
            try {
                if (mime_type_is_text (message_part.mime_type))
                    FileUtils.set_contents (filename, message_part.body.str);
                else
                    FileUtils.set_data (filename,
                        GLib.Base64.decode (message_part.body.str));
                uri = Filename.to_uri (filename, null);
            } catch (GLib.Error error) {
                /* FIXME: Show error dialogue */
                GLib.message (_("Failed to write file: %s"), filename);
                return false;
            }
            if (!should_open)
                return true;
        }

        return Postler.App.show_uri (get_screen (), uri);
    }

    WebKit.WebView new_window_created (WebKit.WebFrame web_frame) {
        /* Forward new windows, we never open new windows of our own */
        return this;
    }

    public override bool scroll_event (Gdk.EventScroll event) {
        if ((event.state & Gdk.ModifierType.CONTROL_MASK) != 0) {
            if (event.direction == Gdk.ScrollDirection.UP) {
                zoom_in ();
                return true;
            } else if (event.direction == Gdk.ScrollDirection.DOWN) {
                zoom_out ();
                return true;
            }
        }
        return false;
    }
}

