/*
** pork_inet.c
** Copyright (C) 2003 Ryan McCabe <ryan@numb.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License, version 2,
** as published by the Free Software Foundation.
*/

#include <config.h>

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <pork_missing.h>
#include <pork_inet.h>

/*
** Write to a socket, deal with interrupted and incomplete writes. Returns
** the number of characters written to the socket on success, -1 on failure.
*/

ssize_t sock_write(int sock, void *buf, size_t len) {
	ssize_t n, written = 0;

	while (len > 0) {
		n = write(sock, buf, len);
		if (n == -1) {
			if (errno == EINTR)
				continue;
			return (-1);
		}

		written += n;
		len -= n;
		buf = (char *) buf + n;
	}

	return (written);
}

/*
** printf-like function that writes to sockets.
*/

#ifdef HAVE_VASPRINTF

int sockprintf(int fd, const char *fmt, ...) {
	va_list ap;
	char *buf;
	ssize_t ret;

	va_start(ap, fmt);
	ret = vasprintf(&buf, fmt, ap);
	va_end(ap);

	ret = sock_write(fd, buf, ret);
	free(buf);

	return (ret);
}

#else

int sockprintf(int fd, const char *fmt, ...) {
	va_list ap;
	char buf[4096];

	va_start(ap, fmt);
	vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);

	return (sock_write(fd, buf, strlen(buf)));
}

#endif

inline int get_peer_addr(int sock, struct sockaddr_storage *ss) {
	socklen_t len = sizeof(*ss);

	return (getpeername(sock, (struct sockaddr *) ss, &len));
}

inline int get_local_addr(int sock, struct sockaddr_storage *ss) {
	socklen_t len = sizeof(*ss);

	return (getsockname(sock, (struct sockaddr *) ss, &len));
}

/*
** Return string IPv4 or IPv6 address.
*/

inline void get_ip(	struct sockaddr_storage *ss,
					char *buf,
					size_t len)
{
	inet_ntop(ss->ss_family, sin_addr(ss), buf, len);
}

/*
** Returns the address set in the appropriate
** sockaddr struct.
*/

inline void *sin_addr(struct sockaddr_storage *ss) {
	if (ss->ss_family == AF_INET6)
		return (&SIN6(ss)->sin6_addr);

	return (&SIN4(ss)->sin_addr);
}

/*
** Returns the length of the sockaddr struct.
*/

inline size_t sin_len(const struct sockaddr_storage *ss) {
	if (ss->ss_family == AF_INET6)
		return (sizeof(struct sockaddr_in6));

	return (sizeof(struct sockaddr_in));
}

/*
** Returns the port set in the sockaddr struct.
*/

inline in_port_t sin_port(const struct sockaddr_storage *ss) {
	if (ss->ss_family == AF_INET6)
		return (SIN6(ss)->sin6_port);

	return (SIN4(ss)->sin_port);
}

/*
** Sets the port for the approprite socket family.
*/

inline void sin_set_port(struct sockaddr_storage *ss, in_port_t port) {
	if (ss->ss_family == AF_INET6)
		SIN6(ss)->sin6_port = port;

	SIN4(ss)->sin_port = port;
}

/*
** Return the canonical hostname of the given address.
*/

inline int get_hostname(struct sockaddr_storage *addr,
						char *hostbuf,
						size_t len)
{
	int ret;

	ret = getnameinfo((struct sockaddr *) addr, sizeof(struct sockaddr_storage),
					hostbuf, len, NULL, 0, NI_NAMEREQD);

	return (ret);
}

/*
** Get the port associated with a tcp service name.
*/

int get_port(const char *name, in_port_t *port) {
	struct servent *servent;

	servent = getservbyname(name, "tcp");
	if (servent != NULL)
		*port = ntohs(servent->s_port);
	else {
		char *end;
		long temp_port;

		temp_port = strtol(name, &end, 10);

		if (*end != '\0')
			return (-1);

		if (!VALID_PORT(temp_port))
			return (-1);

		*port = temp_port;
	}

	return (0);
}

/*
** Return a network byte ordered ipv4 or ipv6 address.
*/

int get_addr(const char *hostname, struct sockaddr_storage *addr) {
	struct addrinfo *res = NULL;
	size_t len;

	if (getaddrinfo(hostname, NULL, NULL, &res) != 0) {
		if (res != NULL)
			freeaddrinfo(res);

		return (-1);
	}

	switch (res->ai_addr->sa_family) {
		case AF_INET:
			len = sizeof(struct sockaddr_in);
			break;
		case AF_INET6:
			len = sizeof(struct sockaddr_in6);
			break;
		default:
			goto out_fail;
	}

	if (len < res->ai_addrlen)
		goto out_fail;

	memcpy(addr, res->ai_addr, res->ai_addrlen);
	freeaddrinfo(res);

	return (0);

out_fail:
	freeaddrinfo(res);
	return (-1);
}

inline int sock_setflags(int sock, u_int32_t flags) {
	return (fcntl(sock, F_SETFL, flags));
}

int nb_connect(	struct sockaddr_storage *ss,
				struct sockaddr_storage *laddr,
				in_port_t port,
				int *dsock)
{
	int sock;
	int ret = 0;

	sock = socket(ss->ss_family, SOCK_STREAM, 0);
	if (sock < 0)
		return (-1);

	if (laddr != NULL) {
		if (bind(sock, (struct sockaddr *) laddr, sin_len(laddr)) != 0)
			goto err_out;
	}

	if (sock_setflags(sock, O_NONBLOCK) != 0)
		goto err_out;

	sin_set_port(ss, htons(port));

	if (connect(sock, (struct sockaddr *) ss, sin_len(ss)) != 0) {
		if (errno != EINPROGRESS)
			goto err_out;

		ret = -EINPROGRESS;
	} else {
		if (sock_setflags(sock, 0) != 0)
			goto err_out;
	}

	*dsock = sock;
	return (ret);

err_out:
	close(sock);
	return (-1);
}

int sock_is_error(int sock) {
	int error;
	socklen_t errlen = sizeof(error);
	int ret;

	ret = getsockopt(sock, SOL_SOCKET, SO_ERROR, &error, &errlen);
	if (ret == -1)
		return (-errno);

	return (-error);
}
