/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-searchmass-options.h"
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxedit-ui-searchmass-results.h"

  



GtkWidget *
polyxedit_searchmass_opt_setup_wnd (PxmEditCtxt *editctxt)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  GtkWidget *combo = NULL;
  GtkWidget *vbox = NULL;
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;


  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  


  gui_file = 
    g_strdup_printf ("%s/polyxedit-searchmass-options.glade", 
		     userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "searchmass_options_wnd", 
		       PACKAGE);
  g_free (gui_file);

  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "searchmass_options_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the search mass window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* Immediately set to the window a pointer to the editctxt:
   */
  g_object_set_data (G_OBJECT (window), "editctxt", editctxt);


  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window), "messages_entry",
		     widget);


  /* Set the polymer name to the relative entry.
   */
  widget = glade_xml_get_widget (xml, "polseq_name_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "polseq_name_entry", 
		     widget);
  
  if (editctxt->polymer->plminfo->name != NULL)
    gtk_entry_set_text (GTK_ENTRY (widget), 
			editctxt->polymer->plminfo->name);
  else
    gtk_entry_set_text (GTK_ENTRY (widget), _("Not set"));
  

  widget = glade_xml_get_widget (xml, "polseq_identity_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "polseq_identity_number_entry", 
		     widget);
  
  /* Set the polymer id number (the pointer to editctxt
     to the relative entry.
  */
  help = g_strdup_printf ("%p", editctxt);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);
  
  /* Now deal systematically with the widgets that we will use later.
   */
  widget = glade_xml_get_widget (xml, 
				 "polseq_data_sel_only_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "polseq_data_sel_only_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "polseq_data_whole_seq_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "polseq_data_whole_seq_radiobutton",
		     widget);


  widget = glade_xml_get_widget (xml, 
				 "mono_unique_mass_entry");
  g_object_set_data (G_OBJECT (window), 
		     "mono_unique_mass_entry",
		     widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "activate",
     G_CALLBACK (polyxedit_searchmass_opt_mass_entry_activate), 
     window);
  
  widget = glade_xml_get_widget (xml, 
				 "mono_masses_textview");
  g_object_set_data (G_OBJECT (window), 
		     "mono_masses_textview",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "avg_unique_mass_entry");
  g_object_set_data (G_OBJECT (window), 
		     "avg_unique_mass_entry",
		     widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "activate",
     G_CALLBACK (polyxedit_searchmass_opt_mass_entry_activate), 
     window);
  

  widget = glade_xml_get_widget (xml, 
				 "avg_masses_textview");
  g_object_set_data (G_OBJECT (window), 
		     "avg_masses_textview",
		     widget);


  /*
    Initial version used when Glade-2 was not faulty.
    
    widget = glade_xml_get_widget (xml, 
    "mono_tolerance_combo");
    g_object_set_data (G_OBJECT (window), 
    "mono_tolerance_combo",
    widget);
  */
  /* Momentaneous manual creation of the GtkComboBox due to 
     a bug in Glade-2 that prevents it from saving a file that contains
     such a widget.
  */
  combo = gtk_combo_box_new_text ();
  g_assert (combo != NULL);
  
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_amu);
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_pct);
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_ppm);
  
  /* The "tolerance_units_amu" should be set active by default.
   */
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);

  g_object_set_data (G_OBJECT (window), 
		     "mono_tolerance_combo",
		     combo);
  /* Get the box where the combo must be packed.
   */
  vbox = glade_xml_get_widget (xml, 
			       "mono_tolerance_vbox");
  g_assert (vbox != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "mono_tolerance_vbox",
		     vbox);
  gtk_box_pack_start (GTK_BOX (vbox), combo, TRUE, TRUE, 0);

  /* The combo has to be before the tolerance GtkEntry box.
   */
  gtk_box_reorder_child (GTK_BOX (vbox), combo, 0);

  widget = glade_xml_get_widget (xml, 
				 "mono_tolerance_entry");
  g_object_set_data (G_OBJECT (window), 
		     "mono_tolerance_entry",
		     widget);

  /*
    Initial version used when Glade-2 was not faulty.
    
    widget = glade_xml_get_widget (xml, 
    "avg_tolerance_combo");
    g_object_set_data (G_OBJECT (window), 
    "avg_tolerance_combo",
    widget);
  */
  /* Momentaneous manual creation of the GtkComboBox due to 
     a bug in Glade-2 that prevents it from saving a file that contains
     such a widget.
  */
  combo = gtk_combo_box_new_text ();
  g_assert (combo != NULL);
  
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_amu);
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_pct);
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_ppm);

  /* The "tolerance_units_amu" should be set active by default.
   */
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);

  g_object_set_data (G_OBJECT (window), 
		     "avg_tolerance_combo",
		     combo);
  /* Get the box where the combo must be packed.
   */
  vbox = glade_xml_get_widget (xml, 
			       "avg_tolerance_vbox");
  g_assert (vbox != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "avg_tolerance_vbox",
		     vbox);
  gtk_box_pack_start (GTK_BOX (vbox), combo, TRUE, TRUE, 0);

  /* The combo has to be before the tolerance GtkEntry box.
   */
  gtk_box_reorder_child (GTK_BOX (vbox), combo, 0);

  widget = glade_xml_get_widget (xml, 
				 "avg_tolerance_entry");
  g_object_set_data (G_OBJECT (window), 
		     "avg_tolerance_entry",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "unique_mass_search_mode_checkbutton");
  g_object_set_data (G_OBJECT (window), 
		     "unique_mass_search_mode_checkbutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "search_button");
  g_object_set_data (G_OBJECT (window), 
		     "search_button",
		     widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxedit_searchmass_opt_search_button), 
		    window);

  widget = glade_xml_get_widget (xml, 
				 "currently_searched_mass_entry");
  g_object_set_data (G_OBJECT (window), 
		     "currently_searched_mass_entry",
		     widget);

  widget = 
    glade_xml_get_widget (xml, 
			  "currently_searched_mass_mono_type_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "currently_searched_mass_mono_type_radiobutton",
		     widget);

  widget = 
    glade_xml_get_widget (xml, 
			  "currently_searched_mass_avg_type_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "currently_searched_mass_avg_type_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "total_number_found_oligomers_entry");
  g_object_set_data (G_OBJECT (window), 
		     "total_number_found_oligomers_entry",
		     widget);



  gtk_widget_show_all (GTK_WIDGET (window));
  
  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));

  /* The signal of the window itself.
   */
  /* Signal / callback connections.
   */
  g_signal_connect 
    (G_OBJECT (window),
     "delete_event",
     G_CALLBACK (polyxedit_searchmass_opt_wnd_delete_event), 
     editctxt);

  /* Set this window pointer as a full datum to the polymer sequence
     editor window, so that when it is closed this window is closed
     also. 

     There might be more than one cleavage options' window opened for
     a given polymer seqence editing window, and we do not want that
     the second window destroys the datum name of the first window, so
     we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("searchmass_options_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (editctxt->seqeditorctxt->sequence_editor_wnd),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxedit_searchmass_opt_wnd_really_close);
  
  g_free (help);

  return window;
}

void
polyxedit_searchmass_opt_mass_entry_activate (GtkEntry *entry,
					      gpointer data)
{
  GtkWidget *check = NULL;
  GtkWidget *window = data;
  GtkWidget *display_window = NULL;

  /* Textview stuff....
   */
  GtkTextBuffer *buffer = NULL;
  GtkTextView *textview = NULL;
  GtkTextIter start;
  GtkTextIter end;

  gchar *new_text = NULL;
  
  PxmSearchmassOpt *smopt = NULL;
  PxmEditCtxt *editctxt = NULL;
  PxmCalcOpt *calcopt = NULL;
  PxmIonizerule *ionizerule = NULL;
  PxmMassType mass_type = PXM_MASS_NONE;
  
  gchar *name = NULL;
  gchar *help = NULL;
  
  GPtrArray *alloligsGPA = NULL;
  GPtrArray *smoptGPA = NULL;

  gint count = 0;
  
  gboolean unique_mode = FALSE;
  
  
  g_assert (window != NULL);
  g_assert (entry != NULL);
  
  /* Get the editctxt pointer immediately.
   */
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);


  /* Immediately establish if we are working on a mono or avg mass.
   */
  name = (gchar *) gtk_widget_get_name (GTK_WIDGET (entry));
  
  if (0 == strcmp ("mono_unique_mass_entry", name))
    mass_type = PXM_MASS_MONO;
  else
    mass_type = PXM_MASS_AVG;
  

  /* Get to know if we are in a multi- or unique-search mass mode.
   */
  check = g_object_get_data (G_OBJECT (window), 
			     "unique_mass_search_mode_checkbutton");
  g_assert (check != NULL);
  unique_mode = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));

  /* Depending on 'unique_mode' we'll have different actions to take.
   */
  if (TRUE == unique_mode)
    {
      /* First step is to allocate a PxmSearchmassOpt that we'll fill
	 with proper data.
      */
      smopt = pxmchem_searchmassopt_new ();

      /* Immediately set the mass type that we already know.
       */
      smopt->mass_type = mass_type;

      /* And also set some other data to allow construction of fully
	 detailed oligomers, so that later we can show the
	 data in the results window.
      */
      smopt->mnm_chement = PXMCHEMENT_MNM_MODIF;
      smopt->plm_chement = PXMCHEMENT_PLM_BOTH_MODIF;
      smopt->put_sequence = TRUE;
      
      /* Now get to have a properly converted mass value from the 
	 corresponding entry:
      */
      if (FALSE == polyxedit_searchmass_opt_get_mass_value (window,
							    mass_type,
							    smopt))
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Failed to get a correct mass value"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_searchmassopt_free (smopt);
	  
	  return;
	}
	 
      /* Now get to have a properly converted tolerance value from the
	 corresponding entry:
      */
      if (FALSE == 
	  polyxedit_searchmass_opt_get_tolerance_value (window,
							mass_type,
							smopt))
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Failed to get a correct tolerance value"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_searchmassopt_free (smopt);
	  
	  return;
	}
      
      /* Now get the tolerance type from the proper GtkOptionMenu widget.
       */
      if (FALSE == 
	  polyxedit_searchmass_opt_get_tolerance_type (window,
						       mass_type,
						       smopt))
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Failed to get a correct tolerance type"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_searchmassopt_free (smopt);
	  
	  return;
	}

      /* Duplicate the editctxt's calcopt (because its start_idx,
	 end_idx members are going to be modified during the mass search
	 operation):
      */
      calcopt = pxmchem_calcopt_dup (editctxt->calcopt);
    
      if (FALSE == polyxedit_searchmass_opt_get_selection_type (window,
								calcopt))
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Failed to get selection type"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_searchmassopt_free (smopt);
	  
	  return;
	}


      /* Allocate the array that will hold the smopt instance, so that
	 we can call a unified function which is expecting an array
	 of smopt objects.
      */
      smoptGPA = g_ptr_array_new ();
      g_ptr_array_add (smoptGPA, smopt);
      
      /* Allocate the array of arrays of oligomers that will hold all
	 the arrays of oligomers organized by searched masses.
      */
      alloligsGPA = g_ptr_array_new ();


      /* Duplicate the polchemdef's ionizerule structure, so that we are 
	 on our own during the display of the different oligomers later.
      */
      ionizerule = pxmchem_ionizerule_dup (editctxt->ionizerule);
      
      /* Perform the search proper.
       */
      count = polyxedit_searchmass_opt_search_masses (editctxt,
						      window,
						      smoptGPA,
						      alloligsGPA,
						      calcopt,
						      ionizerule);
      if (count == -1)
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Failed to search the mass in the polymer"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  pxmchem_searchmassopt_GPA_free (smoptGPA);
	  pxmchem_calcopt_free (calcopt);
	  pxmchem_ionizerule_free (ionizerule);
	  
	  return;
	}
      else if (count == 0)
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Mass not found in the polymer"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  pxmchem_searchmassopt_GPA_free (smoptGPA);
	  pxmchem_calcopt_free (calcopt);
	  pxmchem_ionizerule_free (ionizerule);
	  
	  return;
	}

      /* At this point, we do not need the array of smopt instances,
	 because each oligomer in the arrays that sit in the
	 alloligsGPA has a copy of the smopt object by which it was
	 found.
      */
      pxmchem_searchmassopt_GPA_free (smoptGPA);
	  
      /* At this point we just have to display the oligomers.
       */
      display_window = polyxedit_searchmass_res_wnd_setup (editctxt,
							   alloligsGPA);
      if (display_window == NULL)
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Failed to display the oligomers in a window"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  pxmchem_calcopt_free (calcopt);
	  pxmchem_ionizerule_free (ionizerule);
	  
	  return;
	}
      
      /* Now that we have the certainty that the oligomers' displaying
	 window could be created and populated correctly with the
	 oligomers, we can set the calcopt instance as datum to the
	 results displaying window. This structure will be freed when
	 the window will be destroyed. Note that the oligsGPA pointer
	 was set as a datum during setting up of the displaying
	 window above.
      */
      g_object_set_data_full (G_OBJECT (display_window),
			      "calcopt", calcopt,
			      (GDestroyNotify) pxmchem_calcopt_free);
      
      g_object_set_data_full (G_OBJECT (display_window),
			      "ionizerule", ionizerule,
			      (GDestroyNotify) pxmchem_ionizerule_free);
      
      /* And now we can ask that all the informations contained in these
	 structures be updated in the results' window at proper places.
      */
      polyxedit_searchmass_res_wnd_update_searchmass_data (display_window);
      
      /* Finally we can return, our work here is finished.
       */
      return;
    }
  else
    {
      /* We are working in the multi search mass mode, which means
	 that we just validate the text that is in the entry and 
	 put it in the textview that corresponds to the entry from
	 which the "activate" signal was emitted.
      */
      if (0 == strcmp ("mono_unique_mass_entry", name))
	{
	  textview = g_object_get_data (G_OBJECT (window),
					"mono_masses_textview");
	}
      else if (0 == strcmp ("avg_unique_mass_entry", name))
	{
	  textview = g_object_get_data (G_OBJECT (window),
					"avg_masses_textview");
	}
      else
	g_error (_("%s@%d: failed to interpret the window layout\n"),
	       __FILE__, __LINE__);

      /* Get the buffer that holds the text presently (if any) in the
	 textview.
      */
      buffer = gtk_text_view_get_buffer (textview);

      /* We now know where the text to be read is and where it should
	 be put.
      */
      gtk_text_buffer_get_selection_bounds (buffer, &start, &end);
      
      /* At this point, the 'end' textiter should point to the insertion
	 point. This is where the contents of the entry that emitted 
	 the "activate" signal are to be put.
      */
      help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);

      if (strlen (help) <= 0)
	{
	  g_free (help);
	  
	  return;
	}
      
      /* Append a space followed by a newline to the help text, so
	 that when it is inserted in the textview buffer the line feed
	 ensures that all values succcessively entered are displayed
	 in column. The space character ensures that the user may put
	 the cursor somewhere in the textview buffer and insert a new
	 value through the entry widget.
      */
      new_text = g_strconcat (help, " \n", NULL);
      g_free (help);

      gtk_text_buffer_insert (buffer, &end, new_text, -1);
      
      g_free (new_text);

      /* Now that the entry's contents have been copied to the textview
	 we can erase them from the entry.
      */
      gtk_entry_set_text (GTK_ENTRY (entry), "");
      
    }
  
  return;
}


gint
polyxedit_searchmass_opt_search_masses (PxmEditCtxt *editctxt,
					GtkWidget *window,
					GPtrArray *smoptGPA,
					GPtrArray *alloligsGPA,
					PxmCalcOpt *calcopt,
					PxmIonizerule *ionizerule)
{
  gint count = 0;
  gint total = 0;
  gint iter = 0;

  gchar *prefix = NULL;
  gchar *help = NULL;
  
  GtkWidget *current_mass_entry = NULL;
  GtkWidget *total_oligomers_entry = NULL;
  GtkWidget *mono_radio = NULL;
  GtkWidget *avg_radio = NULL;
   
  
  PxmSearchmassOpt *smopt = NULL;

  GPtrArray *oligsGPA = NULL;
  

  g_assert (editctxt != NULL);
  g_assert (window != NULL);
  g_assert (smoptGPA != NULL);
  g_assert (alloligsGPA != NULL);
  g_assert (calcopt != NULL);
  g_assert (ionizerule != NULL);
  
  
  current_mass_entry = g_object_get_data (G_OBJECT (window),
					  "currently_searched_mass_entry");
  g_assert (current_mass_entry != NULL);
  
  total_oligomers_entry = 
    g_object_get_data (G_OBJECT (window),
		       "total_number_found_oligomers_entry");
  g_assert (total_oligomers_entry != NULL);
  
  mono_radio = 
    g_object_get_data (G_OBJECT (window),
		       "currently_searched_mass_mono_type_radiobutton");
  g_assert (mono_radio != NULL);
  
  avg_radio =
    g_object_get_data (G_OBJECT (window),
		       "currently_searched_mass_avg_type_radiobutton");
  g_assert (avg_radio != NULL);
  
  
  for (iter = 0; iter < smoptGPA->len ; iter++)
    {
      /* Prepare the prefix string to allow for proper naming in the
	 subsequently called function.
      */
      prefix= g_strdup_printf ("%d", iter + 1);

      smopt = g_ptr_array_index (smoptGPA, iter);
      g_assert (smopt != NULL);

      /* Allocate an array for this specific mass search.
       */
      oligsGPA = g_ptr_array_new ();
      

      /* Inform the user that a new mass is being searched.
       */
      help = g_strdup_printf ("%.5lf", smopt->mass);
      gtk_entry_set_text (GTK_ENTRY (current_mass_entry), help);
      g_free (help);

      if (smopt->mass_type == PXM_MASS_MONO)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (mono_radio),
				      TRUE);
      else if (smopt->mass_type == PXM_MASS_AVG)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (avg_radio),
				      TRUE);
      else g_assert_not_reached ();
      
      while (gtk_events_pending ())
	gtk_main_iteration ();

      /* Let's do the search proper !
       */
      count = pxmchem_searchmass_polymer (editctxt->polymer,
					  oligsGPA,
					  editctxt->polchemdefctxt->
					  polchemdef,
					  smopt,
					  calcopt,
					  ionizerule,
					  prefix);
      
      /* Free the prefix string, because we'll need to allocate a new
	 one later, or we'll get out of the current loop.
      */
      g_free (prefix);
      
      if (count == -1)
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Failed to search for one mass in the polymer"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  return -1;
	}
      
      /* At this point we can add the new array to the array of arrays,
	 but only if it has data in it. Otherwise we free it immediately.
      */
      if (oligsGPA->len > 0)
	g_ptr_array_add (alloligsGPA, oligsGPA);
      else
	g_ptr_array_free (oligsGPA, TRUE);
      
      total += count;

      /* Tell the user how many oligomers in total we got.
       */
      help = g_strdup_printf ("%d", total);
      gtk_entry_set_text (GTK_ENTRY (total_oligomers_entry), help);
      g_free (help);
    }
  
  return total;
}

void
polyxedit_searchmass_opt_search_button (GtkWidget *button,
					gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *display_window = NULL;
  GtkWidget *entry = NULL;
  GtkWidget *check = NULL;

  /* Textview stuff....
   */
  GtkTextBuffer *buffer = NULL;
  GtkTextView *textview = NULL;
  GtkTextIter start;
  GtkTextIter end;

  gchar *masses_text = NULL;
  gchar *help = NULL;
  
  gdouble *mass = NULL;

  PxmMassType mass_types [] = 
    {
      PXM_MASS_MONO,
      PXM_MASS_AVG,
      PXM_MASS_NONE
    };
  
  
  PxmSearchmassOpt *smopt_template = NULL;
  PxmSearchmassOpt *smopt = NULL;

  PxmEditCtxt *editctxt = NULL;

  PxmCalcOpt *calcopt = NULL;
  PxmIonizerule *ionizerule = NULL;
  
  GPtrArray *massesGPA = NULL;
  GPtrArray *smoptGPA = NULL;
  GPtrArray *alloligsGPA = NULL;
  
  gint count = 0;
  gint iter = 0;
  
  gboolean mono_ok = FALSE;
  gboolean avg_ok = FALSE;
  gboolean unique_mode = FALSE;
  

  /* The search button was clicked, which means that we have to check
     a number of things:

     - If unique mass mode is activated, we need to know wether the
     entry widgets (mono and/or avg) do contain text. If so, only one
     must contain text. Otherwise we do not know which entry use !

     - If unique mass mode is not activated, we'll have to deal with
     both mono and avg textviews, and parse their content in sequence,
     so that we get all the masses in them. For each successfully
     parsed mass we do have to create a smopt instance to add to the
     array of PxmSearchmassOpt instances.

     - Finally, once all the relevant masses have been converted to 
     smopt objects in the array of smopts, then mass searching can
     go on through the call of the appropriate function.
  */

  g_assert (window != NULL);
  

  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  check = g_object_get_data (G_OBJECT (window), 
			     "unique_mass_search_mode_checkbutton");
  g_assert (check != NULL);
  unique_mode = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));

  if (TRUE == unique_mode)
    {
      /* There must one mass entry, and one only, that has 
	 a string in it. Get the pointer to this entry and call
	 the appropriate function.
      */
      mono_ok = polyxedit_searchmass_opt_get_mass_value (window,
							 PXM_MASS_MONO,
							 NULL);

      avg_ok = polyxedit_searchmass_opt_get_mass_value (window,
							PXM_MASS_AVG,
							NULL);

      /* If both entries did contain valid string, that is an error,
	 because we do not know which one to use for the unique mode
	 mass search.
      */
      if (mono_ok == TRUE && avg_ok == TRUE)
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Both MONO and AVG text entries have a mass value"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	}
      else if (mono_ok == TRUE)
	{
	  /* We have a valid string in the MONO mass entry, which means
	     that we simply get the pointer to the entry in question
	     and call the proper function.
	  */
	  entry = g_object_get_data (G_OBJECT (window), 
				     "mono_unique_mass_entry");
	  g_assert (entry != NULL);
	  
	  polyxedit_searchmass_opt_mass_entry_activate (GTK_ENTRY (entry),
							window);

	  return;
	}
      else /* (avg_ok == TRUE) */
	{
	  /* We have a valid string in the AVG mass entry, which means
	     that we simply get the pointer to the entry in question
	     and call the proper function.
	  */
	  entry = g_object_get_data (G_OBJECT (window), 
				     "avg_unique_mass_entry");
	  g_assert (entry != NULL);
	  
	  polyxedit_searchmass_opt_mass_entry_activate (GTK_ENTRY (entry),
							window);
	  return;
	}
    }
  else /* (FALSE == unique_mode) */
    {
      /* At this point we know that we have to deal with both MONO and
	 AVG textviews, get their contents, in sequence and parse
	 these contents into masses, which are in turn changed into
	 PxmSearchmassOpt instances.
      */


      /* During the process of parsing the contents of the MONO/AVG
	 textview widgets, we'll allocate a number of PxmSearchmassopt
	 object (smopt, for short). Inside one round of parsing (MONO
	 first and AVG next), all these objects will be identical
	 unless for the mass value.

	 So we allocate a template smopt that we'll later use for the
	 duplications that we'll perform each time a valid mass was 
	 parsed. We'll thus only modify the mass in the newly allocated
	 smopt.
      */
      smopt_template = pxmchem_searchmassopt_new ();

      /* Set some other data to allow construction of fully
	 detailed oligomers, so that later we can show the
	 data in the results window.
      */
      smopt_template->mnm_chement = PXMCHEMENT_MNM_MODIF;
      smopt_template->plm_chement = PXMCHEMENT_PLM_BOTH_MODIF;
      smopt_template->put_sequence = TRUE;


      /* For the parsing of the masses in the textview we'll use a 
	 GPtrArray of gdoubles, that we allocated right away.
      */
      massesGPA = g_ptr_array_new ();
      
      /* Now, for each mass in the array, we will need to prepare a
	 smopt object that we'll add to the smoptGPA that we can
	 allocate now.
      */
      smoptGPA = g_ptr_array_new ();


      /* Sequentially study the contents of the MONO and AVG masses'
	 textviews:
      */
      while (mass_types [iter] != PXM_MASS_NONE)
	{
	  g_assert (mass_types [iter] == PXM_MASS_MONO 
		    || mass_types [iter] == PXM_MASS_AVG);
	  
	  smopt_template->mass_type = mass_types [iter];


	  /* We start by checking if the textview has valid text in
	     it.  This is because if it has no text in it it is of no
	     interest to go further for the current mass type : just 
	     increment iter and go to the other mass type if there is 
	     one still available.
	  */
	  /* Now deal with the textview proper.
	   */
	  if (mass_types [iter] == PXM_MASS_MONO)
	    textview = g_object_get_data (G_OBJECT (window), 
					  "mono_masses_textview");
	  else
	    textview = g_object_get_data (G_OBJECT (window), 
					  "avg_masses_textview");
	  
	  g_assert (textview);
	  
	  buffer = gtk_text_view_get_buffer (textview);
	  
	  /* We now know where the text to be read is and where it should
	     be put.
	  */
	  gtk_text_buffer_get_start_iter (buffer, &start);
	  gtk_text_buffer_get_end_iter (buffer, &end);
	  
	  /* masses_text is going to hold an allocated UTF-8 string.
	   */
	  masses_text = gtk_text_buffer_get_text (buffer,
						  &start,
						  &end,
						  FALSE);
	  g_assert (masses_text != NULL);
	  
	  if (FALSE == g_utf8_validate (masses_text, -1, NULL))
	    {
	      if (mass_types [iter] == PXM_MASS_MONO)
		polyxmass_timeoutmsg_message_set
		  ((GtkWindow *) window,
		   _("The data in the MONO text view are not valid UTF-8"),
		   POLYXMASS_NORM_MSG_TIMEOUT);
	      else 
		polyxmass_timeoutmsg_message_set
		  ((GtkWindow *) window,
		   _("The data in the AVG text view are not valid UTF-8"),
		   POLYXMASS_NORM_MSG_TIMEOUT);
	      
	      /* Remember that we encounter this point twice: the first
		 time is for the MONO round, and the second time is
		 for the AVG round. It is not because the MONO textview
		 is empty that we must return with an error: imagine
		 that the user only wanted to search for AVG masses.

		 That means that each time something is "wrong" in a
		 textview, we have to keep in mind that the other
		 textview may very well contain good data. Thus, we do
		 not return upon an error, we simply free the memory
		 that was allocated for the current round, and let the
		 loop go on with a 'continue' statement.
	      */
	      
	      g_free (masses_text);
	      
	      iter++;
	      continue;
	    }
	  
	  if (g_utf8_strlen (masses_text, -1) <= 0)
	    {
	      g_free (masses_text);
	    
	      iter++;
	      continue;
	    }
	  
	  /* At this point we know that there are masses to be searched
	     which means that it is useful to continue getting data 
	     from the other widgets.
	  */
	  
	  /* Now get to have a properly converted tolerance value from the
	     corresponding entry:
	  */
	  if (FALSE == 
	      polyxedit_searchmass_opt_get_tolerance_value (window,
							    mass_types [iter],
							    smopt_template))
	    {
	      if (mass_types [iter] == PXM_MASS_MONO)
		polyxmass_timeoutmsg_message_set
		  ((GtkWindow *) window,
		   _("Failed to get a correct MONO tolerance value"),
		   POLYXMASS_NORM_MSG_TIMEOUT);
	      else
		polyxmass_timeoutmsg_message_set
		  ((GtkWindow *) window,
		   _("Failed to get a correct AVG tolerance value"),
		   POLYXMASS_NORM_MSG_TIMEOUT);

	      g_free (masses_text);
		
	      iter++;
	      continue;
	    }

	  /* Now get the tolerance type from the proper GtkOptionMenu widget.
	   */
	  if (FALSE == 
	      polyxedit_searchmass_opt_get_tolerance_type (window,
							   mass_types [iter],
							   smopt_template))
	    {
	      if (mass_types [iter] == PXM_MASS_MONO)
		polyxmass_timeoutmsg_message_set
		  ((GtkWindow *) window,
		   _("Failed to get a correct MONO tolerance type"),
		   POLYXMASS_NORM_MSG_TIMEOUT);
	      else
		polyxmass_timeoutmsg_message_set
		  ((GtkWindow *) window,
		   _("Failed to get a correct AVG tolerance type"),
		   POLYXMASS_NORM_MSG_TIMEOUT);

	      g_free (masses_text);
		  
	      iter++;
	      continue;
	    }
	  

	  /* We STILL have to parse the text in 'masses_text' and for each
	     successfully parsed mass we get a gdouble value that is
	     allocated and added to the 'massesGPA'.
	  */
	  libpolyxmass_globals_parse_masses_in_text (masses_text, massesGPA);
	  
	  if (massesGPA->len <= 0)
	    {
	      g_free (masses_text);
		
	      iter++;
	      continue;
	    }
	  
	  /* Now that we have our array of masses (gdouble ones), we 
	     can iterate in it and ensure that for each mass we create
	     a smopt object that gets added to smoptGPA. In the process
	     we free all the gdouble masses of the array, that in turn
	     will be freed...
	  */
	  while (massesGPA->len > 0)
	    {
	      mass = g_ptr_array_remove_index (massesGPA, 0);
	      g_assert (mass != NULL);
	      
	      smopt = 
		pxmchem_searchmassopt_dup (smopt_template,
					   PXM_SEARCHMASSOPT_DUP_DEEP_YES);
	      
	      smopt->mass = *mass;
	      
	      g_ptr_array_add (smoptGPA, smopt);
	      
	      g_free (mass);
	    }
	  
	  /* We can free the text that was fetched from the textview.
	   */
	  g_free (masses_text);
	  
	  /* At this point, all the masses are freed and an according number
	     of smopt objects have been allocated into smoptGPA. We thus 
	     are now ready to either go proceed to the other mass type,
	     or to perform the mass search proper.
	  */
	  iter++;
	}
      /* end of
	 while (mass_types [iter] != PXM_MASS_NONE)
      */

      /* At this point, we have an array of smopt objects that reflect
	 the masses that were listed in both the textview widgets:
	 MONO and AVG.
      */

      /* We do not need the template smopt anymore.
       */
      pxmchem_searchmassopt_free (smopt_template);

      /* Since we had emptied the array of masses at each one of the
	 two above rounds, we can free it, because it is empty by now.
      */
      g_ptr_array_free (massesGPA, TRUE);


      /* Duplicate the editctxt's calcopt (because its start_idx,
	 end_idx members are going to be modified during the mass search
	 operation):
      */
      calcopt = pxmchem_calcopt_dup (editctxt->calcopt);
    
      if (FALSE == polyxedit_searchmass_opt_get_selection_type (window,
								calcopt))
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Failed to get the selection type"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_searchmassopt_free (smopt);
	  
	  return;
	}

      /* Allocate the array of arrays of oligomers that will hold all
	 the arrays of oligomers organized by searched masses.
      */
      alloligsGPA = g_ptr_array_new ();
      
      /* Duplicate the polchemdef's ionizerule structure, so that we are 
	 on our own during the display of the different oligomers later.
      */
      ionizerule = pxmchem_ionizerule_dup (editctxt->ionizerule);
      
      /* Perform the search proper.
       */
      count = polyxedit_searchmass_opt_search_masses (editctxt,
						      window,
						      smoptGPA,
						      alloligsGPA,
						      calcopt,
						      ionizerule);
      if (count == -1)
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Failed to search the masses in the polymer"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  pxmchem_searchmassopt_GPA_free (smoptGPA);
	  pxmchem_calcopt_free (calcopt);
	  pxmchem_ionizerule_free (ionizerule);
	  
	  return;
	}
      else if (count == 0)
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("No mass was found in the polymer"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  pxmchem_searchmassopt_GPA_free (smoptGPA);
	  pxmchem_calcopt_free (calcopt);
	  pxmchem_ionizerule_free (ionizerule);
	  
	  return;
	}

      /* At this point, we do not need the array of smopt instances,
	 because each oligomer in the arrays that sit in the
	 alloligsGPA has a copy of the smopt object by which it was
	 found.
      */
      pxmchem_searchmassopt_GPA_free (smoptGPA);
	  
      /* At this point we just have to display the oligomers.
       */
      display_window = polyxedit_searchmass_res_wnd_setup (editctxt,
							   alloligsGPA);
      if (display_window == NULL)
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Failed to display the oligomers in a window"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  pxmchem_calcopt_free (calcopt);
	  pxmchem_ionizerule_free (ionizerule);
	  
	  return;
	}

      /* Register the results' window !
       */
      /* Immediately set a automagically freeable string with the
	 description of the new allocated results window so that later
	 -during window management- we can describe what the window was
	 for. This will be used if the user asks to find masses in the
	 results and then a new window will be displayed with all the
	 masses that were found in the results window, and we will want to
	 display that description string.
      */
      help = g_strdup_printf (_("Search Mass in '%p'"), editctxt);
      
      g_object_set_data_full (G_OBJECT (display_window),
			      "wnd_desc", help, 
			      (GDestroyNotify) free);
      
      if (NULL == polyxmass_winmngmt_register_window 
	  ("POLYXEDIT",
	   display_window,
	   window,
	   editctxt,
	   _("Results of Search Mass"),
	   help,
	   TRUE,
	   polyxedit_searchmass_results_wnd_make_report))
	g_critical (_("%s@%d: failed to register the "
		      "search mass results' window\n"),
		    __FILE__, __LINE__);
      

      /* Do not free help, because we have set it as a full datum to
	 the window where the find mass results are to be displayed !
	 
	 g_free (help);
      */

      
      /* Now that we have the certainty that the oligomers' displaying
	 window could be created and populated correctly with the
	 oligomers, we can set the calcopt instance as datum to the
	 results displaying window. This structure will be freed when
	 the window will be destroyed. Note that the oligsGPA pointer
	 was set as a datum during setting up of the displaying
	 window above.
      */
      g_object_set_data_full (G_OBJECT (display_window),
			      "calcopt", calcopt,
			      (GDestroyNotify) pxmchem_calcopt_free);
      
      g_object_set_data_full (G_OBJECT (display_window),
			      "ionizerule", ionizerule,
			      (GDestroyNotify) pxmchem_ionizerule_free);
      
      /* And now we can ask that all the informations contained in these
	 structures be updated in the results' window at proper places.
      */
      polyxedit_searchmass_res_wnd_update_searchmass_data (display_window);
      
      /* Finally we can return, our work here is finished.
       */
      return;
    }
  /* end of 
     else (FALSE == unique_mode)
  */

  return;
}
 

 
gboolean
polyxedit_searchmass_opt_get_mass_value (GtkWidget *window, 
					 PxmMassType type,
					 PxmSearchmassOpt *smopt)
{
  gdouble mass;
  
  gboolean result = FALSE;

  gchar *help = NULL;

  GtkWidget *entry = NULL;
  
  /* Depending on the value of the 'type' parameter, we have to get
     the proper mass text value from the proper text entry. So,
     first get text, and next put the successfully converted value
     into the smopt structure. Note that if smopt is
     NULL, this function is called only to check that the entry
     contains text that can successfully be converted into gdouble.
  */

  g_assert (window != NULL);

  if (type == PXM_MASS_MONO)
    {
      entry = g_object_get_data (G_OBJECT (window),
				 "mono_unique_mass_entry");
      g_assert (entry != NULL);
    }
  else /* (type == PXM_MASS_AVG) */
    {
      entry = g_object_get_data (G_OBJECT (window),
				 "avg_unique_mass_entry");
      g_assert (entry != NULL);
    }
  
  /* Get the text value and next convert it to a gdouble value.
   */
  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  
  result = libpolyxmass_globals_strtod (help, &mass);
  
  g_free (help);

  /* At this point we have successfully converted the text value  
     into a numerical value, so we can set this to the 
     smopt param if non-NULL.
  */
  if (smopt != NULL)
    smopt->mass = mass;
  
  return result;
}


gboolean
polyxedit_searchmass_opt_get_tolerance_value (GtkWidget *window, 
					      PxmMassType type,
					      PxmSearchmassOpt *smopt)
{
  gdouble tolerance;

  gboolean result = FALSE;
  
  gchar *help = NULL;

  GtkWidget *entry = NULL;
  
  
  /* Depending on the value of the 'type' parameter, we have to get
     the proper tolerance text value from the proper text entry. So,
     first get text, and next put the successfully converted value
     into the smopt structure. Note that if smopt is NULL, this
     function is called only to check that the entry contains text
     that can successfully be converted into gdouble.
  */

  g_assert (window != NULL);


  if (type == PXM_MASS_MONO)
    {
      entry = g_object_get_data (G_OBJECT (window),
				 "mono_tolerance_entry");
      g_assert (entry != NULL);
    }
  else /* (type == PXM_MASS_AVG) */
    {
      entry = g_object_get_data (G_OBJECT (window),
				 "avg_tolerance_entry");
      g_assert (entry != NULL);
    }
  
  /* Get the text value and next convert it to a gdouble value.
   */
  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  
  /* Note that if help is empty, then we set it to 0 (no tolerance, which
     means that the mass search is EXACT.
  */
  if (strlen (help) <= 0)
    {
      tolerance = 0;
      result = TRUE;
    }
  else
    result = libpolyxmass_globals_strtod (help, &tolerance);
  
  g_free (help);
    
  /* At this point we have successfully converted the text value into
     a numerical value, so we can set this to the smopt param if
     non-NULL.
  */
  if (smopt != NULL)
    smopt->tolerance = tolerance;
  
  return result;
}


gboolean
polyxedit_searchmass_opt_get_tolerance_type (GtkWidget *window, 
					     PxmMassType type,
					     PxmSearchmassOpt *smopt)
{
  GtkWidget *combo = NULL;

  gint index = -1;
  
  /* Depending on the value of the 'type' parameter, we have to get
     the proper tolerance type from the proper optionmenu.
  */
  
  g_assert (window != NULL);

  if (type == PXM_MASS_MONO)
    {
      combo = g_object_get_data (G_OBJECT (window),
				 "mono_tolerance_combo");
      g_assert (combo != NULL);
    }
  else /* (type == PXM_MASS_AVG) */
    {
      combo = g_object_get_data (G_OBJECT (window),
				 "avg_tolerance_combo");
      g_assert (combo != NULL);
    }

  /* Get the index of the currently active item in the combo.  We will
     connect that index to the relative tolerance_unit_xxx that was
     appended to the GtkComboBox when it was setup (see above).
  */
  index = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
  
  switch (index)
    {
    case 0:
      if (smopt != NULL)
	smopt->tolerance_type = PXM_MASS_TOLERANCE_AMU;
      /*
      debug_printf (("was PXM_MASS_TOLERANCE_AMU\n"));
      */
      return TRUE;
    case 1:
      if (smopt != NULL)
	smopt->tolerance_type = PXM_MASS_TOLERANCE_PCT;
      /*
	debug_printf (("was PXM_MASS_TOLERANCE_PCT\n"));
      */
      return TRUE;
    case 2:
      if (smopt != NULL)
	smopt->tolerance_type = PXM_MASS_TOLERANCE_PPM;
      /*
	debug_printf (("was PXM_MASS_TOLERANCE_PPM\n"));
      */
      return TRUE;
    }
  
  /* If we did not return TRUE by now, that means that the index was
     incorrect... something erroneous has happened.
  */
  return FALSE;
}


gboolean
polyxedit_searchmass_opt_get_selection_type (GtkWidget *window,
					     PxmCalcOpt *calcopt)
{
  gint start = -1;
  gint end = -1;

  GtkWidget *radio = NULL;
  
  PxmEditCtxt *editctxt = NULL;
  
  
  /* Get the indices of the selection if there is one. This way we can
     put the selection-representing string (like [1..100]) to the
     relevant entry. If a selection is there, the "Selection Only"
     radiobutton is checked, else the "Whole Sequence" radiobutton is
     checked.
  */
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);
  

  radio = g_object_get_data (G_OBJECT (window), 
			     "polseq_data_whole_seq_radiobutton");
  g_assert (radio != NULL);
  
  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio)))
    {
      calcopt->start_idx = 0;
      calcopt->end_idx = editctxt->polymer->monomerGPA->len - 1;
    }
  else
    {
      if (FALSE != polyxedit_seqed_wnd_get_selection_indices (editctxt,
							      &start, &end))
	{
	  /* Remember that due to the handling of the polymer sequence
	     displayed in the sequence editor, the end value that is 
	     returned is indeed corresponding to the real index of the last
	     selected monomer + 1, so care should be taken not to outbound
	     the polymer sequence array.
	  */
	  calcopt->start_idx = start;
	  calcopt->end_idx = end - 1;
	}
      else
	{
	  calcopt->start_idx = 0;
	  calcopt->end_idx = editctxt->polymer->monomerGPA->len - 1;
	}
    }
  
  return TRUE;
}






/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxedit_searchmass_opt_wnd_really_close (GtkWidget *window)
{
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);


  gtk_widget_destroy (window);
}

  

gboolean
polyxedit_searchmass_opt_wnd_delete_event (GtkWidget *window, 
					   GdkEventAny *event, 
					   gpointer data)
{
  gchar *help = NULL;

  PxmEditCtxt *editctxt = NULL;
  
  g_assert (window != NULL);

  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);
 
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);

  /* This window pointer was set as a full datum to the parent window
     (a cleavage or fragmentatioon results window), which means that
     we have to remove that pointer, without triggering the callback
     function call.
  */
  help = g_strdup_printf ("searchmass_options_wnd-%p", window);
  
  window = g_object_steal_data (G_OBJECT (editctxt->seqeditorctxt->
					  sequence_editor_wnd), help);
  g_free (help);
  
  /* Let Gtk+ do the rest of the work.
   */
  return FALSE;
}
