/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxdef-globals.h"

#include "polyxdef-ui-modifs.h"
#include "polyxdef-ui-polchemdef.h"


gboolean 
polyxdef_ui_polchemdef_wnd_setup_modifs_notebook_page (PxmDefCtxt *defctxt,
						       GladeXML *xml)
{
  GtkWidget *widget = NULL;


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->polchemdef != NULL);
  

  /* The modifs' notebook page must be correctly set up:
   */
  widget = glade_xml_get_widget (xml, "modifs_available_vbox");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "modifs_available_vbox", widget);

  if (FALSE == 
      polyxdef_ui_polchemdef_wnd_setup_modifs_treeview (defctxt))
    {
      g_critical (_("%s@%d: failed to set up the modifs' treeview\n"),
	     __FILE__, __LINE__);
      
      return FALSE;
    }

  widget = glade_xml_get_widget (xml, "polyxdef_modif_add_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polyxdef_modif_add_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_modif_add_button),
		    defctxt);

  widget = glade_xml_get_widget (xml, "polyxdef_modif_remove_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polyxdef_modif_remove_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_modif_remove_button),
		    defctxt);

  return TRUE;
}



gboolean
polyxdef_ui_polchemdef_wnd_setup_modifs_treeview (PxmDefCtxt *defctxt)
{
  GtkWidget *vbox = NULL;
  
  GPtrArray *modifGPA = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkWidget *sw = NULL;
  


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  g_assert (defctxt->polchemdef != NULL);

  modifGPA = defctxt->polchemdef->modifGPA;
  g_assert (modifGPA != NULL);
  
  vbox = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			   "modifs_available_vbox");
  g_assert (vbox != NULL);

  
  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = polyxdef_create_modifs_treeview_model (modifGPA);

  /* Set to the model a datum with a pointer to the window!
   */
  g_object_set_data (G_OBJECT (model), "window", defctxt->definition_wnd);

  /* And now set the window a datum with a pointer to the mode, so
     that later the model is accessible (add/remove button handlers).
   */
  g_object_set_data (G_OBJECT (defctxt->definition_wnd), 
		     "modif_treeview_model", model);
  
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);

  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
     is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (defctxt->definition_wnd), 
		     "modif_treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);
  

  /* Modif Name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Name"), renderer,
     "text", COLUMN_MDF_NAME,
     "editable", COLUMN_MDF_EDITABLE,
     NULL);
  
  g_signal_connect 
    (G_OBJECT (renderer), "edited",
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_modifs_treeview_cell_edited), defctxt);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MDF_NAME);


  /* Modif Actform column.
   */
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Actform"), renderer,
     "text", COLUMN_MDF_ACTFORM,
     "editable", COLUMN_MDF_EDITABLE,
     NULL);
  
  g_signal_connect 
    (G_OBJECT (renderer), "edited",
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_modifs_treeview_cell_edited), defctxt);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MDF_ACTFORM);


  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  return TRUE;
}


GtkTreeModel *
polyxdef_create_modifs_treeview_model (GPtrArray *modifGPA)
{
  gint iter = 0;

  GtkListStore *model;
  GtkTreeIter tree_iter;

  PxmModif *modif;
  
    
  /* Create the list store.
   */
  model = gtk_list_store_new (COLUMN_MDF_COL_COUNT, /*Numb. of columns*/
			      G_TYPE_STRING, /* modif name */
			      G_TYPE_STRING /* modif actform */,
			      G_TYPE_BOOLEAN /* editability */);

  /* Add the items if modifGPA is non-null and there are items in it.
   */
  if (modifGPA != NULL)
    {
      for (iter = 0 ; iter < modifGPA->len ; iter++)
	{
	  /* Acquire an iterator.
	   */
	  gtk_list_store_append (model, &tree_iter);
	  
	  modif = g_ptr_array_index (modifGPA, iter);
	  
	  gtk_list_store_set (model, &tree_iter,
			      
			      COLUMN_MDF_NAME, 
			      modif->name,
			      
			      COLUMN_MDF_ACTFORM, 
			      modif->actform,
			      
			      COLUMN_MDF_EDITABLE, TRUE,
			      -1);
	}
    }
  
  return GTK_TREE_MODEL (model);
}


void
polyxdef_ui_polchemdef_wnd_modifs_treeview_cell_edited (GtkCellRendererText *cell,
							const gchar *path_string,
							const gchar *new_text,
							gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeModel *model = NULL;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
  GtkTreeIter tree_iter;
  
  PxmModif *modif = NULL;
  
  GPtrArray *modifGPA = NULL;

  gint *column;
  gint idx;
  
  gboolean result = FALSE;
  
  gchar *old_text = NULL;
  gchar *help = NULL;
  

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  g_assert (defctxt->polchemdef != NULL);

  modifGPA = defctxt->polchemdef->modifGPA;
  g_assert (modifGPA != NULL);

  g_assert (defctxt->polchemdef->atomGPA != NULL);




  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			     "modif_treeview_model");
  g_assert (model != NULL);

  
  column = g_object_get_data (G_OBJECT (cell), "column");

  gtk_tree_model_get_iter (model, &tree_iter, path);

  switch (GPOINTER_TO_INT (column))
    {
    case COLUMN_MDF_NAME:
      {
	/* Before proceeding and accepting the new text, we should 
	 * make sure that no other modif already exists by the same
	 * name.
	 */
	result = pxmchem_modif_get_index_by_name ((gchar *) new_text,
						  modifGPA);
	
	if (-1 != result)
	  {
	    help = g_strdup_printf (_("A modif by the same name"
				      " exists already: '%s'"),
				    new_text);
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return ;
	  }
	/* At this point, all is OK, we can make the replacement.
	 */
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the item that is edited.
	 */
	idx = gtk_tree_path_get_indices (path) [0];

	modif = g_ptr_array_index (modifGPA, idx);
	g_assert (modif != NULL);
	
	pxmchem_modif_set_name (modif, (gchar *) new_text);
	
	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	
	gtk_list_store_set (GTK_LIST_STORE (model), 
			    &tree_iter, column,
			    modif->name,
			    -1);
      }
      break;
      
    case COLUMN_MDF_ACTFORM:
      {
	/* Before proceeding and accepting the new text, we should
	 * check that the actform in it is valid.
	 */
  	result = pxmchem_actform_check ((gchar *) new_text,
					defctxt->polchemdef->atomGPA);
	
	if (FALSE == result)
	  {
	    help = g_strdup_printf (_("The new actform"
				      " is invalid: '%s'"),
				    new_text);
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return ;
	  }

	/* At this point, all is OK, we can make the replacement.
	 */
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the item that is edited.
	 */
	idx = gtk_tree_path_get_indices (path) [0];

	modif = g_ptr_array_index (modifGPA, idx);
	g_assert (modif != NULL);
	
	pxmchem_modif_set_actform (modif, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	
	gtk_list_store_set (GTK_LIST_STORE (model), 
			    &tree_iter, column,
			    modif->actform,
			    -1);
      }
      break;
    }

  gtk_tree_path_free (path);
}


void
polyxdef_ui_polchemdef_wnd_modif_add_button (GtkWidget *widget, 
					     gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;

  GtkTreeIter tree_iter;
  GtkTreeIter tree_iter_sel;

  GtkTreePath *path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;

  gint idx_mdf = -1;
  
  PxmModif *modif = NULL;
  PxmModif *modif_test = NULL;
  
  GPtrArray *modifGPA = NULL;


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  g_assert (defctxt->polchemdef != NULL);

  modifGPA = defctxt->polchemdef->modifGPA;
  g_assert (modifGPA != NULL);
  

  /* Get the modif treeview model from the window pointer passed
   * a data parameter.
   */
  model = (GtkTreeModel *) 
    g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
		       "modif_treeview_model");
  g_assert (model != NULL);

  treeview = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
				"modif_treeview");
  g_assert (treeview != NULL);

  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);

  if (result == FALSE)
    {
      /* If there is no current selection, that means that the user
       * wants to add a modif at then end of the currently displayed 
       * items. Just an append operation.
       */
      gtk_list_store_append (GTK_LIST_STORE (model), &tree_iter);
    }
  else
    {
      /* There is a selection, so we understand that the user
       * wants to actually insert a new modif below the one
       * that is currently selected. Get index of currently
       * selected item.
       */
      path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				      &tree_iter_sel);
      g_assert (path != NULL);

      gtk_list_store_insert_after (GTK_LIST_STORE (model),
				   &tree_iter,
				   &tree_iter_sel);
      gtk_tree_path_free (path);
    }

  /* After one or the other call to append or store, tree_iter
   * contains the location where the placement is going to happen.
   * Get a path for this iter and and index from  it in turn.
   */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);
      
  idx_mdf = gtk_tree_path_get_indices (path) [0];
  //printf ("idx_mdf is %d\n", idx_mdf);
  
  gtk_tree_path_free (path);

  /* Allocate a new modif so that we can later fill its
   * member data.
   */
  modif = pxmchem_modif_new ();
  
  modif->name = g_strdup (_("enter name here"));
  modif->actform = g_strdup (_("enter actform here"));

  if (idx_mdf + 1 >= modifGPA->len)
    {
      g_ptr_array_add (modifGPA, modif);

      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }
  else if (idx_mdf + 1 < modifGPA->len)
    {
      modif_test = g_ptr_array_insert_val (modifGPA, idx_mdf, modif);
      g_assert (modif == modif_test);

      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }
  else
    {
      g_error (_("%s@%d: failed to insert/add modif item to array\n"),
	     __FILE__, __LINE__);
    }
  
  gtk_list_store_set (GTK_LIST_STORE (model), &tree_iter,
		      
		      COLUMN_MDF_NAME, 
		      modif->name,
		      
		      COLUMN_MDF_ACTFORM, 
		      modif->actform,
		      
		      COLUMN_MDF_EDITABLE, TRUE,
		      -1);

  /* Manage to have the new item added visible, so scroll to where
   * it is and select it.
   */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);

  gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				path,
				NULL,
				TRUE,
				0.5,
				0.5);

  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  gtk_tree_selection_select_iter (tree_selection,
				  &tree_iter);
  gtk_tree_path_free (path);
  
  polyxmass_timeoutmsg_message_set ((GtkWindow *) defctxt->definition_wnd, 
				      _("Added new modif; set its data!"), 
				      POLYXMASS_LONG_MSG_TIMEOUT); 
  return;
}



void
polyxdef_ui_polchemdef_wnd_modif_remove_button (GtkWidget *widget, 
						    gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeModel *model = NULL;
  GtkTreeIter tree_iter ;
  GtkTreeView *treeview = NULL;
  GtkTreeSelection *selection = NULL;
  GtkTreePath *path = NULL;
  
  PxmModif *modif = NULL;
  
  GPtrArray *modifGPA = NULL;

  gint idx = -1;

  gchar *help = NULL;
  

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  g_assert (defctxt->polchemdef != NULL);

  modifGPA = defctxt->polchemdef->modifGPA;
  g_assert (modifGPA != NULL);



  /* Get the modif treeview from the window pointer passed
   * a data parameter.
   */
  treeview = (GtkTreeView *) 
    g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
		       "modif_treeview");

  model = gtk_tree_view_get_model (treeview);
  selection = gtk_tree_view_get_selection (treeview);

  if (gtk_tree_selection_get_selected (selection, NULL, &tree_iter))
    {
      path = gtk_tree_model_get_path (model, &tree_iter);

      idx = gtk_tree_path_get_indices (path) [0];

      gtk_list_store_remove (GTK_LIST_STORE (model), &tree_iter);

      modif = g_ptr_array_remove_index (modifGPA, idx);
      g_assert (modif != NULL);
      
      help = g_strdup_printf (_("Removed modif: '%s'"), modif->name);
      
      pxmchem_modif_free (modif);
      
      gtk_tree_path_free (path);

      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 help, 
	 POLYXMASS_NORM_MSG_TIMEOUT); 
      
      g_free (help);
      
      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }

  return;
}


